/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include <stdio.h>
#include <stdlib.h>

#include "wiregl/include/wiregl_util.h"
#include "wiregl/include/wiregl_client.h"
#include "wiregl/include/wiregl_common.h"
#include "wiregl/include/wiregl_instrument.h"

#ifndef WINDOWS
#define WIREGL_DEFAULT_PATH "."
#else
#define WIREGL_DEFAULT_PATH  "." WIREGL_PATH_SEPARATOR "d:\\work\\wiregl\\opengl_stub" WIREGL_PATH_SEPARATOR "c:\\work\\wiregl\\opengl_stub"
#endif
#define WIREGL_DEFAULT_CONF_FILENAME "wiregl.conf"

#ifdef SIMD
	__m128 pad16byte;
#endif
WireGLGlobals __wiregl_globals;

static WireGLPipeServer *cur_pipeserver = NULL;
static WireGLDisplay *cur_display = NULL;

extern GLapi wiregl_compresspack_api[];

static void 
InitiateServerConnections( void )
{
	WireGLConnectionRequest *request;
	int i,j, k;
	int use_ring = __wiregl_globals.use_ring;

	wireGLNetInit( wireGLClientReceive, NULL );

	request = (WireGLConnectionRequest *) 
		wireGLAlloc( sizeof(request[0]) * __wiregl_globals.num_servers );
	memset( request, 0, sizeof(request[0]) * __wiregl_globals.num_servers );
	
	for ( i = 0 ; i < __wiregl_globals.num_servers; i++ )
	{
		WireGLPipeServer *p = __wiregl_globals.servers + i;
		WireGLConnectionRequest *q = request+i;

		q->max_send      = __wiregl_globals.pipe_buffer_size;
		q->window_x      = p->window_x;
		q->window_y      = p->window_y;
		q->feather       = p->feather;
		q->depth_bits    = __wiregl_globals.depth_bits;
		q->stencil_bits  = __wiregl_globals.stencil_bits;

		q->vis_send_swaps = ( i == 0 );

		q->num_extents   = p->num_extents;
		for ( j = 0; j < p->num_extents; j++ ) 
		{
			q->extent[j].x1 = p->mural_x[j];
			q->extent[j].y1 = p->mural_y[j];
			q->extent[j].x2 = p->mural_x[j] + p->mural_w[j];
			q->extent[j].y2 = p->mural_y[j] + p->mural_h[j];
		}

		q->mural_w = __wiregl_globals.mural.width;
		q->mural_h = __wiregl_globals.mural.height;

		q->apply_viewtransform = __wiregl_globals.apply_viewtransform;
		
		q->peer[0] = '\0';

		if ( use_ring && i+1 < __wiregl_globals.num_servers ) 
		{
			if ( strlen( (p+1)->server) > WIREGL_REQUEST_PEER_NAME_LENGTH )
			{
				wireGLAbort( );
			}
			strcpy( q->peer, (p+1)->server );
		}

		q->num_displays = __wiregl_globals.num_displays;
		for ( j = 0; j < __wiregl_globals.num_displays; j++ ) 
		{
			strcpy (q->display[j].name, __wiregl_globals.display[j].name);
			q->display[j].x = __wiregl_globals.display[j].x;
			q->display[j].y = __wiregl_globals.display[j].y;
			q->display[j].w = __wiregl_globals.display[j].w;
			q->display[j].h = __wiregl_globals.display[j].h;
			q->display[j].rotate_color = 
				__wiregl_globals.display[j].rotate_color;
			for ( k = 0; k < 12; k++ )
			{
				q->display[j].colormatrix[k] = 
					__wiregl_globals.display[j].colormatrix[k];
			}
		}

		q->L2_single_buffer = !__wiregl_globals.sync_L2;

		q->num_pipes = __wiregl_globals.num_servers;
		q->pipe_num  = i;

		q->rotate_color = p->rotate_color;
		for ( j = 0; j < 12; j++ )
		{
			q->colormatrix[j] = p->colormatrix[j];
		}
		q->optimize_bucket = __wiregl_globals.optimize_bucket;
	}

	for ( i = 0; i < __wiregl_globals.num_servers; i++ )
	{
		WireGLPipeServer *p = __wiregl_globals.servers + i;
		WireGLConnectionRequest *q = request + i;
		
		p->conn = wireGLConnectToServer( p->server, WIREGL_TCPIP_DEFAULT_PORT,
										 q );

		wireGLSendBufferInit( &p->buffer, wireGLNetAlloc( p->conn ),
							  wireGLNetMaxSend( ) );
		
		if ( use_ring ) break;
	}
	
	wireGLFree(request);

	/* If ring is enabled, reconfigure the globals to send the stream
	 * to only one server. */
	if ( use_ring )
	{
		__wiregl_globals.num_servers = 1;
	}
}

typedef struct WireGLParameterStruct {
	char *str;
	int value;
} WireGLParameterStruct;

static WireGLParameterStruct wiregl_boolean[] = {
	{ "0", 0 },
	{ "1", 1 },
	{ "disable", 0 },
	{ "enable", 1 },
	{ "n", 0 },
	{ "y", 1 },
	{ "no", 0 },
	{ "yes", 1 },
	{ "off", 0 },
	{ "on", 1 },
	{ NULL, 0 }
};

static int
ParseParameter( char *buf , WireGLParameterStruct *parsers )
{
	WireGLParameterStruct *ps;
	for (ps = parsers ; ps->str != NULL ; ps++)
	{
		if ( !strcmp( buf, ps->str ) )
		{
			return ps->value;
		}
	}

	wireGLSimpleError( "Unrecognized parameter: \"%s\"", buf );
	return -1;
}

static void
InitializePipeserver( WireGLPipeServer *pipe, int index )
{
	int i;

	// don't initialize the SendBuffer here, because the
	// pipe_buffer_size isn't (necessarily) finalized

	pipe->index = index;
	for ( i = 0; i < WIREGL_MAX_EXTENTS; i++)
	{
		pipe->mural_x[i] = 0;
		pipe->mural_y[i] = 0;
		pipe->mural_w[i] = 1024;
		pipe->mural_h[i] = 768;
	}
	pipe->num_extents = 0;
	pipe->window_x = 0;
	pipe->window_y = 0;
	pipe->feather.top_dice = 0;
	pipe->feather.top_size = 0;
	pipe->feather.top[0] = 0.0f;
	pipe->feather.top[1] = 0.0f;
	pipe->feather.top[2] = 0.0f;
	pipe->feather.top[3] = 0.0f;
	pipe->feather.bottom_dice = 0;
	pipe->feather.bottom_size = 0;
	pipe->feather.bottom[0] = 0.0f;
	pipe->feather.bottom[1] = 0.0f;
	pipe->feather.bottom[2] = 0.0f;
	pipe->feather.bottom[3] = 0.0f;
	pipe->feather.left_dice = 0;
	pipe->feather.left_size = 0;
	pipe->feather.left[0] = 0.0f;
	pipe->feather.left[1] = 0.0f;
	pipe->feather.left[2] = 0.0f;
	pipe->feather.left[3] = 0.0f;
	pipe->feather.right_dice = 0;
	pipe->feather.right_size = 0;
	pipe->feather.right[0] = 0.0f;
	pipe->feather.right[1] = 0.0f;
	pipe->feather.right[2] = 0.0f;
	pipe->feather.right[3] = 0.0f;
	pipe->rotate_color = 0;
	memset( pipe->feather.overlay, 0, sizeof(pipe->feather.overlay) );
}

static void 
ParseDisplay( char *buf )
{
	if ( __wiregl_globals.num_displays >= WIREGL_MAX_DISPLAYS )
	{
		wireGLSimpleError( "Maxium outputs reached (%d): %s", 
						   WIREGL_MAX_DISPLAYS, buf );
	}
	
	cur_display = &__wiregl_globals.display[__wiregl_globals.num_displays++];

	cur_display->x = 0;
	cur_display->y = 0;
	cur_display->w = 0;
	cur_display->h = 0;

	if ( strlen( buf ) > WIREGL_MAX_DISPLAY_NAME_LEN )
	{
		wireGLSimpleError( "display name too long: display %s\n", buf );
	}

	strcpy( cur_display->name, buf );

	/* if we are doing a vis server or L2 then we should be applying
       the view transform */
	__wiregl_globals.apply_viewtransform = 1;
}

static void 
ParseDisplayExtent( char *buf )
{
	int *x, *y, *w, *h;

	if ( !cur_display )
	{
		wireGLSimpleError( "No Display selected: display_extent %s", buf );
	}

	x = &cur_display->x;
	y = &cur_display->y;
	w = &cur_display->w;
	h = &cur_display->h;

	if ( sscanf( buf, "%d %d %d %d", x, y, w, h ) != 4 )
	{
		wireGLSimpleError( "bad display dims : %s\n", buf );
	}
}

static void
ParsePipe( char *buf )
{
	unsigned int n_bytes = ( sizeof(__wiregl_globals.servers[0]) * 
							 ( __wiregl_globals.num_servers + 1 ) );
	
	wireGLRealloc( (void **) &__wiregl_globals.servers, n_bytes );
	cur_pipeserver = &__wiregl_globals.servers[__wiregl_globals.num_servers];
	InitializePipeserver( cur_pipeserver, __wiregl_globals.num_servers );
	__wiregl_globals.num_servers++;

	cur_pipeserver->server = strdup( buf );
}

static void 
ParsePipeWindow( char *buf )
{
	if ( sscanf( buf, "%d %d", &(cur_pipeserver->window_x), 
				 &(cur_pipeserver->window_y) ) != 2 )
	{
		wireGLSimpleError( "bad pipe_window : %s\n", buf );
	}
}

static void 
ParsePipeExtent( char *buf )
{
	int x, y, w, h, i;
	if ( sscanf( buf, "%d %d %d %d", &x, &y, &w, &h ) != 4 )
	{
		wireGLSimpleError( "bad pipe_extent : %s\n", buf );
	}

	i = cur_pipeserver->num_extents++;
	if ( i >= WIREGL_MAX_EXTENTS )
	{
		wireGLSimpleError( "max extents per server : %d\n", 
						   WIREGL_MAX_EXTENTS );
	}

	cur_pipeserver->mural_x[i] = x;
	cur_pipeserver->mural_y[i] = y;
	cur_pipeserver->mural_w[i] = w;
	cur_pipeserver->mural_h[i] = h;

	if ( x+w > __wiregl_globals.mural.width )
	{
		__wiregl_globals.mural.width = x+w;
	}
	if ( y+h > __wiregl_globals.mural.height )
	{
		__wiregl_globals.mural.height = y+h;
	}

	if ( i > 0 )
	{
		__wiregl_globals.apply_viewtransform = 1;
	}
}

static void 
ParseSwapBarrier( char *buf )
{
#if 0
	if ( __wiregl_globals.num_servers > 1 )
	{
		WireGLConnection *bconn;
		__wiregl_globals.swap_barrier = 1;
		bconn = wireGLBarrierConnect( buf );
		wireGLBarrierCreate( bconn, WIREGL_PIPESERVER_BARRIER,
							 __wiregl_globals.num_servers );
		__wiregl_globals.barrier_server = strdup( buf );
	}
#endif
	WIREGL_UNUSED(buf);
}

static void 
ParsePrintStats( char *buf )
{
	char name[1024];
	int i;

	if (!strcmp( buf, "bytes_per_server" ))
	{
		__wiregl_globals.print.bytes_per_server = 1;
		wireGLInstrumentRegisterPerServerCounter( WIREGL_BYTES_PER_SERVER, __wiregl_globals.num_servers, "Bytes per server" );
	}
	else if (!strcmp( buf, "bytes_per_frame_per_server" ) )
	{
		__wiregl_globals.print.bytes_per_frame_per_server = 1;
		for (i = 0 ; i < __wiregl_globals.num_servers ; i++)
		{
			sprintf( name, "Bytes per frame[%d]", i+1 );
			wireGLInstrumentRegisterPerFrameCounter( WIREGL_BYTES_PER_SERVER_PER_FRAME + i, name );
		}
	}
	else if (!strcmp( buf, "min_bytes_sent" ) )
	{
		__wiregl_globals.print.min_bytes_sent = 1;
		wireGLInstrumentRegisterCounter( WIREGL_MIN_BYTES_SENT, "Minimum bytes to be sent" );
	}
	else if (!strcmp( buf, "frame_rate" ) )
	{
		__wiregl_globals.print.frame_rate = 1;
		wireGLInstrumentRegisterTimer( WIREGL_FRAME_TIMER, "Frame time" );
		wireGLInstrumentRegisterPerFrameCounter( WIREGL_FRAME_TIMER, "Frame rate" );
		wireGLInstrumentStartTimer( WIREGL_FRAME_TIMER );
	}
	else if (!strcmp( buf, "flush_time" ) )
	{
		__wiregl_globals.print.flush_time = 1;
		wireGLInstrumentRegisterTimer( WIREGL_FLUSH_TIMER, "Flush time" );
		wireGLInstrumentRegisterPerFrameCounter( WIREGL_FLUSH_TIMER, "Flush time" );
	}
	else if (!strcmp( buf, "send_time" ) )
	{
		__wiregl_globals.print.send_time = 1;
		wireGLInstrumentRegisterTimer( WIREGL_SEND_TIMER, "Send time" );
		wireGLInstrumentRegisterPerFrameCounter( WIREGL_SEND_TIMER, "Send time" );
	}
	else if (!strcmp( buf, "diff_context_time" ) )
	{
		__wiregl_globals.print.diff_context_time = 1;
		wireGLInstrumentRegisterTimer( WIREGL_DIFF_CONTEXT_TIMER, "Diff context time" );
		wireGLInstrumentRegisterPerFrameCounter( WIREGL_DIFF_CONTEXT_TIMER, "Diff context time" );
	}
	else if (!strcmp( buf, "bytes_sent" ) )
	{
		__wiregl_globals.print.total_bytes_sent = 1;
		wireGLInstrumentRegisterCounter( WIREGL_TOTAL_BYTES_SENT, "Total bytes sent" );
	}
	else if (!strcmp( buf, "total_time" ) )
	{
		__wiregl_globals.print.total_time = 1;
		wireGLInstrumentRegisterTimer( WIREGL_TOTAL_TIME, "Total time" );
		wireGLInstrumentStartTimer( WIREGL_TOTAL_TIME );
	}
	else 
    {
		wireGLSimpleError( "bad print_stats : %s\n", buf );
	}
}

static void 
ParseStatsFile( char *buf ) 
{
	wireGLInstrumentToFile( buf );
}

static void 
ParsePipeBufferSize( char *buf )
{
	int buffer_size;
	if ( sscanf( buf, "%d", &buffer_size ) != 1 )
		wireGLSimpleError( "bad pipe_buffer_size : %s\n", buf );
	__wiregl_globals.pipe_buffer_size = buffer_size;
}

static void 
ParseFeather( char *buf, int *size, float cp[])
{
	sscanf( buf, "%d %f %f %f %f", size, cp, cp+1, cp+2, cp+3);
}
		
static void
ParseFeatherTop( char *buf ) 
{
	ParseFeather( buf, &cur_pipeserver->feather.top_size, 
				  cur_pipeserver->feather.top );
}

static void 
ParseFeatherBottom( char *buf )
{
	ParseFeather( buf, &cur_pipeserver->feather.bottom_size,
				  cur_pipeserver->feather.bottom );
}

static void 
ParseFeatherLeft( char *buf ) 
{
	ParseFeather( buf, &cur_pipeserver->feather.left_size,
				  cur_pipeserver->feather.left );
}

static void 
ParseFeatherRight( char *buf )
{
	ParseFeather( buf, &cur_pipeserver->feather.right_size,
				  cur_pipeserver->feather.right );
}

static void 
ParseFeatherOverlay( char *buf )
{
	strncpy( cur_pipeserver->feather.overlay, buf, 256 );
	cur_pipeserver->feather.overlay[255] = 0;
}

static void 
ParseDepthBits( char *buf ) 
{
	if( sscanf( buf, "%d", &__wiregl_globals.depth_bits) != 1 )
	{
		wireGLSimpleError( "bad depth_bits : %s\n", buf );
	}
}

static void
ParseStencilBits( char *buf )
{
	if( sscanf( buf, "%d", &__wiregl_globals.stencil_bits) != 1 )
	{
		wireGLSimpleError( "bad stencil_bits : %s\n", buf );
	}
}

static void ParseNormalCompression( char *buf )
{
	int i, j;
	
	__wiregl_globals.normal_compression = ParseParameter(buf, wiregl_boolean);
	if ( __wiregl_globals.normal_compression ) 
	{
		for (i=0; wiregl_compresspack_api[i].func; i++)
		{
			for (j=0; wiregl_pack_api[j].func; j++)
			{
				if (!strcmp(wiregl_compresspack_api[i].name, wiregl_pack_api[j].name))
					wiregl_pack_api[j].func = wiregl_compresspack_api[i].func;
			}
		}
	}
}

static void 
ParseVertexCompression( char *buf )
{
#ifdef WINDOWS
	__wiregl_globals.vertex_compression = ParseParameter(buf, wiregl_boolean);
#else
	WIREGL_UNUSED(buf);
	wireGLSimpleError("vertex_compression: only supported in Windows\n");
#endif
}

static void 
ParseUnsafeCompression( char *buf )
{
#ifdef WINDOWS
	if(!__wiregl_globals.vertex_compression)
		wireGLSimpleError("unsafe_compression: vertex_compression not yet enabled\n");
	else
		__wiregl_globals.unsafe_compression = ParseParameter(buf, wiregl_boolean);
#else
	WIREGL_UNUSED(buf);
	wireGLSimpleError("unsafe_compression: only supported in Windows\n");
#endif
}

static void 
ParseCompressionDepthBits( char *buf )
{
#ifdef WINDOWS
	if(!__wiregl_globals.vertex_compression)
		wireGLSimpleError("compression_depthbits: vertex_compression not yet enabled\n");
	else
	__wiregl_globals.compression_depthbits = atoi(buf);
#else
	WIREGL_UNUSED(buf);
	wireGLSimpleError("compression_depthbits: only supported in Windows\n");
#endif
}

static void 
ParseGuiError( char *buf )
{
	__wiregl_globals.gui_error = ParseParameter( buf, wiregl_boolean );
}

static void 
ParseDrawBBox( char *buf )
{
	__wiregl_globals.draw_bbox = ParseParameter( buf, wiregl_boolean );
}

static void 
ParseWarnLevel( char *buf )
{
	__wiregl_globals.warn_level_str = strdup( buf );
}

static void 
ParseBeginEndMax( char *buf )
{
	int size;
	sscanf( buf, "%d", &size );
	if ( size < 1 )
	{
		wireGLSimpleError( "beginend_max: bad value (%s)\n", buf );
	}
	__wiregl_globals.beginend_max = size;
}

static void 
ParseBucketSize( char *buf )
{
	int size;
	if ( sscanf( buf, "%d", &size ) != 1 || size < 3 )
	{
		wireGLSimpleError( "bucket_size: bad value (%s)\n", buf );
	}
	__wiregl_globals.bucket_size = size;
}

static void 
ParseBroadcast( char *buf )
{
	__wiregl_globals.broadcast = ParseParameter( buf, wiregl_boolean );
}

static void 
ParsePackOnly( char *buf )
{
	__wiregl_globals.pack_only = ParseParameter( buf, wiregl_boolean );
}

static void 
ParseResolveArrays( char *buf )
{
	__wiregl_globals.resolve_arrays = ParseParameter( buf, wiregl_boolean );
}

static void 
ParseStateErrorGui( char *buf )
{
	__wiregl_globals.state_error_gui = ParseParameter( buf, wiregl_boolean );
}

static void 
ParseSyncOnFinish( char *buf )
{
	__wiregl_globals.sync_on_finish = ParseParameter( buf, wiregl_boolean );
}

static void 
ParseSyncOnSwap( char *buf )
{
	__wiregl_globals.sync_on_swap = ParseParameter( buf, wiregl_boolean );
}

static void 
ParseSplitBeginEnd( char *buf )
{
	__wiregl_globals.split_begin_end = ParseParameter( buf, wiregl_boolean );
}

static void 
ParseUseRing( char *buf )
{
	__wiregl_globals.use_ring = ParseParameter( buf, wiregl_boolean );
	if (!__wiregl_globals.apply_viewtransform)
		__wiregl_globals.apply_viewtransform = __wiregl_globals.use_ring;
}

static void 
ParseUseSimd( char *buf )
{
	__wiregl_globals.use_simd = ParseParameter( buf, wiregl_boolean );
}

static void 
ParseBroadcastTextures( char *buf )
{
	__wiregl_globals.broadcast_textures = 
		ParseParameter( buf, wiregl_boolean );
}

static void 
ParseColorMatrixAux( char *buf, int *rotate_color, float colormatrix[12] )
{
	float elem[12];
	int i, ret;
	ret = sscanf( buf, "%f %f %f %f %f %f %f %f %f %f %f %f", 
				  &elem[0], &elem[1], &elem[2], &elem[3],
				  &elem[4], &elem[5], &elem[6], &elem[7],
				  &elem[8], &elem[9], &elem[10], &elem[11] );
	if ( ret != 12 )
	{
		wireGLSimpleError( "too few elements to colormatrix (%s)\n", buf );
	}
	*rotate_color = 1;
	for ( i = 0; i < 12; i++ )
	{
		colormatrix[i] = elem[i];
	}
}

static void
ParseColorMatrix( char *buf )
{
	ParseColorMatrixAux( buf, &cur_pipeserver->rotate_color, 
						 cur_pipeserver->colormatrix );
}

static void
ParseDisplayColorMatrix( char *buf )
{
	ParseColorMatrixAux( buf, &cur_display->rotate_color, 
						 cur_display->colormatrix );
}

static void 
ParseSyncL2( char *buf )
{
	__wiregl_globals.sync_L2 = ParseParameter( buf, wiregl_boolean );
	wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "__wiregl_globals.sync_L2=%d",
				   __wiregl_globals.sync_L2 );
}

static void 
ParseOptimizeBucket( char *buf )
{
	__wiregl_globals.optimize_bucket = ParseParameter( buf, wiregl_boolean );
	wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, 
				   "__wiregl_globals.optimize_bucket: %d",
				   __wiregl_globals.optimize_bucket );
}

static WireGLParseStruct wiregl_parse[] = {
	{ "pipe", ParsePipe },
	{ "pipe_extent", ParsePipeExtent },
	{ "pipe_window", ParsePipeWindow },
	{ "swap_barrier", ParseSwapBarrier },
	{ "print_stats", ParsePrintStats },
	{ "stats_file", ParseStatsFile },
	{ "pipe_buffer_size", ParsePipeBufferSize },
	{ "normal_compression", ParseNormalCompression },
	{ "vertex_compression", ParseVertexCompression },
	{ "allow_unsafe_compression", ParseUnsafeCompression },
	{ "compression_depthbits", ParseCompressionDepthBits },
	{ "gui_error", ParseGuiError },
	{ "draw_bbox", ParseDrawBBox },
	{ "warn_level", ParseWarnLevel },
	{ "bucket_size", ParseBucketSize },
	{ "broadcast", ParseBroadcast },
	{ "pack_only", ParsePackOnly },
	{ "resolve_arrays", ParseResolveArrays },
	{ "state_error_gui", ParseStateErrorGui },
    { "sync_on_finish", ParseSyncOnFinish },
    { "sync_on_swap", ParseSyncOnSwap },
	{ "feather_top", ParseFeatherTop },
	{ "feather_bottom", ParseFeatherBottom },
	{ "feather_left", ParseFeatherLeft },
	{ "feather_right", ParseFeatherRight },
	{ "feather_overlay", ParseFeatherOverlay },
	{ "depth_bits", ParseDepthBits },
	{ "stencil_bits", ParseStencilBits },
	{ "split_begin_end", ParseSplitBeginEnd },
	{ "use_ring", ParseUseRing },
	{ "use_simd", ParseUseSimd },
	{ "beginend_max", ParseBeginEndMax },
	{ "display", ParseDisplay},
	{ "display_extent", ParseDisplayExtent},
	{ "display_colormatrix", ParseDisplayColorMatrix },
    { "broadcast_textures", ParseBroadcastTextures },
	{ "colormatrix", ParseColorMatrix },
	{ "sync_L2", ParseSyncL2 },
	{ "optimize_bucket", ParseOptimizeBucket },
	{ NULL, NULL }
};

void 
wireGLInitGlobals( void )
{
	__wiregl_globals.mural.width = 0;
	__wiregl_globals.mural.height = 0;
	__wiregl_globals.print.bytes_per_server = 0;
	__wiregl_globals.print.bytes_per_frame_per_server = 0;
	__wiregl_globals.print.min_bytes_sent = 0;
	__wiregl_globals.print.frame_rate = 0;
	__wiregl_globals.print.flush_time = 0;
	__wiregl_globals.print.send_time = 0;
	__wiregl_globals.print.diff_context_time = 0;
	__wiregl_globals.print.total_bytes_sent = 0;
	__wiregl_globals.print.total_time = 0;
	__wiregl_globals.pipe_buffer_size = 8 * 1024;
	__wiregl_globals.swap_barrier = 0;
	__wiregl_globals.barrier_server = NULL;

#ifndef WINDOWS
	__wiregl_globals.depth_bits = 8;
	__wiregl_globals.stencil_bits = 8;
#else
	__wiregl_globals.depth_bits = 24;
	__wiregl_globals.stencil_bits = 8;
#endif /* !WINDOWS */

	__wiregl_globals.normal_compression = 0;
	__wiregl_globals.vertex_compression = 0;
	__wiregl_globals.unsafe_compression = 0;
	__wiregl_globals.compression_depthbits = -1;
	__wiregl_globals.gui_error       = 0;
	__wiregl_globals.draw_bbox       = 0;
	__wiregl_globals.warn_level_str  = "critical";
	__wiregl_globals.bucket_size     = GL_MAXINT;
	__wiregl_globals.broadcast       = 0;
	__wiregl_globals.pack_only       = 0;
	__wiregl_globals.resolve_arrays  = 1;
	__wiregl_globals.state_error_gui = 0;
	__wiregl_globals.sync_on_finish  = 0;
	__wiregl_globals.sync_on_swap    = 0;
	__wiregl_globals.split_begin_end = 1;
	__wiregl_globals.use_ring        = 0;
	__wiregl_globals.use_simd        = 0;
	__wiregl_globals.sync_L2         = 0;
	__wiregl_globals.beginend_max    = GL_MAXINT;
	__wiregl_globals.num_displays    = 0;
	__wiregl_globals.broadcast_textures = 0;
	__wiregl_globals.optimize_bucket = 0;

	/* Always have the pipeservers set the
	** base matrix */
	__wiregl_globals.apply_viewtransform = 1;

}

static void
DescribeConfig( void )
{
	int i;
	WireGLGlobals *g = &__wiregl_globals;

#define W		wireGLWarning
#define L       WIREGL_WARN_VERBOSE_DEBUG
#define BOOL(x) ((x) ? "yes" : "no" )
#define STR(x)	((x) ? (x) : "<none>" )

	W( L, "mural size=%dx%d", g->mural.width, g->mural.height );
	W( L, "pipe_buffer_size=%d", g->pipe_buffer_size );
	W( L, "swap_barrier=%s  barrier_server=%s",
	   BOOL(g->swap_barrier), STR(g->barrier_server) );
	W( L, "depth_bits=%d  stencil_bits=%d",
	   g->depth_bits, g->stencil_bits );
	W( L, "compression: normal=%s  vertex=%s  unsafe=%s  depthbits=%d",
	   BOOL(g->normal_compression), BOOL(g->vertex_compression),
	   BOOL(g->unsafe_compression), g->compression_depthbits );
	W( L, "warn_level=%s  gui_error=%s  state_error_gui=%s",
	   STR(g->warn_level_str), BOOL(g->gui_error), BOOL(g->state_error_gui) );
	W( L, "split_begin_end=%s  draw_bbox=%s", 
	   BOOL(g->split_begin_end), BOOL(g->draw_bbox) );
	W( L, "broadcast=%s  pack_only=%s  resolve_arrays=%s",
	   BOOL(g->broadcast), BOOL(g->pack_only), BOOL(g->resolve_arrays) );
	W( L, "sync: on_finish=%s on_swap=%s",
	   BOOL(g->sync_on_finish), BOOL(g->sync_on_swap) );
	W( L, "use_ring=%s  use_simd=%s", BOOL(g->use_ring), BOOL(g->use_simd) );
	W( L, "beginend_max=%d  bucket_size=%d", g->beginend_max, g->bucket_size );
	W( L, "num_pipeservers=%d", g->num_servers );
	for ( i = 0; i < g->num_servers; i++ )
	{
		WireGLPipeServer *p = &g->servers[i];
		W( L, "pipeserver[%d] = %d+%d @ mural %d,%d",
		   i + 1, p->window_x, p->window_y, 
		   p->mural_x[0], p->mural_y[0] );
	}

#undef STR
#undef BOOL
#undef L
#undef W
}

void 
wireGLClientParseConfig( void )
{
	FILE *fp = NULL;
	char fname[1024];
	int i;

	char *wiregl_conf, *wiregl_path, *temp;

	int found_it = 0;

	wiregl_conf = getenv( "WIREGL_CONF" );
	if ( !wiregl_conf ) 
	{
		wiregl_path = getenv( "WIREGL_PATH" );
		if ( !wiregl_path )
		{
			wiregl_path = strdup( WIREGL_DEFAULT_PATH );
		}

		temp = wiregl_path;
		while ( temp && *temp )
		{
			char *colon = strchr( temp, WIREGL_PATH_SEPARATOR[0] );
			if ( colon )
			{
				strncpy( fname, temp, colon-temp );
				fname[colon-temp] = '\0';
			}
			else
				strcpy( fname, temp );
			strcat( fname, "/" );
			strcat( fname, WIREGL_DEFAULT_CONF_FILENAME );
			fp = fopen(fname, "r");
			if ( fp )
			{
				fclose(fp);
				found_it = 1;
				break;
			}
			temp = colon ? ( colon + 1 ) : NULL;
		}

		if ( !found_it )
		{
			wireGLSimpleError( "wireGLClientParseConfig: Can't find %s in "
							   "WIREGL_PATH (%s)", 
							   WIREGL_DEFAULT_CONF_FILENAME, wiregl_path );
		}
		wiregl_conf = fname;
	}

	__wiregl_globals.num_servers = 0;

	wireGLParseFile( wiregl_conf, wiregl_parse );

	for ( i = 0; i < __wiregl_globals.num_servers; i++ )
	{
		if ( __wiregl_globals.servers[i].num_extents == 0 )
		{
			wireGLSimpleError( "Missing pipe_extent field for server %d",
							   i + 1 );
		}
	}


	if ( getenv( "WIREGL_WARN_LEVEL" ) )
		__wiregl_globals.warn_level_str = 
			strdup( getenv( "WIREGL_WARN_LEVEL" ) );
	if ( getenv( "WIREGL_GUI_ERROR" ) )
		__wiregl_globals.gui_error = 1;
	if ( getenv( "WIREGL_DRAW_BBOX" ) )
		__wiregl_globals.draw_bbox = 1;
	if ( getenv( "WIREGL_BROADCAST" ) )
		__wiregl_globals.broadcast = 1;
	if ( getenv( "WIREGL_PACK_ONLY" ) )
		__wiregl_globals.pack_only = 1;
	if ( getenv( "WIREGL_RESOLVE_ARRAYS" ) )
		__wiregl_globals.resolve_arrays = 1;
	if ( getenv( "WIREGL_STATE_ERROR_GUI" ) )
		__wiregl_globals.state_error_gui = 1;

	wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "config=\"%s\"", wiregl_conf );
	DescribeConfig( );

	InitiateServerConnections( );

}
