/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include <float.h>

#include "wiregl/include/wiregl_protocol.h"
#include "wiregl/include/wiregl_util.h"
#include "wiregl/include/wiregl_client.h"
#include "wiregl/include/wiregl_instrument.h"

extern char *__wiregl_opcode_names[];

static WireGLPipeServer *state_server = NULL;

/* This is the amount of space we need to save at the buffer */
#define END_FLUFF 4
#define BOUNDINFO_SIZE 24

void 
PackBoundsInfo( WireGLSendBuffer *buf, GLrecti *bounds, 
				void *payload, int length, int num_opcodes )
{
	unsigned char *data = buf->data_current;
	unsigned char *op   = buf->opcode_current;
	int len_aligned     = ( length + 0x3 ) & ~0x3;

	wireGLAssert ( buf->opcode_current <= buf->opcode_start );
	wireGLAssert ( buf->opcode_current > buf->opcode_end );
	wireGLAssert ( buf->data_current >= buf->data_start );
	wireGLAssert ( buf->data_current <= buf->data_end + 
				   BOUNDINFO_SIZE + len_aligned );

	*op = WIREGL_BOUNDSINFO_OPCODE; 
	*((int   *) (data+ 0)) = BOUNDINFO_SIZE + len_aligned;
	*((int   *) (data+ 4)) = bounds->x1;
	*((int   *) (data+ 8)) = bounds->y1;
	*((int   *) (data+12)) = bounds->x2;
	*((int   *) (data+16)) = bounds->y2;
	*((int   *) (data+20)) = num_opcodes;

	/* skip the BOUNDSINFO */
	data += 24;

	/* put in padding opcodes (deliberately bogus) */
	switch ( len_aligned - length )
	{
	  case 3: *data++ = 0xff;
	  case 2: *data++ = 0xff;
	  case 1: *data++ = 0xff;
	  default: break;
	}

	memcpy( data, payload, length );

	buf->opcode_current--;
	buf->data_current += BOUNDINFO_SIZE + len_aligned;
}

void 
wireGLSendBufferInit( WireGLSendBuffer *buf, void *pack, unsigned int size )
{
	unsigned int num_opcodes;

	buf->pack = pack;
	buf->size = size;

	/* each opcode has at least a 1-word payload, so at most the
	 * opcodes are 20% (/ 5) of the space */
	num_opcodes = ( size - sizeof(WireGLMessageOpcodes) ) / 5;

	/* word alignment */
	num_opcodes = ( num_opcodes + 0x3 ) & ~0x3;

	buf->data_start    = 
		(unsigned char *) buf->pack + num_opcodes + sizeof(WireGLMessageOpcodes);
	buf->data_current  = buf->data_start;
	buf->data_end      = (unsigned char *) buf->pack + size;

	buf->opcode_start   = buf->data_start - 1;
	buf->opcode_current = buf->opcode_start;
	buf->opcode_end     = buf->opcode_start - num_opcodes;

}

unsigned int 
wireGLRecommendedPackBufferSize (unsigned int send_size) {
	
	unsigned int ret = 0;

	if ( !__wiregl_globals.apply_viewtransform )
	{
		ret = send_size - sizeof (WireGLMessageOpcodes) - 4;
    }
	else 
	{
		/* This is constructed to ensure that the pack buffer fits
		 * into the data segment of the send buffer.  Note: its rare
		 * that it will fill the data portion of the send buffer since
		 * it's next to impossible to fill the opcode section of the
		 * pack buffer. */
		ret =  
			/* The size of the send buffer data section */
			((send_size - sizeof (WireGLMessageOpcodes)) * 4 / 5)
			/* in addition to the BoundInfo packet */
			- BOUNDINFO_SIZE - 1 
			/* and in case of alignment issues*/
			- 4;
	}

	return ret;
}

void
wireGLPackBufferInit( WireGLPackBuffer *buf, void *pack, unsigned int size )
{
	unsigned int num_opcodes;

	buf->pack = pack;
	buf->size = size;

	/* each opcode has at least a 1-word payload, so at most the
	 * opcodes are 20% (/ 5) of the space */
	num_opcodes = size / 5;

	/* word alignment */
	num_opcodes = ( num_opcodes + 0x3 ) & ~0x3;

	buf->data_start    = 
		(unsigned char *) buf->pack + num_opcodes + sizeof(WireGLMessageOpcodes);
	buf->data_current  = buf->data_start;
	buf->data_end      = (unsigned char *) buf->pack + size;

	buf->opcode_start   = buf->data_start - 1;
	buf->opcode_current = buf->opcode_start;
	buf->opcode_end     = buf->opcode_start - num_opcodes;

	/* hide some space so we can append a glEnd() */
	buf->data_end -= END_FLUFF;

}

static WireGLMessageOpcodes *
ApplySendBufferHeader( WireGLSendBuffer *buf, unsigned int *len )
{
	int num_opcodes;
	WireGLMessageOpcodes *hdr;

	wireGLAssert( buf->opcode_current < buf->opcode_start );
	wireGLAssert( buf->opcode_current >= buf->opcode_end );
	wireGLAssert( buf->data_current > buf->data_start );
	wireGLAssert( buf->data_current <= buf->data_end );

	num_opcodes = buf->opcode_start - buf->opcode_current;
	hdr = (WireGLMessageOpcodes *) 
		( buf->data_start - ( ( num_opcodes + 3 ) & ~0x3 ) - sizeof(*hdr) );

	wireGLAssert( (void *) hdr >= buf->pack );

	hdr->type       = WIREGL_MESSAGE_OPCODES;
	hdr->senderId   = (unsigned int) ~0;  /* to be initialized by caller */
	hdr->numOpcodes = num_opcodes;

	*len = buf->data_current - (unsigned char *) hdr;

	return hdr;
}

void 
wireGLSendPipeServerBuffer( WireGLPipeServer *pipe )
{
	WireGLSendBuffer *buf = &pipe->buffer;
	unsigned int len;
	WireGLMessageOpcodes *hdr;

	if ( buf->opcode_current == buf->opcode_start )
		return;

	hdr = ApplySendBufferHeader( buf, &len );
	hdr->senderId = pipe->conn->sender_id;

#ifdef INSTRUMENT
	if (__wiregl_globals.print.send_time)
	{
		wireGLInstrumentStartTimer( WIREGL_SEND_TIMER );
	}
#endif

#if 0
	wireGLWarning( WIREGL_WARN_DEBUG, "Packet Fullness: %d%%\n",
				   100 - (buf->data_end - buf->data_current)*100/
				         (buf->data_end - buf->data_start) );
	wireGLWarning( WIREGL_WARN_DEBUG, "\tdatafree:   %d\n", buf->data_end - buf->data_current);
	wireGLWarning( WIREGL_WARN_DEBUG, "\topcodefree: %d\n", buf->opcode_current - buf->opcode_end);
#endif

	wireGLNetSend( pipe->conn, &buf->pack, hdr, len );
	wireGLSendBufferInit( buf, wireGLNetAlloc( pipe->conn ), 
						  wireGLNetMaxSend( ) );

#ifdef INSTRUMENT
	if (__wiregl_globals.print.send_time)
	{
		wireGLInstrumentStopTimer( WIREGL_SEND_TIMER );
		wireGLInstrumentIncrPerFrameCounter( WIREGL_SEND_TIMER, wireGLInstrumentGetTime( WIREGL_SEND_TIMER ) );
		wireGLInstrumentResetTimer( WIREGL_SEND_TIMER );
	}
	if (__wiregl_globals.print.total_bytes_sent)
	{
		wireGLInstrumentIncrCounter( WIREGL_TOTAL_BYTES_SENT, len );
	}
	if (__wiregl_globals.print.bytes_per_server)
	{
		wireGLInstrumentIncrPerServerCounter( WIREGL_BYTES_PER_SERVER,
											  pipe->index, len );
	}
	if (__wiregl_globals.print.bytes_per_frame_per_server)
	{
		wireGLInstrumentIncrPerFrameCounter( 
					WIREGL_BYTES_PER_SERVER_PER_FRAME + pipe->index, 
					len );
	}
#endif

	buf->data_current = buf->data_start;
	buf->opcode_current = buf->opcode_start;
}

static void
SendBufferAppend( WireGLSendBuffer *dst, WireGLPackBuffer *src )
{
	int num_data = src->data_current - src->data_start;
	int num_opcode;

	if ( dst->data_current + num_data > dst->data_end )
		wireGLError( "wireGLSendBufferAppend: overflowed the destination!\n" );
	memcpy( dst->data_current, src->data_start, num_data );
	dst->data_current += num_data;

	num_opcode = src->opcode_start - src->opcode_current;
	wireGLAssert( dst->opcode_current - num_opcode >= dst->opcode_end );
	memcpy( dst->opcode_current + 1 - num_opcode, src->opcode_current + 1,
			num_opcode );
	dst->opcode_current -= num_opcode;
}

static void
SendBufferAppendGeom( WireGLSendBuffer *dst, WireGLPackBuffer *src, 
					  GLrecti *bounds )
{
	int length = src->data_current - ( src->opcode_current + 1 );

	if ( dst->data_current + length + BOUNDINFO_SIZE > dst->data_end )
		wireGLError( "wireGLSendBufferAppend: overflowed the destination!\n" );

	PackBoundsInfo( dst, bounds, src->opcode_current + 1, length,
					src->opcode_start - src->opcode_current );
}

void 
wireGLPipeServerBufferAppend( WireGLPipeServer *pipe, WireGLPackBuffer *src )
{
	WireGLSendBuffer *dst = &pipe->buffer;
	int num_data = src->data_current - src->data_start;

	if ( dst->data_current + num_data > dst->data_end )
	{
		/* the pipeserver's buffer doesn't have enough space to tack
		  on the geometry - make space by sending it over the wire */
		wireGLSendPipeServerBuffer( pipe );
	}

	SendBufferAppend( dst, src );
}

void 
wireGLPipeServerBufferAppendGeom( WireGLPipeServer *pipe, 
								  WireGLPackBuffer *src, GLrecti *bounds )
{
	WireGLSendBuffer *dst = &pipe->buffer;
	int num_data = src->data_current - src->opcode_current - 1 +
		BOUNDINFO_SIZE;

	if ( dst->data_current + num_data > dst->data_end )
	{
		/* the pipeserver's buffer doesn't have enough space to tack
		  on the geometry - make space by sending it over the wire */
		wireGLSendPipeServerBuffer( pipe );
	}

	SendBufferAppendGeom( dst, src, bounds );
}

static void SetGlobalsToPipeServerBuffer( WireGLPipeServer *pipe )
{
	wireGLAssert( pipe->buffer.opcode_current <= pipe->buffer.opcode_start );
	wireGLAssert( pipe->buffer.opcode_current >= pipe->buffer.opcode_end );
	wireGLAssert( pipe->buffer.data_current >= pipe->buffer.data_start );
	wireGLAssert( pipe->buffer.data_current <= pipe->buffer.data_end );

	__wiregl_globals.buffer.pack = pipe->buffer.pack;
	__wiregl_globals.buffer.size = pipe->buffer.size;
	__wiregl_globals.buffer.opcode_start = pipe->buffer.opcode_start;
	__wiregl_globals.buffer.opcode_current = pipe->buffer.opcode_current;
	__wiregl_globals.buffer.opcode_end = pipe->buffer.opcode_end;
	__wiregl_globals.buffer.data_start = pipe->buffer.data_start;
	__wiregl_globals.buffer.data_current = pipe->buffer.data_current;
	__wiregl_globals.buffer.data_end = pipe->buffer.data_end;
}

static void RestorePipeServerBufferFromGlobals( WireGLPipeServer *pipe )
{
	wireGLAssert( __wiregl_globals.buffer.opcode_current <= __wiregl_globals.buffer.opcode_start );
	wireGLAssert( __wiregl_globals.buffer.opcode_current >= __wiregl_globals.buffer.opcode_end );
	wireGLAssert( __wiregl_globals.buffer.data_current >= __wiregl_globals.buffer.data_start );
	wireGLAssert( __wiregl_globals.buffer.data_current <= __wiregl_globals.buffer.data_end );

	pipe->buffer.pack = __wiregl_globals.buffer.pack;
	pipe->buffer.size = __wiregl_globals.buffer.size;
	pipe->buffer.opcode_start = __wiregl_globals.buffer.opcode_start;
	pipe->buffer.opcode_current = __wiregl_globals.buffer.opcode_current;
	pipe->buffer.opcode_end = __wiregl_globals.buffer.opcode_end;
	pipe->buffer.data_start = __wiregl_globals.buffer.data_start;
	pipe->buffer.data_current = __wiregl_globals.buffer.data_current;
	pipe->buffer.data_end = __wiregl_globals.buffer.data_end;
}

void wireGLFlushAll( GLcontext *virtual_context )
{
	WireGLGlobals *g = &__wiregl_globals;
	int i;

	wireGLFlushBuffer( virtual_context );

	for ( i = 0; i < g->num_servers; i++ )
	{
		wireGLSendPipeServerBuffer( &g->servers[i] );
	}
}

void wireGLFlushAllWithOpcode( GLcontext *virtual_context, WireGLOpcode op,
							   unsigned int n_data, void *data )
{
	WireGLGlobals *g = &__wiregl_globals;
	int i;
	unsigned char *data_ptr;

	wireGLFlushBuffer( virtual_context );

	/* If someone is forcing us to flush (i.e., SwapBuffers), we need
	 * to get the forced opcode onto the end of each state buffer, and
	 * flush it. */
	if ( n_data )
	{
		GET_BUFFERED_POINTER( n_data );
		memcpy( data_ptr, data, n_data );
	}
	else
	{
		GET_BUFFERED_POINTER_NO_ARGS( );
	}
	WRITE_OPCODE( op );

	for ( i = 0; i < g->num_servers; i++ )
	{
		wireGLPipeServerBufferAppend( &g->servers[i], &g->buffer );
		wireGLSendPipeServerBuffer( &g->servers[i] );
	}

	g->buffer.data_current = g->buffer.data_start;
	g->buffer.opcode_current = g->buffer.opcode_start;

#ifdef INSTRUMENT
	if (__wiregl_globals.print.frame_rate)
	{
		double frame_time = wireGLInstrumentGetTime( WIREGL_FRAME_TIMER );
		wireGLInstrumentIncrPerFrameCounter( WIREGL_FRAME_TIMER, 1.0/frame_time );
		wireGLInstrumentResetTimer( WIREGL_FRAME_TIMER );
		wireGLInstrumentStartTimer( WIREGL_FRAME_TIMER );
	}
	wireGLInstrumentNextFrame();
#endif
}

static GLbitvalue
FlushBuffer( GLcontext *virtual_context )
{
	WireGLGlobals *g = &__wiregl_globals;
	const GLflushinfo * current_bitmask;
	WireGLSendBuffer geom_buffer;
	int i;
	WireGLMessageOpcodes *big_packet_hdr = NULL;
	unsigned int big_packet_len = 0;
	GLrecti bounds;

	if ( state_server )
	{
		/* __glDiffContext was packing into the state_server's buffer
		   via the global pointers when it ran out of space.  Copy the
		   global pointers back into the state_server, send the buffer
		   out the door, and then put it back so we can resume
		   packing.  Word. */
		RestorePipeServerBufferFromGlobals( state_server );
		wireGLSendPipeServerBuffer( state_server );
		SetGlobalsToPipeServerBuffer( state_server );
		return 0;
	}

	if ( g->buffer.size > (unsigned int) g->pipe_buffer_size )
	{
		big_packet_hdr = 
			ApplySendBufferHeader( &g->buffer, &big_packet_len );

#if 0
		wireGLWarning( WIREGL_WARN_DEBUG, "FlushBuffer: handling big "
					   "geometry packet (%u bytes)", big_packet_len );
#endif
	}

	/* First, call the bucketer to find out where we're going. */
	current_bitmask =__glBeginFlush( );

#if 0
	/* comment */
	{
		static int num_buckets = 0;
		static int total_overlap = 0;

		int x = current_bitmask->hits;
		int count = 0;
		while ( x ) {
			x &= ( x - 1 );
			count++;
		}
		total_overlap += count;
		num_buckets++;

		if ( num_buckets == 100000 )
		{
			wireGLWarning( WIREGL_WARN_DEBUG, "num_buckets=%d total_overlap=%d"
						   " average=%.3f", num_buckets, total_overlap,
						   (float) total_overlap / (float) num_buckets );
			num_buckets = 0;
			total_overlap = 0;
		}
	}
#endif

	bounds = current_bitmask->pixelbounds;
	
#if 0
	if ( g->buffer.data_current >= g->buffer.data_end )
	{
		wireGLWarning( WIREGL_WARN_DEBUG, "wireGLFlushBuffer: buffer is very "
					   "full, this could be trouble" );
	}
#endif
	/* HACK HACK HACK */
	/* This extra space has been set aside and hidden so that we'll
	   always have space to append an end when necessary */
	g->buffer.data_end += END_FLUFF;
	__glCloseBeginEnd( virtual_context );

	/* remember the geometry buffer pointers, since we'll repeatedly
       clobber them with pipe server buffer pointers */
	geom_buffer = g->buffer;

	for ( i = 0; i < g->num_servers; i++ )
	{
		if ( !g->use_ring && !( current_bitmask->hits & (1 << i) ) )
			continue;

		/* If the bucketer said we're sending to server "i", do it. */
		state_server = g->servers + i;

#if 0
		/* HACK! */
		if ( state_server->conn->type == WIREGL_DROP_PACKETS )
		{
			continue;
		}
#endif
		
		/* Generate the lazy state evaluation commands into the
		   command buffer for the state_server */
		SetGlobalsToPipeServerBuffer( state_server );
		__glDiffContext( state_server->context, virtual_context );
		RestorePipeServerBufferFromGlobals( state_server );

		/* Append the geometry if it will fit, otherwise send the
	       pipeserver's buffer, and then send the geometry as a
 	       multi-packet */
		if ( big_packet_hdr )
		{
			wireGLSendPipeServerBuffer( state_server );
			big_packet_hdr->senderId = state_server->conn->sender_id;
			wireGLNetSend( state_server->conn, NULL, 
						   big_packet_hdr, big_packet_len );
		}
		else
		{
			if ( current_bitmask->isstate || !g->apply_viewtransform )
			{
				wireGLPipeServerBufferAppend( state_server, &geom_buffer );
			}
			else
			{
				wireGLPipeServerBufferAppendGeom( state_server, &geom_buffer,
												  &bounds );
			}
		}
	}

	/* wipe the state_server global, so we resume "normal" flush mode */
	state_server = NULL;

	/* We're done -- reset all the global pointers so that subsequent
	   geometry commands will pack into the right place. */
	if ( big_packet_hdr )
	{
		wireGLFree( geom_buffer.pack );
		wireGLPackBufferInit( &g->buffer, wireGLAlloc( g->pack_buffer_size ),
							  g->pack_buffer_size );
	}
	else
	{
		g->buffer = geom_buffer;
		g->buffer.data_current = g->buffer.data_start;
		g->buffer.opcode_current = g->buffer.opcode_start;
	}

	/* hide the space to append a glEnd() */
	g->buffer.data_end -= END_FLUFF;

	/* Let the bucketer know that we're done. */
	__glEndFlush( );
	
	return current_bitmask->hits;
}

void wireGLFlushBuffer( GLcontext *ctx )
{
	WireGLSendBuffer  src, *dst;

#if 0
	/* MWE- this breaks any code that is trying to force data over the
       wire, even though there might not be any geometry commands
       currently packed.  For example, when Ian broadcasts textures.
       I think this "optimization" was present in the original version
       of the buffer.c code, is it actually important to anything? */
	WireGLGlobals *g = &__wiregl_globals;
	/* try to outsmart whoever called us... could this be dangerous? */
	if ( g->buffer.opcode_current == g->buffer.opcode_start )
		return;
#endif

	/* only flush if
	 * 1. we aren't in a display list definition
	 * 2. we are splitting begin/ends OR we aren't in a begin/edn
	 */
	if ( !ctx->lists.newend &&
		 ( __wiregl_globals.split_begin_end || !ctx->current.beginend ) )
	{
		(void) FlushBuffer( ctx );
		return;
	}

	/* This should only happen when we are packing geometry, otherwise
       we are splitting a begin/end block, which we don't want to do.
       Right? */
	wireGLAssert( state_server == NULL );

	/* We're inside a begin/end block, and we are not splitting
       those up.  Instead grow the pack buffer and tag the
	   context to be flushed as soon as the glEnd() rolls around. */
	ctx->current.flush_on_end++;

	src = __wiregl_globals.buffer;
	dst = &__wiregl_globals.buffer;
	dst->size <<= 1;
	wireGLPackBufferInit( dst, wireGLAlloc( dst->size ), dst->size );

	SendBufferAppend( dst, &src );

	/* We've changed the location of the geometry buffer.  So we need
	   to notify the current state tracker to update its pointers */
	__glFixupPointers( dst->data_current - src.data_current );

	wireGLFree( src.pack );
}

void *wireGLAllocPacket( unsigned int size )
{
	WireGLGlobals *g = &__wiregl_globals;
	unsigned char *data_ptr;

	/* include space for the length and make the payload word-aligned */
	size = ( size + sizeof(unsigned int) + 0x3 ) & ~0x3;

	if ( g->buffer.data_current + size <= g->buffer.data_end )
	{
		/* we can just put it in the current buffer */
		GET_BUFFERED_POINTER( size );
	}
	else 
	{
		/* Okay, it didn't fit.  Maybe it will after we flush. */
		wireGLFlushBuffer( g->context );
		if ( g->buffer.data_current + size <= g->buffer.data_end )
		{
			GET_BUFFERED_POINTER( size );
		}
		else
		{
			/* It's really way too big, so allocate a temporary packet
			 * with space for the single opcode plus the payload &
			 * header */
			data_ptr = (unsigned char *) 
				wireGLAlloc( sizeof(WireGLMessageOpcodes) + 4 + size );

			/* skip the header & opcode space */
			data_ptr += sizeof(WireGLMessageOpcodes) + 4;
		}
	}

	*((unsigned int *) data_ptr) = size;
	return ( data_ptr + 4 );
}

#define IS_BUFFERED( packet ) \
    ((unsigned char *) (packet) >= __wiregl_globals.buffer.data_start && \
	 (unsigned char *) (packet) < __wiregl_globals.buffer.data_end)

static void
SendPacket( WireGLPipeServer *pipe, WireGLOpcode opcode, void *packet )
{
	unsigned int          len;
	unsigned char        *src;
	WireGLMessageOpcodes *msg;

	wireGLAssert( pipe != NULL );

	/* packet length is indicated by the variable length field, and
	   includes an additional word for the opcode (with alignment) and
	   a header */
	len = ((unsigned int *) packet)[-1] + 4 + sizeof(WireGLMessageOpcodes);

	/* write the opcode in just before the length */
	((unsigned char *) packet)[-5] = (unsigned char) opcode;

#if 0
	wireGLWarning( WIREGL_WARN_DEBUG, "sending a big packet: %s, "
				   "len=%u", __wiregl_opcode_names[opcode], len );
#endif

	/* fix up the pointer to the packet to include the length & opcode
       & header */
	src = (unsigned char *) packet - 8 - sizeof(WireGLMessageOpcodes);

	msg = (WireGLMessageOpcodes *) src;

	msg->type       = WIREGL_MESSAGE_OPCODES;
	msg->senderId   = pipe->conn->sender_id;
	msg->numOpcodes = 1;

	/* the pipeserver's buffer might have data in it, and that should
       go across the wire before this big packet */
	wireGLSendPipeServerBuffer( pipe );

	wireGLNetSend( pipe->conn, NULL, src, len );
}

void 
wireGLSendPacket( WireGLOpcode opcode, void *packet )
{
	if ( IS_BUFFERED( packet ) )
		WRITE_OPCODE( opcode );
	else
		SendPacket( state_server, opcode, packet );
}

void 
wireGLMultiSendPacket( GLcontext *ctx, WireGLOpcode opcode, void *packet )
{
	WireGLGlobals *g = &__wiregl_globals;
	GLbitvalue bitmask;
	int i;

	if ( IS_BUFFERED( packet ) )
	{
		/* the packet is buffered, so just send it the normal way we
           send geometry */
		WRITE_OPCODE( opcode );
		(void) FlushBuffer( ctx );
		return;
	}

	/* use our internal flush mechanism to flush all the pipes that
       this packet overlaps, whether or not there is something in the
       geometry buffer (in fact, we expect it to be empty) */
	wireGLAssert( g->buffer.data_current == g->buffer.data_start );
	bitmask = FlushBuffer( ctx );

	for ( i = 0; i < g->num_servers; i++ )
	{
		if ( !( bitmask & ( 1 << i ) ) )
			continue;

		SendPacket( &g->servers[i], opcode, packet );
	}
}

void 
wireGLFreePacket( void *packet )
{
	if ( IS_BUFFERED( packet ) )
		return;
	
	/* the pointer passed in doesn't include the space for the single
	 * opcode (4 bytes because of the alignment requirement) or the
	 * length field or the header */
	wireGLFree( (unsigned char *) packet - 8 - sizeof(WireGLMessageOpcodes) );
}

void *
wireGLNetworkPointerRead( WireGLNetworkPointer *src )
{
	void *dst;
	memcpy( &dst, src, sizeof(dst) );
	return dst;
}

void 
wireGLNetworkPointerWrite( WireGLNetworkPointer *dst, void *src )
{
	dst->ptrAlign[0] = 0xDeadBeef;
	dst->ptrAlign[1] = 0xCafeBabe;
	memcpy( dst, &src, sizeof(src) );
}

void
wireGLReadPixelsRecv( WireGLMessageReadPixels *msg )
{
	unsigned int y;
	unsigned char *src, *dst;

	src = (unsigned char *) msg + sizeof(*msg);
	dst = (unsigned char *) wireGLNetworkPointerRead( &msg->pixels );
	for ( y = 0; y < msg->rows; y++ )
	{
		memcpy( dst, src, msg->bytes_per_row );
		src += msg->bytes_per_row;
		dst += msg->stride;
	}
}

void 
wireGLClientReceive( WireGLConnection *conn, void *buf, unsigned int len )
{
	WireGLMessage *msg = (WireGLMessage *) buf;

	wireGLAssert( len >= sizeof(msg->type) );
	WIREGL_UNUSED(len);

	switch ( msg->type )
	{
	  case WIREGL_MESSAGE_WRITEBACK:
		conn->pending_writebacks--;
		break;

	  case WIREGL_MESSAGE_READ_PIXELS:
		wireGLReadPixelsRecv( &msg->readPixels );
		conn->pending_writebacks--;
		break;

	  default:
		wireGLError( "wireGLClientReceive: msg->type=0x%x?", msg->type );
		break;
	}

	wireGLNetFree( conn, buf );
}


#if 0
#include <sys/time.h>

static double
SampleTimer( void )
{
	struct timeval now;
	double x;

	gettimeofday( &now, NULL );
	
	x  = (double) now.tv_sec;
	x += 1e-6 * (double) now.tv_usec;

	return x;
}

#endif

void 
wireGLSyncWithPipes( void )
{
	WireGLGlobals *g = &__wiregl_globals;
	int i;
	
	wireGLFlushAll( g->context );

	for ( i = 0; i < g->num_servers; i++ )
	{
		WireGLPipeServer *pipe = &g->servers[i];
		unsigned char *data_ptr;

		wireGLAssert( pipe->conn->pending_writebacks == 0 );

		/* a packet dropping connection is unlikely to respond to a
         * writeback request */
		if ( pipe->conn->type != WIREGL_DROP_PACKETS )
		{
			pipe->conn->pending_writebacks++;
		}

		GET_BUFFERED_POINTER_NO_ARGS( );
		WRITE_OPCODE( WIREGL_WRITEBACK_OPCODE );

		wireGLPipeServerBufferAppend( pipe, &g->buffer );
		wireGLSendPipeServerBuffer( pipe );

		g->buffer.data_current = g->buffer.data_start;
		g->buffer.opcode_current = g->buffer.opcode_start;
	}

	for ( i = 0; i < g->num_servers; i++ )
	{
		WireGLPipeServer *pipe = &g->servers[i];

		while ( pipe->conn->pending_writebacks > 0 )
		{
			wireGLNetRecv( );
		}
	}
}
