/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#ifndef WIREGL_CLIENT_H
#define WIREGL_CLIENT_H

#ifdef WINDOWS
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif

#include "wiregl/include/wiregl_util.h"
#include "wiregl/glcontext/glcontext.h"
#include "wiregl/glcontext/glhw.h"
#include "wiregl/include/wiregl_bbox.h"
#include "wiregl/include/wiregl_opcodes.h"
#include "wiregl/include/wiregl_feather.h"
#include "wiregl/include/wiregl_common.h"

typedef struct WireGLSendBuffer {
	void          *pack;
	unsigned int   size;
	unsigned char *data_start, *data_current, *data_end;
	unsigned char *opcode_start, *opcode_current, *opcode_end;
} WireGLSendBuffer;

typedef WireGLSendBuffer WireGLPackBuffer;

typedef struct WireGLPipeServer {
	WireGLConnection *conn;
	int num_extents;
	int mural_x[WIREGL_MAX_EXTENTS], mural_y[WIREGL_MAX_EXTENTS];
	int mural_w[WIREGL_MAX_EXTENTS], mural_h[WIREGL_MAX_EXTENTS];
	int window_x, window_y;
	WireGLFeatherParam feather;
	GLcontext *context;
	int rotate_color;
	float colormatrix[12];

	char *server;
	int index;

	WireGLSendBuffer buffer;
} WireGLPipeServer;

typedef struct WireGLDisplay {
	char name[WIREGL_MAX_DISPLAY_NAME_LEN];
	int x, y, w, h;
	int rotate_color;
	float colormatrix[12];
} WireGLDisplay;

typedef struct WireGLGlobals {
	/* buffer state */
	WireGLSendBuffer buffer;
	GLcontext       *context;

#ifdef SIMD
	__m128 pad16byte;
#endif
	GLvectorf bounds_min;
	GLvectorf bounds_max;
	/* tracking state */
	GLcurrent current;

	/* pipe server state */
	int num_servers;
	WireGLPipeServer *servers;

	/* client state */
#ifdef WINDOWS
	HDC  client_hdc;
	HWND client_hwnd;
#else
	void         *glx_display;  /* (Display *) */
	unsigned int  glx_drawable; /* (GLXDrawable -> XID -> CARD32) */
#endif

	/* instrumentation */
	struct {
		int bytes_per_server;
		int bytes_per_frame_per_server;
		int min_bytes_sent;
		int frame_rate;
		int flush_time;
		int send_time;
		int diff_context_time;
		int total_bytes_sent;
		int total_time;
	} print;

	/* miscellany */
	struct {
		int width;
		int height;
	} mural;

	
	WireGLDisplay display[WIREGL_MAX_DISPLAYS];
	int num_displays;

	int pipe_buffer_size;
	int pack_buffer_size;
	int depth_bits;
	int stencil_bits;	
	int normal_compression;
	int vertex_compression;
	int unsafe_compression;
	int compression_depthbits;
	int swap_barrier;
	char *barrier_server;
	int gui_error;
	int draw_bbox;
	char *warn_level_str;
	int bucket_size;
	int broadcast;
	int broadcast_textures;
	int pack_only;
	int resolve_arrays;
	int state_error_gui;
	int sync_on_finish;
	int sync_on_swap;
	int split_begin_end;
	int use_ring;
	int apply_viewtransform;
	int use_simd;
	int sync_L2;
	int beginend_max;
	int optimize_bucket;

} WireGLGlobals;

extern WireGLGlobals __wiregl_globals;

/* API interfaces */
extern GLjumptable wiregl_im_api[];
extern GLapi wiregl_pack_api[];
extern GLapi vertexBBapi[];
extern GLapi vertexBBSIMDapi[];
extern GLapi vertexCapi[];
#ifdef WINDOWS
extern GLapi vertexSafeCompressBBapi[];
extern GLapi vertexSafeCompressCapi[];
extern GLapi vertexCompressBBapi[];
extern GLapi vertexCompressCapi[];
#endif


void OPENGL_APIENTRY wireGLCreateContext( int is_parallel );
void OPENGL_APIENTRY wireGLMakeCurrent( void );
void OPENGL_APIENTRY wireGLSwapBuffers( void );
void OPENGL_APIENTRY wireGLUseSystemGL( void );
void OPENGL_APIENTRY wireGLGetIntegerv( GLuint pname, GLint *params );

void wireGLInitGlobals( void );
void wireGLClientParseConfig( void );
void wireGLSendBufferInit( WireGLSendBuffer *send_buf, void *pack, unsigned int size );
void wireGLPackBufferInit( WireGLPackBuffer *send_buf, void *pack, unsigned int size );
unsigned int wireGLRecommendedPackBufferSize (unsigned int send_size);

int  WireGLGetWindowSize( int *width_return, int *height_return );

void *wireGLAllocPacket( unsigned int size );
void wireGLFreePacket( void *packet );
void wireGLSendPacket( WireGLOpcode op, void *packet );
void wireGLMultiSendPacket( GLcontext *ctx, WireGLOpcode op, void *packet );
void wireGLFlushBuffer( GLcontext *ctx );
void wireGLFlushAll( GLcontext *ctx );
void wireGLFlushAllWithOpcode( GLcontext *ctx, WireGLOpcode op, unsigned int n_dat, void *data );
void wireGLClientReceive( WireGLConnection *conn, void *buf, unsigned int len );
void wireGLPipeServerBufferAppend( WireGLPipeServer *pipe, WireGLSendBuffer *src );
void wireGLPipeServerBufferAppendGeom( WireGLPipeServer *pipe, WireGLSendBuffer *src, GLrecti *bounds );
void wireGLSendPipeServerBuffer( WireGLPipeServer *pipe );
void wireGLSyncWithPipes( void );
void wireGLVpinch (GLcontext *ctx);

#define GET_BUFFERED_POINTER( len ) \
  data_ptr = __wiregl_globals.buffer.data_current; \
  if (data_ptr + (len) > __wiregl_globals.buffer.data_end ) \
  { \
    wireGLFlushBuffer( __wiregl_globals.context ); \
	data_ptr = __wiregl_globals.buffer.data_current; \
    wireGLAssert( data_ptr + (len) <= __wiregl_globals.buffer.data_end ); \
  } \
  __wiregl_globals.buffer.data_current += (len)

#define GET_BUFFERED_POINTER_NO_ARGS( ) \
  GET_BUFFERED_POINTER( 4 );  \
  WRITE_DATA( 0, GLuint, 0xdeadbeef )

#if 0

/* Nobody is really using the vertex max so I'm removing it */
#define GET_BUFFERED_COUNT_POINTER( len ) \
  data_ptr = __wiregl_globals.buffer.data_current; \
  if (data_ptr + (len) > __wiregl_globals.buffer.data_end || \
	  __wiregl_globals.current.vtx_count >= __wiregl_globals.current.vtx_max) \
  { \
    wireGLFlushBuffer( __wiregl_globals.context ); \
	data_ptr = __wiregl_globals.buffer.data_current; \
  } \
  __wiregl_globals.buffer.data_current += (len); \
  __wiregl_globals.current.vtx_count++; \

#else

#define GET_BUFFERED_COUNT_POINTER( len ) \
  data_ptr = __wiregl_globals.buffer.data_current; \
  if (data_ptr + (len) > __wiregl_globals.buffer.data_end) \
  { \
    wireGLFlushBuffer( __wiregl_globals.context ); \
	data_ptr = __wiregl_globals.buffer.data_current; \
  } \
  __wiregl_globals.buffer.data_current += (len); \
  __wiregl_globals.current.vtx_count++; \

#endif

#define WRITE_DATA( offset, type, data ) \
  *( (type *) (data_ptr + (offset))) = (data)

#ifdef WIREGL_UNALIGNED_ACCESS_OKAY
#define WRITE_DOUBLE( offset, data ) \
  WRITE_DATA( offset, GLdouble, data )
#else
#define WRITE_DOUBLE( offset, data ) \
  wireGLWriteUnalignedDouble( data_ptr + (offset), (data) )
#endif

#define WRITE_OPCODE( opcode ) \
  *(__wiregl_globals.buffer.opcode_current--) = (unsigned char) opcode

#define WRITE_NETWORK_POINTER( offset, data ) \
  wireGLNetworkPointerWrite( (WireGLNetworkPointer *) ( data_ptr + (offset) ), (data) )

#endif /* WIREGL_CLIENT_H */

