/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include "glcontext.h"
#include "glerror.h"
#include "glhw.h"

void 
__glviewport_initbits (GLviewportbits *v, GLconfig *cfg) {
	UNUSED(v);
	UNUSED(cfg);
	/* do nothing */
}

void
__glviewport_cliptowindow (const GLrecti *imagewindow,
						   GLrecti *q) {
	if (q->x1 < imagewindow->x1) q->x1 = imagewindow->x1;
	if (q->x1 > imagewindow->x2) q->x1 = imagewindow->x2;

	if (q->x2 > imagewindow->x2) q->x2 = imagewindow->x2;
	if (q->x2 < imagewindow->x1) q->x2 = imagewindow->x1;
	
	if (q->y1 < imagewindow->y1) q->y1 = imagewindow->y1;
	if (q->y1 > imagewindow->y2) q->y1 = imagewindow->y2;

	if (q->y2 > imagewindow->y2) q->y2 = imagewindow->y2;
	if (q->y2 < imagewindow->y1) q->y2 = imagewindow->y1;
}

void
__glviewport_converttooutput (const GLrecti *imagewindow,
							  const GLrecti *outputwindow,
							  GLrecti *q) {
	/*
	** Move the imagewindow into outputwindow.
	** So were assuming that the outputwidow is
	** is the same width and height as the imagewindow.
	*/
	q->x1 = q->x1 - imagewindow->x1 + outputwindow->x1;
	q->x2 = q->x2 - imagewindow->x2 + outputwindow->x2;
	q->y1 = q->y1 - imagewindow->y1 + outputwindow->y1;
	q->y2 = q->y2 - imagewindow->y2 + outputwindow->y2;
}

void
__glviewport_setoutputbounds (GLviewportstate *v,
							  const GLrecti *outputwindow,
							  const GLrecti *imagespace,
							  const GLrecti *imagewindow,
							  GLrecti *clipped_imagespace,
							  GLrecti *clipped_imagewindow) {
	GLrecti q;

	v->outputdims = *imagewindow;

	v->widthscale = (GLfloat) ( outputwindow->x2 - outputwindow->x1 );
	v->widthscale /= (GLfloat) ( imagespace->x2 - imagespace->x1 );

	v->heightscale = (GLfloat) ( outputwindow->y2 - outputwindow->y1 );
	v->heightscale /= (GLfloat) ( imagespace->y2 - imagespace->y1 );

	v->x_offset = outputwindow->x1;
	v->y_offset = outputwindow->y1;

	/* If the scissor is invalid 
	** set it to the whole output
	*/
	if (!v->s_valid) {

		__glhw_Scissor (outputwindow->x1, outputwindow->y1,
			outputwindow->x2 - outputwindow->x1,
			outputwindow->y2 - outputwindow->y1);
	} else {
		q.x1 = v->s_x;
		q.x2 = v->s_x + v->s_w;
		q.y1 = v->s_y;
		q.y2 = v->s_y + v->s_h;

		__glviewport_cliptowindow (imagewindow, &q);
		__glviewport_converttooutput (imagewindow, outputwindow, &q);
		__glhw_Scissor  (q.x1,  q.y1, 
			q.x2 - q.x1, q.y2 - q.y1);
	}
	
	/* if the viewport is not valid,
	** set it to the entire output.
	*/
	if (!v->v_valid) {
		if (clipped_imagespace)
			*clipped_imagespace = *imagespace;
		if (clipped_imagewindow)
			*clipped_imagewindow = *imagewindow;

		__glhw_Viewport (outputwindow->x1, outputwindow->y1,
			outputwindow->x2 - outputwindow->x1,
			outputwindow->y2 - outputwindow->y1);

#if 0
		printf ("Invalid: __glhw_Viewport (%d, %d, %d, %d)\n", 
			outputwindow->x1, outputwindow->y1,
			outputwindow->x2 - outputwindow->x1,
			outputwindow->y2 - outputwindow->y1);
#endif

		return;
	}
	
	q.x1 = v->v_x;
	q.x2 = v->v_x + v->v_w;
	q.y1 = v->v_y;
	q.y2 = v->v_y + v->v_h;

	__glviewport_cliptowindow (imagewindow, &q);

	if (clipped_imagespace) {
		clipped_imagespace->x1 = v->v_x;
		clipped_imagespace->x2 = v->v_x + v->v_w;
		clipped_imagespace->y1 = v->v_y;
		clipped_imagespace->y2 = v->v_y + v->v_h;
	}

	if (clipped_imagewindow)
		*clipped_imagewindow = q;

	__glviewport_converttooutput (imagewindow, outputwindow, &q);

	__glhw_Viewport (q.x1,  q.y1, 
		q.x2 - q.x1, q.y2 - q.y1);

#if 0
	printf ("__glhw_Viewport (%d, %d, %d, %d)\n", 
			q.x1,  q.y1, 
			q.x2 - q.x1, q.y2 - q.y1);
#endif
}

void
__glviewport_clamp (GLviewportstate *current, GLviewportstate *target, GLrecti *p,
					 GLint x, GLint y, GLint w, GLint h) {

	p->x1 = x;
	p->y1 = y;
	p->x2 = x+w;
	p->y2 = y+h;

	if (!current->doclamp) 
		return;

	if (current->outputdims.x1 == current->outputdims.x2 &&
		current->outputdims.y1 == current->outputdims.y2)
		return;

	/* Clamp it to the outputdims */
	__glviewport_cliptowindow (&current->outputdims, p);
	
	if (p->x2 == p->x1 ||
		p->y2 == p->y1) {
		p->x1 = 0;
		p->x2 = 0;
		p->y1 = 0;
		p->y2 = 0;
	} else {
		p->x1 -= current->outputdims.x1 - target->x_offset;
		p->y1 -= current->outputdims.y1 - target->y_offset;
		p->x2 -= current->outputdims.x1 - target->x_offset;
		p->y2 -= current->outputdims.y1 - target->y_offset;
	}
}

void
__glviewport_init(GLviewportstate *v, GLconfig *c) {
	int id;	

	v->scissortest = GL_FALSE;
	
	v->maxviewportdims_width = c->maxviewportdims_width;
	v->maxviewportdims_height = c->maxviewportdims_height;

	v->v_valid = c->viewport_valid;
	v->v_x = c->viewport.x1;
	v->v_y = c->viewport.y1;
	v->v_w = c->viewport.x2 - c->viewport.x1;
	v->v_h = c->viewport.y2 - c->viewport.y1;

	v->s_valid = c->viewport_valid;
	v->s_x = c->viewport.x1;
	v->s_y = c->viewport.y1;
	v->s_w = c->viewport.x2 - c->viewport.x1;
	v->s_h = c->viewport.y2 - c->viewport.y1;

	v->getwindowsize = c->getwindowsize;

	id = c->id;
	if (id >= 0 && c->numprojectors > 0)
		v->outputdims = c->bounds[id][0];  /* Is this bad? */
	else
		v->outputdims = c->viewport;

	if (!c->localviewport.x2) 
		v->widthscale = 1.0f;
	else
		v->widthscale = ((GLfloat) v->v_w) / ((GLfloat) c->localviewport.x2);

	if (!c->localviewport.y2)
		v->heightscale = 1.0f;
	else
		v->heightscale = ((GLfloat) v->v_h) / ((GLfloat) c->localviewport.y2);

	v->f = 1.0;
	v->n = 0.0;

	v->x_offset = 0;
	v->y_offset = 0;

	v->doclamp = c->apply_viewtransform;
}

void
__glviewport_destroy(GLviewportstate *v) {
/*
** Do nothing...
*/
	UNUSED(v);
}

void
__glviewport_ApplyViewport(GLviewportstate *v, GLvectorf *p) {
	p->x = (p->x+1.0f)*(v->v_w / 2.0f) + v->v_x;
	p->y = (p->y+1.0f)*(v->v_h / 2.0f) + v->v_y;
}

void
__glviewport_MakeCurrent(GLviewportstate *v, GLviewportbits *vb) {
	if (!v->v_valid && v->getwindowsize) {
		int w, h;
		if (v->getwindowsize(&w, &h)) {
			v->widthscale = (GLfloat) (v->outputdims.x2 - v->outputdims.x1) / (GLfloat) w;
			v->heightscale = (GLfloat) (v->outputdims.y2 - v->outputdims.y1) / (GLfloat) h;

			v->v_x = 0;
			v->v_y = 0;
			v->v_w = v->outputdims.x2 - v->outputdims.x1;
			v->v_h = v->outputdims.y2 - v->outputdims.y1;

			if (!v->s_valid) {
				v->s_x = v->v_x;
				v->s_y = v->v_y;
				v->s_w = v->v_w;
				v->s_h = v->v_h;
				v->s_valid = GL_TRUE;
				vb->s_dims = GLBITS_ONES;
				vb->dirty = GLBITS_ONES;
			}
			
			vb->v_dims = GLBITS_ONES;
			vb->dirty = GLBITS_ONES;
			v->v_valid = GL_TRUE;
		}
	}
}


void GLSTATE_DECL
__glstate_Viewport(GLint x, GLint y, GLsizei width, GLsizei height) {
	GLcontext *g = GetCurrentContext();
	GLviewportstate *v = &(g->viewport);
	GLstatebits *sb = GetStateBits();
	GLviewportbits *vb = &(sb->viewport);
	GLtransbits *tb = &(sb->trans);
	
	if (g->current.beginend)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_OPERATION, "glViewport called in Begin/End"))
			return;
	
	if (IsBucketingActive()) 
		g->bucket.flush(g);

	if (width < 0 || height < 0)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_VALUE, "Negative viewport width or height: %dx%d", width, height))
			return;

	if (x > v->maxviewportdims_width)	x = v->maxviewportdims_width;
	if (x < -v->maxviewportdims_width)	x = -v->maxviewportdims_width;
	if (y > v->maxviewportdims_height)	y = v->maxviewportdims_height;
	if (y < -v->maxviewportdims_height)	y = -v->maxviewportdims_height;
	if (width > v->maxviewportdims_width)	width = v->maxviewportdims_width;
	if (height > v->maxviewportdims_height)	height = v->maxviewportdims_height;

	if (v->getwindowsize) {
		int w, h;
		/* Ok, this is a bit weird.  The output dims of the virtual context should be the
		** the whole output.
		*/
		if (v->getwindowsize(&w, &h)) {
			v->widthscale = (GLfloat) (v->outputdims.x2 - v->outputdims.x1) / (GLfloat) w;
			v->heightscale = (GLfloat) (v->outputdims.y2 - v->outputdims.y1) / (GLfloat) h;
		}

	}

	v->v_x = (GLint) (x*v->widthscale+0.5f);
	v->v_y = (GLint) (y*v->heightscale+0.5f);
	v->v_w = (GLint) (width*v->widthscale+0.5f);
	v->v_h = (GLint) (height*v->heightscale+0.5f);

	v->v_valid = GL_TRUE;
	
	__glbucket_bounds(v->v_x, v->v_y, v->v_w, v->v_h);

	vb->v_dims = g->nbitID;
	vb->dirty = g->nbitID;
	tb->base = g->nbitID;
	tb->compress = g->nbitID;
	tb->dirty = g->nbitID;

}

void GLSTATE_DECL
__glstate_DepthRange(GLclampd znear, GLclampd zfar) {
	GLcontext *g = GetCurrentContext();
	GLviewportstate *v = &(g->viewport);
	GLstatebits *sb = GetStateBits();
	GLviewportbits *vb = &(sb->viewport);
	GLtransbits *tb = &(sb->trans);

	if (g->current.beginend)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_OPERATION, "glDepthRange called in Begin/End"))
			return;

	if (IsBucketingActive()) 
		g->bucket.flush(g);

	v->n = znear;
	v->f = zfar;
	if (v->n < 0.0) v->n = 0.0;
	if (v->n > 1.0) v->n = 1.0;
	if (v->f < 0.0) v->f = 0.0;
	if (v->f > 1.0) v->f = 1.0;

	vb->depth = g->nbitID;
	vb->dirty = g->nbitID;
	tb->compress = g->nbitID;
	tb->dirty = g->nbitID;
}

void GLSTATE_DECL
__glstate_Scissor (GLint x, GLint y, 
					 GLsizei width, GLsizei height) {
	GLcontext *g = GetCurrentContext();
	GLviewportstate *v = &(g->viewport);
	GLstatebits *stateb = GetStateBits();
	GLviewportbits *vb = &(stateb->viewport);

	if (g->current.beginend) 
		if (__glerror(__LINE__, __FILE__, GL_INVALID_OPERATION,
			"glScissor called in begin/end"))
			return;

	if (IsBucketingActive()) 
		g->bucket.flush(g);

	if (width < 0 || height < 0)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_VALUE,
			"glScissor called with negative width/height: %d,%d",
			width, height))
			return;

	if (v->getwindowsize) {
		int w, h;
		/* Ok, this is a bit weird.  The output dims of the virtual context should be the
		** the whole output.
		*/
		if (v->getwindowsize(&w, &h)) {
			v->widthscale = (GLfloat) (v->outputdims.x2 - v->outputdims.x1) / (GLfloat) w;
			v->heightscale = (GLfloat) (v->outputdims.y2 - v->outputdims.y1) / (GLfloat) h;
		}
	}

	v->s_x = (GLint) (x*v->widthscale+0.5f);
	v->s_y = (GLint) (y*v->heightscale+0.5f);
	v->s_w = (GLint) (width*v->widthscale+0.5f);
	v->s_h = (GLint) (height*v->heightscale+0.5f);

	v->s_valid = GL_TRUE;

	vb->s_dims = g->nbitID;
	vb->dirty = g->nbitID;
}

void GLSTATE_DECL
__gltrack_Scissor (GLint x, GLint y, 
					 GLsizei width, GLsizei height) {
	GLcontext *g = GetCurrentContext();
	GLviewportstate *v = &(g->viewport);
	GLstatebits *stateb = GetStateBits();
	GLviewportbits *vb = &(stateb->viewport);

	__glhw_Scissor (x, y, width, height);

	v->s_x = x;
	v->s_y = y;
	v->s_w = width;
	v->s_h = height;
	
	v->s_valid = GL_TRUE;

	vb->s_dims = g->nbitID;
	vb->dirty = g->nbitID;
}

void GLSTATE_DECL
__gltrack_Viewport(GLint x, GLint y, GLsizei width, GLsizei height) {
	GLcontext *g = GetCurrentContext();
	GLviewportstate *v = &(g->viewport);
	GLstatebits *sb = GetStateBits();
	GLviewportbits *vb = &(sb->viewport);

	/* Note that we probably don't 
	** always want to call this 
	** since if we're running in 
	** ring mode, we're going
	** to reissue this.
	*/
	__glhw_Viewport(x, y, width, height);

	v->v_x = x;
	v->v_y = y;
	v->v_w = width;
	v->v_h = height;

	v->v_valid = GL_TRUE;

	vb->v_dims = g->nbitID;
	vb->dirty = g->nbitID;
}

void GLSTATE_DECL
__gltrack_DepthRange(GLclampd znear, GLclampd zfar) {
	GLcontext *g = GetCurrentContext();
	GLviewportstate *v = &(g->viewport);
	GLstatebits *sb = GetStateBits();
	GLviewportbits *vb = &(sb->viewport);

	__glhw_DepthRange( znear,  zfar);

	v->n = znear;
	v->f = zfar;

	vb->depth = g->nbitID;
	vb->dirty = g->nbitID;
}





