/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "glcontext.h"
#include "glerror.h"

#ifdef WINDOWS
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#elif defined(IRIX) || defined(IRIX64) || defined(Linux) || defined(AIX)
#include <dlfcn.h>
#else
#error DSO
#endif

#define GLHWAPI_DEFINE
#include "glhw.h"

#if defined( WINDOWS )
#define SYSTEM_GL_NAME "opengl32"
#else
#define SYSTEM_GL_NAME "libGL.so"
#endif

#define MAXAPI 400
GLfunc *APIbase[MAXAPI+1]			= {NULL};
GLfunc *APIbeginendbase[MAXAPI+1]	= {NULL};
GLfunc *APIvertexbase[MAXAPI+1]		= {NULL};
GLfunc *APIvertexbase_st[MAXAPI+1]	= {NULL};
GLfunc *APIarrayelementbase			= NULL;

GLfunc APIpk[MAXAPI+1] = {NULL};
GLfunc APIst[MAXAPI+1] = {NULL};
GLfunc APIbeginendpk[MAXAPI+1] = {NULL};
GLfunc APIbeginendst[MAXAPI+1] = {NULL};
GLfunc APIvertexpack[MAXAPI+1] = {NULL};
GLfunc APIvertexBBpack[MAXAPI+1] = {NULL};
GLfunc APIvertexCpack[MAXAPI+1] = {NULL};

GLfunc APIarrayelement_locked = (GLfunc) __glstate_ArrayElement_locked;
GLfunc APIarrayelement_unlocked = (GLfunc) __glstate_ArrayElement;

int APIcount = 0;

/* These are defined below */
extern GLjumptable hw_api[];

extern GLapi nop_api[];
extern GLapi glcontext_api[];
extern GLapi system_api[];

extern const char *packonly[];
extern const char *stateonly[];
extern const char *beginend[];
extern const char *vertexfuncs[];

/* LoadApi Functions
** Used during stream processing
** Note: These reimplemented to be a lot faster
** since the funcptrs are stored in a linear
** array, we could just do a memcopy. However,
** the function groups cannot cause non-linear
** groupings.  (I think this isn't a problem
** currently).
*/ 

void
__glhw_LoadPackApi(void) {
	GLfunc **base = APIbase;
	GLfunc *pk = APIpk;
	while (*base)
		**base++ = *pk++;
}

void
__glhw_LoadStateApi(void) {
	GLfunc **base = APIbase;
	GLfunc *st = APIst;
	while (*base)
		**base++ = *st++;
}

void
__glhw_LoadBeginEndApi(void) {
	GLfunc **base = APIbeginendbase;
	GLfunc *pk = APIbeginendpk;
	while (*base)
		**base++ = *pk++;
}

void
__glhw_UnloadBeginEndApi(void) {
	GLfunc **base = APIbeginendbase;
	GLfunc *st = APIbeginendst;
	while (*base)
		**base++ = *st++;
}

void
__glhw_LoadVertexPackApi(void) {
	GLfunc **base= APIvertexbase;
	GLfunc **basest = APIvertexbase_st;
	GLfunc *pk = APIvertexpack;
    while (*base) {
		**base++ = *pk;
		**basest++ = *pk++;
	}
}

void
__glhw_LoadVertexBBPackApi(void) {
	GLfunc **base = APIvertexbase;
	GLfunc **basest = APIvertexbase_st;
	GLfunc *pk = APIvertexBBpack;
    while (*base) {
		**base++ = *pk;
		**basest++ = *pk++;
	}
}

void
__glhw_LoadVertexCPackApi(void) {
	GLfunc **base = APIvertexbase;
	GLfunc **basest = APIvertexbase_st;
	GLfunc *pk = APIvertexCpack;
    while (*base) {
		**base++ = *pk;
		**basest++ = *pk++;
	}
}

void
__glhw_LockArrays(void) {
	*APIarrayelementbase = APIarrayelement_locked;
}

void
__glhw_UnlockArrays(void) {
	*APIarrayelementbase = APIarrayelement_unlocked;
}


/* glhw init function*/
void
__glhw_RegisterApi(GLconfig *c) {
	int i, j, beginend_i=0, vertex_i=0;
	GLapi *p;

	GLjumptable *base = c->immediateapi;
	GLapi *pk = c->packapi;
	GLapi *st = glcontext_api;

	if (!base || !base->func) return;

	/* if they don't give us a pack api,
	** use the systems.
	*/
	if (!pk || !pk->func) {
		__glhw_LoadSystemApi(system_api);
		pk = system_api;
	}

	for (i=0; i < MAXAPI && APIbase[i]; i++);

	for (; i < MAXAPI && base->name; i++, base++) {
		APIbase[i] = base->func;
		APIpk[i] = (GLfunc) __glhw_error;
		APIst[i] = (GLfunc) __glhw_error;

		/* Assign the hwerror */
		for (p=hwerror; p->name; p++) {
			if (!strcmp(base->name, p->name)) {
				APIpk[i] = p->func;
				APIst[i] = p->func;
			}
		}

		/* Assign Pack Function */
		for (p=pk; p->name; p++) {
			if (!strcmp(base->name, p->name)) {
				APIpk[i] = p->func;
				break;
			}
		}

		/* Commands for begin end spliting */
		if (!strcmp(base->name, "glVertex4fv"))
			__glbase_Vertex4fv = (void (GLAPI **)(const GLfloat *)) base->func;
		if (!strcmp(base->name, "glColor4fv"))
			__glbase_Color4fv = (void (GLAPI **)(const GLfloat *)) base->func;
		if (!strcmp(base->name, "glNormal3fv"))
			__glbase_Normal3fv = (void (GLAPI **)(const GLfloat *)) base->func;
		if (!strcmp(base->name, "glTexCoord4fv"))
			__glbase_TexCoord4fv = (void (GLAPI **)(const GLfloat *)) base->func;
		if (!strcmp(base->name, "glEdgeFlag"))
			__glbase_EdgeFlag = (void (GLAPI **)(GLboolean)) base->func;

		/* Assign State Function */
		for (p=st; p->name; p++) {
			if (!strcmp(base->name, p->name)) {
				APIst[i] = p->func;
#if 0
				/* Patch for allowing different ArrayElement call */
				if (!strcmp(base->name, "glArrayElement") &&
					c->resolve_arrays)
					APIst[i] = (GLfunc) __glstate_ArrayElement_resolve;
#endif
				break;
			}
		}

		/* Check Pack only list */
		for (j=0; packonly[j]; j++) {
			if (!strcmp(packonly[j], base->name)) {
				APIst[i] = APIpk[i];
				break;
			}
		}

		/* Check State only list */
		for (j=0; stateonly[j]; j++) {
			if (!strcmp(stateonly[j], base->name)) {
				APIpk[i] = APIst[i];
				break;
			}
		}

		/* ArrayElement (Unimplemented)
**		if (!strcmp(base->name, "ArrayElement")) {
**			APIarrayelementbase = base->func;
**		}
**/

		/* Update Vertex Pack APIs */
		for (j=0; vertexfuncs[j]; j++) {
			if (!strcmp(vertexfuncs[j], base->name)) {
				APIvertexbase[vertex_i] = base->func;
				APIvertexbase_st[vertex_i] = APIst+i;
				APIvertexpack[vertex_i] = APIpk[i];
				if (!c->vertexBBapi) {
					APIvertexBBpack[vertex_i] = APIpk[i];
				} else
					for (j=0; c->vertexBBapi[j].name; j++)
						if (!strcmp(c->vertexBBapi[j].name, base->name)) {
							APIvertexBBpack[vertex_i] = c->vertexBBapi[j].func;
							APIst[i]= c->vertexBBapi[j].func;  //By Default????
							break;
						}
				if (!c->vertexCapi) {
					APIvertexCpack[vertex_i] = APIpk[i];
				} else
					for (j=0; c->vertexCapi[j].name; j++)
						if (!strcmp(c->vertexCapi[j].name, base->name)) {
							APIvertexCpack[vertex_i] = c->vertexCapi[j].func;
							break;
						}
				vertex_i++;
				break;
			}
		}

		/* Update Begin End APIs */
		for (j=0; beginend[j]; j++) {
			if (!strcmp(beginend[j], base->name)) {
				APIbeginendbase[beginend_i] = base->func;
				APIbeginendpk[beginend_i] = APIpk[i];
				APIbeginendst[beginend_i] = APIst[i];
				beginend_i++;
				break;
			}
		}
	}

	/* All lists end in a null */
	APIbase[i] = NULL;
	APIpk[i] = NULL;
	APIst[i] = NULL;
	APIbeginendbase[beginend_i] = NULL;
	APIbeginendpk[beginend_i] = NULL;
	APIbeginendst[beginend_i] = NULL;
	APIvertexbase[vertex_i] = NULL;
	APIvertexbase_st[vertex_i] = NULL;
	APIvertexpack[vertex_i] = NULL;
	APIvertexBBpack[vertex_i] = NULL;
	APIvertexCpack[vertex_i] = NULL;
	APIcount = i;
}

#if defined(WINDOWS)

  #define SYSTEM_OPENGL_LIBRARY "opengl32.dll"

#elif defined(IRIX) || defined(IRIX64)

#if 0
  #if defined(_ABI64)
    #define SYSTEM_OPENGL_LIBRARY "/usr/lib64/libGL.so"
  #elif defined(_ABIN32)
    #define SYSTEM_OPENGL_LIBRARY "/usr/lib32/libGL.so"
  #else
    #define SYSTEM_OPENGL_LIBRARY "/usr/lib/libGL.so"
  #endif
#else
	#define SYSTEM_OPENGL_LIBRARY "libGL.so"
#endif

#elif defined(Linux)

#if 0
  #define SYSTEM_OPENGL_LIBRARY "/usr/lib/libGL.so"
#else
	/* NULL (no filename) means open existing program namespace, which
       is precisely what we want */
	#define SYSTEM_OPENGL_LIBRARY NULL
#endif

#elif defined(AIX)
#define SYSTEM_OPENGL_LIBRARY /usr/lib/libGL.a

#else

  #error Where is the system OpenGL library?

#endif

void __glhw_LoadSystemApi (GLapi *api) {
	int i,j;
#if defined(WINDOWS)
	HMODULE hinstLib;
#else
	void *hinstLib;
#endif

	for (i=0; api[i].func; i++)
		api[i].func = NULL;

#if defined(WINDOWS)

#if 0
	if (!getenv ("WIREGL_USE_LOCAL_GL")) {
		char name[MAX_PATH];
		GetSystemDirectory(name, sizeof(name));
		strcat( name, "\\" );
		strcat( name, SYSTEM_OPENGL_LIBRARY );
		hinstLib = LoadLibrary( name );
	} else {
		hinstLib = LoadLibrary( SYSTEM_OPENGL_LIBRARY );
	}
#endif
	hinstLib = LoadLibrary( SYSTEM_OPENGL_LIBRARY );

	for (i=0; api[i].name; i++)
		api[i].func = (GLfunc) GetProcAddress(hinstLib, api[i].name);

#else
	/* NULL (no filename) means open existing program namespace, which
       is precisely what we want */
	hinstLib = dlopen( NULL, RTLD_LAZY );
	for (i=0; api[i].name; i++) {
		api[i].func = (GLfunc) dlsym(hinstLib, api[i].name);
	}
#endif
	for ( i = 0; api[i].name; i++ ) {
		if ( api[i].func == NULL ) {
			fprintf( stderr, "*** WARNING! Couldn't find \"%s\", "
					 "using a NOP instead!\n", api[i].name );
		}
	}
	
	for (i=0; api[i].name; i++)
		if (api[i].func == NULL)
			for (j=0; nop_api[j].func; j++)
				if (!strcmp (api[i].name, nop_api[j].name)) {
					api[i].func = nop_api[j].func;
					break;
				}
}

void __glhw_LoadSwitchApi (GLapi *api) {
	int i,j;

	if (!api) {
		__glhw_LoadSystemApi(system_api);
		api = system_api;
	}

	/* Initialize everybody to NULL */
	for (i=0; hw_api[i].name; i++)
		*((GLfunc *) hw_api[i].func) = NULL;

	/* Load all api */
	for (i=0; hw_api[i].name; i++)
		for (j=0; api[j].name; j++)
			if (!strcmp (hw_api[i].name, api[j].name)) {
				*((GLfunc *) hw_api[i].func) = api[j].func;
				break;
			}

	/* Load the remaining nops */
	for (i=0; hw_api[i].name; i++)
		if (*((GLfunc *) hw_api[i].func) == NULL)
			for (j=0; nop_api[j].func; j++)
				if (!strcmp (hw_api[i].name, nop_api[j].name)) {
					*((GLfunc *) hw_api[i].func) = nop_api[j].func;
					break;
				}
}

void __glhw_error() {
	__glerror(__LINE__, __FILE__, GL_NO_ERROR,
		"Function API no implemented!");
	exit(1);
}

/* LoadVertex*Funcs functions */
const char *vertexfuncs[] = {
	"glVertex2d",
	"glVertex2dv",
	"glVertex2f",
	"glVertex2fv",
	"glVertex2i",
	"glVertex2iv",
	"glVertex2s",
	"glVertex2sv",
	"glVertex3d",
	"glVertex3dv",
	"glVertex3f",
	"glVertex3fv",
	"glVertex3i",
	"glVertex3iv",
	"glVertex3s",
	"glVertex3sv",
	"glVertex4d",
	"glVertex4dv",
	"glVertex4f",
	"glVertex4fv",
	"glVertex4i",
	"glVertex4iv",
	"glVertex4s",
	"glVertex4sv",
	NULL
};

/* Functions which change from the 
** state to the pack mode in a 
** Begin End block
*/
const char *beginend[] = {
	"glColor3b",
	"glColor3bv",
	"glColor3d",
	"glColor3dv",
	"glColor3f",
	"glColor3fv",
	"glColor3i",
	"glColor3iv",
	"glColor3s",
	"glColor3sv",
	"glColor3ub",
	"glColor3ubv",
	"glColor3ui",
	"glColor3uiv",
	"glColor3us",
	"glColor3usv",
	"glColor4b",
	"glColor4bv",
	"glColor4d",
	"glColor4dv",
	"glColor4f",
	"glColor4fv",
	"glColor4i",
	"glColor4iv",
	"glColor4s",
	"glColor4sv",
	"glColor4ub",
	"glColor4ubv",
	"glColor4ui",
	"glColor4uiv",
	"glColor4us",
	"glColor4usv",
	NULL
};

/* Functions which are always packed
*/
const char *packonly[] = {
	"glBarrierCreate",
	"glBarrierDestroy",
	"glBarrierExec",
	"glCopyPixels",
	"glColor3b",
	"glColor3bv",
	"glColor3d",
	"glColor3dv",
	"glColor3f",
	"glColor3fv",
	"glColor3i",
	"glColor3iv",
	"glColor3s",
	"glColor3sv",
	"glColor3ub",
	"glColor3ubv",
	"glColor3ui",
	"glColor3uiv",
	"glColor3us",
	"glColor3usv",
	"glColor4b",
	"glColor4bv",
	"glColor4d",
	"glColor4dv",
	"glColor4f",
	"glColor4fv",
	"glColor4i",
	"glColor4iv",
	"glColor4s",
	"glColor4sv",
	"glColor4ub",
	"glColor4ubv",
	"glColor4ui",
	"glColor4uiv",
	"glColor4us",
	"glColor4usv",
	"glEdgeFlag",
	"glEdgeFlagv",
	"glEvalCoord1d",
	"glEvalCoord1dv",
	"glEvalCoord1f",
	"glEvalCoord1fv",
	"glEvalCoord2d",
	"glEvalCoord2dv",
	"glEvalCoord2f",
	"glEvalCoord2fv",
	"glEvalPoint1",
	"glEvalPoint2",
	"glFinish",
	"glFlush",
	"glIndexd",
	"glIndexdv",
	"glIndexf",
	"glIndexfv",
	"glIndexi",
	"glIndexiv",
	"glIndexs",
	"glIndexsv",
	"glIndexub",
	"glIndexubv",
#if 0
    /* These have been removed so that the state tracker
    ** can track material properties
    ** Warning, very slow if setting Material properties per 
    ** vertex.  
    ** Need pinch code for glMaterial
    */
	"glMaterialf",
	"glMaterialfv",
	"glMateriali",
	"glMaterialiv",
#endif
    "glNormal3b",
	"glNormal3bv",
	"glNormal3d",
	"glNormal3dv",
	"glNormal3f",
	"glNormal3fv",
	"glNormal3i",
	"glNormal3iv",
	"glNormal3s",
	"glNormal3sv",
	"glReadPixels",
	"glSemaphoreCreate",
	"glSemaphoreDestroy",
	"glSemaphoreP",
	"glSemaphoreV",
	"glTexCoord1d",
	"glTexCoord1dv",
	"glTexCoord1f",
	"glTexCoord1fv",
	"glTexCoord1i",
	"glTexCoord1iv",
	"glTexCoord1s",
	"glTexCoord1sv",
	"glTexCoord2d",
	"glTexCoord2dv",
	"glTexCoord2f",
	"glTexCoord2fv",
	"glTexCoord2i",
	"glTexCoord2iv",
	"glTexCoord2s",
	"glTexCoord2sv",
	"glTexCoord3d",
	"glTexCoord3dv",
	"glTexCoord3f",
	"glTexCoord3fv",
	"glTexCoord3i",
	"glTexCoord3iv",
	"glTexCoord3s",
	"glTexCoord3sv",
	"glTexCoord4d",
	"glTexCoord4dv",
	"glTexCoord4f",
	"glTexCoord4fv",
	"glTexCoord4i",
	"glTexCoord4iv",
	"glTexCoord4s",
	"glTexCoord4sv",
	NULL
};

/* Functions that don't go in a display list */
const char *stateonly[] = {
	"glNewList",
	"glEndList",
	"glAreTexturesResident",
	"glColorPointer",
	"glDeleteLists",
	"glDeleteTextures",
	"glDisableClientState",
	"glEdgeFlagPointer",
	"glEnableClientState",
	"glFeedbackBuffer",
	"glGenLists",
	"glGenTextures",
	"glGetBooleanv",
	"glGetClipPlane",
	"glGetDoublev",
	"glGetError",
	"glGetFloatv",
	"glGetIntegerv",
	"glGetLightfv",
	"glGetLightiv",
	"glGetMapdv",
	"glGetMapfv",
	"glGetMapiv",
	"glGetMaterialfv",
	"glGetMaterialiv",
	"glGetPixelMapfv",
	"glGetPixelMapuiv",
	"glGetPixelMapusv",
	"glGetPointerv",
	"glGetPolygonStipple",
	"glGetString",
	"glGetTexEnvfv",
	"glGetTexEnviv",
	"glGetTexGendv",
	"glGetTexGenfv",
	"glGetTexGeniv",
	"glGetTexImage",
	"glGetTexLevelParameterfv",
	"glGetTexLevelParameteriv",
	"glGetTexParameterfv",
	"glGetTexParameteriv",
	"glHint",
	"glIndexPointer",
	"glInterleavedArrays",
	"glIsEnabled",
	"glIsList",
	"glIsTexture",
	"glNormalPointer",
	"glPixelStoref",
	"glPixelStorei",
	"glPopClientAttrib",
	"glPushClientAttrib",
	"glReadPixels",
	"glRenderMode",
	"glSelectBuffer",
	"glTexCoordPointer",
	"glVertexPointer",
	NULL
};

/* API for state switching */
GLjumptable hw_api[] = {
	{"glAccum", (GLfunc *) &__glhw_Accum},
	{"glAlphaFunc", (GLfunc *) &__glhw_AlphaFunc},
	{"glAreTexturesResident", (GLfunc *) &__glhw_AreTexturesResident},
	{"glArrayElement", (GLfunc *) &__glhw_ArrayElement},
	{"glBegin", (GLfunc *) &__glhw_Begin},
	{"glBindTexture", (GLfunc *) &__glhw_BindTexture},
	{"glBitmap", (GLfunc *) &__glhw_Bitmap},
	{"glBlendFunc", (GLfunc *) &__glhw_BlendFunc},
	{"glCallList", (GLfunc *) &__glhw_CallList},
	{"glCallLists", (GLfunc *) &__glhw_CallLists},
	{"glClear", (GLfunc *) &__glhw_Clear},
	{"glClearAccum", (GLfunc *) &__glhw_ClearAccum},
	{"glClearColor", (GLfunc *) &__glhw_ClearColor},
	{"glClearDepth", (GLfunc *) &__glhw_ClearDepth},
	{"glClearIndex", (GLfunc *) &__glhw_ClearIndex},
	{"glClearStencil", (GLfunc *) &__glhw_ClearStencil},
	{"glClipPlane", (GLfunc *) &__glhw_ClipPlane},
	{"glColor3b", (GLfunc *) &__glhw_Color3b},
	{"glColor3bv", (GLfunc *) &__glhw_Color3bv},
	{"glColor3d", (GLfunc *) &__glhw_Color3d},
	{"glColor3dv", (GLfunc *) &__glhw_Color3dv},
	{"glColor3f", (GLfunc *) &__glhw_Color3f},
	{"glColor3fv", (GLfunc *) &__glhw_Color3fv},
	{"glColor3i", (GLfunc *) &__glhw_Color3i},
	{"glColor3iv", (GLfunc *) &__glhw_Color3iv},
	{"glColor3s", (GLfunc *) &__glhw_Color3s},
	{"glColor3sv", (GLfunc *) &__glhw_Color3sv},
	{"glColor3ub", (GLfunc *) &__glhw_Color3ub},
	{"glColor3ubv", (GLfunc *) &__glhw_Color3ubv},
	{"glColor3ui", (GLfunc *) &__glhw_Color3ui},
	{"glColor3uiv", (GLfunc *) &__glhw_Color3uiv},
	{"glColor3us", (GLfunc *) &__glhw_Color3us},
	{"glColor3usv", (GLfunc *) &__glhw_Color3usv},
	{"glColor4b", (GLfunc *) &__glhw_Color4b},
	{"glColor4bv", (GLfunc *) &__glhw_Color4bv},
	{"glColor4d", (GLfunc *) &__glhw_Color4d},
	{"glColor4dv", (GLfunc *) &__glhw_Color4dv},
	{"glColor4f", (GLfunc *) &__glhw_Color4f},
	{"glColor4fv", (GLfunc *) &__glhw_Color4fv},
	{"glColor4i", (GLfunc *) &__glhw_Color4i},
	{"glColor4iv", (GLfunc *) &__glhw_Color4iv},
	{"glColor4s", (GLfunc *) &__glhw_Color4s},
	{"glColor4sv", (GLfunc *) &__glhw_Color4sv},
	{"glColor4ub", (GLfunc *) &__glhw_Color4ub},
	{"glColor4ubv", (GLfunc *) &__glhw_Color4ubv},
	{"glColor4ui", (GLfunc *) &__glhw_Color4ui},
	{"glColor4uiv", (GLfunc *) &__glhw_Color4uiv},
	{"glColor4us", (GLfunc *) &__glhw_Color4us},
	{"glColor4usv", (GLfunc *) &__glhw_Color4usv},
	{"glColorMask", (GLfunc *) &__glhw_ColorMask},
	{"glColorMaterial", (GLfunc *) &__glhw_ColorMaterial},
	{"glColorPointer", (GLfunc *) &__glhw_ColorPointer},
	{"glCopyPixels", (GLfunc *) &__glhw_CopyPixels},
	{"glCopyTexImage1D", (GLfunc *) &__glhw_CopyTexImage1D},
	{"glCopyTexImage2D", (GLfunc *) &__glhw_CopyTexImage2D},
	{"glCopyTexSubImage1D", (GLfunc *) &__glhw_CopyTexSubImage1D},
	{"glCopyTexSubImage2D", (GLfunc *) &__glhw_CopyTexSubImage2D},
	{"glCullFace", (GLfunc *) &__glhw_CullFace},
	{"glDeleteLists", (GLfunc *) &__glhw_DeleteLists},
	{"glDeleteTextures", (GLfunc *) &__glhw_DeleteTextures},
	{"glDepthFunc", (GLfunc *) &__glhw_DepthFunc},
	{"glDepthMask", (GLfunc *) &__glhw_DepthMask},
	{"glDepthRange", (GLfunc *) &__glhw_DepthRange},
	{"glDisable", (GLfunc *) &__glhw_Disable},
	{"glDisableClientState", (GLfunc *) &__glhw_DisableClientState},
	{"glDrawArrays", (GLfunc *) &__glhw_DrawArrays},
	{"glDrawBuffer", (GLfunc *) &__glhw_DrawBuffer},
	{"glDrawElements", (GLfunc *) &__glhw_DrawElements},
	{"glDrawPixels", (GLfunc *) &__glhw_DrawPixels},
	{"glEdgeFlag", (GLfunc *) &__glhw_EdgeFlag},
	{"glEdgeFlagPointer", (GLfunc *) &__glhw_EdgeFlagPointer},
	{"glEdgeFlagv", (GLfunc *) &__glhw_EdgeFlagv},
	{"glEnable", (GLfunc *) &__glhw_Enable},
	{"glEnableClientState", (GLfunc *) &__glhw_EnableClientState},
	{"glEnd", (GLfunc *) &__glhw_End},
	{"glEndList", (GLfunc *) &__glhw_EndList},
	{"glEvalCoord1d", (GLfunc *) &__glhw_EvalCoord1d},
	{"glEvalCoord1dv", (GLfunc *) &__glhw_EvalCoord1dv},
	{"glEvalCoord1f", (GLfunc *) &__glhw_EvalCoord1f},
	{"glEvalCoord1fv", (GLfunc *) &__glhw_EvalCoord1fv},
	{"glEvalCoord2d", (GLfunc *) &__glhw_EvalCoord2d},
	{"glEvalCoord2dv", (GLfunc *) &__glhw_EvalCoord2dv},
	{"glEvalCoord2f", (GLfunc *) &__glhw_EvalCoord2f},
	{"glEvalCoord2fv", (GLfunc *) &__glhw_EvalCoord2fv},
	{"glEvalMesh1", (GLfunc *) &__glhw_EvalMesh1},
	{"glEvalMesh2", (GLfunc *) &__glhw_EvalMesh2},
	{"glEvalPoint1", (GLfunc *) &__glhw_EvalPoint1},
	{"glEvalPoint2", (GLfunc *) &__glhw_EvalPoint2},
	{"glFeedbackBuffer", (GLfunc *) &__glhw_FeedbackBuffer},
	{"glFinish", (GLfunc *) &__glhw_Finish},
	{"glFlush", (GLfunc *) &__glhw_Flush},
	{"glFogf", (GLfunc *) &__glhw_Fogf},
	{"glFogfv", (GLfunc *) &__glhw_Fogfv},
	{"glFogi", (GLfunc *) &__glhw_Fogi},
	{"glFogiv", (GLfunc *) &__glhw_Fogiv},
	{"glFrontFace", (GLfunc *) &__glhw_FrontFace},
	{"glFrustum", (GLfunc *) &__glhw_Frustum},
	{"glGenLists", (GLfunc *) &__glhw_GenLists},
	{"glGenTextures", (GLfunc *) &__glhw_GenTextures},
	{"glGetBooleanv", (GLfunc *) &__glhw_GetBooleanv},
	{"glGetClipPlane", (GLfunc *) &__glhw_GetClipPlane},
	{"glGetDoublev", (GLfunc *) &__glhw_GetDoublev},
	{"glGetError", (GLfunc *) &__glhw_GetError},
	{"glGetFloatv", (GLfunc *) &__glhw_GetFloatv},
	{"glGetIntegerv", (GLfunc *) &__glhw_GetIntegerv},
	{"glGetLightfv", (GLfunc *) &__glhw_GetLightfv},
	{"glGetLightiv", (GLfunc *) &__glhw_GetLightiv},
	{"glGetMapdv", (GLfunc *) &__glhw_GetMapdv},
	{"glGetMapfv", (GLfunc *) &__glhw_GetMapfv},
	{"glGetMapiv", (GLfunc *) &__glhw_GetMapiv},
	{"glGetMaterialfv", (GLfunc *) &__glhw_GetMaterialfv},
	{"glGetMaterialiv", (GLfunc *) &__glhw_GetMaterialiv},
	{"glGetPixelMapfv", (GLfunc *) &__glhw_GetPixelMapfv},
	{"glGetPixelMapuiv", (GLfunc *) &__glhw_GetPixelMapuiv},
	{"glGetPixelMapusv", (GLfunc *) &__glhw_GetPixelMapusv},
	{"glGetPointerv", (GLfunc *) &__glhw_GetPointerv},
	{"glGetPolygonStipple", (GLfunc *) &__glhw_GetPolygonStipple},
	{"glGetString", (GLfunc *) &__glhw_GetString},
	{"glGetTexEnvfv", (GLfunc *) &__glhw_GetTexEnvfv},
	{"glGetTexEnviv", (GLfunc *) &__glhw_GetTexEnviv},
	{"glGetTexGendv", (GLfunc *) &__glhw_GetTexGendv},
	{"glGetTexGenfv", (GLfunc *) &__glhw_GetTexGenfv},
	{"glGetTexGeniv", (GLfunc *) &__glhw_GetTexGeniv},
	{"glGetTexImage", (GLfunc *) &__glhw_GetTexImage},
	{"glGetTexLevelParameterfv", (GLfunc *) &__glhw_GetTexLevelParameterfv},
	{"glGetTexLevelParameteriv", (GLfunc *) &__glhw_GetTexLevelParameteriv},
	{"glGetTexParameterfv", (GLfunc *) &__glhw_GetTexParameterfv},
	{"glGetTexParameteriv", (GLfunc *) &__glhw_GetTexParameteriv},
	{"glHint", (GLfunc *) &__glhw_Hint},
	{"glIndexMask", (GLfunc *) &__glhw_IndexMask},
	{"glIndexPointer", (GLfunc *) &__glhw_IndexPointer},
	{"glIndexd", (GLfunc *) &__glhw_Indexd},
	{"glIndexdv", (GLfunc *) &__glhw_Indexdv},
	{"glIndexf", (GLfunc *) &__glhw_Indexf},
	{"glIndexfv", (GLfunc *) &__glhw_Indexfv},
	{"glIndexi", (GLfunc *) &__glhw_Indexi},
	{"glIndexiv", (GLfunc *) &__glhw_Indexiv},
	{"glIndexs", (GLfunc *) &__glhw_Indexs},
	{"glIndexsv", (GLfunc *) &__glhw_Indexsv},
	{"glIndexub", (GLfunc *) &__glhw_Indexub},
	{"glIndexubv", (GLfunc *) &__glhw_Indexubv},
	{"glInitNames", (GLfunc *) &__glhw_InitNames},
	{"glInterleavedArrays", (GLfunc *) &__glhw_InterleavedArrays},
	{"glIsEnabled", (GLfunc *) &__glhw_IsEnabled},
	{"glIsList", (GLfunc *) &__glhw_IsList},
	{"glIsTexture", (GLfunc *) &__glhw_IsTexture},
	{"glLightModelf", (GLfunc *) &__glhw_LightModelf},
	{"glLightModelfv", (GLfunc *) &__glhw_LightModelfv},
	{"glLightModeli", (GLfunc *) &__glhw_LightModeli},
	{"glLightModeliv", (GLfunc *) &__glhw_LightModeliv},
	{"glLightf", (GLfunc *) &__glhw_Lightf},
	{"glLightfv", (GLfunc *) &__glhw_Lightfv},
	{"glLighti", (GLfunc *) &__glhw_Lighti},
	{"glLightiv", (GLfunc *) &__glhw_Lightiv},
	{"glLineStipple", (GLfunc *) &__glhw_LineStipple},
	{"glLineWidth", (GLfunc *) &__glhw_LineWidth},
	{"glListBase", (GLfunc *) &__glhw_ListBase},
	{"glLoadIdentity", (GLfunc *) &__glhw_LoadIdentity},
	{"glLoadMatrixd", (GLfunc *) &__glhw_LoadMatrixd},
	{"glLoadMatrixf", (GLfunc *) &__glhw_LoadMatrixf},
	{"glLoadName", (GLfunc *) &__glhw_LoadName},
	{"glLogicOp", (GLfunc *) &__glhw_LogicOp},
	{"glMap1d", (GLfunc *) &__glhw_Map1d},
	{"glMap1f", (GLfunc *) &__glhw_Map1f},
	{"glMap2d", (GLfunc *) &__glhw_Map2d},
	{"glMap2f", (GLfunc *) &__glhw_Map2f},
	{"glMapGrid1d", (GLfunc *) &__glhw_MapGrid1d},
	{"glMapGrid1f", (GLfunc *) &__glhw_MapGrid1f},
	{"glMapGrid2d", (GLfunc *) &__glhw_MapGrid2d},
	{"glMapGrid2f", (GLfunc *) &__glhw_MapGrid2f},
	{"glMaterialf", (GLfunc *) &__glhw_Materialf},
	{"glMaterialfv", (GLfunc *) &__glhw_Materialfv},
	{"glMateriali", (GLfunc *) &__glhw_Materiali},
	{"glMaterialiv", (GLfunc *) &__glhw_Materialiv},
	{"glMatrixMode", (GLfunc *) &__glhw_MatrixMode},
	{"glMultMatrixd", (GLfunc *) &__glhw_MultMatrixd},
	{"glMultMatrixf", (GLfunc *) &__glhw_MultMatrixf},
	{"glNewList", (GLfunc *) &__glhw_NewList},
	{"glNormal3b", (GLfunc *) &__glhw_Normal3b},
	{"glNormal3bv", (GLfunc *) &__glhw_Normal3bv},
	{"glNormal3d", (GLfunc *) &__glhw_Normal3d},
	{"glNormal3dv", (GLfunc *) &__glhw_Normal3dv},
	{"glNormal3f", (GLfunc *) &__glhw_Normal3f},
	{"glNormal3fv", (GLfunc *) &__glhw_Normal3fv},
	{"glNormal3i", (GLfunc *) &__glhw_Normal3i},
	{"glNormal3iv", (GLfunc *) &__glhw_Normal3iv},
	{"glNormal3s", (GLfunc *) &__glhw_Normal3s},
	{"glNormal3sv", (GLfunc *) &__glhw_Normal3sv},
	{"glNormalPointer", (GLfunc *) &__glhw_NormalPointer},
	{"glOrtho", (GLfunc *) &__glhw_Ortho},
	{"glPassThrough", (GLfunc *) &__glhw_PassThrough},
	{"glPixelMapfv", (GLfunc *) &__glhw_PixelMapfv},
	{"glPixelMapuiv", (GLfunc *) &__glhw_PixelMapuiv},
	{"glPixelMapusv", (GLfunc *) &__glhw_PixelMapusv},
	{"glPixelStoref", (GLfunc *) &__glhw_PixelStoref},
	{"glPixelStorei", (GLfunc *) &__glhw_PixelStorei},
	{"glPixelTransferf", (GLfunc *) &__glhw_PixelTransferf},
	{"glPixelTransferi", (GLfunc *) &__glhw_PixelTransferi},
	{"glPixelZoom", (GLfunc *) &__glhw_PixelZoom},
	{"glPointSize", (GLfunc *) &__glhw_PointSize},
	{"glPolygonMode", (GLfunc *) &__glhw_PolygonMode},
	{"glPolygonOffset", (GLfunc *) &__glhw_PolygonOffset},
	{"glPolygonStipple", (GLfunc *) &__glhw_PolygonStipple},
	{"glPopAttrib", (GLfunc *) &__glhw_PopAttrib},
	{"glPopClientAttrib", (GLfunc *) &__glhw_PopClientAttrib},
	{"glPopMatrix", (GLfunc *) &__glhw_PopMatrix},
	{"glPopName", (GLfunc *) &__glhw_PopName},
	{"glPrioritizeTextures", (GLfunc *) &__glhw_PrioritizeTextures},
	{"glPushAttrib", (GLfunc *) &__glhw_PushAttrib},
	{"glPushClientAttrib", (GLfunc *) &__glhw_PushClientAttrib},
	{"glPushMatrix", (GLfunc *) &__glhw_PushMatrix},
	{"glPushName", (GLfunc *) &__glhw_PushName},
	{"glRasterPos2d", (GLfunc *) &__glhw_RasterPos2d},
	{"glRasterPos2dv", (GLfunc *) &__glhw_RasterPos2dv},
	{"glRasterPos2f", (GLfunc *) &__glhw_RasterPos2f},
	{"glRasterPos2fv", (GLfunc *) &__glhw_RasterPos2fv},
	{"glRasterPos2i", (GLfunc *) &__glhw_RasterPos2i},
	{"glRasterPos2iv", (GLfunc *) &__glhw_RasterPos2iv},
	{"glRasterPos2s", (GLfunc *) &__glhw_RasterPos2s},
	{"glRasterPos2sv", (GLfunc *) &__glhw_RasterPos2sv},
	{"glRasterPos3d", (GLfunc *) &__glhw_RasterPos3d},
	{"glRasterPos3dv", (GLfunc *) &__glhw_RasterPos3dv},
	{"glRasterPos3f", (GLfunc *) &__glhw_RasterPos3f},
	{"glRasterPos3fv", (GLfunc *) &__glhw_RasterPos3fv},
	{"glRasterPos3i", (GLfunc *) &__glhw_RasterPos3i},
	{"glRasterPos3iv", (GLfunc *) &__glhw_RasterPos3iv},
	{"glRasterPos3s", (GLfunc *) &__glhw_RasterPos3s},
	{"glRasterPos3sv", (GLfunc *) &__glhw_RasterPos3sv},
	{"glRasterPos4d", (GLfunc *) &__glhw_RasterPos4d},
	{"glRasterPos4dv", (GLfunc *) &__glhw_RasterPos4dv},
	{"glRasterPos4f", (GLfunc *) &__glhw_RasterPos4f},
	{"glRasterPos4fv", (GLfunc *) &__glhw_RasterPos4fv},
	{"glRasterPos4i", (GLfunc *) &__glhw_RasterPos4i},
	{"glRasterPos4iv", (GLfunc *) &__glhw_RasterPos4iv},
	{"glRasterPos4s", (GLfunc *) &__glhw_RasterPos4s},
	{"glRasterPos4sv", (GLfunc *) &__glhw_RasterPos4sv},
	{"glReadBuffer", (GLfunc *) &__glhw_ReadBuffer},
	{"glReadPixels", (GLfunc *) &__glhw_ReadPixels},
	{"glRectd", (GLfunc *) &__glhw_Rectd},
	{"glRectdv", (GLfunc *) &__glhw_Rectdv},
	{"glRectf", (GLfunc *) &__glhw_Rectf},
	{"glRectfv", (GLfunc *) &__glhw_Rectfv},
	{"glRecti", (GLfunc *) &__glhw_Recti},
	{"glRectiv", (GLfunc *) &__glhw_Rectiv},
	{"glRects", (GLfunc *) &__glhw_Rects},
	{"glRectsv", (GLfunc *) &__glhw_Rectsv},
	{"glRenderMode", (GLfunc *) &__glhw_RenderMode},
	{"glRotated", (GLfunc *) &__glhw_Rotated},
	{"glRotatef", (GLfunc *) &__glhw_Rotatef},
	{"glScaled", (GLfunc *) &__glhw_Scaled},
	{"glScalef", (GLfunc *) &__glhw_Scalef},
	{"glScissor", (GLfunc *) &__glhw_Scissor},
	{"glSelectBuffer", (GLfunc *) &__glhw_SelectBuffer},
	{"glShadeModel", (GLfunc *) &__glhw_ShadeModel},
	{"glStencilFunc", (GLfunc *) &__glhw_StencilFunc},
	{"glStencilMask", (GLfunc *) &__glhw_StencilMask},
	{"glStencilOp", (GLfunc *) &__glhw_StencilOp},
	{"glTexCoord1d", (GLfunc *) &__glhw_TexCoord1d},
	{"glTexCoord1dv", (GLfunc *) &__glhw_TexCoord1dv},
	{"glTexCoord1f", (GLfunc *) &__glhw_TexCoord1f},
	{"glTexCoord1fv", (GLfunc *) &__glhw_TexCoord1fv},
	{"glTexCoord1i", (GLfunc *) &__glhw_TexCoord1i},
	{"glTexCoord1iv", (GLfunc *) &__glhw_TexCoord1iv},
	{"glTexCoord1s", (GLfunc *) &__glhw_TexCoord1s},
	{"glTexCoord1sv", (GLfunc *) &__glhw_TexCoord1sv},
	{"glTexCoord2d", (GLfunc *) &__glhw_TexCoord2d},
	{"glTexCoord2dv", (GLfunc *) &__glhw_TexCoord2dv},
	{"glTexCoord2f", (GLfunc *) &__glhw_TexCoord2f},
	{"glTexCoord2fv", (GLfunc *) &__glhw_TexCoord2fv},
	{"glTexCoord2i", (GLfunc *) &__glhw_TexCoord2i},
	{"glTexCoord2iv", (GLfunc *) &__glhw_TexCoord2iv},
	{"glTexCoord2s", (GLfunc *) &__glhw_TexCoord2s},
	{"glTexCoord2sv", (GLfunc *) &__glhw_TexCoord2sv},
	{"glTexCoord3d", (GLfunc *) &__glhw_TexCoord3d},
	{"glTexCoord3dv", (GLfunc *) &__glhw_TexCoord3dv},
	{"glTexCoord3f", (GLfunc *) &__glhw_TexCoord3f},
	{"glTexCoord3fv", (GLfunc *) &__glhw_TexCoord3fv},
	{"glTexCoord3i", (GLfunc *) &__glhw_TexCoord3i},
	{"glTexCoord3iv", (GLfunc *) &__glhw_TexCoord3iv},
	{"glTexCoord3s", (GLfunc *) &__glhw_TexCoord3s},
	{"glTexCoord3sv", (GLfunc *) &__glhw_TexCoord3sv},
	{"glTexCoord4d", (GLfunc *) &__glhw_TexCoord4d},
	{"glTexCoord4dv", (GLfunc *) &__glhw_TexCoord4dv},
	{"glTexCoord4f", (GLfunc *) &__glhw_TexCoord4f},
	{"glTexCoord4fv", (GLfunc *) &__glhw_TexCoord4fv},
	{"glTexCoord4i", (GLfunc *) &__glhw_TexCoord4i},
	{"glTexCoord4iv", (GLfunc *) &__glhw_TexCoord4iv},
	{"glTexCoord4s", (GLfunc *) &__glhw_TexCoord4s},
	{"glTexCoord4sv", (GLfunc *) &__glhw_TexCoord4sv},
	{"glTexCoordPointer", (GLfunc *) &__glhw_TexCoordPointer},
	{"glTexEnvf", (GLfunc *) &__glhw_TexEnvf},
	{"glTexEnvfv", (GLfunc *) &__glhw_TexEnvfv},
	{"glTexEnvi", (GLfunc *) &__glhw_TexEnvi},
	{"glTexEnviv", (GLfunc *) &__glhw_TexEnviv},
	{"glTexGend", (GLfunc *) &__glhw_TexGend},
	{"glTexGendv", (GLfunc *) &__glhw_TexGendv},
	{"glTexGenf", (GLfunc *) &__glhw_TexGenf},
	{"glTexGenfv", (GLfunc *) &__glhw_TexGenfv},
	{"glTexGeni", (GLfunc *) &__glhw_TexGeni},
	{"glTexGeniv", (GLfunc *) &__glhw_TexGeniv},
	{"glTexImage1D", (GLfunc *) &__glhw_TexImage1D},
	{"glTexImage2D", (GLfunc *) &__glhw_TexImage2D},
	{"glTexParameterf", (GLfunc *) &__glhw_TexParameterf},
	{"glTexParameterfv", (GLfunc *) &__glhw_TexParameterfv},
	{"glTexParameteri", (GLfunc *) &__glhw_TexParameteri},
	{"glTexParameteriv", (GLfunc *) &__glhw_TexParameteriv},
	{"glTexSubImage1D", (GLfunc *) &__glhw_TexSubImage1D},
	{"glTexSubImage2D", (GLfunc *) &__glhw_TexSubImage2D},
	{"glTranslated", (GLfunc *) &__glhw_Translated},
	{"glTranslatef", (GLfunc *) &__glhw_Translatef},
	{"glVertex2d", (GLfunc *) &__glhw_Vertex2d},
	{"glVertex2dv", (GLfunc *) &__glhw_Vertex2dv},
	{"glVertex2f", (GLfunc *) &__glhw_Vertex2f},
	{"glVertex2fv", (GLfunc *) &__glhw_Vertex2fv},
	{"glVertex2i", (GLfunc *) &__glhw_Vertex2i},
	{"glVertex2iv", (GLfunc *) &__glhw_Vertex2iv},
	{"glVertex2s", (GLfunc *) &__glhw_Vertex2s},
	{"glVertex2sv", (GLfunc *) &__glhw_Vertex2sv},
	{"glVertex3d", (GLfunc *) &__glhw_Vertex3d},
	{"glVertex3dv", (GLfunc *) &__glhw_Vertex3dv},
	{"glVertex3f", (GLfunc *) &__glhw_Vertex3f},
	{"glVertex3fv", (GLfunc *) &__glhw_Vertex3fv},
	{"glVertex3i", (GLfunc *) &__glhw_Vertex3i},
	{"glVertex3iv", (GLfunc *) &__glhw_Vertex3iv},
	{"glVertex3s", (GLfunc *) &__glhw_Vertex3s},
	{"glVertex3sv", (GLfunc *) &__glhw_Vertex3sv},
	{"glVertex4d", (GLfunc *) &__glhw_Vertex4d},
	{"glVertex4dv", (GLfunc *) &__glhw_Vertex4dv},
	{"glVertex4f", (GLfunc *) &__glhw_Vertex4f},
	{"glVertex4fv", (GLfunc *) &__glhw_Vertex4fv},
	{"glVertex4i", (GLfunc *) &__glhw_Vertex4i},
	{"glVertex4iv", (GLfunc *) &__glhw_Vertex4iv},
	{"glVertex4s", (GLfunc *) &__glhw_Vertex4s},
	{"glVertex4sv", (GLfunc *) &__glhw_Vertex4sv},
	{"glVertexPointer", (GLfunc *) &__glhw_VertexPointer},
	{"glViewport", (GLfunc *) &__glhw_Viewport},
	{NULL, NULL}
};

/* API for system opengl */
GLapi system_api[] = {
	{ "glAccum", NULL},
	{ "glAlphaFunc", NULL},
	{ "glAreTexturesResident", NULL},
	{ "glArrayElement", NULL},
	{ "glBegin", NULL},
	{ "glBindTexture", NULL},
	{ "glBitmap", NULL},
	{ "glBlendFunc", NULL},
	{ "glCallList", NULL},
	{ "glCallLists", NULL},
	{ "glClear", NULL},
	{ "glClearAccum", NULL},
	{ "glClearColor", NULL},
	{ "glClearDepth", NULL},
	{ "glClearIndex", NULL},
	{ "glClearStencil", NULL},
	{ "glClipPlane", NULL},
	{ "glColor3b", NULL},
	{ "glColor3bv", NULL},
	{ "glColor3d", NULL},
	{ "glColor3dv", NULL},
	{ "glColor3f", NULL},
	{ "glColor3fv", NULL},
	{ "glColor3i", NULL},
	{ "glColor3iv", NULL},
	{ "glColor3s", NULL},
	{ "glColor3sv", NULL},
	{ "glColor3ub", NULL},
	{ "glColor3ubv", NULL},
	{ "glColor3ui", NULL},
	{ "glColor3uiv", NULL},
	{ "glColor3us", NULL},
	{ "glColor3usv", NULL},
	{ "glColor4b", NULL},
	{ "glColor4bv", NULL},
	{ "glColor4d", NULL},
	{ "glColor4dv", NULL},
	{ "glColor4f", NULL},
	{ "glColor4fv", NULL},
	{ "glColor4i", NULL},
	{ "glColor4iv", NULL},
	{ "glColor4s", NULL},
	{ "glColor4sv", NULL},
	{ "glColor4ub", NULL},
	{ "glColor4ubv", NULL},
	{ "glColor4ui", NULL},
	{ "glColor4uiv", NULL},
	{ "glColor4us", NULL},
	{ "glColor4usv", NULL},
	{ "glColorMask", NULL},
	{ "glColorMaterial", NULL},
	{ "glColorPointer", NULL},
	{ "glCopyPixels", NULL},
	{ "glCopyTexImage1D", NULL},
	{ "glCopyTexImage2D", NULL},
	{ "glCopyTexSubImage1D", NULL},
	{ "glCopyTexSubImage2D", NULL},
	{ "glCullFace", NULL},
	{ "glDeleteLists", NULL},
	{ "glDeleteTextures", NULL},
	{ "glDepthFunc", NULL},
	{ "glDepthMask", NULL},
	{ "glDepthRange", NULL},
	{ "glDisable", NULL},
	{ "glDisableClientState", NULL},
	{ "glDrawArrays", NULL},
	{ "glDrawBuffer", NULL},
	{ "glDrawElements", NULL},
	{ "glDrawPixels", NULL},
	{ "glEdgeFlag", NULL},
	{ "glEdgeFlagPointer", NULL},
	{ "glEdgeFlagv", NULL},
	{ "glEnable", NULL},
	{ "glEnableClientState", NULL},
	{ "glEnd", NULL},
	{ "glEndList", NULL},
	{ "glEvalCoord1d", NULL},
	{ "glEvalCoord1dv", NULL},
	{ "glEvalCoord1f", NULL},
	{ "glEvalCoord1fv", NULL},
	{ "glEvalCoord2d", NULL},
	{ "glEvalCoord2dv", NULL},
	{ "glEvalCoord2f", NULL},
	{ "glEvalCoord2fv", NULL},
	{ "glEvalMesh1", NULL},
	{ "glEvalMesh2", NULL},
	{ "glEvalPoint1", NULL},
	{ "glEvalPoint2", NULL},
	{ "glFeedbackBuffer", NULL},
	{ "glFinish", NULL},
	{ "glFlush", NULL},
	{ "glFogf", NULL},
	{ "glFogfv", NULL},
	{ "glFogi", NULL},
	{ "glFogiv", NULL},
	{ "glFrontFace", NULL},
	{ "glFrustum", NULL},
	{ "glGenLists", NULL},
	{ "glGenTextures", NULL},
	{ "glGetBooleanv", NULL},
	{ "glGetClipPlane", NULL},
	{ "glGetDoublev", NULL},
	{ "glGetError", NULL},
	{ "glGetFloatv", NULL},
	{ "glGetIntegerv", NULL},
	{ "glGetLightfv", NULL},
	{ "glGetLightiv", NULL},
	{ "glGetMapdv", NULL},
	{ "glGetMapfv", NULL},
	{ "glGetMapiv", NULL},
	{ "glGetMaterialfv", NULL},
	{ "glGetMaterialiv", NULL},
	{ "glGetPixelMapfv", NULL},
	{ "glGetPixelMapuiv", NULL},
	{ "glGetPixelMapusv", NULL},
	{ "glGetPointerv", NULL},
	{ "glGetPolygonStipple", NULL},
	{ "glGetString", NULL},
	{ "glGetTexEnvfv", NULL},
	{ "glGetTexEnviv", NULL},
	{ "glGetTexGendv", NULL},
	{ "glGetTexGenfv", NULL},
	{ "glGetTexGeniv", NULL},
	{ "glGetTexImage", NULL},
	{ "glGetTexLevelParameterfv", NULL},
	{ "glGetTexLevelParameteriv", NULL},
	{ "glGetTexParameterfv", NULL},
	{ "glGetTexParameteriv", NULL},
	{ "glHint", NULL},
	{ "glIndexMask", NULL},
	{ "glIndexPointer", NULL},
	{ "glIndexd", NULL},
	{ "glIndexdv", NULL},
	{ "glIndexf", NULL},
	{ "glIndexfv", NULL},
	{ "glIndexi", NULL},
	{ "glIndexiv", NULL},
	{ "glIndexs", NULL},
	{ "glIndexsv", NULL},
	{ "glIndexub", NULL},
	{ "glIndexubv", NULL},
	{ "glInitNames", NULL},
	{ "glInterleavedArrays", NULL},
	{ "glIsEnabled", NULL},
	{ "glIsList", NULL},
	{ "glIsTexture", NULL},
	{ "glLightModelf", NULL},
	{ "glLightModelfv", NULL},
	{ "glLightModeli", NULL},
	{ "glLightModeliv", NULL},
	{ "glLightf", NULL},
	{ "glLightfv", NULL},
	{ "glLighti", NULL},
	{ "glLightiv", NULL},
	{ "glLineStipple", NULL},
	{ "glLineWidth", NULL},
	{ "glListBase", NULL},
	{ "glLoadIdentity", NULL},
	{ "glLoadMatrixd", NULL},
	{ "glLoadMatrixf", NULL},
	{ "glLoadName", NULL},
	{ "glLogicOp", NULL},
	{ "glMap1d", NULL},
	{ "glMap1f", NULL},
	{ "glMap2d", NULL},
	{ "glMap2f", NULL},
	{ "glMapGrid1d", NULL},
	{ "glMapGrid1f", NULL},
	{ "glMapGrid2d", NULL},
	{ "glMapGrid2f", NULL},
	{ "glMaterialf", NULL},
	{ "glMaterialfv", NULL},
	{ "glMateriali", NULL},
	{ "glMaterialiv", NULL},
	{ "glMatrixMode", NULL},
	{ "glMultMatrixd", NULL},
	{ "glMultMatrixf", NULL},
	{ "glNewList", NULL},
	{ "glNormal3b", NULL},
	{ "glNormal3bv", NULL},
	{ "glNormal3d", NULL},
	{ "glNormal3dv", NULL},
	{ "glNormal3f", NULL},
	{ "glNormal3fv", NULL},
	{ "glNormal3i", NULL},
	{ "glNormal3iv", NULL},
	{ "glNormal3s", NULL},
	{ "glNormal3sv", NULL},
	{ "glNormalPointer", NULL},
	{ "glOrtho", NULL},
	{ "glPassThrough", NULL},
	{ "glPixelMapfv", NULL},
	{ "glPixelMapuiv", NULL},
	{ "glPixelMapusv", NULL},
	{ "glPixelStoref", NULL},
	{ "glPixelStorei", NULL},
	{ "glPixelTransferf", NULL},
	{ "glPixelTransferi", NULL},
	{ "glPixelZoom", NULL},
	{ "glPointSize", NULL},
	{ "glPolygonMode", NULL},
	{ "glPolygonOffset", NULL},
	{ "glPolygonStipple", NULL},
	{ "glPopAttrib", NULL},
	{ "glPopClientAttrib", NULL},
	{ "glPopMatrix", NULL},
	{ "glPopName", NULL},
	{ "glPrioritizeTextures", NULL},
	{ "glPushAttrib", NULL},
	{ "glPushClientAttrib", NULL},
	{ "glPushMatrix", NULL},
	{ "glPushName", NULL},
	{ "glRasterPos2d", NULL},
	{ "glRasterPos2dv", NULL},
	{ "glRasterPos2f", NULL},
	{ "glRasterPos2fv", NULL},
	{ "glRasterPos2i", NULL},
	{ "glRasterPos2iv", NULL},
	{ "glRasterPos2s", NULL},
	{ "glRasterPos2sv", NULL},
	{ "glRasterPos3d", NULL},
	{ "glRasterPos3dv", NULL},
	{ "glRasterPos3f", NULL},
	{ "glRasterPos3fv", NULL},
	{ "glRasterPos3i", NULL},
	{ "glRasterPos3iv", NULL},
	{ "glRasterPos3s", NULL},
	{ "glRasterPos3sv", NULL},
	{ "glRasterPos4d", NULL},
	{ "glRasterPos4dv", NULL},
	{ "glRasterPos4f", NULL},
	{ "glRasterPos4fv", NULL},
	{ "glRasterPos4i", NULL},
	{ "glRasterPos4iv", NULL},
	{ "glRasterPos4s", NULL},
	{ "glRasterPos4sv", NULL},
	{ "glReadBuffer", NULL},
	{ "glReadPixels", NULL},
	{ "glRectd", NULL},
	{ "glRectdv", NULL},
	{ "glRectf", NULL},
	{ "glRectfv", NULL},
	{ "glRecti", NULL},
	{ "glRectiv", NULL},
	{ "glRects", NULL},
	{ "glRectsv", NULL},
	{ "glRenderMode", NULL},
	{ "glRotated", NULL},
	{ "glRotatef", NULL},
	{ "glScaled", NULL},
	{ "glScalef", NULL},
	{ "glScissor", NULL},
	{ "glSelectBuffer", NULL},
	{ "glShadeModel", NULL},
	{ "glStencilFunc", NULL},
	{ "glStencilMask", NULL},
	{ "glStencilOp", NULL},
	{ "glTexCoord1d", NULL},
	{ "glTexCoord1dv", NULL},
	{ "glTexCoord1f", NULL},
	{ "glTexCoord1fv", NULL},
	{ "glTexCoord1i", NULL},
	{ "glTexCoord1iv", NULL},
	{ "glTexCoord1s", NULL},
	{ "glTexCoord1sv", NULL},
	{ "glTexCoord2d", NULL},
	{ "glTexCoord2dv", NULL},
	{ "glTexCoord2f", NULL},
	{ "glTexCoord2fv", NULL},
	{ "glTexCoord2i", NULL},
	{ "glTexCoord2iv", NULL},
	{ "glTexCoord2s", NULL},
	{ "glTexCoord2sv", NULL},
	{ "glTexCoord3d", NULL},
	{ "glTexCoord3dv", NULL},
	{ "glTexCoord3f", NULL},
	{ "glTexCoord3fv", NULL},
	{ "glTexCoord3i", NULL},
	{ "glTexCoord3iv", NULL},
	{ "glTexCoord3s", NULL},
	{ "glTexCoord3sv", NULL},
	{ "glTexCoord4d", NULL},
	{ "glTexCoord4dv", NULL},
	{ "glTexCoord4f", NULL},
	{ "glTexCoord4fv", NULL},
	{ "glTexCoord4i", NULL},
	{ "glTexCoord4iv", NULL},
	{ "glTexCoord4s", NULL},
	{ "glTexCoord4sv", NULL},
	{ "glTexCoordPointer", NULL},
	{ "glTexEnvf", NULL},
	{ "glTexEnvfv", NULL},
	{ "glTexEnvi", NULL},
	{ "glTexEnviv", NULL},
	{ "glTexGend", NULL},
	{ "glTexGendv", NULL},
	{ "glTexGenf", NULL},
	{ "glTexGenfv", NULL},
	{ "glTexGeni", NULL},
	{ "glTexGeniv", NULL},
	{ "glTexImage1D", NULL},
	{ "glTexImage2D", NULL},
	{ "glTexParameterf", NULL},
	{ "glTexParameterfv", NULL},
	{ "glTexParameteri", NULL},
	{ "glTexParameteriv", NULL},
	{ "glTexSubImage1D", NULL},
	{ "glTexSubImage2D", NULL},
	{ "glTranslated", NULL},
	{ "glTranslatef", NULL},
	{ "glVertex2d", NULL},
	{ "glVertex2dv", NULL},
	{ "glVertex2f", NULL},
	{ "glVertex2fv", NULL},
	{ "glVertex2i", NULL},
	{ "glVertex2iv", NULL},
	{ "glVertex2s", NULL},
	{ "glVertex2sv", NULL},
	{ "glVertex3d", NULL},
	{ "glVertex3dv", NULL},
	{ "glVertex3f", NULL},
	{ "glVertex3fv", NULL},
	{ "glVertex3i", NULL},
	{ "glVertex3iv", NULL},
	{ "glVertex3s", NULL},
	{ "glVertex3sv", NULL},
	{ "glVertex4d", NULL},
	{ "glVertex4dv", NULL},
	{ "glVertex4f", NULL},
	{ "glVertex4fv", NULL},
	{ "glVertex4i", NULL},
	{ "glVertex4iv", NULL},
	{ "glVertex4s", NULL},
	{ "glVertex4sv", NULL},
	{ "glVertexPointer", NULL},
	{ "glViewport", NULL},
	{ NULL, NULL}};
	

/* API for default state tracking */
GLapi glcontext_api[] = {
/* glbucket.c */
	{"glAccum", (GLfunc) __glstate_Accum},
	{"glCallList", (GLfunc) __glstate_CallList},
	{"glCallLists", (GLfunc) __glstate_CallLists},
	{"glClear", (GLfunc) __glstate_Clear},
	{"glBitmap", (GLfunc) __glstate_Bitmap},
	{"glDrawPixels", (GLfunc) __glstate_DrawPixels},
	{"glRectd", (GLfunc) __glstate_Rectd},
	{"glRectdv", (GLfunc) __glstate_Rectdv},
	{"glRectf", (GLfunc) __glstate_Rectf},
	{"glRectfv", (GLfunc) __glstate_Rectfv},
	{"glRecti", (GLfunc) __glstate_Recti},
	{"glRectiv", (GLfunc) __glstate_Rectiv},
	{"glRects", (GLfunc) __glstate_Rects},
	{"glRectsv", (GLfunc) __glstate_Rectsv},
/* glbuffer.c */
	{ "glAlphaFunc", (GLfunc) __glstate_AlphaFunc},
	{ "glDepthFunc", (GLfunc) __glstate_DepthFunc},
	{ "glBlendFunc", (GLfunc) __glstate_BlendFunc},
	{ "glLogicOp", (GLfunc) __glstate_LogicOp},
	{ "glDrawBuffer", (GLfunc) __glstate_DrawBuffer},
	{ "glReadBuffer", (GLfunc) __glstate_ReadBuffer},
	{ "glIndexMask", (GLfunc) __glstate_IndexMask},
	{ "glColorMask", (GLfunc) __glstate_ColorMask},
	{ "glClearColor", (GLfunc) __glstate_ClearColor},
	{ "glClearIndex", (GLfunc) __glstate_ClearIndex},
	{ "glClearDepth", (GLfunc) __glstate_ClearDepth},
	{ "glClearAccum", (GLfunc) __glstate_ClearAccum},
	{ "glDepthMask", (GLfunc) __glstate_DepthMask},
/* glclient.c */
	{ "glEnableClientState", (GLfunc) __glstate_EnableClientState},
	{ "glDisableClientState", (GLfunc) __glstate_DisableClientState},
	{ "glVertexPointer", (GLfunc) __glstate_VertexPointer},
	{ "glColorPointer", (GLfunc) __glstate_ColorPointer},
	{ "glIndexPointer", (GLfunc) __glstate_IndexPointer},
	{ "glNormalPointer", (GLfunc) __glstate_NormalPointer},
	{ "glTexCoordPointer", (GLfunc) __glstate_TexCoordPointer},
	{ "glEdgeFlagPointer", (GLfunc) __glstate_EdgeFlagPointer},
	{ "glInterleavedArrays", (GLfunc) __glstate_InterleavedArrays},
	{ "glDrawArrays", (GLfunc) __glstate_DrawArrays}, 
	{ "glArrayElement", (GLfunc) __glstate_ArrayElement}, 
	{ "glGetPointerv", (GLfunc) __glstate_GetPointerv},
	{ "glDrawArrays", (GLfunc) __glstate_DrawArrays},
	{ "glDrawElements", (GLfunc) __glstate_DrawElements},
/* glbucket.c */
	{"glBegin", (GLfunc) __glstate_Begin},
	{"glEnd", (GLfunc) __glstate_End},
/* glenable.c */
	{ "glEnable", (GLfunc) __glstate_Enable},
	{ "glDisable", (GLfunc) __glstate_Disable},
/* gleval.c */
	{"glMap1d", (GLfunc) __glstate_Map1d},
	{"glMap1f", (GLfunc) __glstate_Map1f},
	{"glMap2d", (GLfunc) __glstate_Map2d},
	{"glMap2f", (GLfunc) __glstate_Map2f},
	{"glMapGrid1d", (GLfunc) __glstate_MapGrid1d},
	{"glMapGrid1f", (GLfunc) __glstate_MapGrid1f},
	{"glMapGrid2d", (GLfunc) __glstate_MapGrid2d},
	{"glMapGrid2f", (GLfunc) __glstate_MapGrid2f},
/* glfog.c */
	{ "glFogf", (GLfunc) __glstate_Fogf},
	{ "glFogi", (GLfunc) __glstate_Fogi},
	{ "glFogiv", (GLfunc) __glstate_Fogiv},
	{ "glFogfv", (GLfunc) __glstate_Fogfv},
/* glget.c */
	{"glGetBooleanv", (GLfunc) __glstate_GetBooleanv},
	{"glGetIntegerv", (GLfunc) __glstate_GetIntegerv},
	{"glGetFloatv", (GLfunc) __glstate_GetFloatv},
	{"glGetDoublev", (GLfunc) __glstate_GetDoublev},
	{"glIsEnabled", (GLfunc) __glstate_IsEnabled},
	{"glGetString", (GLfunc) __glstate_GetString},
/* glhint.c */
	{ "glHint", (GLfunc) __glstate_Hint},
/* gllighting.c */
	{ "glShadeModel", (GLfunc) __glstate_ShadeModel},
	{ "glColorMaterial", (GLfunc) __glstate_ColorMaterial},
	{ "glLightModelf", (GLfunc) __glstate_LightModelf},
	{ "glLightModeli", (GLfunc) __glstate_LightModeli},
	{ "glLightModelfv", (GLfunc) __glstate_LightModelfv},
	{ "glLightModeliv", (GLfunc) __glstate_LightModeliv},
	{ "glLightf", (GLfunc) __glstate_Lightf},
	{ "glLighti", (GLfunc) __glstate_Lighti},
	{ "glLightfv", (GLfunc) __glstate_Lightfv},
	{ "glLightiv", (GLfunc) __glstate_Lightiv},
	{ "glMaterialf", (GLfunc) __glstate_Materialf},
	{ "glMateriali", (GLfunc) __glstate_Materiali},
	{ "glMaterialfv", (GLfunc) __glstate_Materialfv},
	{ "glMaterialiv", (GLfunc) __glstate_Materialiv},
	{"glGetMaterialfv", (GLfunc) __glstate_GetMaterialfv},
	{"glGetMaterialiv", (GLfunc) __glstate_GetMaterialiv},
	{"glGetLightfv", (GLfunc) __glstate_GetLightfv},
	{"glGetLightiv", (GLfunc) __glstate_GetLightiv},
/* glline.c */
	{ "glPointSize", (GLfunc) __glstate_PointSize},
	{ "glLineWidth", (GLfunc) __glstate_LineWidth},
	{ "glLineStipple", (GLfunc) __glstate_LineStipple},
/* gllists.c */
	{"glGenLists", (GLfunc) __glstate_GenLists},
	{"glNewList", (GLfunc) __glstate_NewList},
	{"glEndList", (GLfunc) __glstate_EndList},
	{"glDeleteLists", (GLfunc) __glstate_DeleteLists},
	{"glIsList", (GLfunc) __glstate_IsList},
	{"glListBase", (GLfunc) __glstate_ListBase},
/* glpixel.c */
	{ "glPixelStoref", (GLfunc) __glstate_PixelStoref},
	{ "glPixelStorei", (GLfunc) __glstate_PixelStorei},
	{ "glPixelTransferf", (GLfunc) __glstate_PixelTransferf},
	{ "glPixelTransferi", (GLfunc) __glstate_PixelTransferi},
	{ "glPixelZoom", (GLfunc) __glstate_PixelZoom},
/* glpolygon.c */
	{ "glCullFace", (GLfunc) __glstate_CullFace},
	{ "glFrontFace", (GLfunc) __glstate_FrontFace},
	{ "glPolygonMode", (GLfunc) __glstate_PolygonMode},
	{ "glPolygonOffset", (GLfunc) __glstate_PolygonOffset},
	{ "glPolygonStipple", (GLfunc) __glstate_PolygonStipple},
	{ "glGetPolygonStipple", (GLfunc) __glstate_GetPolygonStipple},
/* glrasterpos.c */
	{"glRasterPos2d", (GLfunc) __glstate_RasterPos2d},
	{"glRasterPos2dv", (GLfunc) __glstate_RasterPos2dv},
	{"glRasterPos2f", (GLfunc) __glstate_RasterPos2f},
	{"glRasterPos2fv", (GLfunc) __glstate_RasterPos2fv},
	{"glRasterPos2i", (GLfunc) __glstate_RasterPos2i},
	{"glRasterPos2iv", (GLfunc) __glstate_RasterPos2iv},
	{"glRasterPos2s", (GLfunc) __glstate_RasterPos2s},
	{"glRasterPos2sv", (GLfunc) __glstate_RasterPos2sv},
	{"glRasterPos3d", (GLfunc) __glstate_RasterPos3d},
	{"glRasterPos3dv", (GLfunc) __glstate_RasterPos3dv},
	{"glRasterPos3f", (GLfunc) __glstate_RasterPos3f},
	{"glRasterPos3fv", (GLfunc) __glstate_RasterPos3fv},
	{"glRasterPos3i", (GLfunc) __glstate_RasterPos3i},
	{"glRasterPos3iv", (GLfunc) __glstate_RasterPos3iv},
	{"glRasterPos3s", (GLfunc) __glstate_RasterPos3s},
	{"glRasterPos3sv", (GLfunc) __glstate_RasterPos3sv},
	{"glRasterPos4d", (GLfunc) __glstate_RasterPos4d},
	{"glRasterPos4dv", (GLfunc) __glstate_RasterPos4dv},
	{"glRasterPos4f", (GLfunc) __glstate_RasterPos4f},
	{"glRasterPos4fv", (GLfunc) __glstate_RasterPos4fv},
	{"glRasterPos4i", (GLfunc) __glstate_RasterPos4i},
	{"glRasterPos4iv", (GLfunc) __glstate_RasterPos4iv},
	{"glRasterPos4s", (GLfunc) __glstate_RasterPos4s},
	{"glRasterPos4sv", (GLfunc) __glstate_RasterPos4sv},
/* glscissor.c */
	{ "glScissor", (GLfunc) __glstate_Scissor},
/* glstencil.c */
	{ "glStencilFunc", (GLfunc) __glstate_StencilFunc},
	{ "glStencilOp", (GLfunc) __glstate_StencilOp},
	{ "glClearStencil", (GLfunc) __glstate_ClearStencil},
	{ "glStencilMask", (GLfunc) __glstate_StencilMask},
/* gltexture.c */
	{ "glGenTextures", (GLfunc) __glstate_GenTextures},
	{ "glDeleteTextures", (GLfunc) __glstate_DeleteTextures},
	{ "glBindTexture", (GLfunc) __glstate_BindTexture},
	{ "glTexImage1D", (GLfunc) __glstate_TexImage1D},
	{ "glTexImage2D", (GLfunc) __glstate_TexImage2D},
	{ "glTexSubImage1D", (GLfunc) __glstate_TexSubImage1D},
	{ "glTexSubImage2D", (GLfunc) __glstate_TexSubImage2D},
	{ "glTexImage3D", (GLfunc) __glstate_TexImage3D},
	{ "glTexParameteri", (GLfunc) __glstate_TexParameteri},
	{ "glTexParameterf", (GLfunc) __glstate_TexParameterf},
	{ "glTexParameteriv", (GLfunc) __glstate_TexParameteriv},
	{ "glTexParameterfv", (GLfunc) __glstate_TexParameterfv},
	{ "glGetTexParameteriv", (GLfunc) __glstate_GetTexParameteriv},
	{ "glGetTexParameterfv", (GLfunc) __glstate_GetTexParameterfv},
	{ "glGetTexLevelParameterfv", (GLfunc) __glstate_GetTexLevelParameterfv},
	{ "glGetTexLevelParameteriv", (GLfunc) __glstate_GetTexLevelParameteriv},
	{ "glTexEnvf", (GLfunc) __glstate_TexEnvf},
	{ "glTexEnvi", (GLfunc) __glstate_TexEnvi},
	{ "glTexEnvfv", (GLfunc) __glstate_TexEnvfv},
	{ "glTexEnviv", (GLfunc) __glstate_TexEnviv},
	{ "glGetTexEnvfv", (GLfunc) __glstate_GetTexEnvfv},
	{ "glGetTexEnviv", (GLfunc) __glstate_GetTexEnviv},
	{ "glTexGend", (GLfunc) __glstate_TexGend},
	{ "glTexGenf", (GLfunc) __glstate_TexGenf},
	{ "glTexGeni", (GLfunc) __glstate_TexGeni},
	{ "glTexGendv", (GLfunc) __glstate_TexGendv},
	{ "glTexGenfv", (GLfunc) __glstate_TexGenfv},
	{ "glTexGeniv", (GLfunc) __glstate_TexGeniv},
	{ "glGetTexGendv", (GLfunc) __glstate_GetTexGendv},
	{ "glGetTexGenfv", (GLfunc) __glstate_GetTexGenfv},
	{ "glGetTexGeniv", (GLfunc) __glstate_GetTexGeniv},
	{ "glAreTexturesResident", (GLfunc) __glstate_AreTexturesResident},
	{ "glPrioritizeTextures", (GLfunc) __glstate_PrioritizeTextures},
/* gltrans.c */
	{ "glClipPlane", (GLfunc) __glstate_ClipPlane},
	{ "glGetClipPlane", (GLfunc) __glstate_GetClipPlane},
	{ "glMatrixMode", (GLfunc) __glstate_MatrixMode},
	{ "glLoadIdentity", (GLfunc) __glstate_LoadIdentity},
	{ "glPopMatrix", (GLfunc) __glstate_PopMatrix},
	{ "glPushMatrix", (GLfunc) __glstate_PushMatrix},
	{ "glLoadMatrixf", (GLfunc) __glstate_LoadMatrixf},
	{ "glLoadMatrixd", (GLfunc) __glstate_LoadMatrixd},
	{ "glMultMatrixf", (GLfunc) __glstate_MultMatrixf},
	{ "glMultMatrixd", (GLfunc) __glstate_MultMatrixd},
	{ "glTranslatef", (GLfunc) __glstate_Translatef},
	{ "glTranslated", (GLfunc) __glstate_Translated},
	{ "glRotatef", (GLfunc) __glstate_Rotatef},
	{ "glRotated", (GLfunc) __glstate_Rotated},
	{ "glScalef", (GLfunc) __glstate_Scalef},
	{ "glScaled", (GLfunc) __glstate_Scaled},
	{ "glFrustum", (GLfunc) __glstate_Frustum},
	{ "glOrtho", (GLfunc) __glstate_Ortho},
/* glviewport.c */
	{ "glViewport", (GLfunc) __glstate_Viewport},
	{ "glDepthRange", (GLfunc) __glstate_DepthRange},
	{ "glGetError", (GLfunc) __glstate_GetError},
	{ NULL, NULL}};

void GLAPI 
__glhw_nop() {}

void GLAPI
__glhw_nop3 (GLbyte a, GLbyte b, GLbyte c) {
	UNUSED(a);
	UNUSED(b);
	UNUSED(c);
}

void GLAPI
__glhw_nop4 (GLint a) {
	UNUSED(a);
}

void GLAPI
__glhw_nop6 (GLshort a, GLshort b, GLshort c) {
	UNUSED(a);
	UNUSED(b);
	UNUSED(c);
}

void GLAPI
__glhw_nop8 (GLdouble a) {
	UNUSED(a);
}

void GLAPI
__glhw_nop12 (GLdouble a, GLint b) {
	UNUSED(a);
	UNUSED(b);
}

void GLAPI
__glhw_nop16 (GLdouble a, GLdouble b) {
	UNUSED(a);
	UNUSED(b);
}

void GLAPI 
__glhw_nop20 (GLdouble a, GLdouble b, GLint c) {
	UNUSED(a);
	UNUSED(b);
	UNUSED(c);
}

void GLAPI
__glhw_nop24 (GLdouble a, GLdouble b, GLdouble c) {
	UNUSED(a);
	UNUSED(b);
	UNUSED(c);
}

void GLAPI
__glhw_nop28 (GLdouble a, GLdouble b, GLdouble c, GLint d) {
	UNUSED(a); 
	UNUSED(b);
	UNUSED(c);
	UNUSED(d);
}

void GLAPI
__glhw_nop32 (GLdouble a, GLdouble b, GLdouble c, GLdouble d) {
	UNUSED(a);
	UNUSED(b);
	UNUSED(c);
	UNUSED(d);
}

void GLAPI
__glhw_nop36 (GLdouble a, GLdouble b, GLdouble c, GLdouble d, GLint e) {
	UNUSED(a);
	UNUSED(b);
	UNUSED(c);
	UNUSED(d);
	UNUSED(e);
}

void GLAPI
__glhw_nop40 (GLdouble a, GLdouble b, GLdouble c, GLdouble d, GLdouble e) {
	UNUSED(a); 
	UNUSED(b);
	UNUSED(c);
	UNUSED(d);
	UNUSED(e);
}

void GLAPI
__glhw_nop44 (GLdouble a, GLdouble b, GLdouble c, GLdouble d, GLdouble e, GLint f) {
	UNUSED(a);
	UNUSED(b);
	UNUSED(c);
	UNUSED(d);
	UNUSED(e);
	UNUSED(f);
}

void GLAPI
__glhw_nop48 (GLdouble a, GLdouble b, GLdouble c, GLdouble d, GLdouble e, GLdouble f) {
	UNUSED(a);
	UNUSED(b);
	UNUSED(c);
	UNUSED(d);
	UNUSED(e);
	UNUSED(f);
}

void GLAPI
__glhw_nop52 (GLdouble a, GLdouble b, GLdouble c, GLdouble d, GLdouble e, GLdouble f, GLint g) {
	UNUSED(a);
	UNUSED(b);
	UNUSED(c);
	UNUSED(d);
	UNUSED(e);
	UNUSED(f);
	UNUSED(g);
}

void GLAPI
__glhw_nop56 (GLdouble a, GLdouble b, GLdouble c, GLdouble d, GLdouble e, GLdouble f, GLdouble g) {
	UNUSED(a);
	UNUSED(b);
	UNUSED(c);
	UNUSED(d);
	UNUSED(e);
	UNUSED(f);
	UNUSED(g);
}

/* NOP assignments */
GLapi nop_api[] = {
/* glbuffer.c */
	{ "glAlphaFunc", (GLfunc) __glhw_nop8},
	{ "glDepthFunc", (GLfunc) __glhw_nop4},
	{ "glBlendFunc", (GLfunc) __glhw_nop8},
	{ "glBlendColor", (GLfunc) __glhw_nop16},
	{ "glBlendEquation", (GLfunc) __glhw_nop4},
	{ "glLogicOp", (GLfunc) __glhw_nop4},
	{ "glDrawBuffer", (GLfunc) __glhw_nop4},
	{ "glReadBuffer", (GLfunc) __glhw_nop4},
	{ "glIndexMask", (GLfunc) __glhw_nop4},
	{ "glColorMask", (GLfunc) __glhw_nop4},
	{ "glClearColor", (GLfunc) __glhw_nop16},
	{ "glClearIndex", (GLfunc) __glhw_nop4},
	{ "glClearDepth", (GLfunc) __glhw_nop4},
	{ "glClearAccum", (GLfunc) __glhw_nop16},
	{ "glDepthMask", (GLfunc) __glhw_nop4},
/* glclient.c */
	{ "glEnableClientState", (GLfunc) __glhw_nop4},
	{ "glDisableClientState", (GLfunc) __glhw_nop4},
	{ "glVertexPointer", (GLfunc) __glhw_nop16},
	{ "glColorPointer", (GLfunc) __glhw_nop16},
	{ "glIndexPointer", (GLfunc) __glhw_nop12},
	{ "glNormalPointer", (GLfunc) __glhw_nop12},
	{ "glTexCoordPointer", (GLfunc) __glhw_nop16},
	{ "glEdgeFlagPointer", (GLfunc) __glhw_nop8},
/* glcurrent.c */
	{ "glColor3bv", (GLfunc) __glhw_nop3},
	{ "glColor3dv", (GLfunc) __glhw_nop24},
	{ "glColor3fv", (GLfunc) __glhw_nop12},
	{ "glColor3iv", (GLfunc) __glhw_nop12},
	{ "glColor3sv", (GLfunc) __glhw_nop6},
	{ "glColor3ubv", (GLfunc) __glhw_nop3},
	{ "glColor3uiv", (GLfunc) __glhw_nop12},
	{ "glColor3usv", (GLfunc) __glhw_nop6},
	{ "glColor4bv", (GLfunc) __glhw_nop4},
	{ "glColor4dv", (GLfunc) __glhw_nop32},
	{ "glColor4fv", (GLfunc) __glhw_nop16},
	{ "glColor4iv", (GLfunc) __glhw_nop16},
	{ "glColor4sv", (GLfunc) __glhw_nop8},
	{ "glColor4ubv", (GLfunc) __glhw_nop4},
	{ "glColor4uiv", (GLfunc) __glhw_nop16},
	{ "glColor4usv", (GLfunc) __glhw_nop8},
	{ "glBegin", (GLfunc) __glhw_nop4},
	{ "glEnd", (GLfunc) __glhw_nop},
/* glenable.c */
	{ "glEnable", (GLfunc) __glhw_nop4},
	{ "glDisable", (GLfunc) __glhw_nop4},
	{ "glDepthMask", (GLfunc) __glhw_nop4},
/* gleval.c */
	{ "glMap1d", (GLfunc) __glhw_nop32},
	{ "glMap2d", (GLfunc) __glhw_nop56},
	{ "glMapGrid1d", (GLfunc) __glhw_nop20},
	{ "glMapGrid2d", (GLfunc) __glhw_nop40},
/* glfog.c */
	{ "glFogf", (GLfunc) __glhw_nop8},
	{ "glFogi", (GLfunc) __glhw_nop8},
	{ "glFogiv", (GLfunc) __glhw_nop8},
	{ "glFogfv", (GLfunc) __glhw_nop8},
/* glhint.c */
	{ "glHint", (GLfunc) __glhw_nop8},
/* gllighting.c */
	{ "glShadeModel", (GLfunc) __glhw_nop4},
	{ "glColorMaterial", (GLfunc) __glhw_nop8},
	{ "glLightModelf", (GLfunc) __glhw_nop8},
	{ "glLightModeli", (GLfunc) __glhw_nop8},
	{ "glLightModelfv", (GLfunc) __glhw_nop8},
	{ "glLightModeliv", (GLfunc) __glhw_nop8},
	{ "glLighf", (GLfunc) __glhw_nop12},
	{ "glLighti", (GLfunc) __glhw_nop12},
	{ "glLightfv", (GLfunc) __glhw_nop12},
	{ "glLightiv", (GLfunc) __glhw_nop12},
	{ "glMaterialf", (GLfunc) __glhw_nop12},
	{ "glMateriali", (GLfunc) __glhw_nop12},
	{ "glMaterialfv", (GLfunc) __glhw_nop12},
	{ "glMaterialiv", (GLfunc) __glhw_nop12},
/* glline.c */
	{ "glPointSize", (GLfunc) __glhw_nop4},
	{ "glLineWidth", (GLfunc) __glhw_nop4},
	{ "glLineStipple", (GLfunc) __glhw_nop8},
/* glpixel.c */
	{ "glPixelStoref", (GLfunc) __glhw_nop8},
	{ "glPixelStorei", (GLfunc) __glhw_nop8}, 
	{ "glPixelTransferf", (GLfunc) __glhw_nop8},
	{ "glPixelTransferi", (GLfunc) __glhw_nop8},
	{ "glPixelZoom", (GLfunc) __glhw_nop8},
/* glpolygon.c */
	{ "glCullFace", (GLfunc) __glhw_nop4},
	{ "glFrontFace", (GLfunc) __glhw_nop4},
	{ "glPolygonMode", (GLfunc) __glhw_nop8},
	{ "glPolygonStipple", (GLfunc) __glhw_nop4}, 
/* glscissor.c */
	{ "glScissor", (GLfunc) __glhw_nop16},
/* glstencil.c */
	{ "glStencilFunc", (GLfunc) __glhw_nop12},
	{ "glStencilOp", (GLfunc) __glhw_nop12},
	{ "glClearStencil", (GLfunc) __glhw_nop4},
	{ "glStencilMask", (GLfunc) __glhw_nop4},
/* gltexture.c */
	{ "glBindTexture", (GLfunc) __glhw_nop8},
	{ "glTexParameteri", (GLfunc) __glhw_nop12},
	{ "glTexParameterf", (GLfunc) __glhw_nop12},
	{ "glTexParameteriv", (GLfunc) __glhw_nop12},
	{ "glTexParameterfv", (GLfunc) __glhw_nop12},
	{ "glTexImage1D", (GLfunc) __glhw_nop32},
	{ "glTexImage2D", (GLfunc) __glhw_nop36},
	{ "glTexImage3D", (GLfunc) __glhw_nop40},
	{ "glTexGend", (GLfunc) __glhw_nop16},
	{ "glTexGenf", (GLfunc) __glhw_nop12},
	{ "glTexGeni", (GLfunc) __glhw_nop12},
	{ "glTexGendv", (GLfunc) __glhw_nop12},
	{ "glTexGenfv", (GLfunc) __glhw_nop12},
	{ "glTexEnvf", (GLfunc) __glhw_nop12},
	{ "glTexEnvi", (GLfunc) __glhw_nop12},
	{ "glTexEnvfv", (GLfunc) __glhw_nop12},
	{ "glTexEnviv", (GLfunc) __glhw_nop12},
/* gltrans.c */
	{ "glClipPlane", (GLfunc) __glhw_nop8},
	{ "glGetClipPlane", (GLfunc) __glhw_nop8},
	{ "glMatrixMode", (GLfunc) __glhw_nop},
	{ "glLoadIdentity", (GLfunc) __glhw_nop},
	{ "glPopMatrix", (GLfunc) __glhw_nop},
	{ "glPushMatrix", (GLfunc) __glhw_nop},
	{ "glLoadMatrixf", (GLfunc) __glhw_nop4},
	{ "glLoadMatrixd", (GLfunc) __glhw_nop4},
	{ "glMultMatrixf", (GLfunc) __glhw_nop4},
	{ "glMultMatrixd", (GLfunc) __glhw_nop4},
	{ "glTranslatef", (GLfunc) __glhw_nop12},
	{ "glTranslated", (GLfunc) __glhw_nop24},
	{ "glRotatef", (GLfunc) __glhw_nop16},
	{ "glRotated", (GLfunc) __glhw_nop32},
	{ "glScalef", (GLfunc) __glhw_nop12},
	{ "glScaled", (GLfunc) __glhw_nop24},
	{ "glFrustum", (GLfunc) __glhw_nop48},
	{ "glOrtho", (GLfunc) __glhw_nop48},
/* glviewport.c */
	{ "glViewport", (GLfunc) __glhw_nop16},
	{ "glDepthRange", (GLfunc) __glhw_nop16},
/* glconfig.c */
	{ "glGetBooleanv", (GLfunc) __glhw_nop8}, 
	{ "glGetIntegerv", (GLfunc) __glhw_nop8}, 
	{ "glGetFloatv", (GLfunc) __glhw_nop8},
	{ "glGetConvolutionParameteriv", (GLfunc) __glhw_nop12},
	{ "glGetString", (GLfunc) __glhw_nop4},
	{ NULL, NULL}
};
