/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include <stdlib.h>
#include <stdio.h>
#include "glcontext.h"
#include "glcurrent.h"
#include "glhw.h"
#include "glerror.h"


void 
__glcurrent_initbits (GLcurrentbits *c, GLconfig *cfg) {
	/* do nothing */
	UNUSED(c);
	UNUSED(cfg);
}

void
__glcurrent_switch (GLcurrentbits *c, GLbitvalue bitID,
					 GLcurrentstate *current, GLcurrentstate *target) {
	GLbitvalue nbitID = ~bitID;

	if (c->enable & bitID) {
		if (current->normalize != target->normalize) {
			if (target->normalize == GL_TRUE)
				__glhw_Enable(GL_NORMALIZE);
			else
				__glhw_Disable(GL_NORMALIZE);
			c->enable = GLBITS_ONES;
		}
		c->enable &= nbitID;
		c->dirty = GLBITS_ONES;
	}

	if (c->raster & bitID) {
		if (target->rastervalid) {
			if (target->rasterpos_pre.x != current->rasterpos.x ||
				target->rasterpos_pre.y != current->rasterpos.y) {
					GLvectorf p;
					p.x = target->rasterpos_pre.x - current->rasterpos.x;
					p.y = target->rasterpos_pre.y - current->rasterpos.y;
					__glhw_Bitmap(0, 0, 0.0f, 0.0f, p.x, p.y, 0);
					c->raster = GLBITS_ONES;
					c->dirty  = GLBITS_ONES;
			}
		}
		c->raster &= nbitID;
	}

	/* Vertex Current State Switch Code */

	/* Its important that we don't do a value check here because
	** current may not actaully have the correct values, I think...
	** We also need to restore the current state tracking pointer
	** since the packing functions will set it.
	*/

	/* NEED TO FIX THIS!!!!!! */
	if (c->color & bitID) {
		if (COMPARE_COLOR(current->color,target->color)) {
			__glhw_Color4fv ((GLfloat *) &(target->color));
			c->color = GLBITS_ONES;
			c->dirty = GLBITS_ONES;
		}
		c->color &= nbitID;
	}

	if (c->index & bitID) {
		if (target->index != current->index) {
			__glhw_Indexf (target->index);
			c->index = GLBITS_ONES;
			c->dirty = GLBITS_ONES;
		}
		c->index &= nbitID;
	}

	if (c->normal & bitID) {
		if (COMPARE_VECTOR (current->normal, target->normal)) {
			__glhw_Normal3fv ((GLfloat *) &(target->normal.x));
			c->normal = GLBITS_ONES;
			c->dirty = GLBITS_ONES;
		}
		c->normal &= nbitID;
	}

	if (c->texcoord & bitID) {
		if (COMPARE_TEXCOORD (current->texcoord, target->texcoord_pre)) {
			__glhw_TexCoord4fv ((GLfloat *) &(target->texcoord.s));
			c->normal = GLBITS_ONES;
			c->dirty = GLBITS_ONES;
		}
		c->texcoord &= nbitID;
	}

	c->dirty &= nbitID;
}

void
__glcurrent_sync (GLcurrentbits *c, GLbitvalue bitID,
					 GLcurrentstate *current, GLcurrentstate *target) {
	GLbitvalue nbitID = ~bitID;

	if (c->enable & bitID) {
		if (current->normalize != target->normalize) {
			if (target->normalize == GL_TRUE)
				__glhw_Enable(GL_NORMALIZE);
			else
				__glhw_Disable(GL_NORMALIZE);
			current->normalize = target->normalize;
		}
		c->enable &= nbitID;
	}

	if (c->raster & bitID) {
		current->rastervalid = target->rastervalid;
		if (target->rastervalid) {
			if (target->rasterpos_pre.x != current->rasterpos.x ||
				target->rasterpos_pre.y != current->rasterpos.y) {
					GLvectorf p;
					p.x = target->rasterpos_pre.x - current->rasterpos.x;
					p.y = target->rasterpos_pre.y - current->rasterpos.y;
					__glhw_Bitmap(0, 0, 0.0f, 0.0f, p.x, p.y, 0);
			}
			current->rasterpos = target->rasterpos;
		}
		c->raster &= nbitID;
	}

	/* Vertex Current State Sync Code */
	/* Some things to note here:
	** 1) Compare is done against the pre value since the
	**    current value includes the geometry info.
	** 2) Update is done with the current value since
	**    the server will be getting the geometry block
	** 3) Copy is done outside of the compare to ensure
	**    that it happens.
	*/
	if (c->color & bitID) {
		if (COMPARE_COLOR(current->color,target->color_pre)) {
			__glhw_Color4fv ((GLfloat *) &(target->color_pre.r));
		}
		current->color = target->color;
		c->color &= nbitID;
	}

	if (c->index & bitID) {
		if (current->index != target->index_pre) {
			__glhw_Indexf (target->index);
		}
		current->index = target->index;
		c->index &= nbitID;
	}

	if (c->normal & bitID) {
		if (COMPARE_VECTOR (current->normal, target->normal_pre)) {
			__glhw_Normal3fv ((GLfloat *) &(target->normal_pre.x));
		}
		current->normal = target->normal;
		c->normal &= nbitID;
	}

	if (c->texcoord & bitID) {
		if (COMPARE_TEXCOORD (current->texcoord, target->texcoord_pre)) {
			__glhw_TexCoord4fv ((GLfloat *) &(target->texcoord_pre.s));
		}
		current->texcoord = target->texcoord;
		c->texcoord &= nbitID;
	}

	if (c->edgeflag & bitID) {
		if (current->edgeflag != target->edgeflag_pre) {
			__glhw_EdgeFlag (target->edgeflag_pre);
		}
		current->edgeflag = target->edgeflag;
		c->edgeflag &= nbitID;
	}

	c->dirty &= nbitID;
}

void
__glcurrent_issue_params (GLvertex *vtx) {
	GLfloat val[4];

#if 0
	GLfloat purple[4] = {1.0f, 0.0f, 1.0f, 1.0f};
#endif

	val[0] = vtx->texcoord.s; val[1] = vtx->texcoord.t; 
	val[2] = vtx->texcoord.p; val[3] = vtx->texcoord.q;
	(*__glbase_TexCoord4fv)((const GLfloat *) val);
	val[0] = vtx->normal.x; val[1] = vtx->normal.y; 
	val[2] = vtx->normal.z;
	(*__glbase_Normal3fv)((const GLfloat *) val);
	(*__glbase_EdgeFlag)	(vtx->edgeflag);
	val[0] = vtx->color.r; val[1] = vtx->color.g; 
	val[2] = vtx->color.b; val[3] = vtx->color.a;
	(*__glbase_Color4fv)((const GLfloat *) val);

#if 0
	printf ("Restore Color: %f, %f, %f, %f\n",  
	vtx->color.r,  vtx->color.g,  
	vtx->color.b,  vtx->color.a);
#endif

}

void
__glcurrent_issue_vertex (GLvertex *vtx) {
	GLfloat val[4];
	
	__glcurrent_issue_params (vtx);

	val[0] = vtx->pos.x; val[1] = vtx->pos.y; 
	val[2] = vtx->pos.z; val[3] = vtx->pos.w;
	(*__glbase_Vertex4fv)((const GLfloat *) val);	
}


void
__glcurrent_restore_tri (GLcurrentstate *c) {
	int i;
	GLvertex v;
	static int count = 0;

	if (c->beginend) {
		__glhw_Begin(c->mode);
		c->beginendnum++;

		/* If the winding flag is set, it means
		** that the strip was broken on an odd 
		** vertex number.  To fix, we issue the 
		** first vertex twice to resolve the ordering
		*/
		if (c->wind)
			__glcurrent_issue_vertex(c->vtx);

		for (i=c->isloop?1:0; i<c->num_restore; i++) {
#if 0
			if ((c->vtx[i].pos.y == 0.4f && c->vtx[i].color.b != c->vtx[i].pos.x) ||
				(c->vtx[i].pos.y == 0.6f && c->vtx[i].color.r != c->vtx[i].pos.x))
				printf ("ERROR: %d\n", count);
#endif
			__glcurrent_issue_vertex(c->vtx + i);
		}

		count++;
		
		/* Setup values for next vertex */
		v.texcoord = c->texcoord;
		v.normal   = c->normal;
		v.edgeflag = c->edgeflag;
		v.color    = c->color;
	
		__glcurrent_issue_params (&v);

	}
}

void 
__glcurrent_finish (GLcurrentstate *c, GLcurrentbits *cb, GLbitvalue bitID) {
	UNUSED (cb);
	UNUSED (bitID);
	
	/* Reset the vertex count */
	c->current->vtx_count = 0;
	c->current->vtx_count_begin = 0;

	/* Reset the begin end count */
	c->beginendnum = 0;

	/* Clear the current pointers */
	__glcurrent_null_pointers (c);

	/* Restart the triangle engine */
	__glcurrent_restore_tri (c);
}

void
__glcurrent_init (GLcurrentstate *c, GLconfig *cfg) {
	
	GLvectorf	zero_vector = {0.0f, 0.0f, 0.0f, 1.0f};
	GLcolorf	one_color	= {1.0f, 1.0f, 1.0f, 1.0f};
	GLtexcoordf zero_texcoord = {0.0f, 0.0f, 0.0f, 1.0f};

	c->color	= one_color;
	c->index	= 1.0f;
	c->texcoord = zero_texcoord;
	c->normal	= zero_vector;
	c->normal.z = 1.0f;

	c->num_restore = 0;
	c->wind = 0;
	c->isloop = 0;

    /* Compute the raster pos */
	if (cfg->id >= 0 && cfg->id < cfg->numprojectors) {
		GLrecti *b;
		b = &cfg->bounds[cfg->id][0];
		c->rasterpos.x = (GLfloat) b->x1;
		c->rasterpos.y = (GLfloat) b->y1;
	} else {
		c->rasterpos.x = 0.0f;
		c->rasterpos.y = 0.0f;
	}
	c->rasterpos.z = 0.0f;
	c->rasterpos.w = 1.0f;

	c->rasterpos_pre = c->rasterpos;

	c->rasterdistance = 0.0f;
	c->rastercolor = one_color;
	c->rastertexture = zero_texcoord;
	c->rastervalid = GL_TRUE;
	c->rasterindex = 1.0f;

	c->edgeflag = GL_TRUE;
	c->normalize = GL_FALSE;

	c->beginend = GL_FALSE;
	c->beginendnum = 0;
	c->beginendmax = cfg->beginend_max;
	c->mode = 0x10; /* Undefined Mode */
	c->flush_on_end = 0;

	c->current = cfg->current;
}

void
__glcurrent_destroy (GLcurrentstate *c) {
/*
** Do Nothing
*/
	UNUSED(c);
}


void GLSTATE_DECL
__glstate_Begin (GLenum mode) {
	GLcontext *g = GetCurrentContext();
	GLcurrentstate *c = &g->current;
	GLbucketstate *b = &g->bucket;

	if (mode > GL_POLYGON)
		if (__glerror (__LINE__, __FILE__, GL_INVALID_ENUM,
			"Begin called with invalid mode: %d", mode))
			return;

	if (c->beginend)
		if (__glerror (__LINE__, __FILE__, GL_INVALID_OPERATION,
			"Begin called inside begin/end"))
			return;

	if (*b->count > *b->max)
		b->flush(g);
	
	__glhw_Begin (mode);

	/* Begin the bucketing */
	if (!IsBucketingActive()) 
		__glbucket_begin();

	c->current->vtx_count_begin = c->current->vtx_count;
	c->wind = 0;
	c->isloop = 0;

	c->beginend = GL_TRUE;
	c->mode = mode;
	c->beginendnum++;
}

void GLSTATE_DECL
__glstate_End (void) {
	GLcontext *g = GetCurrentContext();
	GLcurrentstate *c = &g->current;

	if (!c->beginend)
		if (__glerror (__LINE__, __FILE__, GL_INVALID_OPERATION,
			"End called without preceeding Begin"))
			return;

	/* Issue loop vertex */
	if (c->isloop) {
		(*__glbase_TexCoord4fv) ((GLfloat *) &(c->vtx->texcoord.s));
		(*__glbase_Normal3fv)	((GLfloat *) &(c->vtx->normal.x));
		(*__glbase_EdgeFlag)	(c->vtx->edgeflag);
		(*__glbase_Color4fv)	((GLfloat *) &(c->vtx->color.r));

		(*__glbase_Vertex4fv)((GLfloat *) &(c->vtx->pos.x));
		c->isloop = 0;
	}

	__glhw_End();

	c->beginend = GL_FALSE;

	if ( c->flush_on_end ) {
		c->flush_on_end = 0;
		g->bucket.flush(g);
	}

	if ( c->beginendnum >= c->beginendmax) {
		g->bucket.flush(g);
	}
}

void GLSTATE_DECL
__gltrack_Begin (GLenum mode) {
	GLcontext *g = GetCurrentContext();
	GLcurrentstate *c = &g->current;

	__glhw_Begin (mode);

	c->beginend = GL_TRUE;
	c->mode = mode;
}

void GLSTATE_DECL
__gltrack_End (void) {
	GLcontext *g = GetCurrentContext();
	GLcurrentstate *c = &g->current;
	
	__glhw_End();

	c->beginend = GL_FALSE;
}
