/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include <math.h>
#include <float.h>
#include <assert.h>
#include "glcontext.h"
#include "glerror.h"
#include "glhw.h"

#ifdef WINDOWS
#pragma warning( disable : 4514 )  
#endif 

GLint __bktactive = 0;
GLint __bktalwaysbroadcast = 0;
GLint __bktbroadcast = 0;
GLint __bktnobuffer = 0;
GLint __isstate = 0;

GLflushinfo flushinfo;

GLuint __tmp_max;
GLuint __tmp_count;
GLvectorf __tmp_bmin;
GLvectorf __tmp_bmax;

static const GLvectorf maxvector = {FLT_MAX, FLT_MAX, FLT_MAX, FLT_MAX};
static const GLvectorf minvector = {-FLT_MAX, -FLT_MAX, -FLT_MAX, -FLT_MAX};

int optimize_bucket=0;
int numprojectors;
GLrecti p[GLCONFIG_MAX_PROJECTORS][GLCONFIG_MAX_EXTENTS];
int extents[GLCONFIG_MAX_PROJECTORS];
GLrecti imagespace;
float viewport_height_div_2;
float viewport_width_div_2;
float viewport_center_x;
float viewport_center_y;

typedef struct BucketRegion *BucketRegion_ptr;
typedef struct BucketRegion {
	GLbitvalue       id;
	GLrecti          extents;
	BucketRegion_ptr right;
	BucketRegion_ptr up;
} BucketRegion;

#define HASHRANGE 256
BucketRegion *rhash[HASHRANGE][HASHRANGE];
BucketRegion *rlist;
int rlist_alloc = 0;

#define BKT_DOWNHASH(a, range) ((a)*HASHRANGE/(range))
#define BKT_UPHASH(a, range) ((a)*HASHRANGE/(range) + ((a)*HASHRANGE%(range)?1:0))

void 
__glbucket_fillhash (void) {
	int i, j, k, m;
	int r_len=0;
	BucketRegion *rlist;
	GLbitvalue id;

	/* Allocate rlist */
	rlist_alloc = GLCONFIG_MAX_PROJECTORS*GLCONFIG_MAX_EXTENTS;
	rlist = (BucketRegion *) malloc (rlist_alloc * sizeof (*rlist));

	assert (rlist);

	for (i=0; i<HASHRANGE; i++) {
		for (j=0; j<HASHRANGE; j++) {
			rhash[i][j] = NULL;
		}
	}

	/* Fill hash table */
	id = 0;
	for (i=0; i<numprojectors; i++) {
		for (j=0; j<extents[i]; j++) {
			BucketRegion *r = &rlist[id++];
			r->id = 1 << i;
			r->extents = p[i][j];
			
			for (k=BKT_DOWNHASH(r->extents.x1, imagespace.x2);
				 k<=BKT_UPHASH(r->extents.x2, imagespace.x2) && k < HASHRANGE;
				 k++) {
				for (m=BKT_DOWNHASH(r->extents.y1, imagespace.y2);
				     m<=BKT_UPHASH(r->extents.y2, imagespace.y2) && m < HASHRANGE;
					 m++) {
					if ( rhash[m][k] == NULL ||
						(rhash[m][k]->extents.x1 > r->extents.x1 &&
						 rhash[m][k]->extents.y1 > r->extents.y1)) {
						 rhash[m][k] = r;
					}
				}
			}
		}
	}
	r_len = id;

	/* Initialize links */
	for (i=0; i<r_len; i++) {
		BucketRegion *r = &rlist[i];
		r->right = NULL;
		r->up    = NULL;
	}

	/* Build links */
	for (i=0; i<r_len; i++) {
		BucketRegion *r = &rlist[i];
		for (j=0; j<r_len; j++) {
			BucketRegion *q = &rlist[j];
			if (r==q) continue;

			/* Right Edge */
			if (r->extents.x2 == q->extents.x1 &&
				r->extents.y1 == q->extents.y1 &&
				r->extents.y2 == q->extents.y2) {
				r->right = q;
			}

			/* Upper Edge */
			if (r->extents.y2 == q->extents.y1 &&
				r->extents.x1 == q->extents.x1 &&
				r->extents.x2 == q->extents.x2) {
				r->up = q;
			}
		}
	}
}

void
__glstate_SetBroadcast( GLint state )
{
	__bktbroadcast = state;
}

void
__glstate_SendState( GLint state )
{
	__isstate = state;
	__bktalwaysbroadcast = state;
}

void
__glbucket_syncstate ( void ) {
	GLcontext *g = GetCurrentContext ();
   	__bktbroadcast = 1;
	__isstate = 1;
	g->bucket.flush(g);
	__bktbroadcast = 0;
	__isstate = 0;
}

void
__glbucket_init(GLbucketstate *b, GLconfig *c) {
	int i,j;

	b->hint = 0;
	b->active = 0;
	__bktactive = 0;
		
	b->count = &(c->current->vtx_count);
	b->max = &(c->current->vtx_max);
	b->bmin = c->bounds_min;
	b->bmax = c->bounds_max;

	if (!c->current) {
		b->count = &__tmp_count;
		b->max = &__tmp_max;
	}
	if (!b->bmin)
		b->bmin = &__tmp_bmin;
	if (!b->bmax)
		b->bmax = &__tmp_bmax;

	*b->bmin = maxvector;
	*b->bmax = minvector;
	*b->count = 0;
	*b->max = c->bucket_size;

	if (c->numprojectors == 0 || c->broadcast || c->pack_only )
		__bktalwaysbroadcast = 1;

	if (c->numprojectors == 0)
		__bktnobuffer = 1;

	if (c->numprojectors > GLCONFIG_MAX_PROJECTORS)
		__glerror(__LINE__, __FILE__, GL_NO_ERROR, 
		"Cannot support more than %d outputs", GLCONFIG_MAX_PROJECTORS);
	
	numprojectors = c->numprojectors;
	imagespace = c->viewport;
	viewport_width_div_2 = imagespace.x2 / 2.0f;
	viewport_height_div_2 = imagespace.y2 / 2.0f;
	viewport_center_x = viewport_width_div_2;
	viewport_center_y = viewport_height_div_2;

	for (i=0; i<c->numprojectors; i++) {
		
		if (c->numextents[i] > GLCONFIG_MAX_EXTENTS)
			__glerror(__LINE__, __FILE__, GL_NO_ERROR, 
			"Cannot support more than %d extents per output", GLCONFIG_MAX_EXTENTS);

		extents[i] = c->numextents[i];

		for (j=0; j<c->numextents[i]; j++) {
			p[i][j] = c->bounds[i][j];
		}
	}

	if (c->optimize_bucket) {
		optimize_bucket = 1;
		__glbucket_fillhash();
	}
	
	b->flush = c->flush;
}

void
__glbucket_destroy(GLbucketstate *b) {
	UNUSED(b);
}

void __glbucket_bounds(GLint x, GLint y, GLint w, GLint h) {
	
	viewport_width_div_2 = w / 2.0f;
	viewport_height_div_2 = h / 2.0f;
	viewport_center_x = x + viewport_width_div_2;
	viewport_center_y = y + viewport_height_div_2;
}

GLrectf * __glbucket_getclip(GLbitvalue bitID) {

	(void) bitID;
	UNIMPLEMENTED();
	
	return NULL;
}

void
__glbucket_begin() {
	GLcontext *g = GetCurrentContext();
	GLtransstate *t = &(g->trans);
	GLbucketstate *bkt = &(g->bucket);

	if (!t->transformvalid)
		__gltrans_UpdateTransform(t);

	if (__config->compressionpreprocess)
	{
		GLstatebits *sb = GetStateBits();
		sb->trans.dirty = GLBITS_ONES;
		sb->trans.compress = GLBITS_ONES;
		__config->compressionpreprocess(t->transform, bkt->hint == WIREGL_OBJECT_BBOX_HINT);
	}
	
	if (__bktnobuffer)
		bkt->flush(g);

	bkt->active=1;
	__bktactive=1;
}

static 
float _vmult(float *m, float x, float y, float z) {
	float a;
	a =  *m*x;	m += 4;
	a += *m*y;	m += 4;
	a += *m*z;	m += 4;
	a += *m;
	return a;
}

void
__glbucket_drawbbox(void) {
	
	static int init=0;
	static GLfloat c[GLCONFIG_MAX_PROJECTORS][3];
	unsigned int i;
	GLbitvalue a;
	GLfloat outcolor[3] = {0.0f, 0.0f, 0.0f};
	GLfloat tot;
	GLfloat xmin = flushinfo.objectmin.x;
	GLfloat xmax = flushinfo.objectmax.x;
	GLfloat ymin = flushinfo.objectmin.y;
	GLfloat ymax = flushinfo.objectmax.y;
	GLfloat zmin = flushinfo.objectmin.z;
	GLfloat zmax = flushinfo.objectmax.z;

	if (!init) {
		for (i=0; i<GLCONFIG_MAX_PROJECTORS; i++) {
			c[i][0] = (GLfloat) rand();
			c[i][1] = (GLfloat) rand();
			c[i][2] = (GLfloat) rand();
			tot = (GLfloat) sqrt (c[i][0]*c[i][0] + c[i][1]*c[i][1] + c[i][2]*c[i][2]);
			c[i][0] /= tot;
			c[i][1] /= tot;
			c[i][2] /= tot;
		}
		init = 1;
	}		

	tot = 0.0f;
	for (i=0, a=1; i<GLCONFIG_MAX_PROJECTORS; i++, a*=2)
		if (flushinfo.hits&a) {
			outcolor[0] += c[i][0];
			outcolor[1] += c[i][1];
			outcolor[2] += c[i][2];
			tot+=1.0f;
		}
	outcolor[0] /= tot;
	outcolor[1] /= tot;
	outcolor[2] /= tot;

	__glhw_PushAttrib(GL_CURRENT_BIT | GL_ENABLE_BIT);
	__glhw_Disable(GL_TEXTURE_2D);
	__glhw_Disable(GL_TEXTURE_1D);
	__glhw_Disable(GL_LIGHTING);
	__glhw_Disable(GL_BLEND);
	__glhw_Disable(GL_ALPHA_TEST);
	__glhw_Disable(GL_DEPTH_TEST);
	__glhw_Disable(GL_FOG);
	__glhw_Disable(GL_STENCIL_TEST);
	__glhw_Disable(GL_SCISSOR_TEST);
	__glhw_Disable(GL_LOGIC_OP);

	__glhw_Color3fv(outcolor);
	__glhw_Begin(GL_LINE_LOOP);
	__glhw_Vertex3f(xmin, ymin, zmin);
	__glhw_Vertex3f(xmin, ymin, zmax);
	__glhw_Vertex3f(xmin, ymax, zmax);
	__glhw_Vertex3f(xmin, ymax, zmin);
	__glhw_End();
	__glhw_Begin(GL_LINE_LOOP);
	__glhw_Vertex3f(xmax, ymin, zmin);
	__glhw_Vertex3f(xmax, ymin, zmax);
	__glhw_Vertex3f(xmax, ymax, zmax);
	__glhw_Vertex3f(xmax, ymax, zmin);
	__glhw_End();
	__glhw_Begin(GL_LINE_LOOP);
	__glhw_Vertex3f(xmin, ymin, zmin);
	__glhw_Vertex3f(xmax, ymin, zmin);
	__glhw_Vertex3f(xmax, ymax, zmin);
	__glhw_Vertex3f(xmin, ymax, zmin);
	__glhw_End();
	__glhw_Begin(GL_LINE_LOOP);
	__glhw_Vertex3f(xmin, ymin, zmax);
	__glhw_Vertex3f(xmax, ymin, zmax);
	__glhw_Vertex3f(xmax, ymax, zmax);
	__glhw_Vertex3f(xmin, ymax, zmax);
	__glhw_End();

	__glhw_PopAttrib();
}

const GLflushinfo *
__glbucket_flush () {
	GLcontext *g = GetCurrentContext();
	GLbucketstate *bkt = &(g->bucket);
	GLtransstate *t = &(g->trans);
    GLmatrix *m = &(t->transform);
	const GLvectorf zero_vect = {0.0f, 0.0f, 0.0f, 1.0f};
	const GLvectorf one_vect = {1.0f, 1.0f, 1.0f, 1.0f};
	const GLvectorf neg_vect = {-1.0f, -1.0f, -1.0f, 1.0f};
	const GLrecti fullscreen = {-GL_MAXINT, GL_MAXINT, -GL_MAXINT, GL_MAXINT};
	const GLrecti nullscreen = {0, 0, 0, 0};
	float x[8], y[8], z[8], w[8];
	float xmin, ymin, xmax, ymax, zmin, zmax;
	GLrecti ibounds;
	int i,j;

	GLbitvalue retval;
/*  Here is the arragement of the bounding box
**  0 --- 1
**  |\    .\
**  | 2 --- 3 
**  | |   . |
**  | |   . |
**  4.|...5 |
**   \|    .|
**    6 --- 7
**  c array contains the edge connectivitiy list
*/
	static const int c[8][3] = {	{1, 2, 4}, {0, 3, 5}, {0, 3, 6}, {1, 2, 7},
									{0, 5, 6}, {1, 4, 7}, {2, 4, 7}, {3, 5, 6} };

	/* Init flushinfo */
	flushinfo.objectmin = *bkt->bmin;
	flushinfo.objectmax = *bkt->bmax;
	flushinfo.screenmin = zero_vect;
	flushinfo.screenmax = zero_vect;
	flushinfo.pixelbounds = nullscreen;
	flushinfo.hits = 0;
	flushinfo.isstate = __isstate;
		
	/* Check to make sure the transform is valid */
	if (!t->transformvalid)
		__gltrans_UpdateTransform(t);

	if(__config->compressionpostprocess) {
		float objbox[6];
		objbox[0] =	flushinfo.objectmin.x;
		objbox[1] = flushinfo.objectmax.x;
		objbox[2] =	flushinfo.objectmin.y;
		objbox[3] = flushinfo.objectmax.y;
		objbox[4] =	flushinfo.objectmin.z;
		objbox[5] = flushinfo.objectmax.z;
		__config->compressionpostprocess((float *) objbox);
	}

	if (!g->current.beginend) {
		bkt->active = 0;
		__bktactive = 0;
	}

	if (__bktalwaysbroadcast) {
		flushinfo.screenmin = neg_vect;
		flushinfo.screenmax = one_vect;
		flushinfo.pixelbounds = fullscreen;
		flushinfo.hits = GLBITS_ONES;
		return &flushinfo;
	}

	if (bkt->hint != WIREGL_OBJECT_BBOX_HINT &&
        bkt->hint != WIREGL_SCREEN_BBOX_HINT) {
		*bkt->bmin = maxvector;
		*bkt->bmax = minvector;
	}
	
	xmin = flushinfo.objectmin.x;
	ymin = flushinfo.objectmin.y;
	zmin = flushinfo.objectmin.z;
	xmax = flushinfo.objectmax.x;
	ymax = flushinfo.objectmax.y;
	zmax = flushinfo.objectmax.z;

	/* If the screen space hint is turned on skip the transform */
	if (bkt->hint == WIREGL_SCREEN_BBOX_HINT) {
		retval = 0;

		/* triv reject */
		if (xmin > 1.0f || ymin > 1.0f ||
			xmax < -1.0f || ymax < -1.0f) {
			flushinfo.hits = 0;
			return &flushinfo;
		}

		/* clamp */
		if (xmin < -1.0f) xmin = -1.0f;
		if (ymin < -1.0f) ymin = -1.0f;
		if (xmax > 1.0f) xmax = 1.0f;
		if (ymax > 1.0f) xmax = 1.0f;

		ibounds.x1 = (int) (viewport_width_div_2*xmin + viewport_center_x);
		ibounds.x2 = (int) (viewport_width_div_2*xmax + viewport_center_x);
		ibounds.y1 = (int) (viewport_height_div_2*ymin + viewport_center_y);
		ibounds.y2 = (int) (viewport_height_div_2*ymax + viewport_center_y);
		
		for (i=0; i<numprojectors; i++) {
			for (j=0; j<extents[i]; j++) {
				GLrecti *b = &p[i][j];
				if (ibounds.x1 < b->x2 && ibounds.x2 > b->x1 &&
					ibounds.y1 < b->y2 && ibounds.y2 > b->y1) {
					retval |= 1 << i;
					break;
				}
			}
		}


		flushinfo.hits = retval;
		flushinfo.screenmin.x = xmin;
		flushinfo.screenmin.y = ymin;
		flushinfo.screenmax.x = xmax;
		flushinfo.screenmax.y = ymax;
		flushinfo.pixelbounds = ibounds;
		return &flushinfo;
	}

	// if were in broadcast mode, return all ones;
	// If we're defining a display list, it also goes to everybody.
	if (bkt->hint != WIREGL_OBJECT_BBOX_HINT &&
		(__bktbroadcast ||
		__bktalwaysbroadcast ||
		g->lists.newend ||
		bkt->hint == WIREGL_BROADCAST_HINT)) {

		flushinfo.hits = GLBITS_ONES;
		flushinfo.screenmin = neg_vect;
		flushinfo.screenmax = one_vect;
		flushinfo.pixelbounds = fullscreen;
		return &flushinfo;
	}

	// If there are no vertices, return.
	if (bkt->hint != WIREGL_OBJECT_BBOX_HINT &&
		!*(bkt->count)) {		
		flushinfo.hits = 0;
		return &flushinfo;
	}

	//Now transform the bounding box points
	x[0] = _vmult(&(m->m00), xmin, ymin, zmin);
	x[1] = _vmult(&(m->m00), xmax, ymin, zmin);
	x[2] = _vmult(&(m->m00), xmin, ymax, zmin);
	x[3] = _vmult(&(m->m00), xmax, ymax, zmin);
	x[4] = _vmult(&(m->m00), xmin, ymin, zmax);
	x[5] = _vmult(&(m->m00), xmax, ymin, zmax);
	x[6] = _vmult(&(m->m00), xmin, ymax, zmax);
	x[7] = _vmult(&(m->m00), xmax, ymax, zmax);

	y[0] = _vmult(&(m->m01), xmin, ymin, zmin);
	y[1] = _vmult(&(m->m01), xmax, ymin, zmin);
	y[2] = _vmult(&(m->m01), xmin, ymax, zmin);
	y[3] = _vmult(&(m->m01), xmax, ymax, zmin);
	y[4] = _vmult(&(m->m01), xmin, ymin, zmax);
	y[5] = _vmult(&(m->m01), xmax, ymin, zmax);
	y[6] = _vmult(&(m->m01), xmin, ymax, zmax);
	y[7] = _vmult(&(m->m01), xmax, ymax, zmax);

	z[0] = _vmult(&(m->m02), xmin, ymin, zmin);
	z[1] = _vmult(&(m->m02), xmax, ymin, zmin);
	z[2] = _vmult(&(m->m02), xmin, ymax, zmin);
	z[3] = _vmult(&(m->m02), xmax, ymax, zmin);
	z[4] = _vmult(&(m->m02), xmin, ymin, zmax);
	z[5] = _vmult(&(m->m02), xmax, ymin, zmax);
	z[6] = _vmult(&(m->m02), xmin, ymax, zmax);
	z[7] = _vmult(&(m->m02), xmax, ymax, zmax);

	w[0] = _vmult(&(m->m03), xmin, ymin, zmin);
	w[1] = _vmult(&(m->m03), xmax, ymin, zmin);
	w[2] = _vmult(&(m->m03), xmin, ymax, zmin);
	w[3] = _vmult(&(m->m03), xmax, ymax, zmin);
	w[4] = _vmult(&(m->m03), xmin, ymin, zmax);
	w[5] = _vmult(&(m->m03), xmax, ymin, zmax);
	w[6] = _vmult(&(m->m03), xmin, ymax, zmax);
	w[7] = _vmult(&(m->m03), xmax, ymax, zmax);


	/* Find the 2D bounding box of the 3D bounding box */
	xmin = ymin = zmin = FLT_MAX;
	xmax = ymax = zmax = -FLT_MAX;
	
	for (i=0; i<8; i++) {
		float xp = x[i];
		float yp = y[i];
		float zp = z[i];
		float wp = w[i];

		/* If corner is to be clipped... */
		if (zp < -wp) {

			/* Point has three edges */
			for (j=0; j<3; j++) {
				/* Handle the clipping... */
				int k = c[i][j];
				float xk = x[k];
				float yk = y[k];
				float zk = z[k];
				float wk = w[k];
				float t = (wp + zp) / (zp+wp-zk-wk);

				if (t < 0.0f || t > 1.0f)
					continue;
				wp = wp + (wk-wp) * t;
				xp = xp + (xk-xp) * t;
				yp = yp + (yk-yp) * t;
				zp = -wp;

				xp /= wp;
				yp /= wp;
				zp /= wp;
				
				if (xp < xmin) xmin = xp;
				if (xp > xmax) xmax = xp;
				if (yp < ymin) ymin = yp;
				if (yp > ymax) ymax = yp;
				if (zp < zmin) zmin = zp;
				if (zp > zmax) zmax = zp;
			}
		} else {
			/* corner was not clipped.. */
			xp /= wp;
			yp /= wp;
			zp /= wp;
			if (xp < xmin) xmin = xp;
			if (xp > xmax) xmax = xp;
			if (yp < ymin) ymin = yp;
			if (yp > ymax) ymax = yp;
			if (zp < zmin) zmin = zp;
			if (zp > zmax) zmax = zp;
		}
	}

	/* Copy for export */
	flushinfo.screenmin.x = xmin;
	flushinfo.screenmin.y = ymin;
	flushinfo.screenmin.z = zmin;
	flushinfo.screenmax.x = xmax;
	flushinfo.screenmax.y = ymax;
	flushinfo.screenmax.z = zmax;

	/* Now we're down to 4 points
	** (xmin, ymin)
	** (xmax, ymin)
	** (xmin, ymax)
	** (xmax, ymax)
	*/

	/* triv reject */
	if (xmin > 1.0f || ymin > 1.0f ||
		xmax < -1.0f || ymax < -1.0f) {
		flushinfo.hits = 0;
		return &flushinfo;
	}

	/* clamp */
	if (xmin < -1.0f) xmin = -1.0f;
	if (ymin < -1.0f) ymin = -1.0f;
	if (xmax > 1.0f) xmax = 1.0f;
	if (ymax > 1.0f) xmax = 1.0f;

	ibounds.x1 = (int) (viewport_width_div_2*xmin + viewport_center_x);
	ibounds.x2 = (int) (viewport_width_div_2*xmax + viewport_center_x);
	ibounds.y1 = (int) (viewport_height_div_2*ymin + viewport_center_y);
	ibounds.y2 = (int) (viewport_height_div_2*ymax + viewport_center_y);

	flushinfo.pixelbounds = ibounds;

	retval = 0;

	if (!optimize_bucket) {
		for (i=0; i<numprojectors; i++) {
			for (j=0; j<extents[i]; j++) {
				if (ibounds.x1 < p[i][j].x2 && ibounds.x2 > p[i][j].x1 &&
					ibounds.y1 < p[i][j].y2 && ibounds.y2 > p[i][j].y1) {
					retval |= 1 << i;
					break;
				}
			}
		}
	} else {
		BucketRegion *r;
		BucketRegion *q;
		
		for (r = rhash[BKT_DOWNHASH(ibounds.y1, imagespace.y2)][BKT_DOWNHASH(ibounds.x1, imagespace.x2)];
			 r && ibounds.y2 > r->extents.y1;
			 r = r->up) {
			for (q=r; q && ibounds.x2 > q->extents.x1; q = q->right) {
				if (retval & q->id) 
					continue;
				if (ibounds.x1 < q->extents.x2 && /* ibounds.x2 > q->extents.x1 && */
					ibounds.y1 < q->extents.y2 && ibounds.y2 > q->extents.y1) {
					retval |= q->id;
				}
			}
		}
	}

	flushinfo.hits = retval;
	return &flushinfo;
}

GLmatrix *
__glbucket_getcompositematrix(void) {
	GLcontext *g = GetCurrentContext();
	GLtransstate *t = &(g->trans);
    GLmatrix *m = &(t->transform);
	return m;
}

/* Broadcast functions */

void GLSTATE_DECL
__glstate_Accum (GLenum op, GLfloat value) {
	GLcontext *g = GetCurrentContext();
	GLbucketstate *bkt = &(g->bucket);

	if (IsBucketingActive())
		bkt->flush(g);

	if (g->current.beginend)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_OPERATION, 
			"Accum called in begin/end"))
			return;

	if (op != GL_ACCUM &&
		op != GL_LOAD &&
		op != GL_ADD &&
		op != GL_MULT &&
		op != GL_RETURN) {
		if (__glerror(__LINE__, __FILE__, GL_INVALID_ENUM, 
			"Accum called with bogus enum: %d", op))
			return;
	}

	if (__bktnobuffer) {	
		bkt->flush(g);
		__glhw_Accum (op, value);
		return;
	}

	__glhw_Accum (op, value);
	__bktbroadcast = 1;
	__isstate = 1;
	bkt->flush(g);
	__isstate = 0;
	__bktbroadcast = 0;
}


void GLSTATE_DECL
__gltrack_Accum (GLenum op, GLfloat value) {
	__glhw_Accum (op, value);
}

void GLSTATE_DECL
__glstate_CallLists (GLsizei n, GLenum type, const GLvoid *lists) {
	GLcontext *g = GetCurrentContext();
	GLlistsstate *l = &(g->lists);
	GLbucketstate *bkt = &(g->bucket);

	if (bkt->active || __bktnobuffer)
		bkt->flush(g);

	__glhw_ListBase (l->base);
	__glhw_CallLists (n, type, lists);
	__bktbroadcast = 1;
	bkt->flush(g);
	__bktbroadcast = 0;
}

void GLSTATE_DECL
__glstate_CallList (GLuint list){ 
	GLcontext *g = GetCurrentContext();
	GLbucketstate *bkt = &(g->bucket);

	if (bkt->active || __bktnobuffer)
		bkt->flush(g);

	__glhw_CallList (list);
	__bktbroadcast = 1;
	bkt->flush(g);
	__bktbroadcast = 0;
}

void GLSTATE_DECL
__glstate_Clear (GLbitfield mask) {
	GLcontext *g = GetCurrentContext();
	GLbucketstate *bkt = &(g->bucket);
	GLbitvalue update;

	if (g->current.beginend)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_OPERATION, 
			"Clear called in begin/end"))
			return;

	if (mask & 
		~GL_COLOR_BUFFER_BIT & 
		~GL_DEPTH_BUFFER_BIT &
		~GL_ACCUM_BUFFER_BIT &
		~GL_STENCIL_BUFFER_BIT)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_VALUE, 
			"Clear called with bogus mask: %d", mask))
			return;

	if (IsBucketingActive())
		bkt->flush(g);

	if (__bktnobuffer) {	
		update = g->update;
		g->update = GLUPDATE_BUFFER | GLUPDATE_VIEWPORT;
		bkt->flush(g);
		g->update = update;
		__bktbroadcast = 0;
		__glhw_Clear (mask);
		return;
	}

	__glhw_Clear (mask);

	__bktbroadcast = 1;
	__isstate = 1;
	update = g->update;
	g->update = GLUPDATE_BUFFER | GLUPDATE_STENCIL | GLUPDATE_VIEWPORT;
	bkt->flush(g);
	g->update = update;
	__bktbroadcast = 0;
	__isstate = 0;

}

void GLSTATE_DECL
__gltrack_Clear (GLbitfield mask) {
	__glhw_Clear (mask);
}


void GLSTATE_DECL 
__glstate_DrawPixels (GLsizei width, GLsizei height, 
		GLenum format, GLenum type, const GLvoid *pixels)
{
	GLcontext *g = GetCurrentContext();
	GLcurrentstate *c = &(g->current);
	GLbucketstate *bkt = &(g->bucket);
	
	GLviewportstate *v = &(g->viewport);

	GLenum hint;

	if (c->beginend)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_OPERATION, 
			"DrawPixels called in begin/end"))
			return;

	if (width < 0 || height < 0)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_VALUE,
			"DrawPixels called with neg dims: %dx%d", width, height))
			return;

	if (!c->rastervalid)
		return;

	if (IsBucketingActive() || __bktnobuffer)
		bkt->flush(g);

	bkt->bmin->x = (c->rasterpos.x)/v->v_w;
	bkt->bmin->y = (c->rasterpos.y)/v->v_h;
    bkt->bmax->x = (c->rasterpos.x + width)/v->v_w;
    bkt->bmax->y = (c->rasterpos.y + height)/v->v_h;

	bkt->bmin->x *= 2.0f;
	bkt->bmin->y *= 2.0f;
	bkt->bmax->x *= 2.0f;
	bkt->bmax->y *= 2.0f;

	bkt->bmin->x -= 1.0f;
	bkt->bmin->y -= 1.0f;
	bkt->bmax->x -= 1.0f;
	bkt->bmax->y -= 1.0f;

	hint = bkt->hint;
	bkt->hint = WIREGL_SCREEN_BBOX_HINT;

	/* don't do a flush, DrawPixels understand that it needs to flush,
       and will handle all that for us */
	__glhw_DrawPixels (width, height, format, type, pixels);

	bkt->hint = hint;
	*bkt->bmin = maxvector;
    *bkt->bmax = minvector;
}

void GLSTATE_DECL
__glstate_Bitmap (	GLsizei width, GLsizei height,
					GLfloat xorig, GLfloat yorig,
					GLfloat xmove, GLfloat ymove,
					const GLubyte * bitmap) {
	GLcontext *g = GetCurrentContext();
	GLbucketstate *bkt = &(g->bucket);
	GLcurrentstate *c = &(g->current);
	GLviewportstate *v = &(g->viewport);
	GLstatebits *s = GetStateBits();
	GLcurrentbits *cb = &(s->current);
	GLenum hint;
	
	if (g->current.beginend)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_OPERATION, 
			"Bitmap called in begin/end"))
			return;

	if (width < 0 || height < 0)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_VALUE,
			"Bitmap called with neg dims: %dx%d", width, height))
			return;

	if (!c->rastervalid)
		return;

	if (IsBucketingActive() || __bktnobuffer)
		bkt->flush(g);

	bkt->bmin->x = (c->rasterpos.x - xorig)/v->v_w;
	bkt->bmin->y = (c->rasterpos.y - yorig)/v->v_h;
    bkt->bmax->x = (c->rasterpos.x - xorig + width)/v->v_w;
    bkt->bmax->y = (c->rasterpos.y - yorig + height)/v->v_h;

	bkt->bmin->x *= 2.0f;
	bkt->bmin->y *= 2.0f;
	bkt->bmax->x *= 2.0f;
	bkt->bmax->y *= 2.0f;

	bkt->bmin->x -= 1.0f;
	bkt->bmin->y -= 1.0f;
	bkt->bmax->x -= 1.0f;
	bkt->bmax->y -= 1.0f;

	c->rasterpos.x += xmove;
	c->rasterpos.y += ymove;
	cb->raster = g->nbitID;
	cb->dirty = g->nbitID;

	hint = bkt->hint;
	bkt->hint = WIREGL_SCREEN_BBOX_HINT;

	/* don't flush, Bitmap will handle that */
	__glhw_Bitmap (width, height, xorig, yorig, xmove, ymove, bitmap);

	bkt->hint = hint;
	*bkt->bmin = maxvector;
    *bkt->bmax = minvector;

	c->rasterpos_pre.x += xmove;
	c->rasterpos_pre.y += ymove;
}

void GLSTATE_DECL
__gltrack_Bitmap (	GLsizei width, GLsizei height,
					GLfloat xorig, GLfloat yorig,
					GLfloat xmove, GLfloat ymove,
					const GLubyte * bitmap) {
	GLcontext *g = GetCurrentContext();
	GLcurrentstate *c = &(g->current);
	GLstatebits *s = GetStateBits();
	GLcurrentbits *cb = &(s->current);

	c->rasterpos.x += xmove;
	c->rasterpos.y += ymove;
	cb->raster = g->nbitID;
	cb->dirty = g->nbitID;

	__glhw_Bitmap (width, height, xorig, yorig, xmove, ymove, bitmap);

	c->rasterpos_pre.x += xmove;
	c->rasterpos_pre.y += ymove;
}

