/*
 *  view.c, version 1.0alpha1.
 *    View a light field.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <gl/gl.h>
#include <gl/device.h>
#include <gl/image.h>
#include <unistd.h>
#include "lightfield.h"

#define FZ 0.0
#define EPSILON 0.001

#define DEFAULTSIZEX 192
#define DEFAULTSIZEY 192

#define BASENAME "."

Matrix Identity =
	{ 1, 0, 0, 0,  0, 1, 0, 0,  0, 0, 1, 0,  0, 0, 0, 1 };

#define MENU_POINT	1
#define MENU_UVINTER	2
#define MENU_STINTER	3
#define MENU_ZOOM1      4
#define MENU_ZOOM2      5
#define MENU_ZOOM3      6
#define MENU_REFINE1    7
#define MENU_REFINE2    8
#define MENU_REFINE3    9
#define MENU_RESET	10
#define MENU_EXIT	11

static char menustr[] =
    "Options %t|Point|UV Interpolate|ST Interpolate %l|"
    "Pixel Zoom 1|Pixel Zoom 2|PixelZoom 3 %l|"
    "Never Auto-Refine |Auto-Refine UVST|Auto-Refine Zoom %l|"
    "Reset View %l|Exit";

#define RESETXOBS 0.0
#define RESETYOBS 0.0
#define RESETZOBS 1.985

#define INITXCENTER 0.0
#define INITYCENTER 0.0
#define INITZCENTER 0.0

#define	UVLERP	0x1
#define	STLERP	0x2

#define REFINE_INTERP 0x1
#define REFINE_ZOOM   0x2

#define	ROTATEX	0x1
#define	ROTATEY	0x2
#define	ROTATEZ	0x4


#define SIGN(x) (((x)<0) ? -1 : 1)

float xobs = RESETXOBS, yobs = RESETYOBS, zobs = RESETZOBS;
float xcenter = INITXCENTER;
float ycenter = INITYCENTER;
float zcenter = INITZCENTER;
float fov = 40;
long xsize = DEFAULTSIZEX;
long ysize = DEFAULTSIZEY;
extern float pixelzoom;	   /* defined in irisgl_driver.c */
int shift = 0;

/* This is used as the temporary file if we need to gunzip it. */
char tempLif[1000] = "/tmp/current.lif";


typedef enum {
  LF_LID,
  LF_LIF,
  LF_LIFBADVER,
  LF_GZ,
  LF_UNKNOWN
} LF_FileType;

float xrot = 0;
float yrot = 0;
float zrot = 0;

char *baseName = NULL;
int interp = UVLERP;
bool_t autoRefine = REFINE_INTERP;

/* Local helper functions */
void loadview();
void resetview();
void ParseArgs(int argc, char *argv[]);
void PrintUsage(char *programName);
LF_FileType DetermineFileType(char *fileName);

int
main(int argc, char *argv[])
{
  long xorig, yorig;
  float lx, ly, mx, my, dx, dy; 
  float rold, rnew;
  float timing;
  short val;
  long dev;
  int leftdown = 0, middledown = 0;
  int done = 0;
  long menuid;
  int animate = 0;
  int rotate = ROTATEX | ROTATEY;
  int redraw = 1;
  int refineredraw = REFINE_INTERP;
  int frame = 0;
  int single_frame = 0;
  time_t start;
  float clamp_ratio;
  int i;
  float newxobs, newyobs, newzobs;
  float newxcenter, newycenter, newzcenter;
  char *suffixPtr;
  /* Four draw descriptions, for the 4 types of interpolation */
  float draw_descr[4][3] = {
    {LF_DRAW_MAG_FILTER, LF_NEAREST_UVST, LF_NULL },
    {LF_DRAW_MAG_FILTER, LF_BILINEAR_UV_NEAREST_ST, LF_NULL },
    {LF_DRAW_MAG_FILTER, LF_NEAREST_UV_BILINEAR_ST, LF_NULL },
    {LF_DRAW_MAG_FILTER, LF_QUADLINEAR_UVST, LF_NULL }}; 
  float compress_descr[] = {
    LF_NULL,
  };
  float read_descr[] = {
    LF_NULL,
  };
  LF_FileType fileType = LF_UNKNOWN;

  /* Put camera in default position */
  resetview();

  /* Parse the input arguments (verbose flag, input, etc) */
  ParseArgs(argc, argv);

  /* Set window size (if environment variable specifies it) */
  if  (getenv("LF_WIN_SIZE")) {
    sscanf(getenv("LF_WIN_SIZE"), "%d%d", &xsize, &ysize);
  }

  lfInitContext();

  /* Determine the type of file:
   * .lif, .lid, .gz, or other...
   */
  fileType = DetermineFileType(baseName);
  
  /* gunzip, if necessary */
  if (fileType == LF_GZ) {
    char cmdLine[1024];
    char *tempDir=NULL;
    /* Figure out if TEMP environment variable is set */
    tempDir = getenv("TEMP");
    if (tempDir != NULL) {
      sprintf(tempLif, "%s/current.lif", tempDir);
    } else {
      sprintf(tempLif, "current.lif");
    }

    /* gunzip, run DetermineFileType again... */
    sprintf(cmdLine, "gunzip -c %s > %s\n", baseName, tempLif);
    lfOutput("Executing: %s\n", cmdLine);
    if (system(cmdLine) != 0) {
      /* if gunzip fails, try gzip -d */
      sprintf(cmdLine, "gzip -dc %s > %s\n", baseName, tempLif);
      lfOutput("Attempting: %s\n", cmdLine);
      if (system(cmdLine) != 0) {
	/* If both fail, let user know */
	lfError("Unable to run gunzip/%s\n", cmdLine);
	exit(-1);
      }
    }
    lfOutput("Done!\n");
    baseName = strdup(tempLif);
    fileType = DetermineFileType(baseName);
  }


  if (fileType == LF_LIF) {
    /* Read in using new file format */
    lfOutput("Reading light field file %s\n", baseName);
    lfBegin(LF_FIELD, NULL);
    lfRead(LF_LIGHTFIELD, read_descr, baseName);
    lfEnd();
  } else if (fileType == LF_LID) {
    /* Read in using old file format */
    lfOutput("Reading/generating light field from slice images.\n");
    /* Read in the file, VQ compress it */
    lfBegin(LF_FIELD, NULL);
    lfRead(LF_SLICE_ST, read_descr, baseName);
    /*     lfCompress(LF_VQ, compress_descr); */
    lfEnd();
  } else {
    exit(-1);
  }

  /* If we gunzipped to create a temporary file, delete it. */
  if (!strcmp(baseName, tempLif)) {
    char cmdLine[1024];
    sprintf(cmdLine, "rm -f %s\n", tempLif);
    system(cmdLine);
    lfOutput("Removing %s \n", tempLif);
  }

  /* Now set up to start drawing */
  foreground();
  keepaspect(1,1);
  if  (xsize && ysize)
    prefsize(pixelzoom*xsize, pixelzoom*ysize);
  winopen("light");
  winconstraints();
  getorigin(&xorig, &yorig);
  getsize(&xsize, &ysize);
  xsize /= pixelzoom;
  ysize /= pixelzoom;
  RGBmode();
  doublebuffer();
  gconfig();
  subpixel(TRUE);
  backface(TRUE);
  polymode(PYM_FILL);
  blendfunction(BF_ONE, BF_ZERO);
  mmode(MVIEWING);

  /* Set up events to watch                                            */
  /* (Stuff from Marc's viewer) */
  qdevice(REDRAW);		/* Mainly window resizing                    */
  qdevice(LEFTMOUSE);		/* Panning (mouse X) and tilting (mouse Y)   */
  tie(LEFTMOUSE, MOUSEX, MOUSEY);
  qdevice(MIDDLEMOUSE);		/* Zooming (mouse Y)                         */
  tie(MIDDLEMOUSE, MOUSEX, MOUSEY);
  qdevice(RIGHTMOUSE);		/* Bring up popup menu                       */
  qdevice(UPARROWKEY);		/* Unshifted moves observer up/down          */
  qdevice(DOWNARROWKEY);	/* shifted changes fov                       */
  qdevice(LEFTARROWKEY);	/* Make a small change in longitude          */
  qdevice(RIGHTARROWKEY);	/* Same, in the other direction              */
  qdevice(ESCKEY);		/* Exits program                             */
  qdevice(LEFTSHIFTKEY);	/* Switches left and middle mouse buttons    */
  qdevice(RIGHTSHIFTKEY);	/* from moving observer to moving center     */
  qdevice(AKEY);		/* Turn animation on			     */
  qdevice(RKEY);		/* Reset animation                           */
  qdevice(SKEY);                /* Stop animation                            */
  qdevice(SPACEKEY);		/* Reset viewing location		     */
  qdevice(ESCKEY);		/* quit					     */
  menuid = defpup(menustr);

  start = time(NULL);

    /*
     *	main event loop
     */
  while (!done)  { 
    redraw = 0;
    while (qtest() != 0)  {
      dev = qread(&val);
      switch (dev)  {
      case REDRAW:
	  reshapeviewport();
	  getorigin(&xorig, &yorig);
	  getsize(&xsize, &ysize);
	  xsize /= pixelzoom;
	  ysize /= pixelzoom;
	  redraw= 1;
	  break;
      case LEFTMOUSE:
	leftdown = val;
	/* Read current mouse position, which was tied to the mouse button. */
	dev = qread(&val);
	lx = ((float) val-xorig)/(xsize*pixelzoom);
	dev = qread(&val);
	ly = ((float) val-yorig)/(ysize*pixelzoom);
	/* Turn on auto-refine if mouse released */
	if (autoRefine && !leftdown && !middledown) refineredraw = autoRefine;
	else refineredraw = 0;
	break;

      case MIDDLEMOUSE:
	middledown = val;
	/* Read current mouse position, which was tied to the mouse button. */
	dev = qread(&val);
	lx = ((float) val-xorig)/(xsize*pixelzoom);
	dev = qread(&val);
	ly = ((float) val-yorig)/(ysize*pixelzoom);
	/* Turn on auto-refine if mouse released */
	if (autoRefine && !leftdown && !middledown) refineredraw = autoRefine;
	else refineredraw = 0;
	break;

      case RIGHTMOUSE:
	{
	  time_t newt;
	  int menuval;
	  setpup(menuid, MENU_POINT, (interp) ? PUP_BOX : PUP_CHECK);
	  setpup(menuid, MENU_UVINTER, (interp&UVLERP) ? PUP_CHECK : PUP_BOX);
	  setpup(menuid, MENU_STINTER, (interp&STLERP) ? PUP_CHECK : PUP_BOX);
	  setpup(menuid, MENU_ZOOM1, (pixelzoom == 1) ?  PUP_CHECK : PUP_BOX);
	  setpup(menuid, MENU_ZOOM2, (pixelzoom == 2) ?  PUP_CHECK : PUP_BOX);
	  setpup(menuid, MENU_ZOOM3, (pixelzoom == 3) ?  PUP_CHECK : PUP_BOX);
	  setpup(menuid, MENU_REFINE1, (!autoRefine) ?  PUP_CHECK : PUP_BOX);
	  setpup(menuid, MENU_REFINE2, (autoRefine&REFINE_INTERP) ?  PUP_CHECK : 
		 PUP_BOX);
	  setpup(menuid, MENU_REFINE3, (autoRefine&REFINE_ZOOM) ?  PUP_CHECK : 
		 PUP_BOX);
	  menuval = dopup(menuid);
	  if (menuval < 1)  break;

	  /* Figure out current interpolation state */
	  switch (menuval) {
	  case MENU_POINT:
	    interp = 0;
	    break;
	  case MENU_UVINTER:
	    if( interp & UVLERP )
	      interp &= ~UVLERP;
	    else
	      interp |= UVLERP;
	    break;
	  case MENU_STINTER:
	    if( interp & STLERP )
	      interp &= ~STLERP;
	    else
	      interp |= STLERP;
	    break;
	  case MENU_ZOOM1:
	    pixelzoom = 1;
	    break;
	  case MENU_ZOOM2:
	    pixelzoom = 2;
	    break;
	  case MENU_ZOOM3:
	    pixelzoom = 3;
	    break;
	  case MENU_REFINE1:
	    autoRefine = 0;
	    break;
	  case MENU_REFINE2:
	    autoRefine ^= REFINE_INTERP;
	    break;
	  case MENU_REFINE3:
	    autoRefine ^= REFINE_ZOOM;
	    break;
	  case MENU_RESET:
	    resetview();
	    break;
	  case MENU_EXIT:
	    done = 1;
	    break;
	  }

	  /* We no longer need to set the draw_descr variables here,
	   * because it just picks the right one when it draws. 
	   */

	  newt = time(NULL);
	  lfGetParameters(LF_DRAW, LF_DRAW_SCANLINE_CLAMP_RATIO, &clamp_ratio);
	  lfGetParameters(LF_DRAW, LF_DRAW_TIMING, &timing);
	  lfOutput("frame = %.3f clamp ratio = %.3f timing = %.3f\n", 
		 frame / (float)(newt - start), clamp_ratio, 
		 timing * 1000.f / frame);
	  start = newt;
	  frame = 0;

	  /*lfSetParameters(LF_DRAW, draw_descr); */
	}
	redraw = 1;
	break;
      case SPACEKEY:
	if(val) {
	  resetview();
	  redraw=1;
	}
	break;
      case UPARROWKEY:
	if(val) fov *= 1.1;
	redraw=1;
	break;
      case DOWNARROWKEY:
	if(val) fov /= 1.1;
	redraw=1;
	break;
      
      case LEFTARROWKEY:
	if  (val) single_frame = -1;
	redraw= 1;
	break;

      case RIGHTARROWKEY:
	if  (val) single_frame = 1;
	redraw= 1;
	break;

      case LEFTSHIFTKEY:
      case RIGHTSHIFTKEY:
	shift = val;
	break;

      case ESCKEY:
	if (val)  done = 1;
	break;

      case AKEY:
	if (val)  animate = 1;
	break;

      case SKEY:
	if (val)  animate = 0;
	break;

      case RKEY:
	if (val) {
	  /*static float draw_descr[] = {
	    LF_DRAW_STATS_RESET,
	    LF_NULL,
	  };	*/
	  time_t newt = time(NULL);

	  lfGetParameters(LF_DRAW, LF_DRAW_SCANLINE_CLAMP_RATIO, &clamp_ratio);
	  lfGetParameters(LF_DRAW, LF_DRAW_TIMING, &timing);
	  lfOutput("frame = %.3f clamp ratio = %.3f timing = %.3f\n", 
		 frame / (float)(newt - start), clamp_ratio, 
		 timing * 1000.f / frame);
	  /*	  lfSetParameters(LF_DRAW, draw_descr); */

	  start = newt;
	  frame = 0;
	}
	break;

      default:
	break;

      } /* end: switch (dev) */

    } /* end:  while (qtest() != 0) */

    if (leftdown || middledown) {
      /* Only do mouse motion when the buttons are down! */
      mx = ((float)getvaluator(MOUSEX)-xorig)/(xsize*pixelzoom);
      my = ((float)getvaluator(MOUSEY)-yorig)/(ysize*pixelzoom);
      dx = 2*(mx - lx);
      dy = 2*(my - ly);
      lx = mx;
      ly = my;

      /* Mouse interactions -- pasted from Marc's viewer */
      if (leftdown && middledown) {
	/* Mouse Y maps to zooming toward/from center
	 * mouse X does nothing
	 * Movement slows down as you approach the far plane
	 * Don't let observer reach far plane unless multifield
	 */
	if (dy < 0) {
	  xobs = xobs+(xobs-xcenter)*dy;
	  yobs = yobs+(yobs-ycenter)*dy;
	  zobs = zobs+(zobs-zcenter)*dy;
	}
	else {
	  xobs = xcenter+(xobs-xcenter)/(1-dy);
	  yobs = ycenter+(yobs-ycenter)/(1-dy);
	  zobs = zcenter+(zobs-zcenter)/(1-dy);
	}
	/* Restore dy to original sign */
	dy = -dy;
      }
      else if (leftdown) {
	/* Mouse XY maps to longitude and lattitude
	 * as observer orbits center at fixed distance
	 * If shifted, suppress lattitude changes,
	 * easing flyarounds of full-surround fields
	 */
	float aold,anew;
	rold = sqrt((xobs-xcenter)*(xobs-xcenter)+
		    (zobs-zcenter)*(zobs-zcenter));
	aold = fatan((xobs-xcenter)/(zobs-zcenter)) / M_PI*180.0;
	if (ABS(zobs-zcenter) < EPSILON) aold = 90.0 * 
					   SIGN(xobs-xcenter);
	else if (zobs<zcenter) aold = 180.0 + aold;
	anew = aold - dx * 90.0;

	xobs = xcenter + rold * fsin(anew/180.0*M_PI);
	zobs = zcenter + rold * fcos(anew/180.0*M_PI);

	rold = sqrt((xobs-xcenter)*(xobs-xcenter)+
		    (yobs-ycenter)*(yobs-ycenter)+
		    (zobs-zcenter)*(zobs-zcenter));
	newyobs = yobs-dy*rold;
	if (!shift) {
	  yobs = newyobs;
	}
	rnew = sqrt((xobs-xcenter)*(xobs-xcenter)+
		    (yobs-ycenter)*(yobs-ycenter)+
		    (zobs-zcenter)*(zobs-zcenter));
	newxobs = (xobs-xcenter)*(rold/rnew)+xcenter;
	newyobs = (yobs-ycenter)*(rold/rnew)+ycenter;
	newzobs = (zobs-zcenter)*(rold/rnew)+zcenter;
	xobs = newxobs;
	if (!shift) {
	  yobs = newyobs;
	}
	zobs = newzobs;
      }
      else if (middledown) {
	/* Mouse XY maps to translation of viewed image      */
	/* achieved by matched moves of observer and center  */
	/* perpendicular to the ray connecting them          */
	/* Such translation shows off horizontal parallax   */
	if (ABS(zcenter-zobs) > ABS(xcenter-xobs)) {
	  /* Front and back views */
	  newxcenter = xcenter+(zcenter-zobs)*dx/2;
	  newzcenter = zcenter-(xcenter-xobs)*dx/2;
	  newycenter = ycenter-ABS(zcenter-zobs)*dy/2;
	  newzcenter = newzcenter-(ycenter-yobs)*dy/2;
	  newxobs = xobs+(zcenter-zobs)*dx/2;
	  newzobs = zobs-(xcenter-xobs)*dx/2;
	  newyobs = yobs-ABS(zcenter-zobs)*dy/2;
	  newzobs = newzobs-(ycenter-yobs)*dy/2;
	}
	else {
	  /* Side views */
	  newzcenter = zcenter-(xcenter-xobs)*dx/2;
	  newxcenter = xcenter+(zcenter-zobs)*dx/2;
	  newycenter = ycenter-ABS(xcenter-xobs)*dy/2;
	  newxcenter = newxcenter-(ycenter-yobs)*dy/2;
	  newzobs = zobs-(xcenter-xobs)*dx/2;
	  newxobs = xobs+(zcenter-zobs)*dx/2;
	  newyobs = yobs-ABS(xcenter-xobs)*dy/2;
	  newxobs = newxobs-(ycenter-yobs)*dy/2;
	}
	xcenter = newxcenter;
	ycenter = newycenter;
	zcenter = newzcenter;
	/* If shifted, movement of observer is suppressed, */
	/* causing mouse XY to map to viewing direction at */
	/* the (fixed) current position, which exhibits    */
	/* no horizontal parallax but is often useful      */
	if (!shift) {
	  xobs = newxobs;
	  yobs = newyobs;
	  zobs = newzobs;
	} 
      }

    }

    else if (animate || single_frame) {
      if  (animate > 0 || single_frame > 0)  {
	if  (rotate & ROTATEX)
	  xrot += .15;
	if  (rotate & ROTATEY)
	  yrot += .15;
	if  (rotate & ROTATEZ)
	  zrot += .15;
	if  (xrot > 15)
	  animate = -1;
      }
      else {
	if  (rotate & ROTATEX)
	  xrot -= .15;
	if  (rotate & ROTATEY)
	  yrot -= .15;
	if  (rotate & ROTATEZ)
	  zrot -= .15;
	if  (xrot < -15)
	  animate = 1;
      }
      single_frame = 0;
    }

    if( redraw || refineredraw || leftdown || middledown) {
      czclear(0x000000, getgdesc(GD_ZMAX));

      loadview();
      if (refineredraw && autoRefine) {
	int tempzoom = pixelzoom;
	/* mouse not down - high res. redraw */
	if (refineredraw & REFINE_ZOOM) pixelzoom = 1;	
	lfBegin(LF_FIELD, NULL);
	if (refineredraw & REFINE_INTERP) {
	  lfDraw(LF_COLOR, &(draw_descr[UVLERP + STLERP][0]), NULL);
	} else {
	  lfDraw(LF_COLOR, &(draw_descr[interp][0]), NULL);
	}
	lfEnd();
	if (refineredraw & REFINE_ZOOM) pixelzoom = tempzoom;
	refineredraw = 0;
      } else {
	/* normal redraw */
	lfBegin(LF_FIELD, NULL);
	lfDraw(LF_COLOR, &(draw_descr[interp][0]), NULL);
	lfEnd();
	/* If no motion, schedule higher-res redraw */
	if (autoRefine && !leftdown && !middledown) {
	  refineredraw=autoRefine;
	} else {
	  refineredraw=0;
	}
      }
      swapbuffers();
      frame++;
    }
  }

  lfGetParameters(LF_DRAW, LF_DRAW_SCANLINE_CLAMP_RATIO, &clamp_ratio);
  lfGetParameters(LF_DRAW, LF_DRAW_TIMING, &timing);
  lfOutput("frame = %.3f clamp_ratio = %.3f timing = %.3f\n", 
	 frame / (float)(time(NULL) - start), clamp_ratio, 
	 timing * 1000.f / frame);
  
}

void view(float fov, float aspect, float n, float f)
{
  float T[4][4];
#define RADIANS(a) ((3.14159/180)*(a))
  float cotfov2 = 1./tan(RADIANS(fov/2.));
  T[0][0] = cotfov2/aspect;
  T[0][1] = 0.;
  T[0][2] = 0.;
  T[0][3] = 0.;
    
  T[1][0] = 0.;
  T[1][1] = cotfov2;
  T[1][2] = 0.;
  T[1][3] = 0.;

  T[2][0] = 0.;
  T[2][1] = 0.;
  T[2][2] = -1.;
  T[2][3] = -1.;

  T[3][0] = 0.;
  T[3][1] = 0.;
  T[3][2] = -2*n;;
  T[3][3] = 0.;

  mmode(MPROJECTION);
  loadmatrix(T);
  /*
    perspective((int)(10*fov), xsize/(float)ysize, .001, 1000.0);
    */
}

void loadview()
{
  view(fov, xsize/(float)ysize, .001, 1000.0);
  mmode(MVIEWING);
  loadmatrix(Identity);
  lookat(xobs,yobs,zobs, xcenter, ycenter, zcenter, 0.0);
  rot(yrot,'y');
  rot(xrot,'x');
  rot(zrot,'z');
  /* printf("rot %g y\n",yrot); */
}

void resetview()
{
  /* Move the viewer back to the starting point */
  xobs=RESETXOBS;
  yobs=RESETYOBS;
  zobs=RESETZOBS;
  xcenter = INITXCENTER;
  ycenter = INITYCENTER;
  zcenter = INITZCENTER;
  xrot = 0;
  yrot = 0;
  zrot = 0;
}

void ParseArgs(int argc, char *argv[])
{
  char *progName = argv[0];
  
  argc--;
  argv++;

  while (argc > 0) {
    /* Verbose flag */
    if (!strcmp(argv[0], "-v")) {
      lfVerbose = 1;
      argc--;
      argv++;
    } 
    /* size */
    else if (!strcmp(argv[0], "-size")) {
      if (argc < 3) PrintUsage(progName);
      xsize = atoi(argv[1]);
      ysize = atoi(argv[2]);
      /* error checking */
      if (xsize <2 || ysize < 2 || xsize > 2048 || ysize > 2048) {
	lfError("Invalid window size... using defaults\n");
	xsize = DEFAULTSIZEX;
	ysize = DEFAULTSIZEY;
      }
      argc -= 3;
      argv += 3;
    } 

    /* zoom */
    else if (!strcmp(argv[0], "-zoom")) {
      if (argc < 2) PrintUsage(progName);
      pixelzoom = atoi(argv[1]);
      /* error checking */
      if (pixelzoom < 1 || pixelzoom > 100) {
	lfError("Invalid zoom scale... using default\n");
	pixelzoom = 2.0;
      }
      argc -= 2;
      argv += 2;
    } 
    
    /* Interpolation mode */
    else if (!strcmp(argv[0], "-lerp")) {
      if (argc < 2) PrintUsage(progName);
      if (!strcmp(argv[1], "none")) {
	interp = 0;
      } else if (!strcmp(argv[1], "uv")) {
	interp = UVLERP;
      } else if (!strcmp(argv[1], "st")) {
	interp = STLERP;
      } else if (!strcmp(argv[1], "uvst")) {
	interp = STLERP + UVLERP;
      } else {
	lfError("Invalid lerp mode: %s\n", argv[1]);
	PrintUsage(progName);
      }
      argc -= 2;
      argv += 2;
    }

    /* Auto-Refine */
    else if (!strcmp(argv[0], "-refine")) {
      if (argc < 2) PrintUsage(progName);
      if (!strcmp(argv[1], "all")) {
	autoRefine = REFINE_INTERP + REFINE_ZOOM;
      } else if (!strcmp(argv[1], "uvst")) {
	autoRefine = REFINE_INTERP;
      } else if (!strcmp(argv[1], "zoom")) {
	autoRefine = REFINE_ZOOM;
      } else if (!strcmp(argv[1], "off")) {
	autoRefine = 0;
      } else {
	lfError("Invalid auto-refine mode: %s\n", argv[1]);
	PrintUsage(progName);
      }
      argc -= 2;
      argv += 2;
    }
    
    /* filename */
    else if (strncmp(argv[0], "-", 1)) {
      if (baseName == NULL) {
	baseName = argv[0];
      } else {
	lfError("Ignoring extra file %s\n", baseName);
      } 
      argc--;
      argv++;
    } 
    
    /* default */
    else {
      PrintUsage(progName);
      exit(-1);
    }
  }

  /* Check to make sure baseName has been set */
  if (baseName == NULL) {
    PrintUsage(progName);
  }
}
  

void PrintUsage(char *programName)
{
  fprintf(stderr, 
	  "lifview v.1.0.\n"
	  "Usage: %s [-v] [-size <x> <y>] [-zoom <z>] [-lerp <mode>] \n"
	  "               [-refine <mode>] <lightfield>\n"
	  "\n"
	  "     -v              Verbose on\n"
	  "\n"
	  "     -size <x> <y>   Size (in pixels) of display window\n"
	  "\n"
	  "     -zoom <z>       Zoom factor for display window\n"
	  "                     (The actual window size will be x*z, y*z)\n"
	  "\n"
	  "     -lerp uvst      Start with full uvst interpolation filter\n"
	  "     -lerp uv        Start with only uv interpolation (default)\n"
	  "     -lerp st        Start with only st interpolation\n"
	  "     -lerp none      Start with point sampling\n"
	  "\n"
	  "     -refine all     Auto-refine image when viewer isn't moving\n"
	  "                     (this refines both uvst and zoom)\n"
	  "     -refine uvst    Auto-refine uvst interpolation only\n"
	  "     -refine zoom    Auto-refine pixel zoom only\n"
	  "     -refine off     Never auto-refine image\n"
	  "\n"
	  "     <lightfield>    This can be the name of either a .lif\n"
	  "                     file or a .lid file.  If it is a .lid\n"
	  "                     file, it will tell lifview where to\n"
	  "                     find the raw .rgb images.\n"
	  "\n"
	  , programName);
  
  exit(-1);
}

LF_FileType DetermineFileType(char *fileName)
{
  LF_FileType type = LF_UNKNOWN;
  char buffer[20];
  FILE *inFile = NULL;
  char *suffixPtr, *tPtr;

  /* Open the input file */
#ifdef sgi
  inFile = fopen(fileName, "rb");
#else /* WIN32 */
  inFile = fopen(fileName, "r+b");
#endif 
  if (!inFile) {
    lfError("Cannot open file %s\n", fileName);
    exit(-1);
  }
  
  /* Get the first 8 characters of the file to determine its type */
  fgets(buffer, 9, inFile);
  
  switch(buffer[0]) {
  case 'L':  /* LIF1.0 */
    if (!strncmp(buffer, "LIF", 3)) {
      if (!strncmp(buffer, "LIF1.0", 6)) {
	type = LF_LIF;
      } else {
	lfError("Cannot handle LIF version number \"%s\".\n"
		"This is lifview 1.0.\n", &(buffer[3]));
	type = LF_LIFBADVER;
      }
      
    } else {
      type = LF_UNKNOWN;
    }
    break;
  case '#': /* # LIF1.0 */
    type = LF_LID;
    break;

  case 31:  /* gzip magic byte */
    type = LF_GZ;
    break;
  default:
    type = LF_UNKNOWN;
    break;
  }

  /* If the type is still unknown, look at the suffix of the
   * filename, and try to use it if possible, or at least print
   * a reasonable error message. 
   */
  if (type == LF_UNKNOWN) {
    /* Figure out the suffix */
    suffixPtr = fileName;
    /* Make suffixPtr point to character after the last . */
    for (tPtr = fileName; tPtr < (fileName + strlen(fileName)); tPtr++) {
      if (*tPtr == '.') suffixPtr = tPtr+1;
    }
    
    /* Now test it against known types */
    if (!strcmp(".lif", suffixPtr) || !strcmp(".LIF", suffixPtr)) {
      lfError("Bad .lif file, does not start with \"LIF1.0\".\n");
    }
    else if (!strcmp(".gz", suffixPtr) || !strcmp(".GZ", suffixPtr)) {
      lfError("Bad .gz file, does not start with magic characters.\n");
    }
    else if (!strcmp(".lid", suffixPtr) || !strcmp(".LID", suffixPtr)) {
      /* This is ok, I guess.  They might've stripped out the comments. */
      type = LF_LID;
    } else {
      lfError("Unrecognized input format for file \"%s\".\n", fileName);
    }
  }

  /* Close up file, return */
  fclose(inFile);
  return(type);
}
