/*
 *  convert.c: write out a lightfield in the new file format
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include "lightfield.h"

/* Local helper functions */
void Usage(char *progName);
int CheckPow2(float *array, int first, int last);


int 
main(int argc, char *argv[])
{
  char *basename;
  char *destname;
  char *progname;
  char *suffixPtr;
  int i;
  float read_descr[] = {
    LF_NULL,
  };
  float write_descr[] = {
    LF_NULL,
  };
  /* Vector-quantization default settings */
  int vqP = 1;
  float compress_descr[] = {
    LF_VQ_TRAIN_SIZE, .04,
    LF_VQ_TILESIZE, 2, 2, 2, 2,
    LF_VQ_CODESIZE, 16384,
    LF_NULL,
  };

  /* Parse the arguments */
  progname = argv[0];
  argv++;      
  argc--;

  while (argc > 2) {
    /* -raw: No compression */
    if (!strcmp(argv[0], "-novq")) {
      vqP = 0;
      argv++;
      argc--;
      
    }
    /* -vq: Vector quantization compression */
    else if (!strcmp(argv[0], "-vq")) {
      vqP = 1;
      argv++;
      argc--;
    }
    /* -v: verbose */
    else if (!strcmp(argv[0], "-v")) {
      lfVerbose = 1;
      argv++;
      argc--;
    }
    /* -vqtrain:  size of the training set (fraction of entire set) */
    else if (!strcmp(argv[0], "-vqtrain")) {
      compress_descr[1] = atof(argv[1]);
      argv+=2;
      argc-=2;
      if (compress_descr[1] <=0) {
	fprintf(stderr, "Warning: vq training set must be greater than 0.\n");
	compress_descr[1] = .02;
      }
      if (compress_descr[1] > 1) {
	fprintf(stderr, 
		"Warning: vq training set cannot be greater than 1.\n");
	compress_descr[1] = 1.0;
      }	
      
    } 
    /* -vqtile: size (dimensions) of the vq tile */
    else if (!strcmp(argv[0], "-vqtile")) {
      if (argc < 5) Usage(progname);
      compress_descr[3] = atoi(argv[1]);
      compress_descr[4] = atoi(argv[2]);
      compress_descr[5] = atoi(argv[3]);
      compress_descr[6] = atoi(argv[4]);
      argv += 5;
      argc -= 5;
      if (CheckPow2(compress_descr, 3, 6)) {
	fprintf(stderr, "Warning:  vq tile size must be a power of 2 "
		"in each dimension.\n");
      }
    }
    /* -vqcode: size of the codebook (number of entries */
    else if (!strcmp(argv[0], "-vqcode")) {
      compress_descr[8] = atoi(argv[1]);
      argv+= 2;
      argc-= 2;
      /* For now, disallow codebooks smaller than 257.
       * The code doesn't handle 1-byte code indices correctly yet.
       */
      if (compress_descr[8] < 257) {
	fprintf(stderr, "Warning: vqcodebook too small!\n");
	fprintf(stderr, "Warning: Current minimum codebook size: 257\n");
	compress_descr[8] = 257;
      }
      if (compress_descr[8] > 65536) {
	fprintf(stderr, "Warning: vqcodebook is very large. "
		"This may take a while...\n");
      }
    }
    /* default */
    else {
      Usage(progname);
    }
  }
  /* Now only the two file names should be left */
  if( argc != 2) {
    Usage(progname);
    exit(-1);
  }

  /* Ok, now do it. */
  basename = argv[0];
  destname = argv[1];
  
  lfInitContext();
  
  /* Read in the file, VQ compress it */
  lfBegin(LF_FIELD, NULL);
  /* SuffixPtr points to last 4 characters of the file name */
  suffixPtr = &basename[(strlen(basename) > 4) ? (strlen(basename)-4) : 0];
  /* Read in the lightfield */
  if (!strcmp(suffixPtr, ".lif") || !strcmp(suffixPtr, ".LIF")) {
    /* Read in lightfield format */
    printf("Converting lightfield %s to %s.\n", basename, destname);
    lfRead(LF_LIGHTFIELD, read_descr, basename);
  } else if (!strcmp(suffixPtr, ".lid") || !strcmp(suffixPtr, ".LID")) {
    /* Read in .rgb format. */
    printf("Converting lightfield description %s and .rgb's to %s.\n", 
	   basename, destname);
    lfRead(LF_SLICE_ST, read_descr, basename);
  } else {
    fprintf(stderr, "Error: input file %s is not a .lif or .lid file.\n",
	    basename);
    Usage(progname);
    exit(-1);
  }

  /* compress, if required */
  if (vqP) {
    lfCompress(LF_VQ, compress_descr);
  }

  /* Write the file out */
  lfWrite(LF_LIGHTFIELD, write_descr, destname);
  lfEnd();

  /* Notify user that it succeeded. */
  printf("Done.\n");
}

/* This function makes sure that every element of the array between
 * first and last (inclusive) is a power of 2, between 1 and 256.
 */
int CheckPow2(float *array, int first, int last)
{
  int i;
  int pow;
  int errorP = 0;

  for (i=first; i <= last; i++) {
    /* find pow, the ceiling of log (base 2) of array[i]. */
    for (pow = 0; pow <= 8; pow++) {
      if (array[i] <= (1 << pow)) break;
    }	
    if (array[i] != (1 << pow)) {
      errorP = 1;
      array[i] = (1 << pow);
    }
  }
  return errorP;
}



void Usage(char *progName)
{
  fprintf
    (stderr, 
     "Usage:\n"
     "%s [options] <infield> <outfield.lif>\n\n"
     "arguments:\n\n"
     "    -v                      Verbose mode on, prints progress.\n"
     "\n"
     "    -vq                     Use vector-quantization compression.\n"
     "                            This is the default. \n"
     "\n"
     "    -novq                   Don't use vector-quantization compression\n"
     "\n"
     "    -vqtrain <fraction>     Specify fraction of lightfield to be used\n"
     "                            as a training set.  The fraction should be\n"
     "                            between 0 and 1.  Default is 0.04.\n"
     "\n"
     "    -vqtile <u> <v> <s> <t> Specify the size of each vq tile.  Each\n"
     "                            of the four numbers should be a fairly\n"
     "                            small power of 2.  Default is 2 2 2 2.\n"
     "\n"
     "    -vqcode <size>          Specify the number of tiles in the\n"
     "                            codebook.  Depending on the tile size,\n"
     "				  good numbers would be between 256 and\n"
     "                            65536.  Default is 16384.\n"
     "\n"
     "    <infield>		  Should be either a .lid file\n"
     "                            (LIght field Description file), which\n"
     "                            specifies a set of .rgb files, or a .lif\n"
     "                            file, which is the native light field\n"
     "                            format.\n"
     "\n"
     "    <outfield.lif>	  The output file.\n"
     "\n"
     , progName);
  
  exit(-1);
}
