/*
 * vq_iface.c
 *
 * interface to Marc and Navin's VQ code
 */
#include <stdio.h>
#include <stdlib.h>
#include <stream.h>
#include <string.h>
#include <assert.h>
#include <bstring.h>
#include "dataseq.h"
#include "codevec.h"
#include "vq.h"
#include "vq_iface.h"

const int MAXVECTORLEN = 64;
const int MAX_DOMAIN_DIMENS = 64;
const int MAX_TILE_DIMENS = MAX_DOMAIN_DIMENS / 2;

typedef unsigned char CWTYPE;

typedef struct node{
    CWTYPE cw[MAXVECTORLEN];	/* Encoder codebook nodes are big enough     */
    float rate;	
    int leafindex;
    int left;
    int right;
} NODETYPE;

typedef NODETYPE *TREETYPE;

double PERF::lambda = 0;

static FILE *infile;
static NODETYPE *LocalBuffer;
static CWTYPE *LocalPtr;
static long offset;
static int nodecount;
static int leafcount;
static int VectorLen;

static int
Malloc(int size ) {
    int p = offset;
    offset += size;
    nodecount++;
    return p;
}

static void
readCW(CWTYPE *cwstart)
{
    char s[64+1];
    int temp;
  
    while (  fscanf(infile, "%64s", s) > 0
	  && strncmp(s, "CW", 64) != 0
	  );
    fscanf(infile,"%*d%d",&VectorLen);
    for ( int m=0; m<VectorLen; m++) {
	fscanf(infile,"%d",&temp);
	cwstart[m] = temp;
    }
}

static void
writeCW(CWTYPE *cwstart)
{
    for (int m=0; m<VectorLen; m++) {
	*LocalPtr++ = cwstart[m];
    }
}

static void
readtree(TREETYPE parent)
{
    int havechild;
    static char s[64+1];
  
    while (  fscanf(infile,"%64s",s) > 0
	  && strncmp(s, "NTN", 64) != 0
	  && strncmp(s, "LTN", 64) != 0
	  );

    fscanf(infile,"%*d%d",&havechild);
    if (havechild == 2) {
	parent->left =Malloc(1);
	parent->right=Malloc(1);
	readtree(&LocalBuffer[parent->right]);
	readtree(&LocalBuffer[parent->left]);
    }  

    readCW(parent->cw);
    fscanf(infile,"%f",&parent->rate);
  
    if  (havechild == 0) {
	parent->left = parent->right = 0;
	parent->leafindex = leafcount++;
    }
}

static void
convert(TREETYPE parent)
{
    assert((parent->left && parent->right) ||
	   (!parent->left && !parent->right));

    if  (! parent->left && ! parent->right)  {  
	writeCW(parent->cw);
	return;
    }

    convert(&LocalBuffer[parent->right]);
    convert(&LocalBuffer[parent->left]);
}

/*
 * function to generate a codebook given a data array 
 */
VQ_t *
vqGenTreeCodebook(const unsigned char *data_array, int data_size, int dw_size,
	int max_code_size, int cw_size)
{
    // cannot handle generic component size yet
    assert(dw_size == 1);

    // create a configuration file
    char *config_file = tempnam(getenv("TMPDIR"), "lfvq");
    filebuf fb;
    if (! fb.open(config_file,output))
	Error("can't open vq output file ", config_file);
    ostream os(&fb);
    os << "MTSVQ" << " " << max_code_size << "\n";
    os << "NTN" << " " << dw_size << " 0\n";
    os << "CW" << " " << cw_size << " 0\n";
    fb.close();

    // perform VQ training
    VQ* vqp = new_VQ(config_file);
    DATASEQ ds(data_array, data_size);
    vqp->train(ds);

    // output trained VQ codebook into a file
    char *tree_vq_file = tempnam(getenv("TMPDIR"), "lfvq");
    vqp->put(tree_vq_file);

    // build a linear codebook out of a tree structure codebook
    char total_str[64+1];
    infile = fopen(tree_vq_file, "r");
    while (  fscanf(infile, "%64s", total_str) > 0
	  && strncmp(total_str, "Total_codewords", 64) != 0
	  );

    int codebook_size;
    fscanf(infile, "%d%s", &codebook_size, total_str);

    LocalBuffer = new NODETYPE[codebook_size];
    nodecount = offset = leafcount = 0;
    
    int root = Malloc(1);
    readtree(&LocalBuffer[root]);

    fclose(infile);

    char sys_cmd[256];
    sprintf(sys_cmd, "rm -f %s", config_file);
    system(sys_cmd);
    sprintf(sys_cmd, "rm -f %s", tree_vq_file);
    system(sys_cmd);

    VQ_t *vq = new VQ_t;
    vq->type = VQ_FIXED_TREE;
    vq->size = nodecount;
    vq->nwords = leafcount;
    vq->cw_size = cw_size;
    vq->codebook = LocalBuffer;

    return vq;
}

VQ_t *
vqTreeCodebookToLinearCodebook(VQ_t *tree_vq)
{
    assert(tree_vq->type == VQ_FIXED_TREE);

    VQ_t *linear_vq = new VQ_t;

    LocalBuffer = (TREETYPE)tree_vq->codebook;
    linear_vq->codebook = LocalPtr =
			new CWTYPE[tree_vq->nwords * tree_vq->cw_size];
    VectorLen = tree_vq->cw_size;

    convert(&LocalBuffer[0]);

    linear_vq->type = VQ_FIXED_LINEAR;
    linear_vq->size = linear_vq->nwords = tree_vq->nwords;
    linear_vq->cw_size = tree_vq->cw_size;

    return linear_vq;
}

void
vqFreeCodebook(VQ_t *vq)
{
    delete vq->codebook;
    delete vq;
}

static int
Tile_Length(int len[], int start, int width)
{
    int length = 1;
    int i;

    for ( i=0 ; i<width ; i++ )
        length *= len[start+i];
    return length;
}

static int
Image_Dimens( int domain_dimens, int len[], int image_len[])
{
    int i;
    int image_domain_dimens = domain_dimens / 2;

    assert(! (domain_dimens&1));
    for ( i=0; i<image_domain_dimens; i++)
	image_len[i] = len[image_domain_dimens+i];
    return image_domain_dimens;
}

static int
Num_Bits(int size)
{
    int i;
    if  (size <= 0) return 0;
    i = 1; size--;
    while ((size = size>>1) > 0) i++;
    return i;
}

static int
Find_Code_Word_Fullsearch(unsigned char* codebook,
	int nwords, unsigned char* tilep, int dim)
{
	unsigned char*	wordp;
	unsigned char*	pixelp;
	int		pixel,word,diff,dist,mindist,bestn;
	int		n,i;

	wordp = codebook;
	mindist = 1 << 30;
	bestn = -1;
	for (n=0; n<nwords; n++) {
		pixelp = tilep;
		dist = 0;
		for (i=0; i<dim; i++) {
			pixel = *pixelp++;
			word = *wordp++;
			diff = word - pixel;
			dist += diff * diff;
		}
		if (dist < mindist) {
			bestn = n;
			mindist = dist;
		}
	}
	return bestn;
}

static int
Find_Code_Word_Treesearch(TREETYPE cb, int pidx,
	CWTYPE *vector, int VectorLen, CWTYPE **cw)
{
	TREETYPE parent = &cb[pidx];
	unsigned long	mse0,mse1;
	CWTYPE *cw0, *cw1;
	long temp;
	int m;

	if (!parent->left && !parent->right) {
		*cw = parent->cw;
		return (parent->leafindex);
	}

	cw0 = cb[parent->left].cw;
	for (m=0,mse0=0; m<VectorLen; m++) {
		temp = vector[m]-cw0[m];
		mse0 +=temp*temp;
	}

	cw1 = cb[parent->right].cw;
	for (m=0,mse1=0; m<VectorLen; m++) {
		temp = vector[m]-cw1[m];
		mse1 +=temp*temp;
	}	

	if (mse0<mse1)
		return Find_Code_Word_Treesearch(
			cb,parent->left,vector,VectorLen,cw);
	else
		return Find_Code_Word_Treesearch(
			cb,parent->right,vector,VectorLen,cw);
}

/*
 * put a interger Value in a bit stream buffer StreamBuf using
 * exactly Nbits. It is the Count's value in the stream, i.e. The
 * offset of the first bit of the value is (Nbits * Count)
 */
static void
PutStream(unsigned char *StreamBuf,int Value,int Nbits,int Count)
{
    int Offset,StartByte,StartBit,BitsLeft,BitsAvailInByte;
    unsigned int TempValue;
    unsigned char mask;

    if (Nbits ==8) 
	StreamBuf[Count] = Value;
    else {
	Offset = Nbits * Count;
	StartByte = Offset >> 3;
	StartBit = Offset & 0x7;
	BitsLeft = Nbits;
	TempValue = (unsigned int)Value;
	while (BitsLeft > 0) {
	    BitsAvailInByte = 8-StartBit;
	    if (BitsLeft >= BitsAvailInByte) {
		/* have to break the data in to segments */
		mask = TempValue >> (BitsLeft - BitsAvailInByte);
		StreamBuf[StartByte] |= mask;

		TempValue -= (mask <<(BitsLeft - BitsAvailInByte));
		BitsLeft -= BitsAvailInByte;
		StartByte ++;
		StartBit =0;
	    }
	    else {
		/* (BitsLeft < BitsAvailInByte), all bits written at once*/
		mask = TempValue << (BitsAvailInByte - BitsLeft);
		StreamBuf[StartByte] |= mask;

		BitsLeft =0; /* get out of while loop */
	    }
	}
    }
}

/*
 * Extract a Nbit-integer Value from a bit stream bufffer StreamBuf;
 * It is the Count's value in the stream, i.e. The
 * offset of the first bit of the value is (Nbits * Count)
*/
static int
ExtractStream(unsigned char *StreamBuf,int Nbits,int Count)
{
    int Value;
    int Offset,StartByte,StartBit,BitsLeft,BitsAvailInByte;
    unsigned int TempValue;
    unsigned char mask;

    if (Nbits == 8) {
	Value = StreamBuf[Count];
    }
    else {
	Offset = Nbits * Count;
	StartByte = Offset >> 3;
	StartBit = Offset & 0x7;
	BitsLeft = Nbits;
	TempValue =0;
	while (BitsLeft >0) {
	    BitsAvailInByte = 8-StartBit;
	    if (BitsLeft > BitsAvailInByte) {
		mask =(unsigned char)(StreamBuf[StartByte]<<StartBit) >>StartBit;
		TempValue = (TempValue<< BitsAvailInByte) | mask;

		BitsLeft -= BitsAvailInByte;
		StartByte++;
		StartBit =0;
	    }
	    else { /* if BitsLeft < BitsAvailInByte */
		mask =(unsigned char)(StreamBuf[StartByte]<<StartBit) >> StartBit; /* set MSB bits to zero */
		mask = mask >>(BitsAvailInByte - BitsLeft); /* align to LSB */
		TempValue = (TempValue<< BitsLeft) | mask;

		BitsLeft =0;
	    }
	}
	Value = TempValue;
    }
    return(Value);
}

void
vqEncode(VQ_t *vq, unsigned char *image, int domain_dimens, int len[], int clen,
	int padded, int padded_nbits, int *codearray, int *bitoffset_p)
{
    int image_domain_dimens,image_len[MAX_DOMAIN_DIMENS];
    unsigned char *tilep = image;
    unsigned char *wordp;
    unsigned char *vectorp;
    int tile[MAX_TILE_DIMENS];
    int index;
    int dimen,i;
    int nbits = padded ? padded_nbits : Num_Bits(vq->nwords);
    int dim = Tile_Length(len, 0, domain_dimens>>1) * clen;
    int ntiles = 0;
    int bitoffset = 0;

    assert(vq->type == VQ_FIXED_LINEAR || vq->type == VQ_FIXED_TREE);
    assert(dim == vq->cw_size);

    image_domain_dimens = Image_Dimens(domain_dimens, len, image_len);
    for (i=0; i<image_domain_dimens; i++)
    	tile[i] = 0;
    bzero(codearray, Tile_Length(image_len,0,image_domain_dimens) * nbits / 8);

    do {
    	vectorp = tilep;

    	if  (vq->type == VQ_FIXED_TREE)
      	    index = Find_Code_Word_Treesearch((TREETYPE)vq->codebook,
			0, (CWTYPE*)vectorp, dim, (CWTYPE**)&wordp);
	else
	    index = Find_Code_Word_Fullsearch((CWTYPE *)vq->codebook,
			vq->nwords, vectorp, dim);
	PutStream((unsigned char*)codearray,index,nbits,ntiles);

	bitoffset += nbits;
	tilep += dim;
	ntiles++;

	dimen = 0;
	while (++tile[dimen] == image_len[dimen] &&
		dimen < image_domain_dimens) dimen++;
	for (i=0; i<dimen; i++)
	    tile[i] = 0;
    }
    while (dimen < image_domain_dimens);
    *bitoffset_p = bitoffset;

    char *buf = new char[1 << 16];
    delete buf;
}

void
vqDecodeLinear(VQ_t *vq, int *codearray, int domain_dimens, int len[],
	int clen, int padded, int padded_nbits, unsigned char *image)
{
    int image_domain_dimens,image_len[MAX_DOMAIN_DIMENS];
    unsigned char *wordp;
    unsigned char *tilep = image;
    int tile[MAX_TILE_DIMENS];
    int dim = Tile_Length(len, 0, domain_dimens>>1) * clen;
    int index;
    int dimen,i;
    int nbits = padded ? padded_nbits : Num_Bits(vq->nwords);
    int ntiles = 0;
    int bitoffset = 0;

    assert(vq->type == VQ_FIXED_LINEAR);
    assert(dim == vq->cw_size);

    image_domain_dimens = Image_Dimens(domain_dimens,len, image_len);
    for (i=0; i<image_domain_dimens; i++)
	tile[i] = 0;

    do {
	index = ExtractStream((unsigned char*)codearray,
			nbits,ntiles);
	assert(index < vq->nwords);

	wordp = (unsigned char *)vq->codebook + index * dim;
	for (i=0; i<dim; i++)
	    *tilep++ = *wordp++;
	bitoffset += nbits;
	ntiles++;

	dimen = 0;
	while (tile[dimen]++ == image_len[dimen]-1 &&
	       dimen < image_domain_dimens) dimen++;
	for (i=0; i<dimen; i++)
	    tile[i] = 0;
    }
    while (dimen < image_domain_dimens);
}

