#ifndef vq_h
#define vq_h

#include <stream.h>
#include "error.h"
#include "perf.h"
#include "tvq.h"

class DATABLK;				// forward reference
class DATASEQ;				// forward reference
class VQ;				// forward reference
class TREENODE;				// forward reference
class NTN;                              // forward reference
class TreeStructVQ;			// forward reference
class MTSVQ;                            // forward reference

VQ* new_VQ(char*);
VQ* new_VQ(istream&);
TREENODE* new_TREENODE(istream&);

class VQ {
public:
    VQ() { }
    virtual ~VQ() { }
    virtual void train(const DATASEQ&, double precision=EPSILON) = 0;
    virtual PERF test(const DATABLK&) const = 0;
    virtual VQ*  copy() const { return 0; }
    virtual void get(istream&) = 0;
    virtual void put(ostream&) const = 0;

    PERF test(const DATASEQ&) const;
    void get(char*);
    void put(char*);
    };

class FullSearchVQ: public VQ {
protected:
    int width;		// width of codewords (dimension in datavecs)
    int size;		// number of codewords
    VQ** cw;		// array of pointers to codewords
    double* rate;	// array of codelengths
    double logsize;	// rate of codebook
    double entropy;	// entropy of codebook
    int fullcnt;	// number of full codewords

    virtual PERF nearest(const DATABLK&, int&) const = 0;
    void train(const DATASEQ&, double precision, int fillEmptyCellsFlag);
public:
    FullSearchVQ() { size=0; cw=0; rate=0; entropy=0; fullcnt=0; }
    FullSearchVQ(const FullSearchVQ&);
    ~FullSearchVQ();
    virtual void train(const DATASEQ&, double precision=EPSILON) = 0;
    PERF test(const DATABLK&) const;
    virtual VQ*  copy() const = 0;
    void get(istream&);
    virtual void put(ostream&) const;
    };

class STDVQ: public FullSearchVQ {
protected:
    PERF nearest(const DATABLK&, int&) const;
public:
    STDVQ() { }
    STDVQ(istream& is) { get(is); }
    STDVQ(const STDVQ& lbgvq) : FullSearchVQ(lbgvq) { }
    virtual void train(const DATASEQ&, double precision=EPSILON);
    virtual VQ*  copy() const;
    virtual void put(ostream&) const;
    };

class TREENODE: public STDVQ {
    friend NTN;
    friend TreeStructVQ;
    friend MTSVQ;

protected:
    DATASEQ vqds;	// dataseq for this node
    VQ* nodecw;		// codeword at the node (not necessarily treenode)
    long nodepop;	// population of this node
    PERF nodeperf;	// performance at node
    PERF deltaperf;	// change in performance due to node (used by prune)
    double lambda;	// negative change in distortion over change in rate
    double lambdaopt;	// max for greedy growing, min for prune
    char subtree;	// 1 to treat as root of subtree, 0 as single node
    char split;		// 1 for node which has been split, 0 otherwise

    void splitData();
    void splitNode(double precision, int default_size=2);
    PERF doBestSplit(double precision);
    void delete_unsplit();
    void delete_vqds();
    long countCWs();

    virtual double deltaRate() const = 0;
public:
    TREENODE() { nodecw=0; }
    TREENODE(istream& is) { get(is); }
    TREENODE(const TREENODE&);
    ~TREENODE();
    void train(const DATASEQ&, double precision=EPSILON);
    virtual PERF test(const DATABLK&) const = 0;
    virtual VQ*  copy() const { return 0; }
    void get(istream&);
    virtual void put(ostream&) const;
    };

class NTN: public TREENODE {
protected:
    double deltaRate() const;
public:
    NTN() { }
    NTN(istream& is) { get(is); }
    NTN(const NTN& ntn) : TREENODE(ntn) { }
    PERF test(const DATABLK&) const;
    VQ*  copy() const;
    void put(ostream&) const;
    };

class TreeStructVQ: public VQ {
    friend MTSVQ;
protected:
    double constraint;
    TREENODE* root;
public:
    TreeStructVQ() { root=0; }
    TreeStructVQ(const TreeStructVQ&);
    ~TreeStructVQ();
    virtual void train(const DATASEQ&, double precision=EPSILON) = 0;
    PERF test(const DATABLK& db) const;
    virtual VQ*  copy() const = 0;
    void get(istream&);
    virtual void put(ostream&) const;
    };

class MTSVQ: public TreeStructVQ {
public:
    MTSVQ() { }
    MTSVQ(istream& is) { get(is); }
    MTSVQ(const MTSVQ& mtsvq) : TreeStructVQ(mtsvq) { }
    void train(const DATASEQ&, double precision=EPSILON);
    VQ*  copy() const;
    void put(ostream&) const;
    };

#endif
