#include <stdio.h>
#include <stream.h>
#include <string.h>
#include "error.h"
#include "perf.h"
#include "dataseq.h"
#include "codevec.h"
#include "tvq.h"

extern "C" void bcopy(const void *, void *, int);

long N = 0;			// number of vectors in source (global)
DATAVEC* dvp = 0;		// array of DATAVECs (global)

static int dim = MAXDIM;	// dimension of codeword (local)

/*
 *  construct a DATASEQ from an array.
 */
void ReadData(const unsigned char *data_array, int size, DATASEQ& ds)
{
    if  (size % dim != 0)
	Error("ReadData: Invalid data array dimension\n");

    N = size / dim;
    dvp = new DATAVEC[N];

    for ( int i=0,w=0 ; i<size ; i+=dim,w++ )
	bcopy(data_array+i, dvp[w].vec, dim);
    ds.push(new DATABLK(dvp,N));
}

/*
 *  Construct a CODEVEC from another CODEVEC.
 */
CODEVEC::CODEVEC(const CODEVEC& codevec) {

    for (register int n=0; n<dim; n++)
	vec[n] = codevec.vec[n];
    }

/*
 *  Destruct a CODEVEC. 
 */
CODEVEC::~CODEVEC() { }

/*
 *  Compute squared-error distortion between source and reproduction vector.
 */
double CODEVEC::dist(const DATAVEC& dv) const {

    double dist = 0.0;
    for (register int n=0; n<dim; n++) {
	double tmp = double(dv.vec[n]) - double(vec[n]);
	dist += tmp * tmp;
	}
    return dist;
    }

/*
 *  Total the distortion to this codeword from each vector in the block.
 */
PERF CODEVEC::test(const DATABLK& db) const {

    PERF perf = 0;
    for (long w=0; w<db.width; w++)
	perf += PERF(dist(db.loc[w]));
    return perf;
    }

/*
 *  Replace the codeword by the centroid of all vectors in the sequence.
 */
void CODEVEC::train(const DATASEQ& ds, double precision) {

    // Initialize.
    long pop = 0;
    double accum[MAXDIM];
    for (register int n=0; n<dim; n++) accum[n] = 0;

    // Accumulate.
    for (DATABLK* dbp=ds.head; dbp; dbp=dbp->next) {
	pop += dbp->width;
	for (long w=0; w<dbp->width; w++)
	    for (n=0; n<dim; n++)
		accum[n] += dbp->loc[w].vec[n];
	}

    // Normalize.
    for (n=0; n<dim; n++)
	vec[n] = round(accum[n]/pop);
    }

/*
 * Copy a CODEVEC.
 */
VQ* CODEVEC::copy() const {

    return new CODEVEC(*this);
    }

/*
 *  Input a CODEVEC.
 */
void CODEVEC::get(istream& is) {

    is >> dim;
    if (dim > MAXDIM) Error("constant MAXDIM exceeded");

    // Next d entries specify the coefficients.
    int d;
    is >> d;
    if (d<0 || d>dim) Error("bad format for CW");
    for (int n=0; n<d; n++) is >> vec[n];

    // Remaining entries are 0.
    for (; n<dim; n++) vec[n] = 0;
    }

/*
 *  Output a CODEVEC.
 */
void CODEVEC::put(ostream& os) const {

    os << "CW " << dim << " " << dim << "\n";
    for (int n=0; n<dim; n++)
	os << vec[n] << "\n";
    }
