/*
 * util.c
 *
 * utility stuff
 */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include "lightfield.h"

/* Global variable, controls whether or not to print debug messages */
bool_t lfVerbose = 0;

/* Print error messages (if lfVerbose) */
void
lfError(char *fmt, ...)
{
    va_list args;

    va_start(args, fmt);
    fprintf(stderr, "Error: ");
    vfprintf(stderr, fmt, args);
    fprintf(stderr, "\n");
    va_end(args);
}

/* Print output messages */
void
lfOutput(char *fmt, ...)
{
    va_list args;

    if (lfVerbose) {
      va_start(args, fmt);
      vfprintf(stdout, fmt, args);
      va_end(args);
    }
}

/* Calculate the floor of the log (base 2) of a number */
int
log2(int size)
{
    int i = 0;
    while ((size >>= 1) > 0) i++;
    return i;
}

/* Set up the indices for the offset tables for indexing into the
 * light field data array */
void
lfMakeIndices(LFShared *shared)
{
    LFView *view = &__LF_GET_CONTEXT()->view;
    int nu = shared->nu;
    int nv = shared->nv;
    int ns = shared->ns;
    int nt = shared->nt;
    int u, v, s, t;

    if  (shared->uindex)
	free(shared->uindex);

    if  (! shared->vq)  {
	int tile_u_bits = view->tile_u_bits;
	int tile_v_bits = view->tile_v_bits;
	int tile_s_bits = view->tile_s_bits;
	int tile_t_bits = view->tile_t_bits;
	int tile_u_mask = ((1 << tile_u_bits) - 1);
	int tile_v_mask = ((1 << tile_v_bits) - 1);
	int tile_s_mask = ((1 << tile_s_bits) - 1);
	int tile_t_mask = ((1 << tile_t_bits) - 1);
	int tile_st_bits = tile_s_bits + tile_t_bits;
	int tile_vst_bits = tile_v_bits + tile_st_bits;
	int tile_uvst_bits = tile_u_bits + tile_vst_bits;

	if  (  (nu & tile_u_mask)
	    || (nv & tile_v_mask)
	    || (ns & tile_s_mask)
	    || (nt & tile_t_mask)
	    )  {
	    fprintf(stderr, "lfMakeIndices: nu = %d, nv = %d, ns = %d, nt = %d\n"
			    "not multiples of tile size [%d, %d, %d, %d]\n",
			    nu, nv, ns, nt,
			    tile_u_mask+1, tile_v_mask+1,tile_s_mask+1,tile_t_mask+1);
	    exit(1);
	}
	    
	shared->uindex = calloc((nu+nv+ns+nt+4)<<1, sizeof(int));
	if  (! shared->uindex)  {
	    fprintf(stderr,"lfMakeIndices: can't allocate storage\n");
	    exit(1);
	}

	shared->vindex = shared->uindex + nu + 1;
	shared->sindex = shared->vindex + nv + 1;
	shared->tindex = shared->sindex + ns + 1;
	shared->vqtile_uindex = shared->tindex + nt + 1;
	shared->vqtile_vindex = shared->vqtile_uindex + nu + 1;
	shared->vqtile_sindex = shared->vqtile_vindex + nv + 1;
	shared->vqtile_tindex = shared->vqtile_sindex + ns + 1;

	for ( u=0 ; u<nu ; u++ )  {
	    shared->uindex[u] = ((u>>tile_u_bits)*nv*ns*nt) << (tile_uvst_bits - tile_vst_bits);
	    shared->vqtile_uindex[u] = (u & tile_u_mask) << tile_vst_bits;
	}
	shared->uindex[u] = shared->uindex[u-1];
	shared->vqtile_uindex[u] = shared->vqtile_uindex[u-1];

	for ( v=0 ; v<nv ; v++ ) {
	    shared->vindex[v] = ((v>>tile_v_bits)*ns*nt) << (tile_uvst_bits - tile_st_bits);
	    shared->vqtile_vindex[v] = (v & tile_v_mask) << tile_st_bits;
	}
	shared->vindex[v] = shared->vindex[v-1];
	shared->vqtile_vindex[v] = shared->vqtile_vindex[v-1];

	for ( s=0 ; s<ns ; s++ ) {
	    shared->sindex[s] = ((s>>tile_s_bits)*nt) << (tile_uvst_bits - tile_t_bits);
	    shared->vqtile_sindex[s] = (s & tile_s_mask) << tile_t_bits;
	}
	shared->sindex[s] = shared->sindex[s-1];
	shared->vqtile_sindex[s] = shared->vqtile_sindex[s-1];

	for ( t=0 ; t<nt ; t++ ) {
	    shared->tindex[t] = (t>>tile_t_bits) << tile_uvst_bits;
	    shared->vqtile_tindex[t] = (t & tile_t_mask);
	}
	shared->tindex[t] = shared->tindex[t-1];
	shared->vqtile_tindex[t] = shared->vqtile_tindex[t-1];
    }
    else {
	int vq_ubits = shared->vq->ubits;
	int vq_vbits = shared->vq->vbits;
	int vq_sbits = shared->vq->sbits;
	int vq_tbits = shared->vq->tbits;
	int vq_umsk = (1 << vq_ubits) - 1;
	int vq_vmsk = (1 << vq_vbits) - 1;
	int vq_smsk = (1 << vq_sbits) - 1;
	int vq_tmsk = (1 << vq_tbits) - 1;

	/* XXX: need to patch in 1 extra entry to avoid checking later */
	shared->uindex = calloc((nu+nv+ns+nt+4)<<1, sizeof(int));
	if  (! shared->uindex)  {
	    fprintf(stderr,"lfMakeIndices: can't allocate storage\n");
	    exit(1);
	}

	shared->vindex = shared->uindex + nu + 1;
	shared->sindex = shared->vindex + nv + 1;
	shared->tindex = shared->sindex + ns + 1;
	shared->vqtile_uindex = shared->tindex + nt + 1;
	shared->vqtile_vindex = shared->vqtile_uindex + nu + 1;
	shared->vqtile_sindex = shared->vqtile_vindex + nv + 1;
	shared->vqtile_tindex = shared->vqtile_sindex + ns + 1;

	for ( u=0 ; u<nu ; u++ ) {
	    shared->uindex[u] = ((u>>vq_ubits)*nv*ns*nt) >> (vq_vbits+vq_sbits+vq_tbits);
	    shared->vqtile_uindex[u] = (u&vq_umsk) << (vq_vbits+vq_sbits+vq_tbits);
	}
	shared->uindex[u] = shared->uindex[u-1];
	shared->vqtile_uindex[u] = shared->vqtile_uindex[u-1];

	for ( v=0 ; v<nv ; v++ ) {
	    shared->vindex[v] = ((v>>vq_vbits)*ns*nt) >> (vq_sbits+vq_tbits);
	    shared->vqtile_vindex[v] = (v&vq_vmsk) << (vq_sbits+vq_tbits);
	}
	shared->vindex[v] = shared->vindex[v-1];
	shared->vqtile_vindex[v] = shared->vqtile_vindex[v-1];

	for ( s=0 ; s<ns ; s++ ) {
	    shared->sindex[s] = ((s>>vq_sbits)*nt) >> vq_tbits;
	    shared->vqtile_sindex[s] = (s&vq_smsk) << vq_tbits;
	}
	shared->sindex[s] = shared->sindex[s-1];
	shared->vqtile_sindex[s] = shared->vqtile_sindex[s-1];

	for ( t=0 ; t<nt ; t++ ) {
	    shared->tindex[t] = (t>>vq_tbits);
	    shared->vqtile_tindex[t] = (t&vq_tmsk);
	}
	shared->tindex[t] = shared->tindex[t-1];
	shared->vqtile_tindex[t] = shared->vqtile_tindex[t-1];
    }
}

/*
 * routine to pad data.
 * We do this to improve cache performance.  If the 
 * image size is a multiple of the cache size (or vice
 * versa), we may get excessive cache swapping.
 */
void
lfPadData(LFSlab *slab, int pad_byte, int new_size, int mode)
{
    LFShared *shared = slab->shared;
    LFVQCodebook *vq = shared->vq;
    char *freed, *src;
    char *alloced, *dst;
    int total_size;
    int old_size;
    int pad_size;
    int i, j;

    if  (vq) {
	if  (vq->sample_size == new_size)
	    return;
	freed = src = vq->codebook;
	old_size = vq->sample_size;
	total_size = (vq->size << (vq->ubits+vq->vbits+vq->sbits+vq->tbits))
			* new_size;
    }
    else {
	if  (slab->sample_size == new_size)
	    return;
	freed = src = slab->lightfield;
	old_size = shared->sample_size;
	total_size = shared->nu * shared->nv * shared->ns * shared->nt
			* new_size;
    }
    pad_size = new_size - old_size;
    dst = alloced = malloc(total_size);

    if  (mode == LF_PAD_FIRST)  {
	for ( i=0 ; i<total_size ; i+=new_size ) {
	    for ( j=0 ; j<pad_size ; j++ )
		*dst++ = pad_byte;
	    for ( j=0 ; j<old_size ; j++ )
		*dst++ = *src++;
	}
    }
    else {
	for ( i=0 ; i<total_size ; i+=new_size ) {
	    for ( j=0 ; j<old_size ; j++ )
		*dst++ = *src++;
	    for ( j=0 ; j<pad_size ; j++ )
		*dst++ = pad_byte;
	}
    }

    free(freed);
    if  (vq) {
	vq->codebook = alloced;
	vq->sample_size = new_size;
    }
    else {
	slab->lightfield = alloced;
	slab->sample_size = new_size;
    }
}
