/*
 * resample.c
 *
 * various routines for resampling the lightfield when drawing
 */
#include "lightfield.h"
#include "lerp.h"

#define	NEEDCLAMP(x, xlo, xhi)	\
	((x < xlo) || (x > xhi))

#define CLAMP(x, xlo, xhi)      \
	if  (x > xhi) x = xhi;  \
	else if  (x < xlo) x = xlo

#define	NEEDCLAMP_SAMPLE(mmap)	\
	(  NEEDCLAMP((mmap)->u, 0, uclamp) \
	|| NEEDCLAMP((mmap)->v, 0, vclamp) \
	|| NEEDCLAMP((mmap)->s, 0, sclamp) \
	|| NEEDCLAMP((mmap)->t, 0, tclamp) )

static void
clampScanline(LFView *view, LFPixelUVST *scanline, int w,
	int uclamp, int vclamp, int sclamp, int tclamp)
{
    int cur_x;

#ifdef	STATS
    view->scanline_clamped++;
#endif
    for ( cur_x=0 ; cur_x<w ; cur_x++,scanline++ ) {
	CLAMP(scanline->u, 0, uclamp);
	CLAMP(scanline->v, 0, vclamp);
	CLAMP(scanline->s, 0, sclamp);
	CLAMP(scanline->t, 0, tclamp);
    }
}

void
lfResampleRGBA8(const LFSlab *slab, LFPixelUVST *scanline, int x, int y, int w)
{
    LFView *view = &__LF_GET_CONTEXT()->view;
    LFPixelUVST *mapp  = scanline;
    LFPixelRGBA8 *imgp = (LFPixelRGBA8 *)view->pixel_buffer + y*view->width + x;
#ifndef	SCAN_ONLY
    bool_t uvlerp = (view->filter & LF_LERP_UV) == LF_LERP_UV;
    bool_t stlerp = (view->filter & LF_LERP_ST) == LF_LERP_ST;
    LFShared *shared = slab->shared;
    int frac_ubits = shared->u_frac_bits;
    int frac_vbits = shared->v_frac_bits;
    int frac_sbits = shared->s_frac_bits;
    int frac_tbits = shared->t_frac_bits;
    int frac_umsk = (1 << frac_ubits) - 1;
    int frac_vmsk = (1 << frac_vbits) - 1;
    int frac_smsk = (1 << frac_sbits) - 1;
    int frac_tmsk = (1 << frac_tbits) - 1;
    int u, v, s, t;
    int fu, fv, fs, ft;
    /* BEGIN stuff used inside macros */
    LFPixelRGBA8 *lightfield = slab->lightfield;
    LFPixelRGBA8 *p00, *p10, *p01, *p11;
    LFPixelRGBA8 c00,c10,c01,c11;
    int *uindex = shared->uindex;
    int *vindex = shared->vindex;
    int *sindex = shared->sindex;
    int *tindex = shared->tindex;
    int *vqtile_uindex = shared->vqtile_uindex;
    int *vqtile_vindex = shared->vqtile_vindex;
    int *vqtile_sindex = shared->vqtile_sindex;
    int *vqtile_tindex = shared->vqtile_tindex;
    int uclamp = (shared->nu << shared->u_frac_bits) - 1;
    int vclamp = (shared->nv << shared->v_frac_bits) - 1;
    int sclamp = (shared->ns << shared->s_frac_bits) - 1;
    int tclamp = (shared->nt << shared->t_frac_bits) - 1;
    /* END for macros */

    if  (NEEDCLAMP_SAMPLE(mapp) || NEEDCLAMP_SAMPLE(mapp+w-1)) {
	clampScanline(view, scanline, w, uclamp, vclamp, sclamp, tclamp);
    }

    if  (uvlerp && stlerp) {
	for ( x=0 ; x<w ; x++,mapp++,imgp++ ) {
	    u = mapp->u >> frac_ubits;
	    v = mapp->v >> frac_vbits;
	    s = mapp->s >> frac_sbits;
	    t = mapp->t >> frac_tbits;

	    fu = mapp->u & frac_umsk;
	    fv = mapp->v & frac_vmsk;
	    fs = mapp->s & frac_smsk;
	    ft = mapp->t & frac_tmsk;

	    QUADLERPRGB(u,v,s,t,fu,fv,fs,ft,imgp);
	}
    }
    else if  (uvlerp) {
	for ( x=0 ; x<w ; x++,mapp++,imgp++ ) {
	    u = mapp->u >> frac_ubits;
	    v = mapp->v >> frac_vbits;
	    s = mapp->s >> frac_sbits;
	    t = mapp->t >> frac_tbits;

	    fu = mapp->u & frac_umsk;
	    fv = mapp->v & frac_vmsk;

	    BILERPUVRGB(u,v,s,t,fu,fv,imgp);
        }
    }
    else if  (stlerp) {
	for ( x=0 ; x<w ; x++,mapp++,imgp++ ) {
	    u = mapp->u >> frac_ubits;
	    v = mapp->v >> frac_vbits;
	    s = mapp->s >> frac_sbits;
	    t = mapp->t >> frac_tbits;

	    fs = mapp->s & frac_smsk;
	    ft = mapp->t & frac_tmsk;

	    BILERPSTRGB(u,v,s,t,fs,ft,imgp);
        }
    }
    else {
	for ( x=0 ; x<w ; x++,mapp++,imgp++ ) {
	    u = mapp->u >> frac_ubits;
	    v = mapp->v >> frac_vbits;
	    s = mapp->s >> frac_sbits;
	    t = mapp->t >> frac_tbits;

	    POINTSAMPLERGB(u,v,s,t,imgp);
	}
    }
#else
    for ( x=0 ; x<w ; x++,imgp++ )
	imgp->r = imgp->g = imgp->b = 0xff;
#endif
}

void
lfResampleVQRGBA8(const LFSlab *slab, LFPixelUVST *scanline, int x, int y, int w)
{
    LFView *view = &__LF_GET_CONTEXT()->view;
    LFPixelUVST *mapp  = scanline;
    LFPixelRGBA8 *imgp = (LFPixelRGBA8 *)view->pixel_buffer + y*view->width + x;
#ifndef	SCAN_ONLY
    bool_t uvlerp = (view->filter & LF_LERP_UV) == LF_LERP_UV;
    bool_t stlerp = (view->filter & LF_LERP_ST) == LF_LERP_ST;
    LFShared *shared = slab->shared;
    LFVQCodebook *vq = shared->vq;
    int frac_ubits = shared->u_frac_bits;
    int frac_vbits = shared->v_frac_bits;
    int frac_sbits = shared->s_frac_bits;
    int frac_tbits = shared->t_frac_bits;
    int frac_umsk = (1 << frac_ubits) - 1;
    int frac_vmsk = (1 << frac_vbits) - 1;
    int frac_smsk = (1 << frac_sbits) - 1;
    int frac_tmsk = (1 << frac_tbits) - 1;
    int u, v, s, t;
    int fu, fv, fs, ft;
    /* BEGIN stuff used inside macros */
    short *lightfield = slab->lightfield;
    LFPixelRGBA8 *codebook = shared->vq->codebook;
    LFPixelRGBA8 *p00, *p10, *p01, *p11;
    LFPixelRGBA8 c00,c10,c01,c11;
    int tile_shift = vq->ubits + vq->vbits + vq->sbits + vq->tbits;
    int *uindex = shared->uindex;
    int *vindex = shared->vindex;
    int *sindex = shared->sindex;
    int *tindex = shared->tindex;
    int *vqtile_uindex = shared->vqtile_uindex;
    int *vqtile_vindex = shared->vqtile_vindex;
    int *vqtile_sindex = shared->vqtile_sindex;
    int *vqtile_tindex = shared->vqtile_tindex;
    int uclamp = (shared->nu << shared->u_frac_bits) - 1;
    int vclamp = (shared->nv << shared->v_frac_bits) - 1;
    int sclamp = (shared->ns << shared->s_frac_bits) - 1;
    int tclamp = (shared->nt << shared->t_frac_bits) - 1;
    /* END for macros */

    if  (NEEDCLAMP_SAMPLE(mapp) || NEEDCLAMP_SAMPLE(mapp+w-1)) {
	clampScanline(view, scanline, w, uclamp, vclamp, sclamp, tclamp);
    }

    if  (uvlerp && stlerp) {
	for ( x=0 ; x<w ; x++,mapp++,imgp++ ) {
	    u = mapp->u >> frac_ubits;
	    v = mapp->v >> frac_vbits;
	    s = mapp->s >> frac_sbits;
	    t = mapp->t >> frac_tbits;

	    fu = mapp->u & frac_umsk;
	    fv = mapp->v & frac_vmsk;
	    fs = mapp->s & frac_smsk;
	    ft = mapp->t & frac_tmsk;

	    QUADLERPVQRGB(u,v,s,t,fu,fv,fs,ft,imgp);
	}
    }
    else if  (uvlerp) {
	for ( x=0 ; x<w ; x++,mapp++,imgp++ ) {
	    u = mapp->u >> frac_ubits;
	    v = mapp->v >> frac_vbits;
	    s = mapp->s >> frac_sbits;
	    t = mapp->t >> frac_tbits;

	    fu = mapp->u & frac_umsk;
	    fv = mapp->v & frac_vmsk;

	    BILERPUVVQRGB(u,v,s,t,fu,fv,imgp);
        }
    }
    else if  (stlerp) {
	for ( x=0 ; x<w ; x++,mapp++,imgp++ ) {
	    u = mapp->u >> frac_ubits;
	    v = mapp->v >> frac_vbits;
	    s = mapp->s >> frac_sbits;
	    t = mapp->t >> frac_tbits;

	    fs = mapp->s & frac_smsk;
	    ft = mapp->t & frac_tmsk;

	    BILERPSTVQRGB(u,v,s,t,fs,ft,imgp);
        }
    }
    else {
	for ( x=0 ; x<w ; x++,mapp++,imgp++ ) {
	    u = mapp->u >> frac_ubits;
	    v = mapp->v >> frac_vbits;
	    s = mapp->s >> frac_sbits;
	    t = mapp->t >> frac_tbits;

	    POINTSAMPLEVQRGB(u,v,s,t,imgp);
	}
    }
#else
    for ( x=0 ; x<w ; x++,imgp++ )
	imgp->r = imgp->g = imgp->b = 0xff;
#endif
}


