/*
 * read.c
 *
 * stuff to input lightfields
 */
#include <stdio.h>
#include <stdlib.h>
#ifdef sgi
#include <gl/image.h>
#endif /* sgi */
#include "lightfield.h"
#include "lerp.h"

static int
readvert(FILE *file, LFVtx *v)
{
    return fscanf(file,"%f %f %f %f %f %f",
        &v->ox, &v->oy, &v->oz, &v->ow, &v->s, &v->t) == 6;
}

static int
readquad(FILE *file, LFVtx *v)
{
    return (
	readvert(file, v+0) &&
	readvert(file, v+1) &&
	readvert(file, v+2) &&
	readvert(file, v+3));
}

static void
readslab(char *basename, LFSlab *slab)
{
#ifndef sgi
  /* Cannot read an array of rgb files if we're not on an sgi, because
   * this relies on the <image.h> library.
   */
  exit(12);
#else /* sgi */
    LFShared *shared = slab->shared;
    LFPixelRGBA8 *lightfield;
    LFPixelRGBA8 *lp;
    short rbuf[8192];
    short gbuf[8192];
    short bbuf[8192];
    IMAGE *image;
    char name[120];
    int u, v;
    int x, xsize;
    int y, ysize;
    int zsize;
    int nu = shared->nu;
    int nv = shared->nv;
    int ns = shared->ns;
    int nt = shared->nt;
    int *uindex;
    int *vindex;
    int *sindex;
    int *tindex;
    int *vqtile_uindex;
    int *vqtile_vindex;
    int *vqtile_sindex;
    int *vqtile_tindex;

    shared->sample_size = 4;
    slab->lightfield = lightfield = 
       (LFPixelRGBA8 *)malloc( nu*nv*ns*nt*sizeof(LFPixelRGBA8) );
    if  (lightfield == NULL) {
	fprintf(stderr,"makeslab: can't allocate storage\n");
	exit(1);
    }
    lfMakeIndices(shared);
    uindex = shared->uindex;
    vindex = shared->vindex;
    sindex = shared->sindex;
    tindex = shared->tindex;
    vqtile_uindex = shared->vqtile_uindex;
    vqtile_vindex = shared->vqtile_vindex;
    vqtile_sindex = shared->vqtile_sindex;
    vqtile_tindex = shared->vqtile_tindex;

    for( u=0; u<nu; u++ )
	for( v=0; v<nv; v++ ) {
	    sprintf(name,"%s.%d.%d.rgb",basename,u,v);
	    printf("reading %s ...\n", name );
            if( (image=iopen(name,"r")) == NULL ) {
		fprintf(stderr,"readlight: can't open input file %s\n",name);
		exit(1);
	    }
	    xsize = image->xsize;
	    ysize = image->ysize;
	    zsize = image->zsize;
	    if(zsize<3) {
		fprintf(stderr,"readlight: this is not an RGB image file\n");
		exit(1);
	    }
	    if( xsize != ns || ysize != nt ) {
		fprintf(stderr,"readlight: this is not the right size image\n");
		exit(1);
	    }
            for( y=0; y<ysize; y++ ) {
		getrow(image,rbuf,y,0);
		getrow(image,gbuf,y,1);
		getrow(image,bbuf,y,2);
		for( x=0; x<xsize; x++ ) {
		    lp = LIGHTPTR(u,v,x,y);
		    lp->r = rbuf[x];
		    lp->g = gbuf[x];
		    lp->b = bbuf[x];
		}
	    }
	    iclose(image);
	}
#endif /* sgi */
}

extern int
    Load_Reg_File(const char* filespec,
	int* domain_dimens, int* range_dimens, int len[],
	int* pixelsize, unsigned char** pixels);

static void
readcodebook(char* filespec, LFVQCodebook *codebook)
{
    LFPixelRGBA8 *code;
    unsigned char *codebook_bitstream, *codebook_bitstream_start;
    int	codebook_domain_dimens;
    int	codebook_range_dimens;
    int	codebook_len[6];
    int	codebook_pixelsize;
    int dim;
    int i;

    if (!Load_Reg_File(filespec,
	&codebook_domain_dimens,&codebook_range_dimens,
	codebook_len,&codebook_pixelsize,(unsigned char **)&codebook_bitstream_start)) {
	fprintf(stderr,"readcodebook: cannot load codebook %s\n", filespec);
	exit(1);
    }

    if ((codebook_domain_dimens&1) != 1 ||	/* nD hypervol, n odd */
	codebook_range_dimens != 1 ||		/* Vector-valued */
	codebook_len[codebook_domain_dimens] != 3) {
	fprintf(stderr,"readcodebook:%s is not an rgb codebook?\n", filespec);
	exit(1);
    }

    codebook->ubits = log2(codebook_len[3]);
    codebook->vbits = log2(codebook_len[2]);
    codebook->sbits = log2(codebook_len[1]);
    codebook->tbits = log2(codebook_len[0]);

    codebook->size = codebook_len[codebook_domain_dimens-1];
    /* For now, codebook size is always 4 bites */
    codebook->sample_size = 4;

    dim = codebook_len[3] * codebook_len[2] * codebook_len[1] * codebook_len[0] *
	  codebook->size;
    codebook->codebook = code = malloc(sizeof(LFPixelRGBA8) * dim);
    codebook_bitstream = codebook_bitstream_start;
    for ( i=0 ; i<dim ; i++,code++ ) {
	code->b = *codebook_bitstream++;
	code->g = *codebook_bitstream++;
	code->r = *codebook_bitstream++;
	code->a = 0xff;
    }
    free(codebook_bitstream_start);
}

static void
readcodearray(char* filespec, LFSlab *slab)
{
    int *codearray_bitstream_address;

    {
	int codearray_domain_dimens;
	int codearray_range_dimens;
	int codearray_len[1];
	int codearray_pixelsize;

	if (!Load_Reg_File(filespec,
	    &codearray_domain_dimens,&codearray_range_dimens,
	    codearray_len, &codearray_pixelsize,
	    (unsigned char **)&codearray_bitstream_address))  {
	    fprintf(stderr,"readcodearray: cannot load codearray %s\n", filespec);
	    exit(1);
	}

	/* Prefix and bitstream are a vector register */
	if (codearray_domain_dimens != 1 ||	/* Vector */
	    codearray_range_dimens != 0 ||
	    codearray_pixelsize != 1) {		/* Byte */
	    fprintf(stderr,"%s is not a codearray?\n", filespec);
	    exit(1);
	}
    }

    /* Retrieve tiled image dimensions from prefix to bitstream */
    {
	LFShared *shared = slab->shared;
	unsigned short *lightfield;
	LFVQCodebook *vq = shared->vq;
	LFPixelRGBA8 *codebook = vq->codebook;
	int *codearray_bitstream_start = codearray_bitstream_address;
	unsigned short *codearray_data;
	int image_domain_dimens = *codearray_bitstream_address++;
	int image_range_dimens = *codearray_bitstream_address++;
	int image_dimens = image_domain_dimens + image_range_dimens;
	int image_pixelsize;
	int image_len[9];	  /* 4D tile + 4D lightfield + color */
	int dim;
	int inc;
	int i;

	for ( i=0; i<image_dimens; i++)
	    image_len[i] = *codearray_bitstream_address++;
	image_pixelsize = *codearray_bitstream_address++;

	if ((image_domain_dimens&1) != 0 ||	/* nD hypervol, n even */
	    image_range_dimens != 1 ||		/* Vector-valued */
	    image_len[image_domain_dimens] != 3 ||	/* rgb */
	    image_pixelsize != 1) {			/* Bytes */
	    fprintf(stderr,"%s not an encoded rgb nD image, n even?\n", filespec);
	    exit(1);
	}

	if (image_len[3] != (1 << vq->ubits) ||
	    image_len[2] != (1 << vq->vbits) ||
	    image_len[1] != (1 << vq->sbits) ||
	    image_len[0] != (1 << vq->tbits)) {
	    fprintf(stderr,"Tile size in %s disagrees with codebook?\n", filespec);
	    exit(1);
	}

	if (image_len[3] * image_len[7] != shared->nu ||
	    image_len[2] * image_len[6] != shared->nv ||
	    image_len[1] * image_len[5] != shared->ns ||
	    image_len[0] * image_len[4] != shared->nt) {
	    fprintf(stderr,"codearray dimension in %s disagrees with lightfield dimension?\n", filespec);
	    exit(1);
	}

	dim = image_len[7] * image_len[6] * image_len[5] * image_len[4];
	inc = image_len[3] * image_len[2] * image_len[1] * image_len[0];
	lightfield = malloc(sizeof(lightfield[0]) * dim);
	codearray_data = (unsigned short *)codearray_bitstream_address;
	for ( i=0 ; i<dim ; i++ )
	    lightfield[i] = codearray_data[i];
	slab->lightfield = lightfield;
	free(codearray_bitstream_start);

	lfMakeIndices(shared);
    }
}

static void
readslabheader(LFField *field, char *name, LFSlab *slab)
{
    FILE *file;
    LFShared *shared = slab->shared;
    int nu, nv, ns, nt;
    char codebook_cmd[32];
    char codebook_name[128];
    char codearray_cmd[32];
    char codearray_name[128];

#ifdef sgi
    if( (file = fopen(name,"r")) == NULL ) {
#else  /* WIN32 */
    if( (file = fopen(name,"r+b")) == NULL ) {
#endif /* sgi */
        fprintf(stderr, "readslabheader: cannot read %s\n", name);
	exit(1);
    }

    if( !readquad(file,slab->uv) || !readquad(file,slab->st) )  {
	fprintf(stderr, "readslabheader: invalid quads\n");
	exit(1);
    }

    if( fscanf(file,"%d %d %d %d", &nu, &nv, &ns, &nt ) != 4 ) {
	fprintf(stderr, "readslabheader: invalid file format\n");
	exit(1);
    }

    shared->nu = nu;
    shared->nv = nv;
    shared->ns = ns;
    shared->nt = nt;

    shared->u_frac_bits = 6;
    shared->v_frac_bits = 6;
    shared->s_frac_bits = 6;
    shared->t_frac_bits = 6;

    if  (  fscanf(file,"%s%s", codebook_cmd, codebook_name) == 2
	&& strcmp(codebook_cmd, "codebook") == 0
	&& fscanf(file,"%s%s", codearray_cmd, codearray_name) == 2
	&& strcmp(codearray_cmd, "codearray") == 0
	)  {
	if  (! field->shared)  {
	    shared->vq = calloc(1, sizeof(LFVQCodebook));
	    readcodebook(codebook_name, shared->vq);
	    field->shared = shared;
	}
	else {
	    shared->vq = field->shared->vq;
	}
	readcodearray(codearray_name, slab);
    }

    fclose(file);
}

LFSlab
*makeslab(LFField *field, char *name)
{
    LFSlab *slab;

    slab = calloc(1, sizeof(LFSlab));
    slab->shared = calloc(1, sizeof(LFShared));

    readslabheader(field, name, slab);

    if  (! slab->shared->vq)
	readslab(name,slab);

    return slab;
}

static int
readslabname(FILE *file, char *name)
{
    return fscanf(file,"%s",name) == 1;
}

LFField
*lfReadField(char *name)
{
#ifndef sgi
  /* Cannot read an array of rgb files if we're not on an sgi, because
   * this relies on the <image.h> library.
   */
  exit(13);
#else /* sgi */
    FILE *file;
    LFField *field = NULL;
    char dir[128];
    char suffix[128];
    char slabname[128];
    int i, n;

    file = fopen(name,"r");
    if( file == NULL ) {
	fprintf(stderr, "cannot read %s\n", name);
	exit(1);
    }

    fscanf(file,"%d",&n);
    if( n > 0 ) {
        if( n > LF_MAXSLABS ) {
	    fprintf(stderr,"Max slabs = %d\n", LF_MAXSLABS );
 	    n = LF_MAXSLABS;
        }
	field = calloc(1, sizeof(LFField));
        field->nslabs = n;
	field->enable_msk = (1 << n) - 1;
        printf("Reading %d slabs ... \n", n );
        for( i=0; i<n; i++ ) {
	    strcpy(dir, name);
	    if( readslabname(file,suffix) )  {
		sprintf(slabname, "%s/%s", dirname(dir), suffix);
		field->slabs[i] = makeslab(field, slabname);
	    }
        }
    }

    fclose(file);

    return field;
#endif /* sgi */
}
