/*
 * rasterize.c
 *
 * program to scan convert triangles
 */

#include "lightfield.h"

/*
 * notice the definition of ceiling and floor mandates that the
 * triangle rasterization is half open from bottom, and half
 * closed from top
 */
#define CEIL(x)  	((int) (x + 1.f))

#define VSWAP(a, b) { LFScreenVtx *tmp = a; a = b; b = tmp; }

#define	DIFF_HI_MD_LO(x)	\
	d##x##_hi_lo = v_hi->##x - v_lo->##x,	\
	d##x##_md_lo = v_md->##x - v_lo->##x

#define	EQUATION(U, u)		\
	cur##U = v_lo->u,\
	d##U##dx = (dy_md_lo * d##u##_hi_lo - dy_hi_lo * d##u##_md_lo) / area,	\
	d##U##dy = (dx_hi_lo * d##u##_md_lo - dx_md_lo * d##u##_hi_lo) / area

/*
 * function to rasterize a point-sampled triangle and generate corresponding
 * texture coordinates [u, v, s, t] for each pixel.
 *
 * the algorithm works as follows: it first finds the major edge e1
 * (from the top-most vertex to the bottom-most vertex); it then
 * walk along e1, depending on the relative orientation of the middle
 * vertex, the DDA might be performed from left-to-right, or right-
 * to-left.
 *
 *	vhi
 *	x  x e2
 *	x    x
 *	x      vmd
 *   e1	x     x
 *	x    x
 *	x   x e0
 *	x  x
 *	vlo
 *
 * the derivation for dudx and dudy is as follows:
 *
 * u = u_lo + a_md * (u_md - u_lo) / area + a_hi * (u_hi - u_lo) / area
 *
 * where a_md = (x_lo-x) * (y_hi-y) - (x_hi-x) * (y_lo-y)
 *	 a_hi = (x_md-x) * (y_lo-y) - (x_lo-x) * (y_md-y)
 *	 area = (x_hi-x_lo) * (y_md-y_lo) - (x_md-x_lo) * (y_hi_y_lo)
 *
 * d(a_md)/dx = y_lo - y_hi
 * d(a_md)/dy = x_hi - x_lo
 * d(a_hi)/dx = y_md - y_lo
 * d(a_hi)/dy = x_lo - x_md
 *
 * let u' = u * area
 *     dx_hi_lo = x_hi - x_lo
 *     dy_hi_lo = y_hi - y_lo
 *     dx_md_lo = x_md - x_lo
 *     dy_md_lo = y_md - y_lo
 *     du_hi_lo = u_hi - u_lo
 *     du_md_lo = u_md - u_lo
 *
 * then
 *
 * area = dx_hi_lo * dy_md_lo - dx_md_lo * dy_hi_lo
 *
 * u' = u_lo * area + a_md * du_md_lo + a_hi * du_hi_lo
 *
 * du'dx = dy_md_lo * du_hi_lo - dy_hi_lo * du_md_lo
 * du'dy = dx_hi_lo * du_md_lo - dx_md_lo * du_hi_lo
 *
 * it's ok to multiply area since we always compute u/w
 * and areas on the enumerator and denominator cancel out
 */
void
lfRasterizeTriPntSample( const LFSlab *slab,
	const LFScreenVtx *v0, const LFScreenVtx *v1, const LFScreenVtx *v2)
{
#ifndef OVERHEAD_ONLY
    LFView *view = &__LF_GET_CONTEXT()->view;
    LFScreenVtx *v_lo = (LFScreenVtx *)v0;
    LFScreenVtx *v_md = (LFScreenVtx *)v1;
    LFScreenVtx *v_hi = (LFScreenVtx *)v2;
    /* Added these 2 to center point-sampling */
    float uvhalfpixel = (view->filter & LF_LERP_UV) ? 0.0 : 
      1 << (slab->shared->u_frac_bits-1);
    float sthalfpixel = (view->filter & LF_LERP_ST) ? 0.0 :
      1 << (slab->shared->s_frac_bits-1);

    int y_lo;
    int y_md;
    int y_hi;

    int left_to_right;
    int first_time;

    int cur_x, cur_y, top_y;

    LFPixelUVST *scanline;
    void (*resample)(const LFSlab *, LFPixelUVST *, int, int, int);

    float xfrac, yfrac;
    float lx, rx, dlx, drx;

    float e0_dxdy, e1_dxdy, e2_dxdy;

    float curU, curxU, dUdx, dUdy;
    float curV, curxV, dVdx, dVdy;
    float curW, curxW, dWdx, dWdy;
    float curS, curxS, dSdx, dSdy;
    float curT, curxT, dTdx, dTdy;
    float curQ, curxQ, dQdx, dQdy;

    if  (v_lo->y > v_md->y) {
	VSWAP(v_lo, v_md);
    }
    if  (v_md->y > v_hi->y) {
	VSWAP(v_md, v_hi);
    }
    if  (v_lo->y > v_md->y) {
	VSWAP(v_lo, v_md);
    }

    y_lo = CEIL(v_lo->y);
    y_md = CEIL(v_md->y);
    y_hi = CEIL(v_hi->y);

    /* trivial testing if any pixel should be drawn */
    if  (y_lo == y_hi)
	return;
    
    /* compute iteration parameters */
    {
	float area;
	float dx_hi_lo, dx_md_lo;
	float dy_hi_lo, dy_md_lo;
	float du_hi_lo, du_md_lo;
	float dv_hi_lo, dv_md_lo;
	float dw_hi_lo, dw_md_lo;
	float ds_hi_lo, ds_md_lo;
	float dt_hi_lo, dt_md_lo;
	float dq_hi_lo, dq_md_lo;

	/* differences */
	DIFF_HI_MD_LO(x);
	DIFF_HI_MD_LO(y);
	DIFF_HI_MD_LO(u);
	DIFF_HI_MD_LO(v);
	DIFF_HI_MD_LO(w);
	DIFF_HI_MD_LO(s);
	DIFF_HI_MD_LO(t);
	DIFF_HI_MD_LO(q);

	/* area and slopes */
	area = dx_hi_lo * dy_md_lo - dx_md_lo * dy_hi_lo;
	e1_dxdy =  dx_hi_lo / dy_hi_lo;

	if  (y_lo == y_md) {
	    e0_dxdy = 0;
	    e2_dxdy = (v_hi->x - v_md->x) / (v_hi->y - v_md->y);
	    left_to_right = e1_dxdy > e2_dxdy;
	}
	else {
	    e0_dxdy = dx_md_lo / dy_md_lo;
	    e2_dxdy = (y_md != y_hi) ?
		      (v_hi->x - v_md->x) / (v_hi->y - v_md->y) : 0;
	    left_to_right = e1_dxdy <= e0_dxdy;
	}

	/*
	 * all iteration parameters w.r.t. u, v, w, s, t, q are scaled
	 * up by area
	 */
	EQUATION(U, u);
	EQUATION(V, v);
	EQUATION(W, w);
	EQUATION(S, s);
	EQUATION(T, t);
	EQUATION(Q, q);
    }

    cur_y = MAX(y_lo, 0);
    top_y = MIN(y_md, view->height);
    yfrac = (float) cur_y - v_lo->y;

    curU += yfrac * dUdy;
    curV += yfrac * dVdy;
    curW += yfrac * dWdy;
    curS += yfrac * dSdy;
    curT += yfrac * dTdy;
    curQ += yfrac * dQdy;

    if  (left_to_right)  {
	dlx = e1_dxdy;
	drx = e0_dxdy;
    }
    else {
	drx = e1_dxdy;
	dlx = e0_dxdy;
    }
    
    lx = v_lo->x + yfrac * dlx;
    rx = v_lo->x + yfrac * drx;

    scanline = view->sample_buffer;
    resample = slab->shared->vq ? lfResampleVQRGBA8 : lfResampleRGBA8;

    first_time = 1;

#ifdef	STATS
    view->scanline_total -= cur_y; 
#endif

loop_twice:

    for ( ; cur_y<top_y ; cur_y++,lx+=dlx,rx+=drx,
	curU+=dUdy,curV+=dVdy,curW+=dWdy,curS+=dSdy,curT+=dTdy,curQ+=dQdy ) {
	int cxl = CEIL(lx);
	int cxr = CEIL(rx);

	int xl = MAX(cxl, 0);
	int xr = MIN(cxr, view->width);
	int width = xr - xl;

	if  (width <= 0)
	    continue;

	xfrac = (float)xl - v_lo->x;

	curxU = curU + xfrac * dUdx;
	curxV = curV + xfrac * dVdx;
	curxW = curW + xfrac * dWdx;
	curxS = curS + xfrac * dSdx;
	curxT = curT + xfrac * dTdx;
	curxQ = curQ + xfrac * dQdx;

	for ( cur_x=0 ; cur_x<width ; cur_x++ ) {
	    float inv_w = 1.f / curxW;
	    float inv_q = 1.f / curxQ;

	    scanline[cur_x].u = (curxU * inv_w + uvhalfpixel);
	    scanline[cur_x].v = (curxV * inv_w + uvhalfpixel);
	    scanline[cur_x].s = (curxS * inv_q + sthalfpixel);
	    scanline[cur_x].t = (curxT * inv_q + sthalfpixel);
	    
	    curxU += dUdx;
	    curxV += dVdx;
	    curxW += dWdx;
	    curxS += dSdx;
	    curxT += dTdx;
	    curxQ += dQdx;
	}

	(*resample)(slab, scanline, xl, cur_y, width);
    }
    
    if  (first_time) {
	cur_y = MAX(y_md, 0);
	top_y = MIN(y_hi, view->height);
	yfrac = (float) cur_y - v_md->y;
	if  (left_to_right) {
	    drx = e2_dxdy;
	    rx = v_md->x + yfrac * drx;
	}
	else {
	    dlx = e2_dxdy;
	    lx = v_md->x + yfrac * dlx;
	}
	first_time = 0;
	goto loop_twice;
    }

#ifdef	STATS
    view->scanline_total += top_y; 
#endif
#endif	/* ! OVERHEAD_ONLY */
}
