/*
 * lif_api.c
 *
 * API to read/write lif files
 */
#include <stdio.h>
#include <stdlib.h>
#include "lightfield.h"
#include "fileIO.h"

static void
readLifFillSlab(LFOps *ops, LFInternOp *op, LFFileHdr *hdr, LFSlab **slabs)
{
    LFShared *shared;
    LFVQCodebook *vq;
    int i, n, id;

    /* determine slabs to be operated on */
    for ( i=n=0 ; i<ops->op_slab_cnt ; i++ )  {
	id = ops->op_slabs[i]->id;
	if  (hdr->slabHdr[id].size)
	    ops->op_slabs[n++] = ops->op_slabs[i];
    }
    ops->op_slab_cnt = n;

    for ( i=0 ; i<ops->op_slab_cnt ; i++ )  {
	id = ops->op_slabs[i]->id;

	/* copy slab attribute information over */
	*ops->op_slabs[i] = *slabs[id];

	/* read codebook information if needed */
	shared = ops->op_slabs[i]->shared;
	vq = shared->vq;
	if  (hdr->slabHdr[id].format == LF_INDEXCHANNEL && ! vq->codebook)  {
	    lfReadCodebook(vq, shared, hdr, hdr->file);
	    vq->type = LF_VQ_LINEAR_DATA;
	}
    }
}

void *
lfBeginReadLif(LFOps *ops, LFInternOp *op)
{
    LFFileHdr *hdr = malloc(sizeof(LFFileHdr));
    LFSlab *slabs[LF_MAXSLABS];
    int i;

    /* allocate temporary slabs */
    slabs[0] = calloc(LF_MAXSLABS, sizeof(LFSlab));
    for ( i=0 ; i<LF_MAXSLABS ; i++ )
	slabs[i] = &slabs[0][i];

    /* Open the file and read the header */
    if  (! (hdr->file = lfOpenInFile(op->op_extra, &hdr, slabs, LF_MAXSLABS))) {
	lfError("Cannot read %s\n", op->op_extra);
	exit(1);
    }

    /* fill in slab data structure */
    if  (ops->chain_msk & LF_SET_SLAB)
	readLifFillSlab(ops, op, hdr, slabs);

    /* determine dataflow per this operation */
    op->op_func   = lfReadLif;
    op->op_end    = lfEndReadLif;
    op->op_update = ops->chain_msk &
		    (LF_GEN_VQ_TRAINSET|LF_GEN_VQ_CODEARRAY) ?
		    lfSerialUpdateBlockSlice : lfUpdateNoop;

    /* free temporary slabs */
    free(slabs[0]);

    /* return lightfield file header */
    return hdr;
}

void
lfReadLif(LFSlab **slabs, int nslab, const int *pos,
	const float *descr, const void *input, int input_size,
	void *output, int output_size, void *aux_ptr)
{
    LFSlab *slab = slabs[pos[LF_SLAB_POS]];
    int u = pos[LF_U_POS];
    int v = pos[LF_V_POS];
    int id = slab->id;
    LFFileHdr *hdr = aux_ptr;

    if  (hdr->slabHdr[id].format == LF_INDEXCHANNEL) {
	LFVQCodebook *vq = slab->shared->vq;
	lfReadBlock(output, id, slab, hdr, hdr->file,
		u, u+(1<<vq->ubits)-1, v, v+(1<<vq->vbits)-1,
		LF_INDEXCHANNEL, LF_INT16);
    }
    else {
	lfReadBlock(output, id, slab, hdr, hdr->file, u, u, v, v,
		LF_RGBACHANNEL, LF_INT8x4);
    }
}

void
lfEndReadLif(LFOps *ops, LFInternOp *op, void *aux_ptr)
{
    LFFileHdr *hdr = aux_ptr;
    lfCloseInFile(&hdr, hdr->file);
}

void *
lfBeginWriteLif(LFOps *ops, LFInternOp *op)
{
    LFFileHdr *hdr = malloc(sizeof(LFFileHdr));

    /* Open the file */
    if  (! (hdr->file = lfOpenOutFile(op->op_extra, &hdr)))  {
	lfError("Cannot read %s\n", op->op_extra);
	exit(1);
    }

    /* determine dataflow per this operation */
    op->op_func   = lfWriteLif;
    op->op_end    = lfEndWriteLif;
    op->op_update = ops->chain_msk &
		    (LF_GEN_VQ_TRAINSET|LF_GEN_VQ_CODEARRAY) ?
		    lfSerialUpdateBlockSlice : lfUpdateNoop;

    /* return lightfield file header */
    return hdr;
}

void
lfWriteLif(LFSlab **slabs, int nslab, const int *pos,
	const float *descr, const void *input, int input_size,
	void *output, int output_size, void *aux_ptr)
{
    LFSlab *slab = slabs[pos[LF_SLAB_POS]];
    int u = pos[LF_U_POS];
    int v = pos[LF_V_POS];
    int id = slab->id;
    LFFileHdr *hdr = aux_ptr;

    if  (slab->shared->vq) {
	LFVQCodebook *vq = slab->shared->vq;
	lfWriteBlock((void *)input, id, slab, hdr, hdr->file,
		u, u+(1<<vq->ubits)-1, v, v+(1<<vq->vbits)-1,
		LF_INDEXCHANNEL, LF_INT16);
    }
    else {
	LFShared *shared = slab->shared;
	lfWriteBlock((void *)input, id, slab, hdr, hdr->file, u, u, v, v,
		shared->sample_size == 3 ? LF_RGBACHANNEL : LF_RGBACHANNEL,
		shared->sample_size == 3 ? LF_INT8x3 : LF_INT8x4);
    }
}

void
lfEndWriteLif(LFOps *ops, LFInternOp *op, void *aux_ptr)
{
    LFFileHdr *hdr = aux_ptr;
    LFSlab *slab;
    LFVQCodebook *vq;
    int i;

    /* output VQ codebook if necessary */
    for ( i=0 ; i<ops->op_slab_cnt ; i++ )  {
	slab = ops->op_slabs[i];
	if  (! (vq = slab->shared->vq))
	    continue;

	lfLinearizeVQCodebook(vq);
	lfWriteCodebook(vq->codebook, vq->id, slab, hdr, hdr->file,
		LF_CODEBOOKCHANNEL,
		vq->sample_size == 3 ? LF_INT8x3 : LF_INT8x4);
    }

    lfCloseOutFile(&hdr, hdr->file);
}

