/*
 * io.c
 *
 * stuff related to file I/O
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef sgi
#include <libgen.h>
#include <gl/image.h>
#endif

#include "lightfield.h"

LFSlab *__lf_descr_slabs;

typedef struct {
    char *slab_names[LF_MAXSLABS];
    int nslab;
} LFSliceHeader;

static void
fillSlabDefault(LFInternOp *op)
{
    LFShared *shared;
    float *descr = op->op_descr;
    int u_size, v_size;
    int s_size, t_size;
    int sample_size;
    int i;

    u_size = v_size = 16;
    s_size = t_size = 256;
    sample_size = 3;
    while (*descr != LF_NULL) {
	switch((int)*descr++) {
	case LF_READ_SAMPLES_UV:
	    u_size = *descr++;
	    v_size = *descr++;
	    break;

	case LF_READ_SAMPLES_ST:
	    s_size = *descr++;
	    t_size = *descr++;
	    break;

	case LF_READ_FORMAT:
	    switch((int)*descr++) {
	    case LF_RGB:
		sample_size = 3;
		break;

	    case LF_RGBA:
		sample_size = 4;
		break;
	    }
	    break;
	}
    }

    for ( i=0 ; i<LF_MAXSLABS ; i++ )  {
	shared = __lf_descr_slabs[i].shared;
	shared->sample_size = sample_size;
	shared->nu = u_size;
	shared->nv = v_size;
	shared->ns = s_size;
	shared->nt = t_size;
    }
}

/*
 * function to begin inputing all slice files in a lightfield
 */
void *
lfBeginReadSliceFile(LFOps *ops, LFInternOp *op)
{
#ifndef sgi
  /* Authoring functions are not supported on non-sgi machines, due to 
   * the lack of support for rgb files <image.h>.
   */
  exit(38);
#else /* sgi */
  char *index_fnm = op->op_extra;
  FILE *fp;
  LFSliceHeader *hdr = calloc(1, sizeof(LFSliceHeader));
  LFSlab *slab;
  char fnm[256];
  char name[128], *dir;
  char suffix[128];
  int i, n, id;
  
  __lf_descr_slabs = calloc(LF_MAXSLABS, sizeof(LFSlab));
  for ( i=0 ; i<LF_MAXSLABS ; i++ )
    __lf_descr_slabs[i].shared = calloc(1, sizeof(LFShared));
  
  /* fill in the defaults */
  fillSlabDefault(op);
  
  /* determine available slabs */
  if  ((fp=freopen(index_fnm, "r", stdin)) == NULL) {
    lfError("lfBeginReadSliceFile: cannot read %s\n", index_fnm);
    exit(1);
  }
  yyparse();
  fclose(fp);

  /* create a file header to access slab slices */
  strcpy(name, index_fnm);
  dir = dirname(name);
  for ( i=0 ; i<LF_MAXSLABS ; i++ ) {
    slab = &__lf_descr_slabs[i];
    if  (! slab->lightfield) 
      continue;
    if (((char *)slab->lightfield)[0] == '/') {
      sprintf(fnm, "%s", (char *)slab->lightfield);
    } else {
      sprintf(fnm, "%s/%s", dir, (char *)slab->lightfield);
    }
    hdr->slab_names[i] = strdup(fnm);
    free(slab->lightfield);
    slab->lightfield = 0;
  }
  
  /* determine what slabs are operable */
  if  (ops->chain_msk & LF_SET_SLAB)  {
    for ( i=n=0 ; i<ops->op_slab_cnt ; i++ ) {
      id = ops->op_slabs[i]->id;
      if  (! hdr->slab_names[id])
	continue;
      
      if  (ops->op_slabs[i]->shared)  {
	if  (ops->op_slabs[i]->shared->vq)
	  free(ops->op_slabs[i]->shared->vq);
	free(ops->op_slabs[i]->shared);
      }
      *ops->op_slabs[i] = __lf_descr_slabs[id];
      ops->op_slabs[i]->id = id;
      ops->op_slabs[n++] = ops->op_slabs[i];
    }
    ops->op_slab_cnt = n;
  }
  free(__lf_descr_slabs);
  
  /* fill in functions */
  op->op_func  = lfReadSliceFile;
  op->op_end   = lfEndReadSliceFile;
  op->op_update = ops->chain_msk & (LF_GEN_VQ_TRAINSET|LF_GEN_VQ_CODEARRAY) ?
    lfSerialUpdateBlockSlice : lfUpdateNoop;
  
  return hdr;
#endif /* sgi */
}

/*
 * function to input a slice file
 */
void
lfReadSliceFile(LFSlab **slabs, int nslab, const int *pos,
	const float *descr, const void *input, int input_size,
	void *output, int output_size, void *aux_ptr)
{
#ifndef sgi
  /* Authoring functions are not supported on non-sgi machines, due to 
   * the lack of support for rgb files <image.h>.
   */
  exit(49);
#else /* sgi */
  IMAGE *image;
  LFSliceHeader *hdr = aux_ptr;
  LFSlab *slab = slabs[pos[LF_SLAB_POS]];
  LFShared *shared = slab->shared;
  char *ip = output, *lp;
  int slab_id = slab->id;
  short rbuf[8192];
  short gbuf[8192];
  short bbuf[8192];
  short abuf[8192];
  int line_size;
  int x, xsize;
  int y, ysize;
  int zsize;
  char slice_name[256];

  sprintf(slice_name, "%s.%d.%d.rgb",
	  hdr->slab_names[slab_id], pos[LF_U_POS], pos[LF_V_POS]);

  if  ((image=iopen(slice_name, "r")) == NULL) {
    lfError("lfReadSliceFile: cannot read %s\n", slice_name);
    exit(1);
  }

  xsize = image->xsize;
  ysize = image->ysize;
  zsize = image->zsize;

  if  (zsize < 3) {
    lfError("lfReadSliceFile: %s is not an RGB image file\n",
	    slice_name);
    exit(1);
  }

  if  (xsize != shared->ns || ysize != shared->nt) {
    lfError("lfReadSliceFile: size of %s mismatches size of slab %d\n",
	    slice_name, slab_id);
    exit(1);
  }

  line_size = shared->nt * shared->sample_size;
  if  (shared->sample_size == 3)  {
    for ( y=0 ; y<ysize ; y++,ip+=shared->sample_size ) {
      getrow(image,rbuf,y,0);
      getrow(image,gbuf,y,1);
      getrow(image,bbuf,y,2);
      for ( x=0,lp=ip ; x<xsize ; x++,lp+=line_size ) {
	lp[2] = rbuf[x];
	lp[1] = gbuf[x];
	lp[0] = bbuf[x];
      }
    }
  }
  else {
    for ( y=0 ; y<ysize ; y++,ip+=shared->sample_size ) {
      getrow(image,rbuf,y,0);
      getrow(image,gbuf,y,1);
      getrow(image,bbuf,y,2);
      getrow(image,abuf,y,3);
      for ( x=0,lp=ip ; x<xsize ; x++,lp+=line_size ) {
	lp[3] = rbuf[x];
	lp[2] = gbuf[x];
	lp[1] = bbuf[x];
	lp[0] = abuf[x];
      }
    }
  }
  iclose(image);
#endif /* sgi */
}
 
/*
 * function to end inputing all slice files in a lightfield
 */
void
lfEndReadSliceFile(LFOps *ops, LFInternOp *op, void *aux_ptr)
{
    LFSliceHeader *hdr = aux_ptr;
    int i;

    for ( i=0 ; i<hdr->nslab ; i++ ) {
	free(hdr->slab_names[i]);
    }
    free(hdr);
}
