/*
 *  fileIO.h - Read/Write a lightfield in the new format
 *
 */

#ifndef _FILEIO_H_
#define _FILEIO_H_

#include "lightfield.h"

/* Currently, we set aside this much room for the header when we
 * write a file.  For reading, it will accept any sized header.
 * This header should be big enough to hold information about 
 * the maximum number of slabs, and a separate vq codebook for
 * every slab.
 */
#define LF_DEFAULTHDRLEN 2000*LF_MAXSLABS

/* This character marks the end of the header.  When it appears in
 * the file, the next character should be the first byte of binary
 * data.
 */
#define LF_HDRTERMINATOR '\000'
#define LF_HDRFILLER     ' '

/* These data structures record information about the file that is
 * not necessarily stored (or not easily computed) from the 
 * regular lightfield / lightslab structres.
 */
typedef enum {
  LF_RGBCHANNEL, 
  LF_RGBACHANNEL, 
  LF_INDEXCHANNEL, 
  LF_CODEBOOKCHANNEL
} LF_ChannelFormat;

typedef enum {
  LF_INT8,
  LF_INT8x3,
  LF_INT8x4,
  LF_INT16,
  LF_INT32
} LF_ChannelType;

typedef struct LFSlabHdr_tag {
  LF_ChannelFormat   format;
  LF_ChannelType     type;
  int 		     offset;
  int                size;
  LFSlab	     *slab;
} LFSlabHdr;

typedef struct LFVQHdr_tag {
  LF_ChannelFormat   format;
  LF_ChannelType     type;
  int 		     offset;
  int                size;
  LFVQCodebook	     *codebook;
} LFVQHdr;


typedef struct LFFileHdr_tag {
  int hdrSize;
  int dataSize;
  LFSlabHdr slabHdr[LF_MAXSLABS];
  LFVQHdr   vqHdr[LF_MAXSLABS];
  FILE		     *file;
} LFFileHdr;


/* Prototypes */
 
/* ----- High-level Functions ----- */
/* Return 0 on failure */
/* These two functions are currently not dataflow-oriented.  
 * However, it should be easily converted.
 */
LFField *lfReadEntireField(char *name);

int     lfWriteEntireField(LFField *field, char *name);


/* ----- lower-level functions ----- */
/* Return 1 on success */

/* Open a Lightslab file, malloc the LFField and LFFileHdr structs,
 * and initialize them from the file header info.
 */
FILE *lfOpenInFile(char *fileName, LFFileHdr **hdr, LFSlab *slabs[], 
		   int nslabs);

/* Open a Lightslab file, malloc the LFFileHdr struct,
 * and fill it in with data from the LFField structure.
 */
FILE *lfOpenOutFile(char *fileName, LFFileHdr **hdr);


/* Write a portion (possibly all) of a slab of a lightfield to file */
int lfWriteBlock(void *block, int slabn, LFSlab *slab, LFFileHdr *hdr, 
		 FILE *file, int uFirst, int uLast, int vFirst, int vLast,
		 LF_ChannelFormat format, LF_ChannelType type);

/* Read a portion (possibly all) of a slab of a lightfield from file */
int lfReadBlock(void *block, int slabn, LFSlab *slab, LFFileHdr *hdr, 
		FILE *file, int uFirst, int uLast, int vFirst, int vLast,
		 LF_ChannelFormat format, LF_ChannelType type);

/* Write an entire codebook to file */
int lfReadCodebook(LFVQCodebook *vq, LFShared *shared, LFFileHdr *hdr, 
		   FILE *file);

/* Write an entire codebook to file */
int lfWriteCodebook(void *block, int vqn, LFSlab *slab, LFFileHdr *hdr, 
		 FILE *file, LF_ChannelFormat format, LF_ChannelType type);

/* Write the header (if necessary) and close the file.
 * NOTE -- This frees the LFFileHdr Structure! 
 */
int lfCloseInFile( LFFileHdr **hdr, FILE *inFile);
int lfCloseOutFile(LFFileHdr **hdr, FILE *outFile);

#endif /* _FILEIO_H_ */
