/*
 *  fileIO.c: read / write a lightfield in the new .lif file format
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include "lightfield.h"
#include "fileIO.h"
#include "lerp.h"


/* Local helper functions */
void *SlabSlice(LFSlab *slab, int u, int v);
void ClearHdr(LFFileHdr *hdr, FILE *file);
int GetHdrLen(FILE *file);
int WriteRawBlock(FILE *outFile, int offset, char *buffer, int elemCount,
		  int elemInSize, int elemOutSize);
int ReadRawBlock(FILE *inFile, int offset, char *buffer, int elemCount,
		  int elemInSize, int elemOutSize);
int NextHdrToken(FILE *file, char *buffer, int maxTokenSize);
int AssertToken(FILE *file, char *token);
int SkipSegment(FILE *inFile);
int ParseSlabHdr(FILE *inFile, LFFileHdr *hdr, LFSlab *slabs[], int nslabs);
int ParseVQHdr(FILE *inFile, LFFileHdr *hdr, LFSlab *slabs[], int nslabs);
int Log2(unsigned long n);


/* ----------------------------------------------------------------------
 * ---------------         High-level Functions           --------------- 
 * ----------------------------------------------------------------------
 */
/* Return 0 on failure */
/* These two functions are currently not dataflow-oriented.  
 * However, it should be easily converted.
 */

LFField *lfReadEntireField(char *name)
{
  LFField *field = NULL;
  LFFileHdr *hdr = NULL;
  FILE *file = NULL;
  int i, slabn, u, v;
  LFSlab *slab;
  void *block;
  int vqIndexBytes = 0;
  bool_t vqP = FALSE;


  /* ------------ Initialization Stage ---------------- */
  /* Allocate lightfield & slabs */
  if ((field = calloc(1, sizeof(LFField))) == NULL) {
    lfError("No memory for LFField.\n");
    return(0);
  }
  if ((field->slabs[0] = calloc(LF_MAXSLABS, sizeof(LFSlab))) == NULL) {
    lfError("Error: No memory for LFField slabs.\n");
    return(0);
  }
  /* Now make the slab pointers point to each element of slab array */
  for (i=0; i < LF_MAXSLABS; i++) {
    field->slabs[i] = &((field->slabs[0])[i]);
  }
  /* Set # of slabs */
  field->nslabs = LF_MAXSLABS;

  /* Open the file, read the header, allocate field & hdr structures, etc */
  if ((file = lfOpenInFile(name, &hdr, field->slabs, LF_MAXSLABS)) == NULL) {
    return(0);
  }

  /* Set the current fields active (so it will write them out again) */
  for (i=0; i< LF_MAXSLABS; i++) {
    if (hdr->slabHdr[i].size) {
      lfOutput("setting slab %d active\n", i);
      __LF_SET_SLAB_ACTIVE(field, i, 1);
    } else {
      lfOutput("Not setting slab %d active\n", i);
      __LF_SET_SLAB_ACTIVE(field, i, 0);
    }
  }

							   
  /* ------------ Main Body Stage ---------------- */
  /* --- Read in all the blocks --- */
  for (slabn = 0; slabn < field->nslabs; slabn++) {
    /* don't read it if slab is not active */
    if (!(__LF_SLAB_ACTIVE(field, slabn))) continue;
    /* don't read it if slab is not contained in the file */
    if (hdr->slabHdr[slabn].size == 0) continue;
    /* ok, it's active, and present in file.  read it in. */
    /* First, allocate lightfield array */
    lfOutput("Allocating field %d: %d bytes \n", slabn, 
	   hdr->slabHdr[slabn].size);
    field->slabs[slabn]->lightfield = (void *) 
      calloc(1, hdr->slabHdr[slabn].size);
    if (field->slabs[slabn]->lightfield == NULL) {
      lfError("Error: cannot alloc memory for lightfield array\n");
      return (0);
    } 
    lfOutput("lightfield array allocated. ptr: %x\n", 
	   (int) field->slabs[slabn]->lightfield); 

    /* Fill in the tables for indexing into this lightslab array */
    lfMakeIndices(field->slabs[slabn]->shared);

    /* Read in the vq codebook, if we haven't done so already */
    slab = field->slabs[slabn];
    vqP = (hdr->slabHdr[slabn].format == LF_INDEXCHANNEL) ? TRUE : FALSE;
    /* Figure out vqIndexBytes, if its vq Compressed */
    if (vqP) vqIndexBytes = (slab->shared->vq->size < 257) ? 1 :
      (slab->shared->vq->size < 65537) ? 2: 4;
    lfOutput("slabn %d, vqP %d\n", slabn, vqP);
    if (vqP && slab->shared->vq->codebook == NULL) {
      lfOutput("vq codebook %d was null for slab %d\n", slab->shared->vq->id, 
	     slabn);
      /* Read the codebook */
      /* (This will allocate the memory needed for the codebook array) */
      lfReadCodebook(slab->shared->vq, slab->shared, hdr, file);
      lfOutput("slab[%d]->shared->vq->codebook: 0x%x\n", slabn, 
	     (int) slab->shared->vq->codebook);
    }

    /* Now read in the actual slab  */
    for (u=0; u < slab->shared->nu; u++) {
      for (v=0; v < slab->shared->nv; v++) {
	/* Get block ptr */
	block =	SlabSlice(slab, u, v);

	/* Read block */
	if (vqP) {
	  /* Read block... (read slices at a time) */
	  if (!(u%(1<<slab->shared->vq->ubits)) && 
	      !(v%(1<<slab->shared->vq->vbits))) {
	    lfReadBlock(block, slabn, slab, hdr, file, 
			 u, u+slab->shared->vq->ubits, 
			 v, v+slab->shared->vq->vbits,
			 LF_INDEXCHANNEL, 
			 (vqIndexBytes == 1) ? LF_INT8 :
			 (vqIndexBytes == 2) ? LF_INT16 : LF_INT32);
	  }
	} else { 
	  /* Not vqP -- just read it out */
	  lfReadBlock(block, slabn, slab, hdr, file, u, u, v, v,
		       (slab->shared->sample_size == 3)? LF_RGBCHANNEL :
		       LF_RGBACHANNEL,
		       (slab->shared->sample_size == 3)? LF_INT8x3 :
		       LF_INT8x4);	
	}
      }
    }
  }

  /* ------------ Ending Stage ---------------- */
  /* Close the file, free hdr structure */
  if (!(lfCloseInFile(&hdr, file))) {
    return(0);
  }

  lfOutput("--------- ReadEntireField done. --------------------------------\n");
  return field;
}

/* Returns 0 on failure */
int lfWriteEntireField(LFField *field, char *name)
{
  LFFileHdr *hdr = NULL;
  FILE *file = NULL;
  int slabn, u, v;
  LFSlab *slab;
  void *block;
  bool_t vqP = FALSE;
  int vqIndexBytes = 0;

  /* ------------ Initialization Stage ---------------- */
  /* Open the file, read the header, allocate hdr structure, etc */
  if ((file = lfOpenOutFile(name, &hdr)) == NULL) {
    return(0);
  }
							   
  /* ------------ Main Body Stage ---------------- */
  /* Write out all the blocks */
  for (slabn = 0; slabn < field->nslabs; slabn++) {
    /* Don't write it if slab is not active */
    if (!(__LF_SLAB_ACTIVE(field, slabn))) {
      lfOutput("%d inactive\n", slabn);
      continue;
    }
    lfOutput("Active: %d\n", slabn);
    /* ok -- write it out */
    slab = field->slabs[slabn];
    vqP = (slab->shared->vq) ? TRUE : FALSE;
    /* Figure out vqIndexBytes, if its vq Compressed */
    if (vqP) vqIndexBytes = (slab->shared->vq->size < 257) ? 1 :
      (slab->shared->vq->size < 65537) ? 2: 4;

    for (u=0; u < slab->shared->nu; u++) {
      for (v=0; v < slab->shared->nv; v++) {
	lfOutput(" ------ %d %d ----\n", u, v);
	/* Get block ptr */
	block = SlabSlice(slab, u, v);

	if (vqP) {
	  /* Write block... (write slices at a time) */
	  if (!(u%(1<<slab->shared->vq->ubits)) && 
	      !(v%(1<<slab->shared->vq->vbits))) {
	    lfWriteBlock(block, slabn, slab, hdr, file, 
			 u, u+slab->shared->vq->ubits, 
			 v, v+slab->shared->vq->vbits,
			 LF_INDEXCHANNEL, 
			 (vqIndexBytes == 1) ? LF_INT8 :
			 (vqIndexBytes == 2) ? LF_INT16 : LF_INT32);
	  }
	} else { 
	  /* Not vqP -- just write it out */
	  lfWriteBlock(block, slabn, slab, hdr, file, u, u, v, v,
		       (slab->shared->sample_size == 3)? LF_RGBCHANNEL :
		       LF_RGBACHANNEL,
		       (slab->shared->sample_size == 3)? LF_INT8x3 :
		       LF_INT8x4);	
	}
      }
    }
    /* Write out the codebook, if vq'd */
    /* It's ok to call lfWriteCodebook multiple times on the same
     * codebook -- it remembers and only writes out the codebook once.
     */
    if (vqP) {
      lfWriteCodebook(slab->shared->vq->codebook, slab->shared->vq->id,
		      slab, hdr, file, LF_CODEBOOKCHANNEL, 
		      (slab->shared->vq->sample_size == 3) ? 
		      LF_INT8x3 : LF_INT8x4);
    }
  }

  /* ------------ Ending Stage ---------------- */
  /* Close the file, free hdr structure */
  if (!(lfCloseOutFile(&hdr, file))) {
    return(0);
  }
}


/* ----------------------------------------------------------------------
 * ---------------          Low-level Functions           --------------- 
 * ----------------------------------------------------------------------
 */



/* Open a Lightslab file, malloc the LFField and LFFileHdr structs,
 * and initialize them from the file header info.
 */
FILE *lfOpenInFile(char *fileName, LFFileHdr **hdr, LFSlab *slabs[], 
		   int nslabs)
{
  FILE *inFile;
  char buffer[200];
  int slabn;
  int vtx, dim;
  long currPos;

  /* Open the input file */
#ifdef sgi
  inFile = fopen(fileName, "rb");
#else /* WIN32 */
  inFile = fopen(fileName, "r+b");
#endif 
  if (!inFile) {
    lfError("Error: cannot open file %s\n", fileName);
    return(NULL);
  }
  /* Malloc the slabHdr array */
  *hdr = (LFFileHdr *) calloc(1, sizeof(LFFileHdr));
  if (*hdr == NULL) {
    lfError("Error: No memory for slabHdr struct.\n");
    return(NULL);
  }

  /* Figure out hdr length */
  (*hdr)->hdrSize = GetHdrLen(inFile);
  lfOutput("Reading header of length %d\n", (*hdr)->hdrSize);
  
  /* Allocate a shared structure */
  for (slabn = 0; slabn < LF_MAXSLABS; slabn++) {
    slabs[slabn]->shared = (LFShared *) calloc(1, sizeof(LFShared));
  }

  /* Make sure we're at beginning of the file */
  rewind(inFile);

  /* Read the hdr, and use it to fill in the data for enabled slabs */
  AssertToken(inFile, "LIF1.0");
  AssertToken(inFile, "datasize");
  NextHdrToken(inFile, buffer, 200);
  (*hdr)->dataSize = atoi(buffer);
  AssertToken(inFile, "bgnlightfield");
  NextHdrToken(inFile, buffer, 200);
  /* Note: ignore lightfield number for now */

  /* First, read all the codebook headers */
  currPos = ftell(inFile);
  while (NextHdrToken(inFile, buffer, 200)) {
    if (!strcmp(buffer, "endlightfield")) break;
    if (!strcmp(buffer, "bgnsegment")) {
      NextHdrToken(inFile, buffer, 200);
      /* if it's a vq segment, parse the header */
      if (!strcmp(buffer, "vq")) {
	ParseVQHdr(inFile, *hdr, slabs, nslabs);
	/* If it's not a vq segment, skip it for now... */
      } else {
	SkipSegment(inFile);
      }
    }
  }
  /* Restore file position to start of the lightfield */
  fseek(inFile, currPos, SEEK_SET);
	
  /* Now we can parse all the slabs */
  while (1) {
    /* Read next token */
    if (!NextHdrToken(inFile, buffer, 200)) {
      lfError("Error: Unexpected end of file header.\n");
      return(0);
    }
    /* Check if endlightfield */
    if (!strcmp(buffer, "endlightfield")) {
      break;
    }
    /* Check if bgnsegment */
    if (!strcmp(buffer, "bgnsegment")) {
      NextHdrToken(inFile, buffer, 200);
      if (!strcmp(buffer, "slab")) {
	ParseSlabHdr(inFile, *hdr, slabs, nslabs);
      } else {
	SkipSegment(inFile);
      }
    } else {
      /* default -- unrecognized token */
      lfError("Error: unexpected token: \"%s\"\n", buffer);
      return 0;
    }
  }
  /* Return file */
  return(inFile);
}

/* This routine reads the rest of the segment. */
int SkipSegment(FILE *inFile)
{
  char buffer[200];

  while (NextHdrToken(inFile, buffer, 200)) {
    if (!strcmp(buffer, "endsegment")) return 1;
  }
  /* if we get here, ran out of header without endsegment */
  return 0;
}

/* Parse a VQ codebook header.  It assumes that "bgnsegment vq" has already
 * been parsed.
 */
int ParseVQHdr(FILE *inFile, LFFileHdr *hdr, LFSlab *slabs[], int nslabs)
{
  char buffer[200];
  int vqn;
  int vtx, dim;
  int tmp;


  /* get codebook number */
  NextHdrToken(inFile, buffer, 200);
  vqn = atoi(buffer);
  lfOutput("ParseVQHdr %d\n", vqn);
  /* Note: does not check to see if slab is enabled... */
  lfOutput("reading slab hdr info for VQ codebook %d\n", vqn);

  /* Allocate the codebook structure */
  if (!((hdr->vqHdr[vqn].codebook = calloc(1, sizeof(LFVQCodebook))))) {
    lfError("Error: Out of memory in ParseVQHdr\n");
    return 0;
  }
  /* Parse the rest of the segment */
  while (NextHdrToken(inFile, buffer, 200)) {
    if (!strcmp(buffer, "endsegment")) return 1;

    if (!strcmp(buffer, "format")) {
      NextHdrToken(inFile, buffer, 200);
      if (!strcmp(buffer, "index")) {
	lfError("Error: Codebook %d has format index?!?\n", vqn);
	hdr->vqHdr[vqn].format = LF_INDEXCHANNEL;
	return 0;
      } else if (!strcmp(buffer, "rgb")) {
	hdr->vqHdr[vqn].format = LF_RGBCHANNEL;
	hdr->vqHdr[vqn].codebook->sample_size = 3;
      } else if (!strcmp(buffer, "rgba")) {
	hdr->vqHdr[vqn].format = LF_RGBACHANNEL;
	hdr->vqHdr[vqn].codebook->sample_size = 4;
      }
    } 

    if (!strcmp(buffer, "bgnchannel")) {
      NextHdrToken(inFile, buffer, 200);
      /* VQ codebook shouldn't be vq encoded! */
      AssertToken(inFile, "type");
      NextHdrToken(inFile, buffer, 200);
      if (!strcmp(buffer, "int8x3")) {
	hdr->vqHdr[vqn].type = LF_INT8x3;
      } else if (!strcmp(buffer, "int8x4")) {
	hdr->vqHdr[vqn].type = LF_INT8x4;
      }
      /* offset */
      AssertToken(inFile, "offset");
      NextHdrToken(inFile, buffer, 200);
      hdr->vqHdr[vqn].offset = atoi(buffer);
      /* size */
      AssertToken(inFile, "size");
      NextHdrToken(inFile, buffer, 200);
      hdr->vqHdr[vqn].size = atoi(buffer);
      lfOutput("vq %d size %d\n", vqn, hdr->vqHdr[vqn].size);
      /* end channel */
      AssertToken(inFile, "endchannel");
    }

    /* number of tiles in codebook */
    if (!strcmp(buffer, "tiles")) {
      NextHdrToken(inFile, buffer, 200);
      hdr->vqHdr[vqn].codebook->size = atoi(buffer);
    }

    /* Size of tiles in codebook */
    if (!strcmp(buffer, "tilesize")) {
      NextHdrToken(inFile, buffer, 200);
      hdr->vqHdr[vqn].codebook->ubits = Log2(atoi(buffer));
      NextHdrToken(inFile, buffer, 200);
      hdr->vqHdr[vqn].codebook->vbits = Log2(atoi(buffer));
      NextHdrToken(inFile, buffer, 200);
      hdr->vqHdr[vqn].codebook->sbits = Log2(atoi(buffer));
      NextHdrToken(inFile, buffer, 200);
      hdr->vqHdr[vqn].codebook->tbits = Log2(atoi(buffer));
    }
  }
  /* if we get here, then we ran out of header without endsegment */
  return(0);
}

/* Parse a slab header.  It assumes that "bgnsegment slab" has already
 * been parsed.
 */
int ParseSlabHdr(FILE *inFile, LFFileHdr *hdr, LFSlab *slabs[],
		 int nslabs)
{
  char buffer[200];
  int slabn, vqn;
  int vtx, dim;

  /* get slab number */
  NextHdrToken(inFile, buffer, 200);
  slabn = atoi(buffer);
  /* Note: does not check to see if slab is enabled... */
  lfOutput("reading slab hdr info for slab %d\n", slabn);
  slabs[slabn]->id = slabn;

  /* Parse the rest of the segment */
  while (NextHdrToken(inFile, buffer, 200)) {
    if (!strcmp(buffer, "endsegment")) return 1;

    if (!strcmp(buffer, "compression")) {
      NextHdrToken(inFile, buffer, 200);
      if (!strcmp(buffer, "vq")) {
	NextHdrToken(inFile, buffer, 200);
	vqn = atoi(buffer);
	slabs[slabn]->shared->vq = hdr->vqHdr[vqn].codebook;
      } else {
	/* no compression */
	slabs[slabn]->shared->vq = NULL;
      }
    }

    if (!strcmp(buffer, "format")) {
      NextHdrToken(inFile, buffer, 200);
      if (!strcmp(buffer, "index")) {
	hdr->slabHdr[slabn].format = LF_INDEXCHANNEL;
	slabs[slabn]->shared->sample_size = 
	  (slabs[slabn]->shared->vq->size < 257) ? 1 :
	  (slabs[slabn]->shared->vq->size < 65537) ? 2 : 4;
      } else if (!strcmp(buffer, "rgb")) {
	hdr->slabHdr[slabn].format = LF_RGBCHANNEL;
	slabs[slabn]->shared->sample_size = 3;
      } else if (!strcmp(buffer, "rgba")) {
	hdr->slabHdr[slabn].format = LF_RGBACHANNEL;
	slabs[slabn]->shared->sample_size = 4;
      }
    } 

    if (!strcmp(buffer, "bgnchannel")) {
      NextHdrToken(inFile, buffer, 200);
      if (!strcmp(buffer, "index")) {
	AssertToken(inFile, "type");
	NextHdrToken(inFile, buffer, 200);
	if (!strcmp(buffer, "int8")) {
	  hdr->slabHdr[slabn].type = LF_INT8;
	} else if (!strcmp(buffer, "int16")) {
	  hdr->slabHdr[slabn].type = LF_INT16;
	} else if (!strcmp(buffer, "int32")) {
	  hdr->slabHdr[slabn].type = LF_INT32;
	}
      } else {
	/* Not vq encoded */
	AssertToken(inFile, "type");
	NextHdrToken(inFile, buffer, 200);
	if (!strcmp(buffer, "int8x3")) {
	  hdr->slabHdr[slabn].type = LF_INT8x3;
	} else if (!strcmp(buffer, "int8x4")) {
	  hdr->slabHdr[slabn].type = LF_INT8x4;
	}
      }
      /* offset */
      AssertToken(inFile, "offset");
      NextHdrToken(inFile, buffer, 200);
      hdr->slabHdr[slabn].offset = atoi(buffer);
      /* size */
      AssertToken(inFile, "size");
      NextHdrToken(inFile, buffer, 200);
      hdr->slabHdr[slabn].size = atoi(buffer);
      lfOutput("slab %d size %d\n", slabn, hdr->slabHdr[slabn].size);
      /* end channel */
      AssertToken(inFile, "endchannel");
    }
	  
    /* samples size */
    if (!strcmp(buffer, "samples_uv")) {
      NextHdrToken(inFile, buffer, 200);
      slabs[slabn]->shared->nu = atoi(buffer);
      NextHdrToken(inFile, buffer, 200);
      slabs[slabn]->shared->nv = atoi(buffer);
    }
    if (!strcmp(buffer, "samples_st")) {
      NextHdrToken(inFile, buffer, 200);
      slabs[slabn]->shared->ns = atoi(buffer);
      NextHdrToken(inFile, buffer, 200);
      slabs[slabn]->shared->nt = atoi(buffer);
    }
	
    /* geometry */
    if (!strcmp(buffer, "geometry_uv")) {
      for (vtx=0; vtx < 4; vtx++) {
	NextHdrToken(inFile, buffer, 200);
	slabs[slabn]->uv[vtx].ox = atof(buffer);
	NextHdrToken(inFile, buffer, 200);
	slabs[slabn]->uv[vtx].oy = atof(buffer);
	NextHdrToken(inFile, buffer, 200);
	slabs[slabn]->uv[vtx].oz = atof(buffer);
	NextHdrToken(inFile, buffer, 200);
	slabs[slabn]->uv[vtx].ow = atof(buffer);
	NextHdrToken(inFile, buffer, 200);
	slabs[slabn]->uv[vtx].s = atof(buffer);
	NextHdrToken(inFile, buffer, 200);
	slabs[slabn]->uv[vtx].t = atof(buffer);
      }
    }
    if (!strcmp(buffer, "geometry_st")) {
      for (vtx=0; vtx < 4; vtx++) {
	NextHdrToken(inFile, buffer, 200);
	slabs[slabn]->st[vtx].ox = atof(buffer);
	NextHdrToken(inFile, buffer, 200);
	slabs[slabn]->st[vtx].oy = atof(buffer);
	NextHdrToken(inFile, buffer, 200);
	slabs[slabn]->st[vtx].oz = atof(buffer);
	NextHdrToken(inFile, buffer, 200);
	slabs[slabn]->st[vtx].ow = atof(buffer);
	NextHdrToken(inFile, buffer, 200);
	slabs[slabn]->st[vtx].s = atof(buffer);
	NextHdrToken(inFile, buffer, 200);
	slabs[slabn]->st[vtx].t = atof(buffer);
      }
    }
  }
  /* if we get here, then we ran out of header without endsegment */
  return(0);
}


/* Open a Lightslab file, malloc the LFFileHdr struct,
 * and fill it in with data from the LFField structure.
 */
FILE *lfOpenOutFile(char *fileName, LFFileHdr **hdr)
{
  FILE *outFile;

  /* Open the output file */
#ifdef sgi
  outFile = fopen(fileName, "w");
#else /* WIN32 */
  outFile = fopen(fileName, "w+r");
#endif  
  if (!outFile) {
    lfError("Error: cannot open file %s\n", fileName);
    return(NULL);
  }
  /* Malloc the slabHdr array */
  *hdr = (LFFileHdr *) calloc(1, sizeof(LFFileHdr));
  if (*hdr == NULL) {
    lfError("Error: No memory for slabHdr struct.\n");
    return(NULL);
  }
  /* reserve enough hdr space to hold the header */
  (*hdr)->hdrSize = LF_DEFAULTHDRLEN;
  /* Initialize the header (clear it, and write terminating character) */
  ClearHdr(*hdr, outFile);

  return outFile;
}


/* Write the header (if necessary) and close the file.
 * NOTE -- This frees the LFFileHdr Structure! 
 */
int lfCloseInFile(LFFileHdr **hdr, FILE *inFile)
{
  /* Free/ Clean-up the hdr structure */
  free(*hdr);
  *hdr = NULL;

  /* Close the output file */
  fclose(inFile);

  return 1;
}


int lfCloseOutFile(LFFileHdr **hdr, FILE *outFile)
{
  LFSlab *slab;
  LFVQCodebook *vq;
  LFSlabHdr *slabHdr;
  LFVQHdr *vqHdr;
  int slabno, vqno;
  int i, chan;
  char *uvgeomStr[4] = {"geometry_uv", "           ", 
			"           ", "           "};
  char *stgeomStr[4] = {"geometry_st", "           ", 
			"           ", "           "};

  /* ---- Write the header ---- */
  rewind(outFile);
  fprintf(outFile, "LIF1.0\n");

  /* Print the datasize */
  fprintf(outFile, "datasize %d\n\n", (*hdr)->dataSize);

  /* print the lightfield */
  fprintf(outFile, "bgnlightfield 1\n");

  /* Go through the slabs one by one */
  for (slabno=0; slabno < LF_MAXSLABS; slabno++) {
    slabHdr = &((*hdr)->slabHdr[slabno]);
    slab = slabHdr->slab;
    if (slab==NULL) continue;
    fprintf(outFile, "  bgnsegment slab %d\n", slabno); 
    if ((vq=slab->shared->vq) != NULL) {
      /* Output VQ index channel */
      fprintf(outFile, "    compression vq %d\n", slab->shared->vq->id);
      fprintf(outFile, "    format index\n");
      fprintf(outFile, "    bgnchannel index    # vq codebook indices\n");
      fprintf(outFile, "      type %s\n", 
	      (slabHdr->type == LF_INT8) ? "int8" :
	      (slabHdr->type == LF_INT16) ? "int16" :
	      /*(slabHdr->type==LF_INT32)?*/ "int32");
      fprintf(outFile, "      offset %d\n", slabHdr->offset);
      fprintf(outFile, "      size %d\n", slabHdr->size);
      fprintf(outFile, "    endchannel\n");
    } else { 
      fprintf(outFile, "    compression none\n");
      switch (slabHdr->format) {
      case LF_RGBCHANNEL:	
	/* red, green, and blue */
	fprintf(outFile, "    format rgb\n");
	fprintf(outFile, "    bgnchannel rgb\n");
	fprintf(outFile, "      type int8x3\n");
	fprintf(outFile, "      offset %d\n", slabHdr->offset);
	fprintf(outFile, "      size %d\n", slabHdr->size);
	fprintf(outFile, "    endchannel\n");
	break;
      case LF_RGBACHANNEL:	
	/* red, green, blue, and alpha */
	fprintf(outFile, "    format rgba\n");
	fprintf(outFile, "    bgnchannel rgba\n");
	fprintf(outFile, "      type int8x4\n");
	fprintf(outFile, "      offset %d\n", slabHdr->offset);
	fprintf(outFile, "      size %d\n", slabHdr->size);
	fprintf(outFile, "    endchannel\n");
	break;
      }
    }	
    /* Print out dimension in u, v, s, t */
    fprintf(outFile, "    samples_uv %d %d\n", slab->shared->nu,
	    slab->shared->nv);
    fprintf(outFile, "    samples_st %d %d\n", slab->shared->ns, 
	    slab->shared->nt);
    /* Print out uv slab coordinates */
    for (i=0; i < 4; i++) {
      fprintf (outFile, "    %s % f % f % f % f % f % f\n", uvgeomStr[i],
	       slab->uv[i].ox, slab->uv[i].oy, slab->uv[i].oz, slab->uv[i].ow,
	       slab->uv[i].s, slab->uv[i].t);
    }
    /* Print out st slab coordinates */
    for (i=0; i < 4; i++) {
      fprintf (outFile, "    %s % f % f % f % f % f % f\n", stgeomStr[i],
	       slab->st[i].ox, slab->st[i].oy, slab->st[i].oz, slab->st[i].ow,
	       slab->st[i].s, slab->st[i].t);
    }

    /* End of segment */
    fprintf(outFile, "  endsegment # %d\n", slabno); /* +1 to start at 1 */
  }

  /* Go through the vqs one by one */
  for (vqno=0; vqno < LF_MAXSLABS; vqno++) {
    vqHdr = &((*hdr)->vqHdr[vqno]);
    vq = vqHdr->codebook;
    if (vq==NULL) continue;
    fprintf(outFile, "  bgnsegment vq %d\n", vqno); 
    switch (vqHdr->format) {
    case LF_RGBCHANNEL:	
      /* red, green, and blue */
      fprintf(outFile, "    format rgb\n");
      fprintf(outFile, "    bgnchannel rgb\n");
      fprintf(outFile, "      type int8x3\n");
      fprintf(outFile, "      offset %d\n", vqHdr->offset);
      fprintf(outFile, "      size %d\n", vqHdr->size);
      fprintf(outFile, "    endchannel\n");
      break;
    case LF_RGBACHANNEL:	
      /* red, green, blue, and alpha */
      fprintf(outFile, "    format rgba\n");
      fprintf(outFile, "    bgnchannel rgba\n");
      fprintf(outFile, "      type int8x4\n");
      fprintf(outFile, "      offset %d\n", vqHdr->offset);
      fprintf(outFile, "      size %d\n", vqHdr->size);
      fprintf(outFile, "    endchannel\n");
      break;
    }
    /* print tile info */
    fprintf(outFile, "    tiles %d\n", vq->size);
    fprintf(outFile, "    tilesize %d %d %d %d\n",
	    1 << vq->ubits, 1 << vq->vbits, 1 << vq->sbits, 1 << vq->tbits);
	
    /* End of segment */
    fprintf(outFile, "  endsegment # %d\n", vqno); /* +1 to start at 1 */
  }

  /* End of lightfield entry */
  fprintf(outFile, "endlightfield # 1\n");
  fprintf(outFile, "endheader\n");

  /* Free/ Clean-up the hdr structure */
  free(*hdr);
  *hdr = NULL;

  /* Close the output file */
  fclose(outFile);
  return 1;
}

/* Write a portion (possibly all) of a slab of a lightfield to file */
int lfWriteBlock(void *block, int slabn, LFSlab *slab, LFFileHdr *hdr, 
		 FILE *file, int uFirst, int uLast, int vFirst, int vLast,
		 LF_ChannelFormat format, LF_ChannelType type)
{
  LFShared *shared = slab->shared;
  LFSlabHdr *slabHdr = &(hdr->slabHdr[slabn]);
  int indexBytes = 0;  /* Number of bytes necessary to index codebook */
  int tilePixels = 0;  /* Number of pixels contained in each tile */
  LFVQCodebook *vq;		/* vq codebook */
  int channelSize;
  int rawOffset, rawLen, inElemBytes, outElemBytes;
  int blockn, blockLen; /* used for VQ index arrays */

  if ((vq = shared->vq) != NULL) {
    /* indexBytes = # of bytes needed to index into the codebook array */
    indexBytes = (vq->size > 256) ? (vq->size > 65536) ? 4 : 2 : 1;
    /* tilePixels is the number of pixels that are in each tile */
    tilePixels = (1 << (vq->ubits + vq->vbits + vq->sbits + vq->tbits));
  }

  if (format == LF_CODEBOOKCHANNEL) {
    /* Codebook*/
    lfError("Error: lfWriteBlock called on a codebook!\n");
    return 0;
  } else {
    /* Handle slab segment (not codebook) */
    /* Set hdr info, if this is the first write */
    if (slabHdr->size == 0) {
      slabHdr->format = format;
      slabHdr->type   = type;
      slabHdr->slab   = slab;
      slabHdr->offset = hdr->dataSize;
      if (vq != NULL) {
	/* Channelsize */
	channelSize =  slab->shared->nu * slab->shared->nv * 
	  slab->shared->ns * slab->shared->nt * indexBytes / tilePixels;
	slabHdr->size = channelSize;
      } else {
	/* Set size big enough to hold raw file */
	slabHdr->size = 
	  ((type == LF_INT8x4) ? 4 : /*(type == LF_INT8x3)?*/  3) *
	  (shared->nu * shared->nv * shared->ns * shared->nt);
      }
      lfOutput("upping datasize by %d\n", slabHdr->size);
      hdr->dataSize += slabHdr->size;
    }
    /* Now write the binary section to file */
    switch (slabHdr->format) {
    case LF_RGBCHANNEL:
    case LF_RGBACHANNEL:
      rawOffset = hdr->hdrSize + slabHdr->offset +
	((slabHdr->type == LF_INT8x3) ? 3 : 4) * 
	(uFirst * slab->shared->nv + vFirst) *
	slab->shared->ns * slab->shared->nt;
      rawLen = ((slabHdr->type == LF_INT8x3) ? 3 : 4) *
	(1+uLast-uFirst) * (1+vLast-vFirst) * 
	slab->shared->ns * slab->shared->nt;
      inElemBytes = outElemBytes = 1;
      break;
    case LF_INDEXCHANNEL:
      /* Note: this requires that the slabslice in u and v is a multiple
       * of the tile size.  (So, for example, if you have 2x2x2x2 blocks,
       * then uFirst and vFirst must be even numbers, and uLast and vLast
       * must be odd numbers.)
       */

      /* Check to make sure that uFirst and vFirst are divisible by 
       * the tile size in u and v.
       */
      if ((uFirst % (1<<slab->shared->vq->ubits)) ||
	  (vFirst % (1<<slab->shared->vq->vbits))) {
	lfError(
		"Error:  for writing out VQ indices, the slice must start\n"
		"        on an even boundary (Divisible by the tile size)\n");
	return(0);
      }
      /* Check to make sure that slice size (in u and v) is divisible
       * by the tile size.
       */
      if (((uLast+1) % (1<<slab->shared->vq->ubits)) ||
	  ((vLast+1) % (1<<slab->shared->vq->vbits))) {
	lfError(
		"Error:  for writing out VQ indices, the size of the slice\n"
		"        must be evenly divisible by the tile size.\n");
	return(0);
      }

      /* Calculate the block offset */
      blockn = ((uFirst >> slab->shared->vq->ubits) *
		(slab->shared->nv >> slab->shared->vq->vbits)) +
	(vFirst >> slab->shared->vq->vbits);
      /* The length of a single tile slice in uv */
      blockLen = ((1<<slab->shared->vq->ubits) *
		  (1<<slab->shared->vq->vbits)) *
	slab->shared->ns * slab->shared->nt / tilePixels;
      /* offset */
      rawOffset = hdr->hdrSize + slabHdr->offset + 
	indexBytes * blockn * blockLen;
      
      /* Calculate the length of the block */
      rawLen = (1+uLast-uFirst) * (1+vLast-vFirst) * 
	slab->shared->ns * slab->shared->nt / tilePixels;

      inElemBytes = outElemBytes = indexBytes;
      break;
    default:
      lfError("Error: unrecognized data format %d\n", 
	      slabHdr->format);
      break;
    }

    WriteRawBlock(file, rawOffset, block,
		  rawLen, inElemBytes, outElemBytes);
  }	
  return 1;
}

/* Read an entire codebook from the file */
/* (This will allocate the memory needed for the codebook array) */
int lfReadCodebook(LFVQCodebook *vq, LFShared *shared, LFFileHdr *hdr, 
		   FILE *file)
{
  LFVQHdr *vqHdr = &(hdr->vqHdr[vq->id]);
  int indexBytes = 0;  /* Number of bytes necessary to index codebook */
  int tilePixels = 0;  /* Number of pixels contained in each tile */
  int channelSize;
  int rawOffset, rawLen, inElemBytes, outElemBytes;
  void *block;

  lfOutput("lfReadCodebook %d\n", vq->id);

  /* indexBytes = # of bytes needed to index into the codebook array */
  indexBytes = (vq->size > 256) ? (vq->size > 65536) ? 4 : 2 : 1;
  /* tilePixels is the number of pixels that are in each tile */
  tilePixels = (1 << (vq->ubits + vq->vbits + vq->sbits + vq->tbits));
  /* Allocate space for the codebook's array */
  if (vq->codebook != NULL) {
    lfError("Warning: lfReadCodebook:"
	    "codebook already contains data!\n");
  }
  if ((vq->codebook = (void *) 
       malloc(vq->size * tilePixels * vq->sample_size)) == NULL) {
    lfError("Error:  Not enough memory to allocate codebook %d!\n",
	    vq->id);
    return 0;
  }

  block = vq->codebook;
  rawOffset = hdr->hdrSize + vqHdr->offset;
  /* vqHdr->size is already in bytes */
  rawLen = vqHdr->size;
  inElemBytes = 1;
  outElemBytes = 1;
  ReadRawBlock(file, rawOffset, block, rawLen, inElemBytes, outElemBytes);
  lfOutput("Done reading %d bytes of codebook.\n", rawLen);
  return 1;
}

/* Write an entire codebook to file */
int lfWriteCodebook(void *block, int vqn, LFSlab *slab, LFFileHdr *hdr, 
		 FILE *file, LF_ChannelFormat format, LF_ChannelType type)
{
  LFShared *shared = slab->shared;
  LFVQHdr *vqHdr = &(hdr->vqHdr[vqn]);
  int indexBytes = 0;  /* Number of bytes necessary to index codebook */
  int tilePixels = 0;  /* Number of pixels contained in each tile */
  LFVQCodebook *vq;		/* vq codebook */
  int channelSize;
  int rawOffset, rawLen, inElemBytes, outElemBytes;

  lfOutput("lfWriteCodebook, format %d\n", format);

  if ((vq = shared->vq) != NULL) {
    /* indexBytes = # of bytes needed to index into the codebook array */
    indexBytes = (vq->size > 256) ? (vq->size > 65536) ? 4 : 2 : 1;
    /* tilePixels is the number of pixels that are in each tile */
    tilePixels = (1 << (vq->ubits + vq->vbits + vq->sbits + vq->tbits));
  }

  if (format == LF_CODEBOOKCHANNEL) {
    /* Handle codebook */
    vq = slab->shared->vq;

    /* Don't re-set the hdr info (which would allocate *another* block
     * for the codebook), if the codebook is already used by another
     * slab.
     */
    if (vqHdr->size) {
      /* debug message */
      lfOutput("skipping codebook %d -- it's shared by another slab.\n",
	     slab->shared->vq->id);
      return 1;
    }

    /* Set hdr info, since we assume that we write the entire codebook
       * out at one time.
       */
    vqHdr->format = (type == LF_INT8x3) ? LF_RGBCHANNEL : LF_RGBACHANNEL;
    vqHdr->type = type;
    vqHdr->codebook = vq;
    vqHdr->offset = hdr->dataSize;
    vqHdr->size = vq->size * 
      (vq->sample_size << (vq->ubits + vq->vbits + vq->sbits + vq->tbits));
    lfOutput("upping datasize by VQ-codebook: %d\n", vqHdr->size);
    hdr->dataSize += vqHdr->size;
    rawOffset = hdr->hdrSize + vqHdr->offset;
    block = vq->codebook;
    rawLen = vqHdr->size;
    inElemBytes = 1;
    outElemBytes = 1;
    WriteRawBlock(file, rawOffset, block,
		  rawLen, inElemBytes, outElemBytes);
  } else {
    lfError("Error: lfWriteCodebook called on a non-codebook.\n");
    return 0;
  }
  return 1;
}


/* **********************************************************************
 *
 *	       Unwritten Functions Below
 *
 * **********************************************************************
 */


/* Read a portion (possibly all) of a slab of a lightfield from file */
int lfReadBlock(void *block, int slabn, LFSlab *slab, LFFileHdr *hdr, 
		 FILE *file, int uFirst, int uLast, int vFirst, int vLast,
		 LF_ChannelFormat format, LF_ChannelType type)
{
  LFShared *shared = slab->shared;
  LFSlabHdr *slabHdr = &(hdr->slabHdr[slabn]);
  int indexBytes = 0;  /* Number of bytes necessary to index codebook */
  int tilePixels = 0;  /* Number of pixels contained in each tile */
  LFVQCodebook *vq;		/* vq codebook */
  int channelSize;
  int rawOffset, rawLen, inElemBytes, outElemBytes;
  int blockn, blockLen; /* used for VQ index arrays */

  if ((vq = shared->vq) != NULL) {
    /* indexBytes = # of bytes needed to index into the codebook array */
    indexBytes = (vq->size > 256) ? (vq->size > 65536) ? 4 : 2 : 1;
    /* tilePixels is the number of pixels that are in each tile */
    tilePixels = (1 << (vq->ubits + vq->vbits + vq->sbits + vq->tbits));
  }

  if (format == LF_CODEBOOKCHANNEL) {
    /* Codebook*/
    lfError("Error: lfReadBlock called on a codebook!\n");
    return 0;
  } else {
    /* Now read the binary section from file */
    switch (slabHdr->format) {
    case LF_RGBCHANNEL:
    case LF_RGBACHANNEL:
      rawOffset = hdr->hdrSize + slabHdr->offset +
	((slabHdr->type == LF_INT8x3) ? 3 : 4) * 
	(uFirst * slab->shared->nv + vFirst) *
	slab->shared->ns * slab->shared->nt;
      rawLen = ((slabHdr->type == LF_INT8x3) ? 3 : 4) *
	(1+uLast-uFirst) * (1+vLast-vFirst) * 
	slab->shared->ns * slab->shared->nt;
      inElemBytes = outElemBytes = 1;
      break;
    case LF_INDEXCHANNEL:
      /* Note: this requires that the slabslice in u and v is a multiple
       * of the tile size.  (So, for example, if you have 2x2x2x2 blocks,
       * then uFirst and vFirst must be even numbers, and uLast and vLast
       * must be odd numbers.)
       */

      /* Check to make sure that uFirst and vFirst are divisible by 
       * the tile size in u and v.
       */
      if ((uFirst % (1<<slab->shared->vq->ubits)) ||
	  (vFirst % (1<<slab->shared->vq->vbits))) {
	lfError(
		"Error:  for reading in VQ indices, the slice must start\n"
		"        on an even boundary (Divisible by the tile size)\n");
	return(0);
      }
      /* Check to make sure that slice size (in u and v) is divisible
       * by the tile size.
       */
      if (((uLast+1) % (1<<slab->shared->vq->ubits)) ||
	  ((vLast+1) % (1<<slab->shared->vq->vbits))) {
	lfError(
		"Error:  for reading in VQ indices, the size of the slice\n"
		"        must be evenly divisible by the tile size.\n");
	return(0);
      }

      /* Calculate the block offset */
      blockn = ((uFirst >> slab->shared->vq->ubits) *
		(slab->shared->nv >> slab->shared->vq->vbits)) +
	(vFirst >> slab->shared->vq->vbits);
      /* The length of a single tile slice in uv */
      blockLen = ((1<<slab->shared->vq->ubits) *
		  (1<<slab->shared->vq->vbits)) *
	slab->shared->ns * slab->shared->nt / tilePixels;
      /* offset */
      rawOffset = hdr->hdrSize + slabHdr->offset + 
	indexBytes * blockn * blockLen;
      
      rawLen = (1+uLast-uFirst) * (1+vLast-vFirst) * 
	slab->shared->ns * slab->shared->nt / tilePixels;
      inElemBytes = outElemBytes = indexBytes;
      break;
    }

    ReadRawBlock(file, rawOffset, block, rawLen, inElemBytes, outElemBytes);
  }	
  return 1;
}


/* **********************************************************************
 *
 *	       local helper functions
 *
 * **********************************************************************
 */


void *SlabSlice(LFSlab *slab, int u, int v)
{
  LFShared *shared = slab->shared;
  int *uindex = shared->uindex;
  int *vindex = shared->vindex;
  int *sindex = shared->sindex;
  int *tindex = shared->tindex;
  int *vqtile_uindex = shared->vqtile_uindex;
  int *vqtile_vindex = shared->vqtile_vindex;
  int *vqtile_sindex = shared->vqtile_sindex;
  int *vqtile_tindex = shared->vqtile_tindex;


  /* We always want to call LIGHTPTR, not VQLIGHTPTR, because
   * we want a pointer into the actual lightfield array, not into
   * the codebook.
   */
  if (shared->vq != NULL) {
    /* Figure out how the codebook indices are packed into the lightfield:
     *    codebook size less than 257      -> 1-byte indices
     *    codebook size less than 65537    -> 2-byte indices
     *    codebook size greater than 65536 -> 4-byte indices
     */
    if (slab->shared->vq->size <= 256) {
      unsigned char *lightfield = slab->lightfield;
      return LIGHTPTR(u, v, 0, 0);
    } else if (slab->shared->vq->size <= 65536) {
      unsigned short *lightfield = slab->lightfield;
      return LIGHTPTR(u, v, 0, 0);
    } else {
      unsigned int *lightfield = slab->lightfield;
      return LIGHTPTR(u, v, 0, 0);
    }
  } else {
    /* Figure out how the data is packed, and get a pointer to it */
    switch (slab->shared->sample_size) {
    case 1: {
      unsigned char *lightfield = slab->lightfield;
      return LIGHTPTR(u, v, 0, 0);
      break;
    }
    case 2: {
      unsigned short *lightfield = slab->lightfield;
      return LIGHTPTR(u, v, 0, 0);
      break;
    }
    case 4: {
      LFPixelRGBA8 *lightfield = slab->lightfield;
      return LIGHTPTR(u, v, 0, 0);
      break;
    }
    case 8: {
      unsigned long *lightfield = slab->lightfield;
      return LIGHTPTR(u, v, 0, 0);
      break;
    }
    default: {
      lfError("Error:  slab packed in memory, %d bytes per pixel.\n"
	      "How the heck can I typecast that array???\n",
	      slab->shared->sample_size);
      return 0;
      break;
    }
    }
  }
}

void ClearHdr(LFFileHdr *hdr, FILE *file)
{
  int i;
  
  /* Go to start of file */
  rewind(file);

  /* Clear hdr space */
  for (i=0; i < (hdr->hdrSize-1); i++) {
    fputc(LF_HDRFILLER, file);
  }
  /* Terminating character */
  fputc(LF_HDRTERMINATOR, file);

  /* Now go back to start for writing hdr over it. */
  rewind(file);


}

int GetHdrLen(FILE *file)
{
  int i=0;
  int c;

  rewind(file);
  /* Count up characters in header until we hit terminator character */
  while ((c=fgetc(file)) != EOF) {
    i++;
    if (c == LF_HDRTERMINATOR) {
      break;
    }
  }
  rewind(file);
  return i;
}


/* Write out raw block of data.  It always writes little-endian,
 * for cross-platform compatibility.
 */
int WriteRawBlock(FILE *outFile, int offset, char *buffer, int elemCount,
		  int elemInSize, int elemOutSize)
{
  int i, j;
  unsigned long data;
  unsigned char c;
  int currPos;

  /*  lfOutput("WriteRawBlock off %d, size %d (%d,%d) into buf %d\n", 
	 offset, elemCount, elemInSize, elemOutSize, (int) buffer);
	 */

  /* Sanity check to avoid bus errors */
  if (elemInSize != 1 && elemInSize != 2 && elemInSize != 4 &&
      elemInSize != 8) {
    lfError(
	    "Error: WriteRawBlock:  What kind of number uses %d bytes??\n", 
	    elemInSize);
    return 0;
  }

  /* Go to starting offset in the file */
  fseek(outFile, offset, SEEK_SET);

  /* Now write */
  for (i=0; i<elemCount; i++) {
    /* Set data = element value */
    switch (elemInSize) {
    case 1:
      data = * (buffer);
      break;
    case 2:
      data = * ((unsigned short *) ((void *) buffer));
      break;
    case 4:
      data = * ((unsigned int *)   ((void *) buffer));
      break;
    case 8:
      data = * ((unsigned long *)  ((void *) buffer));
      break;
    }
    /*    lfOutput("w %d\n", data); */
    buffer += elemInSize;

    /* Write the data, little-endian */
    for (j=0; j < elemOutSize; j++) {
      fputc(data & 0x0ff, outFile);
      data >>= 8;
    }
  }
}
  
/* Read in raw block of data.  It reads writes little-endian,
 * for cross-platform compatibility.
 */
int ReadRawBlock(FILE *inFile, int offset, char *buffer, int elemCount,
		  int elemInSize, int elemOutSize)
{
  int i, j;
  unsigned long data;
  unsigned int tmpdata;
  unsigned char c;
  int currPos;
  int shiftBits = 0;

  /* Sanity check to avoid bus errors */
  if (elemOutSize != 1 && elemOutSize != 2 && elemOutSize != 4 &&
      elemOutSize != 8) {
    lfError(
	    "Error: ReadRawBlock:  What kind of number uses %d bytes??\n", 
	    elemOutSize);
    return 0;
  }

  /* Go to starting offset in the file */
  fseek(inFile, offset, SEEK_SET);

  /* Now read */
  for (i=0; i<elemCount; i++) {
    /* Read the data, little-endian */
    data = tmpdata = 0;
    for (j=0; j < elemInSize; j++) {
      tmpdata = fgetc(inFile);
      /* put the character in the jth byte */
      data += tmpdata << (j<<3);
    }

    /* Set data = element value */
    switch (elemOutSize) {
    case 1:
      * buffer = data;
      break;
    case 2:
      * ((short *) ((void *) buffer)) = data;
      break;
    case 4:
      * ((int *)   ((void *) buffer)) = data;
      break;
    case 8:
      * ((long *)  ((void *) buffer)) = data;
      break;
    }
    buffer += elemOutSize;
  }
}
  
/* This routine returns 1 on success, 0 on failure (end-of-hdr).
 * It assumes that buffer has been allocated and is large enough
 * to hold a token of maxTokenSize-1 + null-terminator.
 */
int NextHdrToken(FILE *file, char *buffer, int maxTokenSize)
{
  int index=0;		/* Index into the buffer */
  int c;
  /* State machine:
   * 0 = haven't seen start of token yet
   * 1 = reading token
   * 2 = token finished
   * 3 = inside a comment
   */
  int state = 0;

  while (state != 2) {
    c = fgetc(file);

    /* Handle EOF */
    if (c == EOF || c == LF_HDRTERMINATOR) {
      buffer[index] = '\0';
      /* Return 1 iff we've seen part of a token, 0 otherwise */
      if (state == 1) {
	return 1;
      } else {
	return 0;
      }
    }
    
    /* Handle comments */
    if (c == '#') {
      if (state == 1) {
	buffer[index] = '\0';
	/* Finish reading the comment, so we start on next line next time */
	while ((c != '\n') && (c != EOF)) {
	  c = fgetc(file);
	}
	/* now return token that we read. */
	return 1;
      } else {
	/* We're inside a comment now */
	state=3;
	continue;
      }
    }

    /* Handle end-of-comment? */
    if (c == '\n') {
      if (state == 1) {
	buffer[index] = '\0';
	return 1;
      } else {
	/* End of comment, back into whitespace */
	state=0;
	continue;
      }
    }

    /* Handle whitespace */
    if (isspace(c)) {
      if (state == 1) {
	buffer[index] = '\0';
	return 1;
      } else {
	/* do nothing -- go on to next character */
	continue;
      }
    }
	
    /* Default case */
    {
      if (state == 0) {
	state = 1;
      }
      if (state == 1 && index < maxTokenSize-1) {
	buffer[index++] = c;
      }
    }
  }
}

/* Read a token from input file, check to make sure it matches what
 * we were expecting.
 */
int AssertToken(FILE *file, char *token)
{
  char buf[200];
  /* Read the next token */
  if (!NextHdrToken(file, buf, 200)) {
    lfError("Error: expected token \"%s\", got EOF.\n", token);
    return 0;
  }
  /* Check to make sure it matches what we expect */
  if (strcmp(token, buf)) {
    lfError("Error: expected token \"%s\", got \"%s\".\n", 
	    token, buf);
    return 0;
  }
  return 1;
}

/* This function takes the log (base2) of n, assuming n is a power of 2 */
int Log2(unsigned long n)
{
  int log;

  for (log=0; log < 32; log++) {
    if (1 & (n >> log)) return log;
  }
  /* if we get here, n was 0 */
  return(-1);
}
