/*
 * draw.c
 *
 * stuff for drawing light slabs
 */
#ifdef sgi
#include <bstring.h>
#endif /* sgi */
#include <stdlib.h>
#include "lightfield.h"

static void
deviceNormalize(LFSlab *slab, LFView *view, LFScreenVtx *vp, int nv,
	int xy_info[4])
{
    LFShared *shared = slab->shared;
    Viewport *viewport = &view->viewport;
    float xmin = 1.f + viewport->xscale + viewport->xcenter;
    float xmax = 1.f - viewport->xscale + viewport->xcenter;
    float ymin = 1.f + viewport->yscale + viewport->ycenter;
    float ymax = 1.f - viewport->yscale + viewport->ycenter;
    float usize = (shared->nu - 1) << shared->u_frac_bits;
    float vsize = (shared->nv - 1) << shared->v_frac_bits;
    float ssize = (shared->ns - 1) << shared->s_frac_bits;
    float tsize = (shared->nt - 1) << shared->t_frac_bits;
    int i;

    for ( i=0 ; i<nv ; i++,vp++ ) {
	/* compute normalized viewport coordinates */
	vp->x = vp->x * viewport->xscale + viewport->xcenter;
        vp->y = vp->y * viewport->yscale + viewport->ycenter;

        if  (xmin > vp->x)
            xmin = vp->x;
        if  (xmax < vp->x)
            xmax = vp->x;

        if  (ymin > vp->y)
            ymin = vp->y;
        if  (ymax < vp->y)
            ymax = vp->y;

	/* compute normalized texture coordinates */
	vp->u *= usize;
	vp->v *= vsize;
	vp->s *= ssize;
	vp->t *= tsize;
    }

    /* compute bounding region */
    xy_info[0] = (int)xmin;
    xy_info[1] = (int)ymin;
    if  (xy_info[0] < 0)
	xy_info[0] = 0;
    if  (xy_info[1] < 0)
	xy_info[1] = 0;

    xy_info[2] = (int)(xmax + 1.f) + 1;
    xy_info[3] = (int)(ymax + 1.f) + 1;
    xy_info[2] = xy_info[2] > view->width  ?
	(view->width -xy_info[0]) : (xy_info[2]-xy_info[0]);
    xy_info[3] = xy_info[3] > view->height ?
	(view->height-xy_info[1]) : (xy_info[3]-xy_info[1]);
}

/* function to draw a light slab */
bool_t
lfDrawSlab(LFSlab *slab)
{
    LFView *view = &__LF_GET_CONTEXT()->view;
    LFScreenVtx inter[32], *poly;
    int ninter[2] = {0, 0};
    int times;
    int i;

    /*
     * perform slab culling and intersection. if slab can be culled or
     * if there is no intersection, return.
     */
    if  (! lfSlabIntersecting(slab->uv, slab->st,
		view->mat_model, view->mat_proj, inter, ninter))
	return FALSE;

    /*
     * otherwise, the intersection exists. in fact, there might be two
     * intersections.
     */
    poly = inter;
    for ( times=0 ; times<2 ; times++ ) {
	int xy_info[4];

	if  (! ninter[times])
	    break;

	/* perform device-dependent normalization */
	deviceNormalize(slab, view, poly, ninter[times], xy_info);

#ifndef	OVERHEAD_ONLY
	/* scan convert the intersecting region */
	for ( i=2 ; i<ninter[times] ; i++ )
	    lfRasterizeTriPntSample(slab, &poly[0], &poly[i-1], &poly[i]);
#endif	/* ! OVERHEAD_ONLY */

	/* increase the pointer for next iteration */
	poly += ninter[times];
    }

    return TRUE;
}

void
configViewport(int x, int y, int width, int height, LFView *view)
{
    float w_2 = .5f * (float)(width - 1);
    float h_2 = .5f * (float)(height - 1);

    view->viewport.xcenter = (float)x + w_2;
    view->viewport.ycenter = (float)y + h_2;
    view->viewport.xscale  = w_2;
    view->viewport.yscale  = h_2;

    view->x = x;
    view->y = y;
    view->width  = width;
    view->height = height;

    if  (width*height*view->pixel_size != view->pixel_buffer_size) {
	if  (view->pixel_buffer)
	    free(view->pixel_buffer);
	view->pixel_buffer_size = width*height*view->pixel_size;
	view->pixel_buffer = malloc(view->pixel_buffer_size);
    }

    if  (width*sizeof(LFPixelUVST) != view->sample_buffer_size) {
	if  (view->sample_buffer)
	    free(view->sample_buffer);
	view->sample_buffer_size = width*sizeof(LFPixelUVST);
	view->sample_buffer = malloc(view->sample_buffer_size);
    }
}

void
lfDrawFramebuffer(LFSlab **slabs, int nslabs)
{
    LFDriver *driver = __LF_GET_DRIVER();
    LFView *view = &__LF_GET_CONTEXT()->view;
    int x, y, width, height;
    int slab_i;

#ifdef	STATS
    lfStartTimer();
#endif

    /* get matrices and viewport */
    (*driver->get_transform)(view->mat_model, view->mat_proj,
	&x, &y, &width, &height);
    configViewport(x, y, width, height, view);

    /* clear the viewport pixel buffer */
    bzero(view->pixel_buffer, view->pixel_buffer_size); 

    /* draw each slab in the slab list */
    for ( slab_i=0 ; slab_i<nslabs ; slab_i++ )
	lfDrawSlab(slabs[slab_i]);

    /* write the resulting looked up stuff into the framebuffer */
    (*driver->write_pixrect)(view->x, view->y, view->width, view->height,
	view->format, view->type, view->pixel_buffer);

#ifdef	STATS
    view->draw_time = view->draw_time + lfEndTimer();
#endif

}

void
lfDrawFieldFramebuffer(LFField *field)
{
    LFSlab *active_slabs[LF_MAXSLABS];
    LFSlab **sp = field->slabs;
    int n_active_slabs = 0;
    int slab_i;

    for ( slab_i=0 ; slab_i<field->nslabs ; slab_i++,sp++ )
	if  (__LF_SLAB_ACTIVE(field, slab_i)) {
	    active_slabs[n_active_slabs++] = *sp;
	}

    lfDrawFramebuffer(active_slabs, n_active_slabs);
}
