/*
 * context.c
 *
 * stuff related to lightfield context
 */
#include <stdlib.h>
#include "lightfield.h"

LFContext *__lf_context;

#define TILE_U_BITS_DEF	0
#define TILE_V_BITS_DEF	0
#define TILE_S_BITS_DEF	0
#define TILE_T_BITS_DEF	0

/* XXX: more of a hack now */
static void
initView(void)
{
    LFView *view = &__LF_GET_CONTEXT()->view;
    char *tile_str = getenv("LF_TILE");

    if  (! tile_str || sscanf(tile_str, "%d%d%d%d", &view->tile_u_bits,
	&view->tile_v_bits, &view->tile_s_bits, &view->tile_t_bits) < 4) {
	view->tile_u_bits = TILE_U_BITS_DEF;
	view->tile_v_bits = TILE_V_BITS_DEF;
	view->tile_s_bits = TILE_S_BITS_DEF;
	view->tile_t_bits = TILE_T_BITS_DEF;
    }
    view->pixel_size = 4;
    view->filter = LF_LERP_UV;
}

static void
initTimer(void)
{
#ifdef sgi
    struct itimerval timer;

    timer.it_interval.tv_sec = 10000;
    timer.it_interval.tv_usec = 0;
    timer.it_value.tv_sec = 10000;
    timer.it_value.tv_usec = 0;
    setitimer(ITIMER_VIRTUAL, &timer, NULL);
#endif
}

static LFField *
allocField(int id)
{
    LFField *field = calloc(1, sizeof(LFField));

    field->id = id;
    return field;
}

/* function to initialize a lightfield context */
void
lfInitContext(void)
{
    /* allocate memory */
    __lf_context = calloc(1, sizeof(LFContext));

    /* initialize timer */
    initTimer();

    /* initialize necessary defaults */
    initView();

    /* initialize default lightfield */
    __lf_context->lf_list = 
    __lf_context->field = allocField(0);
}

/* functions to set/get a list of drawing parameters */
void
lfSetDrawingParameters(float *param_list)
{
    LFView *view = &__LF_GET_CONTEXT()->view;
    float *ptr = param_list;
    int pname;

    for ( ; ; ) {
	pname = *ptr++;
	switch(pname) {
	case LF_DRAW_STATS_RESET:
#ifdef	STATS
	    view->scanline_total = view->scanline_clamped = 0;
	    view->draw_time = 0.f;
	    break;
#else
	    lfError("Need to compile with -DSTATS\n");
	    return;
#endif

	case LF_DRAW_FILTER:
	    view->filter = *ptr++;
	    break;

	case LF_NULL:
	    return;
	}
    }
}

void
lfGetDrawingParameters(int pname, float *param)
{
    LFView *view = &__LF_GET_CONTEXT()->view;

    switch(pname) {
    case LF_DRAW_SCANLINE_CLAMP_RATIO:
#ifdef	STATS
	*param = ! view->scanline_total ? 0.f :
		 (float)view->scanline_clamped / (float)view->scanline_total;
#else
	goto get_stats_warning;
#endif
	break;

    case LF_DRAW_TIMING:
#ifdef	STATS
	*param = view->draw_time;
#else
	goto get_stats_warning;
#endif
	break;

    case LF_NULL:
	return;
    }

    return;

get_stats_warning:
    lfError("Need to compile with -DSTATS\n");
}

/* functions to set/get a list of parameters */
void
lfSetParameters(int op_name, float *param_list)
{
    static void (*op_array[])(float *) = {
	lfSetDrawingParameters,
    };
    if  (op_name < LF_OP_MIN || op_name > LF_OP_MAX)
	lfError("lfSetParameters: unknown operation %d\n", op_name);
    (*op_array[op_name])(param_list);
}

void
lfGetParameters(int op_name, int pname, float *param)
{
    static void (*op_array[])(int, float *) = {
	lfGetDrawingParameters,
    };
    if  (op_name < LF_OP_MIN || op_name > LF_OP_MAX)
	lfError("lfSetParameters: unknown operation %d\n", op_name);
    (*op_array[op_name])(pname, param);
}

void
lfStartTimer(void)
{
#ifdef STATS
    struct itimerval *timer = &__LF_GET_CONTEXT()->timer;
    getitimer(ITIMER_VIRTUAL, timer);
#else
    lfError("lfStartTimer: Need to compile with -DSTATS\n");
#endif
}

float
lfEndTimer(void)
{
#ifdef sgi
    struct itimerval *timer = &__LF_GET_CONTEXT()->timer;
    struct itimerval new;
    float diff;

#ifdef STATS
    getitimer(ITIMER_VIRTUAL, &new);
    diff = timer->it_value.tv_sec - new.it_value.tv_sec;
    diff += (float)(timer->it_value.tv_usec - new.it_value.tv_usec) / 1000000.f;
    return diff;
#else /* no STATS */
    lfError("lfEndTimer: Need to compile with -DSTATS\n");

    return 0.f;
#endif /* STATS */
#endif /* sgi */
}

static void
set_quad_geom(const float *values, LFVtx *v)
{
    int v_i;

    for ( v_i=0 ; v_i<4 ; v_i++,v++ ) {
	v->ox = *values++;
	v->oy = *values++;
	v->oz = *values++;
	v->ow = *values++;
	v->s  = *values++;
	v->t  = *values++;
    }
}

void
lfAttribute(int attrib_name, const float *values)
{
    LFContext *ctx = __LF_GET_CONTEXT();
    LFField *field = ctx->field;
    LFSlab *slab;
    LFOps *ops = &ctx->ops;
    int slab_i;

    if  (! ops->op_slab_cnt)
	return;

    switch(attrib_name) {
    case LF_SLAB_UV:
	for ( slab_i=0 ; slab_i<ops->op_slab_cnt ; slab_i++ ) {
	    slab = ops->op_slabs[slab_i];
	    set_quad_geom(values, slab->uv);
	}
	break;

    case LF_SLAB_ST:
	for ( slab_i=0 ; slab_i<ops->op_slab_cnt ; slab_i++ ) {
	    slab = ops->op_slabs[slab_i];
	    set_quad_geom(values, slab->st);
	}
	break;

    default:
	lfError("Unknown attribute name %d\n", attrib_name);
	return;
    }
}

void
lfBindLightField(int target, int id)
{
    LFContext *ctx = __LF_GET_CONTEXT();
    LFField *field = ctx->field;

    if  (field->id == id)
	return;

    for ( field=ctx->lf_list ; field ; field=field->next )
	if  (field->id == id)  {
	    ctx->field = field;
	    return;
	}

    field = allocField(id);
    field->next = ctx->field;
    ctx->field = field;
}
