/*
 * compression.c
 *
 * stuff to deal with compression (mainly VQ for now)
 */
#include <stdio.h>
#include <stdlib.h>
#ifdef sgi
#include <bstring.h>
#endif

#include "vq_iface.h"
#include "lightfield.h"

static void
fillVQStruct(LFOps *ops, LFInternOp *op)
{
    LFSlab **slabs = ops->op_slabs;
    int nslab = ops->op_slab_cnt;
    float *descr = op->op_descr;
    LFVQCodebook *vq;
    int tile_u_bits = 0;
    int tile_v_bits = 0;
    int tile_s_bits = 2;
    int tile_t_bits = 2;
    float train_pct = .01f;
    int code_size = 1 << 16;
    int i;

    while (*descr != LF_NULL) {
        switch((int)*descr++) {
        case LF_VQ_TILESIZE:
	    tile_u_bits = log2(*descr++);
	    tile_v_bits = log2(*descr++);
	    tile_s_bits = log2(*descr++);
	    tile_t_bits = log2(*descr++);
	    break;

        case LF_VQ_TRAIN_SIZE:
	    train_pct = *descr++;
	    break;

        case LF_VQ_CODESIZE:
	    code_size = *descr++;
	    break;

        default:
	    descr++;
	    break;
        }
    }

    for ( i=0 ; i<nslab ; i++ )
	if  (slabs[i]->shared->vq)
	    free(slabs[i]->shared->vq);

    vq = malloc(sizeof(LFVQCodebook));
    vq->ubits = tile_u_bits;
    vq->vbits = tile_v_bits;
    vq->sbits = tile_s_bits;
    vq->tbits = tile_t_bits;
    vq->size  = code_size;
    vq->train_pct = train_pct;
    vq->codebook = NULL;
    vq->sample_size = slabs[0]->shared->sample_size;

    for ( i=0 ; i<nslab ; i++ )  {
	slabs[i]->shared->vq = vq;
	/* lightfield data stores data now */
	slabs[i]->sample_size = 2;
    }
}

/* function to reshuffle data for VQ processing */
void
lfMemShuffle(LFSlab **slabs, int nslab, const int *pos,
	const float *descr, const void *input, int input_size,
	void *output, int output_size, void *aux_ptr)
{
    LFSlab *slab = slabs[pos[LF_SLAB_POS]];
    LFShared *shared = slab->shared;
    LFVQCodebook *vq = shared->vq;
    unsigned char *src_ptr, *dst_ptr;
    int tblk_mul = shared->nt >> vq->tbits;
    int tile_shift = vq->ubits + vq->vbits + vq->sbits + vq->tbits;
    int smsk = (1 << vq->sbits) - 1;
    int tmsk = (1 << vq->tbits) - 1;
    int sblk, tblk;
    int ssmp, tsmp;
    int u, v, s, t;
    int i, idx;
    int block_size = (shared->ns*shared->nt*shared->sample_size) <<
		     (vq->ubits+vq->vbits);

    assert(block_size == input_size);
    assert(block_size == output_size);

    /*
     * repack training set so that all samples of the same VQ block
     * sit near one another
     */
    for ( u=0,src_ptr=(char *)input ; u<(1<<vq->ubits) ; u++ )
	for ( v=0 ; v<(1<<vq->vbits) ; v++ )
	    for ( s=0 ; s<shared->ns ; s++ )  {
		sblk = s >> vq->sbits;
		ssmp = s & smsk;
		for ( t=0 ; t<shared->nt ; t++ )  {
		    tblk = t >> vq->tbits;
		    tsmp = t & tmsk;
		    idx = ((sblk * tblk_mul + tblk) << tile_shift) + (((((
			u<<vq->vbits)+v)<<vq->sbits)+ssmp)<<vq->tbits)+tsmp;
		    assert(idx < ((shared->ns * shared->nt) <<
				  (vq->ubits + vq->vbits)));
		    dst_ptr  = (char *)output + shared->sample_size * idx;
		    for ( i=0 ; i<shared->sample_size ; i++ )
			*dst_ptr++ = *src_ptr++;
		}
	    }
}

/* function to initialize memory shuffling routine */
void *
lfBeginMemShuffle(LFOps *ops, LFInternOp *op)
{
    if  (ops->chain_msk & LF_SET_SLAB)
	fillVQStruct(ops, op);

    op->op_func  = lfMemShuffle;
    op->op_end   = lfEndDefault;
    op->op_update = lfUpdateNoop;

    return NULL;
}

/* function to generate a codebook */
void
lfVQGenCodebook(LFSlab **slabs, int nslab, const int *pos,
	const float *descr, const void *input, int input_size,
	void *output, int output_size, void *aux_ptr)
{
    LFVQCodebook *vq = slabs[0]->shared->vq;
    VQ_t *tree_vq;

    /* do VQ training */
    tree_vq = vqGenTreeCodebook(input, input_size, 1, vq->size,
	vq->sample_size << (vq->ubits+vq->vbits+vq->sbits+vq->tbits));
    vq->codebook = tree_vq;
    vq->size = tree_vq->nwords;	/* may generate a smaller codebook */
    vq->type = LF_VQ_TREE_VQ;
}

/* function to generate VQ code array */
void
lfVQCompress(LFSlab **slabs, int nslab, const int *pos,
	const float *descr, const void *input, int input_size,
	void *output, int output_size, void *aux_ptr)
{
    LFSlab *slab = slabs[pos[LF_SLAB_POS]];
    LFShared *shared = slab->shared;
    LFVQCodebook *vq = shared->vq;
    int block_size = (shared->ns * shared->nt * shared->sample_size) <<
		     (vq->ubits + vq->vbits);
    int last_bit_offset;
    int arr[8];

    assert(input_size == block_size);
    assert(vq->type == LF_VQ_LINEAR_VQ || vq->type == LF_VQ_TREE_VQ);

    /* do VQ classification */
    arr[0] = 1 << vq->tbits;
    arr[1] = 1 << vq->sbits,
    arr[2] = 1 << vq->vbits,
    arr[3] = 1 << vq->ubits,
    arr[4] = shared->nt >> vq->tbits,
    arr[5] = shared->ns >> vq->sbits,
    arr[6] = 1,
    arr[7] = 1;

    vqEncode(vq->codebook, (unsigned char *)input, 8, arr,
	vq->sample_size /* pixel size */,
	TRUE /* do padding */, 16 /* pad-to-short */, output, &last_bit_offset);

    assert(last_bit_offset == (output_size * 8));
}

void
lfEndVQCompress(LFOps *ops, LFInternOp *op, void *aux_ptr)
{
    LFShared *shared;
    int i;

    for ( i=0 ; i<ops->op_slab_cnt ; i++ )  {
	shared = ops->op_slabs[i]->shared;
	if  (shared->vq)
	    shared->sample_size = 2;
    }
}

/* function to make a linear VQ codebook array */
void
lfLinearizeVQCodebook(LFVQCodebook *vq)
{
    VQ_t *veeq;

    if  (! vq || vq->type == LF_VQ_LINEAR_DATA)
	return;

    if  (vq->type == LF_VQ_TREE_VQ) {
	veeq = vqTreeCodebookToLinearCodebook(vq->codebook);
	vqFreeCodebook(vq->codebook);
    }
    else  if  (vq->type == LF_VQ_LINEAR_VQ)
	veeq = vq->codebook;
    vq->type = LF_VQ_LINEAR_DATA;
    vq->codebook = veeq->codebook;
}
