/*
 * kd.c
 *
 * functions related to multi-dimensional data
 */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include "common.h"
#include "geom_kd.h"

/*
 * function to allocate and free a k-d bounding box
 */
BoundingBox
*geomAllocBoundingBox(int dim)
{
    BoundingBox *box;

    assert(dim > 0);
    box = (BoundingBox *)malloc(sizeof(BoundingBox));
    box->dim = dim;
    box->min_max = (float (*)[2])malloc(sizeof(float [2]) * dim);
    box->min_max_idx = (int (*)[2])malloc(sizeof(int [2]) * dim);

    return box;
}

void
geomFreeBoundingBox(BoundingBox *box)
{
    free(box->min_max);
    free(box->min_max_idx);
    free(box);
}

void
geomBoundingInterval(int dim, int comp_idx, float *pt, int n,
	float *min_max, int *min_max_idx)
{
    float *p = pt + comp_idx;
    float min = *p, max = *p;
    int i;

    min_max_idx[0] = min_max_idx[1] = 0;
    for ( i=1,p+=dim ; i<n ; i++,p+=dim )
	if  (*p < min) {
	    min = *p;
	    min_max_idx[0] = i;
	}
	else if  (*p > max) {
	    max = *p;
	    min_max_idx[1] = i;
	}
    min_max[0] = min;
    min_max[1] = max;
}

/*
 * function to generate a bounding box of a list of points
 */
void
geomBoundingBox(int dim, float *pt, int n, BoundingBox *box)
{
    int dim_i;

    assert(dim == box->dim);
    assert(n > 0);
    for ( dim_i=0 ; dim_i<dim ; dim_i++ )
	geomBoundingInterval(dim, dim_i, pt, n,
		box->min_max[dim_i], box->min_max_idx[dim_i]);
}

/*
 * function to determine if bounding boxes of two clusters
 * are separable in kd.
 * 
 * epsilon: a fudge factor introduced to deal with numerical
 * issues as well. the net effect is to check if two fattened
 * (by epsilon/2) intersect.
 */
bool_t
geomBoundingBoxSeparable(int dim, float *cluster0, int n0,
	float *cluster1, int n1, float epsilon)
{
    float min_max0[2];
    int min_max_idx0[2];
    float min_max1[2];
    int min_max_idx1[2];
    int dim_i;

    assert(dim > 0);
    assert(n0 > 0);
    assert(n1 > 0);

    for ( dim_i=0 ; dim_i<dim ; dim_i++ ) {
	geomBoundingInterval(dim, dim_i, cluster0, n0,
		min_max0, min_max_idx0);
	geomBoundingInterval(dim, dim_i, cluster1, n1,
		min_max1, min_max_idx1);
	if  (  min_max0[1] + epsilon < min_max1[0] 
	    || min_max0[0] - epsilon > min_max1[1]
	    )
	    return TRUE;
    }

    return FALSE;
}
