/*
 * star.c
 *
 * a bunch of functions with respect to 2d star-shaped polygons
 */
#include <assert.h>
#include "common.h"
#include "geom_type.h"
#include "geom_2d.h"
#include "geom_kd.h"
#include "vec2.h"

/*
 * function to determine if a point is inside a star-shaped polygon.
 * here, an interior point in the kernel of the star-shaped polygon
 * is given, and the decision complexity is O(log(n))
 */
bool_t
geom2dPntInStarPolygon(float (*poly)[2], int n,
	float interior_pnt[2], float pnt[2], float epsilon)
{
    float area;
    bool_t sign;
    int lo, up;
    int mid;

    /*
     * sort-of determine the orientation of the polygon with respect to
     * the given point
     */
    area = AREA_PNT2(interior_pnt, poly[0], pnt);
    if  (ABS(area) <= epsilon)
	return geom2dPntInConvexPolygon(poly, n, pnt);
    sign = area >= 0.f;
    area = AREA_PNT2(interior_pnt, poly[0], poly[1]);
    if  ((area >= 0.f) == sign)  {
	lo = 1, up = n - 1;
    }
    else {
	lo = n - 1, up = 1;
    }
    
    /*
     * using binary-search to decide at most a 180-degree space so that
     * the given point lies inside this space.
     */
    for ( ; ABS(up-lo)>1 ; ) {
	mid = (lo + up) >> 1;
	area = AREA_PNT2(interior_pnt, poly[0], poly[mid]);
	if  ((area >= 0.f) == sign)  {
	    lo = mid;
	    area = AREA_PNT2(interior_pnt, pnt, poly[mid]);
	    if  (ABS(area) <= epsilon)
		return geom2dPntInConvexPolygon(poly, n, pnt);
	    if  ((area >= 0.f) == sign)
		goto startBinarySearch;
	}
	else
	    up = mid;
    }

    /*
     * exception case: see if the point is inside the last triangle of
     * the 180-degree space.
     */
    area = AREA_PNT2(interior_pnt, poly[lo], pnt);
    if  ((area >= 0.f) == sign)
	goto donePointInStarPolygon;

    /*
     * binary search the triangle so that the given point is inside
     * two edges of the triangle
     */
startBinarySearch:
    if  (lo < up)  {
	up = lo, lo = 0;
    }
    else {
	if  ((AREA_PNT2(interior_pnt, pnt, poly[n-1]) >= 0.f) == sign) {
	    lo = 0, up = n - 1;
	    goto donePointInStarPolygon;
	}
	up = lo, lo = n - 1;
    }

    for ( ; ABS(up-lo)>1 ; ) {
	mid = (lo + up) >> 1;
	area = AREA_PNT2(interior_pnt, poly[mid], pnt);
	if  (ABS(area) <= epsilon)
	    return geom2dPntInConvexPolygon(poly, n, pnt);
	if  ((area >= 0.f) == sign)
	    lo = mid;
	else 
	    up = mid;
    }

donePointInStarPolygon:
    /*
     * now we just need to check if the point is inside the outside edge
     * of the triangle;
     */
    area = AREA_PNT2(pnt, poly[lo], poly[up]);
    return (area >= 0.f) == sign;
}
