/*
 * convex.c
 *
 * a bunch of functions with respect to 2d convex regions
 */
#include <assert.h>
#include "common.h"
#include "geom_type.h"
#include "geom_2d.h"
#include "geom_kd.h"
#include "vec2.h"

/*
 * function to determine if a polygon is convex
 */
bool_t
geom2dPolygonConvex(float (*poly)[2], int n)
{
  return FALSE;
}

/*
 * function to determine if a point is inside a convex polygon
 * the algorithm used does not use the interior-point method for 
 * checking if a point is inside a star-shaped polygon
 */
bool_t
geom2dPntInConvexPolygon(float (*poly)[2], int n, float pnt[2])
{
 
    float (*prev)[2] = poly + n - 1;
    float (*curr)[2] = poly;
    float area = 0.0;
    bool_t all_pos = TRUE;
    bool_t all_neg = TRUE;
    int i;
    
    for ( i=0 ; i<n ; i++,prev=curr++ /* ,curr++ */ ) {
	area = AREA_PNT2(pnt, prev[0], curr[0]);
	if  (area < 0) {
	    all_pos = FALSE;
	    if  (! all_neg)
		break;
	}
	else  {
	    all_neg = FALSE;
	    if  (! all_pos)
		break;
	}
    }

    return (all_pos || all_neg);
}

/*
 * function to determine if a polygon is CW, or CCW, assuming
 * vertices are in the ascending order. under extreme case,
 * it may return GEOM_ORIENT_AREA0 (area < epsilon); this is
 * the case when the polygon is a degenerated into a line
 */
int
geom2dOrientConvexPolygon(float (*poly)[2], int n, int start, float epsilon)
{
    float *prev = poly[start ? (start - 1) : (n - 1)];
    float *curr = poly[start];
    float *next;
    int next_idx = start < n - 1 ? (start + 1) : 0;
    float area=0.0;

    do {
	next = poly[next_idx];

	area = AREA_PNT2(prev, curr, next);
	if  (ABS(area) >= epsilon)
	    return area > 0.f ? GEOM_ORIENT_CCW : GEOM_ORIENT_CW;

	/*
	 * if for whatever reason, the 3 vertices of almost
	 * colinear, go for the next three vertices
	 */
	prev = curr;
	curr = next;
	next_idx = next_idx < n - 1 ? (next_idx + 1) : 0;
    }
    while (next_idx != start);

    /*
     * if we still have not determined the orientation 
     * after looping thru all vertices of a polygon, declare
     * it as a zero-areaed polygon.
     */
    return GEOM_ORIENT_AREA0;
}

#define	NEXT_IDX(n, cur_i, dir, nxt_i){	\
	nxt_i = cur_i + dir;		\
	if  (nxt_i < 0) nxt_i = n - 1;	\
	else if  (nxt_i >= n) nxt_i = 0; }

static int
searchChainX(float (*poly)[2], int n, int start, int dir, float x)
{
    float *curr = poly[start];
    int cur = start;
    float *next;
    int nxt=0;

    /* not me */
    for ( ; ; ) {
	NEXT_IDX(n, cur, dir, nxt);

	next = poly[nxt];
	if  (curr[0] >= next[0])
	    break;
	if  (next[0] > x)
	    break;
	cur = nxt;
	curr = next;
    }

    return cur;
}

static float
relativePos(float (*poly)[2], int n, int start, int dir, float *pt)
{
    float *curr = poly[start];
    float *next;
    int nxt=0;

    /* not me */
    NEXT_IDX(n, start, dir, nxt);
    next = poly[nxt];

    return AREA_PNT2(curr, next, pt);
}

/*
 * function to determine if two convex polygons are separable
 */
bool_t
geom2dConvexPolygonSeparable(float (*poly0)[2], int n0,
	float (*poly1)[2], int n1, float epsilon)
{
    float (*poly[2])[2];
    int n[2];
    bool_t ccw_dir[2];
    bool_t lo_dir=0;
    bool_t hi_dir=0;
    int lo_idx=0;
    int hi_idx=0;
    BoundingBox *box[2];
    bool_t separable = TRUE;
    bool_t map=0;
    int i;
    
    int orient;
    float poly1_minx;
    int top_chain_idx;
    int bot_chain_idx;
    float *lft_pt;
    
    float hi_edge_dir[2];
    float lo_edge_dir[2];
    int hi_next=0;
    int lo_next=0;
    float inter[2];
    float params[2];

    /*
     * move polygons into an array so that we can play pointers
     * from now on.
     */

    /* not me */
    poly[0] = poly0, n[0] = n0;
    poly[1] = poly1, n[1] = n1;
    ccw_dir[0] = 0;
    ccw_dir[1] = 0;

    /* compute bounding boxes */
    for ( i=0 ; i<2 ; i++ ) {
	/* compute bounding boxes */
	box[i] = geomAllocBoundingBox(2);
	geomBoundingBox(2, (float *)poly[i], n[i], box[i]);
    }

    /* do trivial rejection on y dimension */
    if  (  box[0]->min_max[1][1] + epsilon < box[1]->min_max[1][0]
	|| box[1]->min_max[1][1] + epsilon < box[0]->min_max[1][0]
	)
	goto done_ConvexPolygonSeparable;

    /*
     * find the polygon with the left-most x coordinate,
     * and order two polygons so that polygon 0 is always 
     * the one with the leftmost x coordinate.
     */
    map = box[0]->min_max[0][0] <= box[1]->min_max[0][0];

    /*
     * do 2d trivial rejection on x dimension. notice because of
     * swapping in the previous step, only one comparison is needed.
     */
    if  (box[!map]->min_max[0][1] + epsilon < box[map]->min_max[0][0])
	goto done_ConvexPolygonSeparable;

    /*
     * find two chains of each polygon so that chain 0 
     * is the on the bottom, and chain 1 is on the top
     * this is equivalent to finding the orientation of
     * a polygon.
     */
    for ( i=0 ; i<2 ; i++ ) {
     
      orient = geom2dOrientConvexPolygon(poly[i], n[i],
			box[i]->min_max_idx[0][0], epsilon);
	if  (orient == GEOM_ORIENT_AREA0) {
	    separable = FALSE;
	    goto done_ConvexPolygonSeparable;
	}
	ccw_dir[i] = (orient == GEOM_ORIENT_CCW) ? 1 : -1;
    }

    /*
     * walk along the top and bottom chain of polygon 0 until
     * reaching the left-most x coordinate of polygon 1.
     * determine the relative position of the left-most vertex
     * of poly 1 with respect to the top and bottom chains.
     * this can be transformed into signed area computation.
     * now if the left-most vertex of poly 1 is in between the 
     * top and bottom chains of poly 0, then they intersect.
     * otherwise, we only need to be concerned with one chain
     * from each polygon; call one high chain and one low
     * chain with respect to the vertex of poly 1.
     */
    {
      
      poly1_minx = box[map]->min_max[0][0];
      top_chain_idx = searchChainX(poly[!map], n[!map],
				   box[!map]->min_max_idx[0][0], -ccw_dir[!map],
				   poly1_minx);
	bot_chain_idx = searchChainX(poly[!map], n[!map],
		box[!map]->min_max_idx[0][0], ccw_dir[!map],
		poly1_minx);
	lft_pt = poly[map][box[map]->min_max_idx[0][0]];

	if  (relativePos(poly[!map], n[!map], bot_chain_idx,
		ccw_dir[!map], lft_pt) >= 0.f) {
	    if  (relativePos(poly[!map], n[!map], top_chain_idx,
			-ccw_dir[!map], lft_pt) <= 0.f) {
		separable = FALSE;
		goto done_ConvexPolygonSeparable;
	    }

	    /*
	     * top chain of poly 0 is low chain and bottom
	     * chain of poly 1 is high chain.
	     */
	    hi_idx = box[map]->min_max_idx[0][0];
	    lo_idx = top_chain_idx;
	    hi_dir = ccw_dir[map];
	    lo_dir = -ccw_dir[!map];
	}
	else {
	    if  (relativePos(poly[!map], n[!map], top_chain_idx,
			-ccw_dir[!map], lft_pt) >= 0.f) {
		separable = FALSE;
		goto done_ConvexPolygonSeparable;
	    }
	    
	    /*
	     * bottom chain of poly 0 is high chain and top
	     * chain of poly 1 is low chain. reverse map.
	     */
	    hi_idx = bot_chain_idx;
	    lo_idx = box[map]->min_max_idx[0][0];
	    hi_dir = ccw_dir[!map];
	    lo_dir = -ccw_dir[map];
	    map = !map;
	}
    }

    /*
     * from now on, !map is for the low chain polygon, and map is
     * for the high chain polygon.
     * we start walking on these two chains in lock steps.
     * anytime, we detect the current pair of edges are diverging,
     * we declare that these polygons cannot intersect.
     * anytime, the current pair of edges cross, we declare
     * them intersecting. if finally, coordinates start dropping
     * for any chain, we quit and  return non-intersecting.
     */
    {
	NEXT_IDX(n[map], hi_idx, hi_dir, hi_next);
	NEXT_IDX(n[!map], lo_idx, lo_dir, lo_next);
	SUBVEC2(poly[map][hi_next], poly[map][hi_idx], hi_edge_dir);
	SUBVEC2(poly[!map][lo_next], poly[!map][lo_idx], lo_edge_dir);

	for ( ; ; )  {
	    inter[0] = inter[1] = 0;
	    params[0] = params[1] = 0;

	    /* check for divergence */
	    if  (AREA_VEC2(hi_edge_dir, lo_edge_dir) < 0.f)
		goto done_ConvexPolygonSeparable;

	    /* check for intersection */
	    if  (geom2dSegmentSegmentRelation(
			poly[map][hi_idx], poly[map][hi_next],
			poly[!map][lo_idx], poly[!map][lo_next],
			inter, params, epsilon) == GEOM_FIGURE_INTERSECTING)  {
		separable = FALSE;
		goto done_ConvexPolygonSeparable;
	    }

	    /* determine which edge(s) to move along for the next step */ 
	    if  (poly[map][hi_next][0] >= poly[!map][lo_next][0]) {
		lo_idx = lo_next;
		NEXT_IDX(n[!map], lo_idx, lo_dir, lo_next);
		if  (poly[!map][lo_idx][0] > poly[!map][lo_next][0])
		    break;
		SUBVEC2(poly[!map][lo_next], poly[!map][lo_idx], lo_edge_dir);
	    }
	    else {
		hi_idx = hi_next;
		NEXT_IDX(n[map], hi_idx, hi_dir, hi_next);
		if  (poly[map][hi_idx][0] > poly[map][hi_next][0])
		    break;
		SUBVEC2(poly[map][hi_next], poly[map][hi_idx], hi_edge_dir);
	    }
	}
    }

done_ConvexPolygonSeparable:
    /* free bounding boxes */
    geomFreeBoundingBox(box[0]);
    geomFreeBoundingBox(box[1]);

    return separable;
}

int
copyPolygon(float (*ps)[2], int ns, float (*pd)[2])
{
    int i;

    for ( i=0 ; i<ns ; i++ ) {
	COPYVEC2(ps[i], pd[i]);
    }
    return ns;
}

#define	EPSILON_0	.000001f

int geom2dConvexPolygonIntersect(float (*poly0)[2], int n0,
	float (*poly1)[2], int n1, float (*new)[2])
{
  int newgeom2dConvexPolygonIntersect(float (*poly0)[2], int n0,
				      float (*poly1)[2], int n1, float (*new)[2]);
  void displayPolygon(float (*poly)[2], int inCount);
  int result;

  result = newgeom2dConvexPolygonIntersect(poly0, n0, poly1, n1, new);
 
  return result;
}


/* my own version of polygon intersection 
 */

int newgeom2dConvexPolygonIntersect( float (*poly0)[2], int n0,
				     float (*poly1)[2], int n1, float (*new)[2])
{
  int pingpong;
  float polyTemp[2][20][2];
  float localPoly0[20][2],  localPoly1[20][2];
  float polyCount[2];
  int prev, curr;
  int i;
  int orient;

  /* functions */
  int clipPolygonAgainstEdge(float (*inPoly)[2], int inCount, float *p1, float *p2,
			   float (*outPoly)[2] );

  void displayPolygon(float (*poly)[2], int inCount);

  /* code */
  pingpong = 0;

  /* change the orientation of both to be CCW */
  orient = geom2dOrientConvexPolygon(poly0, n0, 0, 0);
  if( orient != GEOM_ORIENT_CCW )
    {
      for(i=0; i<n0; i++)
	{
	  localPoly0[i][0] = poly0[n0-i-1][0];
	  localPoly0[i][1] = poly0[n0-i-1][1];
	  
	}
    }
  else
    {
       for(i=0; i<n0; i++)
	{
	  localPoly0[i][0] = poly0[i][0];
	  localPoly0[i][1] = poly0[i][1];
	  
	}
    }

  orient = geom2dOrientConvexPolygon(poly1, n1, 0, 0);
  if( orient != GEOM_ORIENT_CCW )
    {
      for(i=0; i<n1; i++)
	{
	  localPoly1[i][0] = poly1[n1-i-1][0];
	  localPoly1[i][1] = poly1[n1-i-1][1];
	  
	}
    }
  else
    {
       for(i=0; i<n1; i++)
	{
	  localPoly1[i][0] = poly1[i][0];
	  localPoly1[i][1] = poly1[i][1];
	  
	}
    }

  /* copy from poly0 to polyTemp[pintpong] */
  for(i=0; i<n0; i++)
    {
      polyTemp[pingpong][i][0] = localPoly0[i][0];
      polyTemp[pingpong][i][1] = localPoly0[i][1];
      
    }
  polyCount[pingpong] = n0;

  /* clip polyTemp[pingpont] against each edge of poly1 */
  for(prev = n1-1, curr=0; curr<n1; prev=curr, curr++, pingpong=1-pingpong)
    {
      if( polyCount[pingpong] == 0 )
	break;

      polyCount[1-pingpong] = 
	clipPolygonAgainstEdge(polyTemp[pingpong], polyCount[pingpong], 
			       localPoly1[prev], localPoly1[curr], polyTemp[1-pingpong] );
     
      }

  /* fill stuff to new */
  for(i=0; i<polyCount[pingpong]; i++)
    {
      new[i][0] = polyTemp[pingpong][i][0];
      new[i][1] = polyTemp[pingpong][i][1];
      
    }
  return polyCount[pingpong];
}


/* 
 * clip a convex, CCW polygon against an edge, the edge is also CCW oriented, i.e.
 * if we walk along the edge from p1 to p2 then the left hand side will be the "inside"
 * of the edge
 *
 * Input: inPoly: input polygon
 *        inCount: # of vertices of inPoly
 *        p1, p2: points of the edge
 *
 * Output: outPoly: the clipped polygon
 * Return: the # of vertices in outPoly( 0 indicates all clipped out )
 */
int clipPolygonAgainstEdge(float (*inPoly)[2], int inCount, float *p1, float *p2,
			   float (*outPoly)[2] )
{
  /* coefficients of the edge, ax+by+c=0 */
  float a, b, c;
  /* index into the polygon array */
  int curr, prev;
  /* index into the polygon where intersection occurs */
  int intersect1, intersect2;
  /* the 2 intersections */
  float cross[2][2];
  /* the # of intersections for outPoly */
  int outCount;
  /* the start and end of the vertices of inPoly which are inside the edge */
  int indexStart, indexEnd;

  int i,j;

  /* functions */
  void findIntersection(float *p1, float *p2, float a, float b, float c, float *cross);
  int pointInsideEdge(float *point, float a, float b, float c);
  void displayPolygon(float (*poly)[2], int count);

  
  /* code */
  if( inCount <= 2 )
    {
      printf("clipPolygonAgainstEdge: not a polygon\n");
      exit(1);
    }

  /* find a, b, c */
  a = -p2[1]+p1[1];
  b = -p1[0]+p2[0];
  c = -a*p1[0] - b*p1[1];

  /* find out the intersections */
  for(prev = inCount-1, curr=0, intersect1=-1, intersect2=-1;
      curr<inCount; 
      prev = curr, curr++ )
    {
      if( pointInsideEdge(inPoly[prev], a, b, c)*pointInsideEdge(inPoly[curr], a, b, c) < 0 )
	{
	  /* intersect occured */
	  if( intersect1 == -1 )
	    {
	      intersect1 = prev;
	      findIntersection(inPoly[prev], inPoly[curr], a, b, c, cross[0]);
	    }
	  else
	    {
	      intersect2 = prev;
	      findIntersection(inPoly[prev], inPoly[curr], a, b, c, cross[1]);
	    }
	}
    }

  if( intersect2 == -1 )
    {
      /* no clipping or all clipping */
      if( pointInsideEdge(inPoly[0], a, b, c) == 1 )
	{
	  /* no clipping */
	  for(i=0; i<inCount; i++)
	    {
	      outPoly[i][0] = inPoly[i][0];
	      outPoly[i][1] = inPoly[i][1];
	  
	    }
      
	  return inCount;
	}
      else
	{
	  /* all clipping */
	  return 0;
	}
    }
  else
    {
      /* partially clipped */
      outCount = 0;

      if( pointInsideEdge(inPoly[intersect1], a, b, c) == -1 )
	{
	  indexStart = (intersect1+1)%inCount;
	  indexEnd = intersect2;
	  if( indexEnd > indexStart )
	    {
	      outCount = indexEnd - indexStart+3;
	    }
	  else
	    {
	      outCount = indexEnd - indexStart+3+inCount;
	    }

	  outPoly[0][0] = cross[0][0];
	  outPoly[0][1] = cross[0][1];
	  outPoly[outCount-1][0] = cross[1][0];
	  outPoly[outCount-1][1] = cross[1][1];
	  
	}
      else
	{
	  indexEnd = intersect1;
	  indexStart = (intersect2+1)%inCount;
	  if( indexEnd > indexStart )
	    {
	      outCount = indexEnd - indexStart+3;
	    }
	  else
	    {
	      outCount = indexEnd - indexStart+3+inCount;
	    }
	  outPoly[0][0] = cross[1][0];
	  outPoly[0][1] = cross[1][1];
	  outPoly[outCount-1][0] = cross[0][0];
	  outPoly[outCount-1][1] = cross[0][1];
	}

      /* copy the rest of the vertices */
      for(i=1, j=indexStart; i<outCount-1; i++, j=(j+1)%inCount)
	{
	  outPoly[i][0] = inPoly[j][0];
	  outPoly[i][1] = inPoly[j][1];
	}

      return outCount;
    }
}

/* decide if a point is inside an edge or not */
int pointInsideEdge(float *point, float a, float b, float c)
{
  float value;

  value = point[0]*a + point[1]*b + c;

  if(value < 0)
    return -1;
  else
    return 1;
}

/* find the intersection of 2 edges */
void findIntersection(float *p1, float *p2, float a, float b, float c, float *cross)
{
  float d1, d2;
  float e1, e2;

  d1 = p1[0]*a+p1[1]*b+c;
  d2 = p2[0]*a+p2[1]*b+c;

  if(d1-d2 == 0)
    {
      printf("findIntersection(): d1==d2\n");
      exit(1);
    }
  
  e2 = d1/(d1-d2);
  e1 = 1.0-e2;
  
  cross[0] = (float)(e1*p1[0]+e2*p2[0]);
  cross[1] = (float)(e1*p1[1]+e2*p2[1]);
}

/* display a polygon */
void displayPolygon(float (*poly)[2], int inCount)
{
  int i;
  printf("*************************************************\n");
  for(i=0; i<inCount; i++)
    {
      printf("%d: x:%f, y:%f\n", i, poly[i][0], poly[i][1] );
    }
  printf("************************************************\n");
}
