#ifndef	LF_UTIL_H
#define	LF_UTIL_H

/* files included */
#include <stdio.h>
#ifdef sgi
#include <sys/time.h>
#endif /* sgi */
#include "common.h"
#include "geom.h"
#include "lf_consts.h"

/* definitions */

/* definitions related to different lightfield operations */
#define	LF_OP_DRAW	0
#define	LF_OP_CREATE	1
#define	LF_OP_COMPRESS	2
#define	LF_OP_WRITE	3
#define	LF_OP_READ	4
#define	LF_OP_MIN	LF_OP_DRAW
#define	LF_OP_MAX	LF_OP_READ
#define	LF_OP_COMPRESS_TRAIN	5
#define	LF_OP_MEM_INPUT	6
#define	LF_OP_SHUFFLE	7

#define	LF_DRAW		LF_OP_DRAW

/* definitions related to drawing description */
#define	LF_DRAW_FILTER	0x1
#define	LF_DRAW_SCANLINE_CLAMP_RATIO 0x2
#define	LF_DRAW_TIMING	0x3
#define	LF_DRAW_STATS_RESET 0x4

/* definitions related to slab culling and intersection */
#define	LF_CULL_NO	0
#define	LF_CULL_1	1
#define	LF_CULL_2	2
#define	LF_CULL_3	3
#define	LF_CULL_4	4
#define	LF_NO_INTERSECT	5

/* definitions related to data flow of an internal operator chain */
#define	LF_SRC_SLABMEM	0
#define	LF_SRC_FILE	1
#define	LF_SRC_CREATE	2
#define	LF_SRC_DFMEM	3

#define	LF_DST_SLABMEM	0
#define	LF_DST_FILE	1
#define	LF_DST_SLABVQ	2
#define	LF_DST_FRAMEBUF	3
#define	LF_DST_DFMEM	4

#define LF_DF_CREATE_COMPRESS		0x0001
#define LF_DF_CREATE_WRITE		0x0002
#define LF_DF_COMPRESS_WRITE		0x0004
#define LF_DF_READ_COMPRESS		0x0008
#define LF_DF_CREATE_COMPRESS_WRITE	0x0010
#define LF_DF_READ_COMPRESS_WRITE	0x0020
#define LF_DF_DRAW			0x0040
#define LF_DF_CREATE			0x0080
#define LF_DF_COMPRESS			0x0100
#define LF_DF_WRITE			0x0200
#define LF_DF_READ			0x0400
#define	LF_DF_CREATE_COMPRESS_TRAIN	0x0800
#define	LF_DF_READ_COMPRESS_TRAIN	0x1000
#define	LF_DF_COMPRESS_TRAIN		0x2000

#define	LF_DF_FIELD	0
#define	LF_DF_SLAB	1
#define	LF_DF_SLICE_ST	2

/* handy aliases for addressing dataflow position array */
#define	LF_SLAB_POS	0
#define	LF_U_POS	1
#define	LF_V_POS	2
#define	LF_S_POS	3
#define	LF_T_POS	4

/* definition related to filtering */
#define	LF_LERP_UV	0x1
#define LF_LERP_ST	0x2

/* padding definitions */
#define	LF_PAD_FIRST	0x0
#define	LF_PAD_LAST	0x1

/* types */
typedef struct {
    short u, v, s, t;
} LFPixelUVST;

typedef struct {
    unsigned char a, b, g, r;
} LFPixelRGBA8;

typedef struct {
    float ox, oy, oz, ow;
    float x, y, z, w;
    float s, t;
} LFVtx;

typedef struct {
    float x, y;		/* screen space vertex coordinates */
    /* here w and q are really 1/w and 1/q */
    float z1, w;	/* z and w with respect to quad UV */
    float z2, q;	/* z and w with respect to quad ST */
    /* here u, v, s, and t are really u/w, v/w, s/q, and t/q */
    float u, v, s, t;
} LFScreenVtx;

typedef struct {
    int id;
    int size;			/* size of the code book */
    int ubits, vbits, sbits, tbits;
    int intern_format;		/* internal format for code words */
    float train_pct;		/* percentage of training set */
    void *codebook;
    int type;			/* what type of data in codebook */
#define	LF_VQ_LINEAR_DATA	0x0
#define	LF_VQ_LINEAR_VQ		0x1
#define	LF_VQ_TREE_VQ		0x2
    int sample_size;		/* per [u,v,s,t] sample size */
}  LFVQCodebook;

/*
 * the following struture contains information that might be 
 * shared by multiple slabs (for example, all slabs in a lightfield).
 */
typedef struct {
    /* current dimensions of u, v, s, t */
    int nu, nv, ns, nt;
    int u_frac_bits, v_frac_bits, s_frac_bits, t_frac_bits;
    int *uindex, *vindex, *sindex, *tindex;
    /* how (u, v, s, t) is translated into tile indices */
    int *vqtile_uindex, *vqtile_vindex, *vqtile_sindex, *vqtile_tindex;
    LFVQCodebook *vq;		/* vq codebook */
    int intern_format;		/* internal format for lightfield data */
    int sample_size;		/* per [u,v,s,t] sample size */
    int flag;			/* flag */
#define	LF_SLAB_VALID_AUX	0x1
}  LFShared;

typedef struct {
    int id;
    LFVtx uv[4];
    LFVtx st[4];
    void *lightfield;
    int sample_size;
    LFShared *shared;
} LFSlab;

typedef struct LFField {
    LFSlab *slabs[LF_MAXSLABS];
    int nslabs;
    unsigned enable_msk;	/* how many slabs are currently enabled */
    LFShared *shared;
    int id;
    struct LFField *next;
} LFField;

#define	__LF_SLAB_ACTIVE(field,i)		\
	(field->enable_msk & (1 << i))
#define	__LF_SET_SLAB_ACTIVE(field,i,active)	\
	(field->enable_msk = (field->enable_msk	\
		& ~(1 << (i))) | ((active) << i))

typedef struct {
    /* external structure */
    int filter;
    int format;
    int type;
#ifdef	STATS
    float draw_time;
    int scanline_clamped;
    int scanline_total;
#endif

    /* internal structure */
    int tile_u_bits, tile_v_bits, tile_s_bits, tile_t_bits;
    int pixel_size;
    int x, y;
    int width, height; 
    Matrix4 mat_model;
    Matrix4 mat_proj;
    Viewport viewport;
    LFPixelUVST *sample_buffer;
    void *pixel_buffer;
    int sample_buffer_size;
    int pixel_buffer_size;
} LFView;

/* an operation applied to a lightfield */
typedef struct LFOp {
    int op_id;
    int op_method;
    float *op_descr;
    void *op_extra;
} LFOp;

#define	LF_MAX_OPS		10
#define	LF_MAX_INTERNOPS	LF_MAX_OPS

struct LFOps;			/* forward declaration */

typedef struct LFInternOp {
    int op_id;
    int op_method;
    float *op_descr;
    void *op_extra;

    /* really an operator on the data flow */
    void (*op_func)(LFSlab ** /*slabs*/, int /*nslab*/, const int * /*pos*/,
	const float * /*descr*/, const void * /*input*/, int /*input_size*/,
	void * /*output*/, int /*output_size*/, void * /*aux_ptr*/);

    /* functions to call after data flow ends */
    void (*op_end)(struct LFOps * /*ops*/, struct LFInternOp * /*op*/, void * /*aux_ptr*/);

    /*
     * update function for next loop, this hides the internal looping and
     * I/O mechanism of dataflow from the operator
     */
    void (*op_update)(LFSlab ** /*slabs*/, int /*nslab*/, struct LFInternOp * /*op*/);
    int op_loop;
    int op_pos[7];

    /* pre-determined I/O memory and size for data flow of each operator */
    char *op_input_alloced;
    char *op_input;
    int op_input_size;
    char *op_output;
    int op_output_size;

    /* auxiliary data structure hanging off an internal operator */
    void *op_aux;
} LFInternOp;

/* stuff related to lfBegin/lfEnd */
typedef struct LFOps {
    /* stuff describing slabs involved for lfBegin/lfEnd */
    LFSlab *op_slabs[LF_MAXSLABS];
    int op_slab_cnt;

    /* stuff describing an external operator chain */
    float *op_hints;
    LFOp ops[LF_MAX_OPS];
    int op_cnt;

    /* stuff describing an internal operator chain */
    int chain_src;		/* LF_SRC_SLABMEM/LF_SRC_FILE/LF_SRC_CREATE/LF_SRC_MEM */
    int chain_dst;		/* LF_DST_SLABMEM/LF_DST_FILE/LF_SRC_MEM/LF_FRAMEBUF */
    int chain_flow;		/* LF_DF_VOL4D, LF_DF_SLICE_ST */
    int chain_type;		/* what type of chain is it */
    LFInternOp chain_ops[LF_MAX_INTERNOPS];
    int chain_cnt;
    int chain_pos[7];		/* current position in terms of source data */
    void *chain_input;		/* source memory */
    void *chain_output;		/* destination memory */
    void *chain_mem;		/* temporary memory in data-flow analysis */
    int chain_msk;		/* various mask */
#define	LF_SHARED		0x01
#define	LF_GEN_VQ_TRAINSET	0x02
#define	LF_GEN_VQ_CODEARRAY	0x04
#define	LF_SET_SLAB		0x08
#define	LF_NEW_SLAB		0x10
#define	LF_GEN_VQ_CODEBOOK	0x20
} LFOps;

typedef struct {
    LFOps ops;
    LFField *field;
    LFField *lf_list;
    LFView view;
#ifdef sgi
    struct itimerval timer;
#endif
} LFContext;

typedef struct {
    void (*get_transform)(Matrix4 /*mat_model*/, Matrix4 /*mat_proj*/,
	int * /*x*/, int * /*y*/, int * /*width*/, int * /*height*/);
    void (*write_pixrect)(int /*x*/, int /*y*/, int /*w*/, int /*h*/,
	int /*format*/, int /*type*/, void * /*pixels*/);
} LFDriver;

/* macros */

#define	__LF_GET_CONTEXT()	__lf_context
#define	__LF_GET_DRIVER()	__lf_driver

/* external variables */
extern LFContext	*__lf_context;
extern LFDriver		*__lf_driver;
extern LFSlab		*__lf_descr_slabs;
extern bool_t          lfVerbose;

/* prototypes */
extern LFField
    *lfReadField(char *name);

extern float
    lfEndTimer(void);

extern int
    log2(int /*size*/);
extern bool_t 
    lfSlabIntersecting(LFVtx /*uv*/[4], LFVtx /*st*/[4],
	const Matrix4 /*model*/, const Matrix4 /*proj*/,
	LFScreenVtx * /*inter*/, int /*ninter*/[2]);

extern void
    *lfBeginCreate(LFOps * /*ops*/, LFInternOp * /*op*/),
    *lfBeginDrawFB(LFOps * /*ops*/, LFInternOp * /*op*/),
    *lfBeginMemShuffle(LFOps * /*ops*/, LFInternOp * /*op*/),
    *lfBeginReadLif(LFOps * /*ops*/, LFInternOp * /*op*/),
    *lfBeginReadSliceFile(LFOps * /*ops*/, LFInternOp * /*op*/),
    *lfBeginWriteLif(LFOps * /*ops*/, LFInternOp * /*op*/),
    lfDataflowAnalysis(LFOps * /*ops*/),
    lfDrawFB(LFSlab ** /*slabs*/, int /*nslab*/, const int * /*pos*/,
	const float * /*descr*/, const void * /*input*/, int /*input_size*/,
	void * /*output*/, int /*output_size*/, void * /*aux_ptr*/),
    lfDrawFieldFramebuffer(LFField * /*field*/),
    lfEndDefault(LFOps * /*ops*/, LFInternOp * /*op*/, void * /*aux_ptr*/),
    lfEndDrawFB(LFOps * /*ops*/, LFInternOp * /*op*/, void * /*aux_ptr*/),
    lfEndReadLif(LFOps * /*ops*/, LFInternOp * /*op*/, void * /*aux_ptr*/),
    lfEndReadSliceFile(LFOps * /*ops*/, LFInternOp * /*op*/, void * /*aux_ptr*/),
    lfEndVQCompress(LFOps * /*ops*/, LFInternOp * /*op*/, void * /*aux_ptr*/),
    lfEndWriteLif(LFOps * /*ops*/, LFInternOp * /*op*/, void * /*aux_ptr*/),
    lfError(char * /*fmt*/, ...),
    lfGetParameters(int /*op_name*/, int /*pname*/, float * /*param*/),
    lfLinearizeVQCodebook(LFVQCodebook * /*vq*/),
    lfMakeIndices(LFShared * /*shared*/),
    lfMemInputSlice(LFSlab ** /*slabs*/, int /*nslab*/, const int * /*pos*/,
	const float * /*descr*/, const void * /*input*/, int /*input_size*/,
	void * /*output*/, int /*output_size*/, void * /*aux_ptr*/),
    lfMemShuffle(LFSlab ** /*slabs*/, int /*nslab*/, const int * /*pos*/,
	const float * /*descr*/, const void * /*input*/, int /*input_size*/,
	void * /*output*/, int /*output_size*/, void * /*aux_ptr*/),
    lfOutput(char * /*fmt*/, ...),
    lfPadData(LFSlab * /*slab*/, int /*pad_type*/, int /*new_size*/, int /*mode*/),
    lfRasterizeTriPntSample(const LFSlab * /*slab*/, const LFScreenVtx * /*v0*/,
	const LFScreenVtx * /*v1*/, const LFScreenVtx * /*v2*/),
    lfReadLif(LFSlab ** /*slabs*/, int /*nslab*/, const int * /*pos*/,
	const float * /*descr*/, const void * /*input*/, int /*input_size*/,
	void * /*output*/, int /*output_size*/, void * /*aux_ptr*/),
    lfReadSliceFile(LFSlab ** /*slabs*/, int /*nslab*/, const int * /*pos*/,
	const float * /*descr*/, const void * /*input*/, int /*input_size*/,
	void * /*output*/, int /*output_size*/, void * /*aux_ptr*/),
    lfResampleRGBA8(const LFSlab * /*slab*/, LFPixelUVST * /*scanline*/,
	int /*x*/, int /*y*/, int /*w*/),
    lfResampleVQRGBA8(const LFSlab * /*slab*/, LFPixelUVST * /*scanline*/,
	int /*x*/, int /*y*/, int /*w*/),
    lfSerialUpdateBlockPoint(LFSlab ** /*slabs*/, int /*nslab*/,
	LFInternOp * /*op*/),
    lfSerialUpdateBlockSlice(LFSlab ** /*slabs*/, int /*nslab*/,
	LFInternOp * /*op*/),
    lfSerialUpdatePoint(LFSlab ** /*slabs*/, int /*nslab*/, LFInternOp * /*op*/),
    lfSetParameters(int /*op_name*/, float * /*param_list*/),
    lfStartTimer(void),
    lfUpdateNoop(LFSlab ** /*slabs*/, int /*nslab*/, LFInternOp * /*op*/),
    lfVQGenCodebook(LFSlab ** /*slabs*/, int /*nslab*/, const int * /*pos*/,
	const float * /*descr*/, const void * /*input*/, int /*input_size*/,
	void * /*output*/, int /*output_size*/, void * /*aux_ptr*/),
    lfVQCompress(LFSlab ** /*slabs*/, int /*nslab*/, const int * /*pos*/,
	const float * /*descr*/, const void * /*input*/, int /*input_size*/,
	void * /*output*/, int /*output_size*/, void * /*aux_ptr*/),
    lfWriteLif(LFSlab ** /*slabs*/, int /*nslab*/, const int * /*pos*/,
	const float * /*descr*/, const void * /*input*/, int /*input_size*/,
	void * /*output*/, int /*output_size*/, void * /*aux_ptr*/);

#include "lf_funcs.h"
#include "fileIO.h"

#endif	/* ! LF_UTIL_H */
