/*
 *  win.c
 *
 *  GLT window routines, specialized for Windows NT
 *
 *  Kekoa Proudfoot
 *  6/19/98, 6/21/98
 */

/* Include files */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <windows.h>
#include <GL/gl.h>

#include "glt.h"
#include "win.h"

/* Type definitions */

#define MAX_CONTEXTS 32
#define MAX_WINDOWS 32

typedef struct {
    int          id;
    int          visualinfo;
    HGLRC        context;
} GLT_context;

typedef struct {
    int          id;
    int          width;
    int          height;
    HWND         window;
    /* Hack? */
    int          context_list[MAX_CONTEXTS + 1];
} GLT_window;

/* Global variables */

static char *windownamepattern = NULL;
static GLT_context contexts[MAX_CONTEXTS + 1];
static GLT_window windows[MAX_WINDOWS + 1];
static int curcontext;
static int curwindow;

/* Function prototypes */

LONG WINAPI wndproc(HWND hwnd, UINT umsg, WPARAM wparam, LPARAM lparam);

/* Macros */

#define DEFAULT_WINDOW_STYLE ( \
    WS_OVERLAPPED | \
    WS_CAPTION | \
    WS_SYSMENU | \
    WS_MINIMIZEBOX | \
    WS_CLIPSIBLINGS | \
    WS_CLIPCHILDREN \
)

/* Utility functions */

static const char *
error_string( int err )
{
    static char buf[256];
    FormatMessage( FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_MAX_WIDTH_MASK,
                   NULL, err, MAKELANGID(LANG_NEUTRAL,SUBLANG_DEFAULT),
                   buf, sizeof(buf) - 1, NULL );
    return buf;
}

static void
evaluate_window_name( GLT_window *win, char *buf )
{
    char *src, *dst;

    if ( windownamepattern )
        src = windownamepattern;
    else
        src = "glplay %w [ctx %c]";

    dst = buf;

    while ( *src ) {
        if ( *src == '%' ) {
            src++;
            switch ( *src ) {

              case 'w':
                dst += sprintf( dst, "%d", win->id );
                break;

              case 'c':
                {
                    int i, n;
                    n = 0;
                    for ( i = 0; i <= MAX_CONTEXTS; i++ ) {
                        if ( win->context_list[i] ) {
                            if ( n )
                                *dst++ = ' ';
                            dst += sprintf( dst, "%d", i );
                            n++;
                        }
                    }
                }
                break;

              case '%':
                *dst++ = '%';
                break;

              default:
                *dst++ = '%';
                *dst++ = *src;
            }
            src++;
        } else {
            *dst++ = *src++;
        }
    }

    *dst = 0;
}

static void
set_window_name( GLT_window *win, char *name )
{
    SetWindowText( win->window, name );
}

static void
set_all_window_names( void )
{
    int  i;
    char buf[256];

    for ( i = 0; i <= MAX_WINDOWS; i++ ) {
        if ( windows[i].id ) {
            evaluate_window_name( &windows[i], buf );
            set_window_name( &windows[i], buf );
        }
    }
}

static int
get_visual(HWND window, GLT_create_context *data)
{
    PIXELFORMATDESCRIPTOR pfd, pfdd;
    int visual;

    memset(&pfd, 0, sizeof(pfd));

    pfd.nSize = sizeof(pfd);
    pfd.nVersion = 1;

    /* Hack? Always draw to window is an assumption, though a good one */
    pfd.dwFlags = PFD_SUPPORT_OPENGL | PFD_DRAW_TO_WINDOW;
    pfd.iPixelType = PFD_TYPE_COLORINDEX;
    pfd.cColorBits = 32;
    pfd.cDepthBits = 1;

    if (data->doublebuffer) pfd.dwFlags |= PFD_DOUBLEBUFFER;
    if (data->rgba) pfd.iPixelType = PFD_TYPE_RGBA;
    if (data->red) pfd.cRedBits = data->red;
    if (data->green) pfd.cGreenBits = data->green;
    if (data->blue) pfd.cBlueBits = data->blue;
    if (data->alpha) pfd.cAlphaBits = data->alpha;
    if (data->accumred || data->accumgreen || data->accumblue ||
        data->accumalpha) pfd.cAccumBits = 1;
    if (data->buffersize) pfd.cColorBits = data->buffersize;
    if (data->depth) pfd.cDepthBits = data->depth;
    if (data->stencil) pfd.cStencilBits = data->stencil;

    if (!(visual = ChoosePixelFormat(GetDC(window), &pfd)))
        return 0;

    if (!DescribePixelFormat(GetDC(window), visual, sizeof(pfd), &pfd))
        glt_fatal("Couldn't get pixel format descritption!");

    if (!(pfd.dwFlags & PFD_SUPPORT_OPENGL))
        return 0;

    if (!(pfd.dwFlags & PFD_DRAW_TO_WINDOW))
        return 0;

    return visual;
}

static void
describe_visual(HWND window, int visual, GLT_create_context *compare)
{
    PIXELFORMATDESCRIPTOR pfd;
    char *fmt = "    %-15s\t%d\t[%d]";
    int rgba, doublebuffer;

    if (!DescribePixelFormat(GetDC(window), visual, sizeof(pfd), &pfd))
        glt_fatal("Couldn't get pixel format descritption!");

    rgba = (pfd.iPixelType == PFD_TYPE_RGBA);
    doublebuffer = (pfd.dwFlags & PFD_DOUBLEBUFFER);

    glt_warn(fmt, "rgba", rgba, compare->rgba);
    glt_warn(fmt, "red", pfd.cRedBits, compare->red);
    glt_warn(fmt, "green", pfd.cGreenBits, compare->green);
    glt_warn(fmt, "blue", pfd.cBlueBits, compare->blue);
    glt_warn(fmt, "alpha", pfd.cAlphaBits, compare->alpha);
    glt_warn(fmt, "buffersize", pfd.cColorBits, compare->buffersize);
    glt_warn(fmt, "doublebuffer", doublebuffer, compare->doublebuffer);
    glt_warn(fmt, "depth", pfd.cDepthBits, compare->depth);
    glt_warn(fmt, "stencil", pfd.cStencilBits, compare->stencil);
    glt_warn(fmt, "accumred", pfd.cAccumRedBits, compare->accumred);
    glt_warn(fmt, "accumgreen", pfd.cAccumGreenBits, compare->accumgreen);
    glt_warn(fmt, "accumblue", pfd.cAccumBlueBits, compare->accumblue);
    glt_warn(fmt, "accumalpha", pfd.cAccumAlphaBits, compare->accumalpha);
}

static int
find_visual(HWND window, GLT_create_context *data)
{
    GLT_create_context tmp;
    int visual;
    int orig;

    visual = get_visual(window, data);

    /* Be less aggressive if no visual if at this point? */

    if (visual && (__glt_debug & GLT_DEBUG)) {
        glt_warn("Found format for context %d:", data->ctx);
        describe_visual(window, visual, data);
    }

    return visual;
}

static void
create_context(GLT_context *ctx, GLT_create_context *data)
{
    HINSTANCE hinstance = GetModuleHandle(NULL);
    WNDCLASS wndclass;
    HWND window;
    int visualinfo;
    PIXELFORMATDESCRIPTOR pfd;
    HGLRC context;

    if (!data->ctx)
        glt_fatal("Zero create context index!");

    if (!GetClassInfo(hinstance, "GLPLAY", &wndclass)) {
        wndclass.style = CS_OWNDC;
        wndclass.lpfnWndProc = (WNDPROC)wndproc;
        wndclass.cbClsExtra = 0;
        wndclass.cbWndExtra = 0;
        wndclass.hInstance = hinstance;
        wndclass.hIcon = LoadIcon(NULL, IDI_WINLOGO);
        wndclass.hCursor = LoadCursor(NULL, IDC_ARROW);
        wndclass.hbrBackground = NULL;
        wndclass.lpszMenuName = NULL;
        wndclass.lpszClassName = "GLPLAY";

        if (!RegisterClass(&wndclass))
            glt_fatal("Couldn't register winclass!");
    }

#if 0
    window = CreateWindow("GLPLAY", "glplay", DEFAULT_WINDOW_STYLE,
                          CW_USEDEFAULT, CW_USEDEFAULT, 1, 1, NULL, NULL,
                          hinstance, NULL);
#else
    window = CreateWindow("GLPLAY", "glplay", DEFAULT_WINDOW_STYLE,
                          10, 10, 1, 1, NULL, NULL,
                          hinstance, NULL);
#endif

    if (!window)
        glt_fatal("Couldn't create window!");
    if (!(visualinfo = find_visual(window, data)))
        glt_fatal("Couldn't find pixel format!");
    if (!DescribePixelFormat(GetDC(window), visualinfo, sizeof(pfd), &pfd))
        glt_fatal("Couldn't get pixel format description!");
    if (!(SetPixelFormat(GetDC(window), visualinfo, &pfd)))
        glt_fatal("Couldn't set pixel format!");
    if (!(context = wglCreateContext(GetDC(window))))
        glt_fatal("Couldn't create context!");
    if (!DestroyWindow(window))
        glt_fatal("Couldn't destroy temporary window!");

    /* Currently, no sharing done, but that would be done here if anywhere */

    ctx->id = data->ctx;
    ctx->visualinfo = visualinfo;
    ctx->context = context;
}

static void
destroy_context(GLT_context *ctx, GLT_destroy_context *data)
{
    /* Hack? Is this correct? */

    if (!wglDeleteContext(ctx->context))
        glt_fatal("Couldn't destroy context!");

    ctx->id = 0;
    ctx->visualinfo = 0;
    ctx->context = 0;
}

static void
create_window(GLT_window *win, int visualinfo, GLT_make_current *data)
{
    char buf[256];
    HINSTANCE hinstance = GetModuleHandle(NULL);
    PIXELFORMATDESCRIPTOR pfd;
    RECT rect = {0, 0, data->width, data->height};
    int width, height;
    HWND window;

    if (!data->win)
        glt_fatal("Zero create window index!");
    if (!data->width)
        glt_fatal("Zero create window width!");
    if (!data->height)
        glt_fatal("Zero create window height!");

    win->id = data->win;
    win->width = data->width;
    win->height = data->height;
    memset(win->context_list, 0, sizeof(win->context_list));

    evaluate_window_name( win, buf );

    /* Yuck. This imposes a minimum window size. */
    AdjustWindowRect(&rect, WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU |
                     WS_MINIMIZEBOX | WS_CLIPSIBLINGS | WS_CLIPCHILDREN,
                     FALSE);

    width = rect.right - rect.left;
    height = rect.bottom - rect.top;

    window = CreateWindow("GLPLAY", buf, WS_OVERLAPPED | WS_CAPTION |
                          WS_SYSMENU | WS_MINIMIZEBOX | WS_CLIPSIBLINGS |
                          WS_CLIPCHILDREN, CW_USEDEFAULT, CW_USEDEFAULT,
                          width, height, NULL, NULL, hinstance, NULL);

    win->window = window;

    if (!window)
        glt_fatal("Couldn't create window!");

    if (!DescribePixelFormat(GetDC(window), visualinfo, sizeof(pfd), &pfd))
        glt_fatal("Couldn't get pixel format description!");

    if (!(SetPixelFormat(GetDC(window), visualinfo, &pfd)))
        glt_fatal("Couldn't set pixel format!");

    /* Do we need to create or set a colormap under Windows NT? */

    ShowWindow(window, SW_SHOWNORMAL);
}

static void
resize_window(GLT_window *win, GLT_make_current *data)
{
    RECT rect = {0, 0, data->width, data->height};
    int width, height;

    if (!data->width)
        glt_fatal("Zero resize window width!");
    if (!data->height)
        glt_fatal("Zero resize window height!");

    /* Yuck. This imposes a minimum window size. */
    AdjustWindowRect(&rect, WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU |
                     WS_MINIMIZEBOX | WS_CLIPSIBLINGS | WS_CLIPCHILDREN,
                     FALSE);

    width = rect.right - rect.left;
    height = rect.bottom - rect.top;

    SetWindowPos(win->window, NULL, 0, 0, width, height, SWP_NOMOVE |
                 SWP_NOZORDER | SWP_NOACTIVATE);

    win->width = data->width;
    win->height = data->height;
}

static void
make_current(GLT_window *win, GLT_context *ctx, GLT_make_current *data)
{
    if ( data->fail ) {
        glt_warn("Make current has failed flag set, skipping call!");
        return;
    }
    /* Hack? */
    if (win->id && ctx->id) {
        char buf[256];
        win->context_list[ctx->id] = 1;
        evaluate_window_name( win, buf );
        set_window_name( win, buf );
    }
    if (!wglMakeCurrent(GetDC(win->window), ctx->context)) {
        int err = GetLastError( );
        glt_warn("Make current failed! %s", error_string( err ) );
    }
}

static void
swap_buffers(GLT_window *win)
{
    SwapBuffers(GetDC(win->window));
}

static LONG WINAPI
wndproc(HWND hwnd, UINT umsg, WPARAM wparam, LPARAM lparam)
{
    PAINTSTRUCT ps;
    int i;

    switch(umsg) {
    case WM_CREATE:
        return 0;
    case WM_SIZE:
        return 0;
    case WM_PAINT:
        BeginPaint(hwnd, &ps);
        EndPaint(hwnd, &ps);
        return 0;
    case WM_DESTROY:
        for (i = 0; i < MAX_WINDOWS; i++) {
            if (hwnd == windows[i].window)
                PostQuitMessage(0);
        }
        return 0;
    case WM_KEYDOWN:
        switch(wparam) {
        case VK_ESCAPE:
        case 'Q':
        case 'q':
            PostQuitMessage(0);
        }
        return 0;
    default:
        return DefWindowProc(hwnd, umsg, wparam, lparam);
    }
}

static void
process_events(void)
{
    MSG msg;

    while (PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE)) {
        if (GetMessage(&msg, NULL, 0, 0) <= 0)
            exit(0);
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }
}

/* Interface functions */

void
glt_set_window_name_pattern( const char *pattern )
{
    if ( !windownamepattern || strcmp( pattern, windownamepattern ) ) {
        if ( windownamepattern )
            glt_free( windownamepattern );
        windownamepattern = glt_malloc( strlen(pattern) + 1 );
        strcpy( windownamepattern, pattern );
    }

    set_all_window_names( );
}

void
__glt_create_context(GLT_create_context *data)
{
    int ctxid = data->ctx;
    GLT_context *ctx;
    if (ctxid > MAX_CONTEXTS)
        glt_fatal("Context index exceeded %d", MAX_CONTEXTS);
    ctx = &contexts[ctxid];
    if (ctx->id)
        glt_warn("Context %d already created!", ctx->id);
    /* Create context */
    create_context(ctx, data);
}

void
__glt_destroy_context(GLT_destroy_context *data)
{
    int ctxid = data->ctx;
    GLT_context *ctx;
    if (ctxid > MAX_CONTEXTS)
        glt_fatal("Context index exceeded %d", MAX_CONTEXTS);
    ctx = &contexts[ctxid];
    if (ctxid == 0 || ctx->id != ctxid)
        glt_fatal("Can't destroy invalid context %d", ctxid);
    /* Destroy context */
    destroy_context(ctx, data);
}

void
__glt_make_current(GLT_make_current *data)
{
    int winid = data->win;
    int ctxid = data->ctx;
    int winwidth = data->width;
    int winheight = data->height;
    GLT_window *win;
    GLT_context *ctx;
    if (winid > MAX_WINDOWS)
        glt_fatal("Window index exceeded %d", MAX_WINDOWS);
    if (ctxid > MAX_CONTEXTS)
        glt_fatal("Context index exceeded %d", MAX_CONTEXTS);
    win = &windows[winid];
    ctx = &contexts[ctxid];
    if (winid && !win->id)
        /* Create window */
        create_window(win, ctx->visualinfo, data);
    if (win->id != winid)
        glt_fatal("Make current with invalid window %d!", winid);
    if (ctx->id != ctxid)
        glt_fatal("Make current with invalid context %d!", ctxid);
    if (winid && (win->width != winwidth || win->height != winheight)) {
        /* Resize window */
        resize_window(win, data);
    }
    if (curwindow != winid || curcontext != ctxid) {
        /* Make current */
        glt_mesg_va(GLT_DEBUG, ("Making context %d current to window %d",
                                ctxid, winid));
        make_current(win, ctx, data);
    }
    curwindow = winid;
    curcontext = ctxid;
}

void
__glt_swap_buffers(void)
{
    GLT_window *win = &windows[curwindow];
    if (curwindow == 0 || win->id != curwindow)
        glt_fatal("Swap buffers with invalid current window!");
    /* Swap buffers */
    swap_buffers(win);
}

void
__glt_process_events(void)
{
    /* Process events */
    process_events();
}
