/*
 *  clparse.c
 *
 *  A second attempt at making a generic command line parser
 *  See clparsemain.c for some example code.
 *
 *  Kekoa Proudfoot
 *  10/7/97
 */

/*****************************************************************************
 * Include files
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include "clparse.h"

/*****************************************************************************
 * Static global variables
 *****************************************************************************/

static int    argc     = 0;
static char **argv     = { NULL };
static opt_t *opts     = { NULL };
static int    nopts    = 0;

static int    pos      = 1;
static int    error    = 0;
static char  *lastopt  = NULL;
static char  *progname = NULL;

/*****************************************************************************
 * Static function definitions
 *****************************************************************************/

static void
clp_missingparam(void)
{
    if (lastopt == NULL)
	clp_fail("missing parameter");
    else
	clp_fail("missing parameter for %s", lastopt);
}

static void
clp_invalidparam(char *param)
{
    if (lastopt == NULL)
	clp_fail("%s: invalid parameter", param);
    else
	clp_fail("%s: invalid parameter for %s", param, lastopt);
}

/*****************************************************************************
 * Interface function definitions
 *****************************************************************************/

void
clp_init(int _argc, char **_argv, opt_t *_opts, int _nopts)
{
    argc     = _argc;
    argv     = _argv;
    opts     = _opts;
    nopts    = _nopts;

    pos      = 1;
    error    = 0;
    lastopt  = NULL;
    progname = argv[0];
}

int
clp_getopt(void)
{
    int i;

    if (pos >= argc)
	return -2;

    lastopt = argv[pos];

    if (lastopt[0] != '-')
	return -2;

    pos++;

    if (lastopt[1] == '-') {
	return -2;
    }

    for (i = 0; i < nopts; i++) {
	if (!strcmp(opts[i].option, &lastopt[1]))
	    return opts[i].index;
    }

    clp_fail("unrecognized option %s", lastopt);

    return -1;
}

int
clp_getint(void)
{
    char  *lastchar;
    int    value;

    if (pos >= argc) {
	clp_missingparam();
	return 0;
    }

    value = strtol(argv[pos], &lastchar, 10);

    if (lastchar == argv[pos]) {
	clp_invalidparam(argv[pos]);
	pos++;
	return 0;
    }
    else {
	pos++;
	return value;
    }
}

int
clp_maybegetint(int failvalue)
{
    char  *lastchar;
    int    value;

    if (pos >= argc)
	return failvalue;

    value = strtol(argv[pos], &lastchar, 10);

    if (lastchar == argv[pos])
	return failvalue;
    else {
	pos++;
	return value;
    }
}

float
clp_getfloat(void)
{
    char  *lastchar;
    float  value;

    if (pos >= argc) {
	clp_missingparam();
	return 0.0f;
    }

    value = strtod(argv[pos], &lastchar);

    if (lastchar == argv[pos]) {
	clp_invalidparam(argv[pos]);
	pos++;
	return 0.0f;
    }
    else {
	pos++;
	return value;
    }
}

float
clp_maybegetfloat(float failvalue)
{
    char  *lastchar;
    float  value;

    if (pos >= argc)
	return failvalue;

    value = strtod(argv[pos], &lastchar);

    if (lastchar == argv[pos])
	return failvalue;
    else {
	pos++;
	return value;
    }
}

char *
clp_getstring(void)
{
    if (pos >= argc) {
	clp_missingparam();
	return "";
    }

    return argv[pos++];
}

char *
clp_maybegetstring(char *failvalue)
{
    if (pos >= argc)
	return failvalue;
    else
	return argv[pos++];
}

int
clp_getpos(void)
{
    return pos;
}

int
clp_geterror(void)
{
    return error;
}

void
clp_fail(char *format, ...)
{
    va_list  list;
    
    if (progname)
	fprintf(stderr, "%s: ", argv[0]);

    va_start(list, format);
    vfprintf(stderr, format, list);
    va_end(list);

    putc('\n', stderr);

    error = 1;
}

