/*
 *  state.h
 *
 *  Header file GL state tracking routines.
 *
 *  Kekoa Proudfoot
 *  2/17/99
 */

#ifndef STATE_H_DEFINED
#define STATE_H_DEFINED

#include <glt.h>

typedef unsigned char GL_bool;
typedef int GL_int;
typedef unsigned int GL_uint;
typedef float GL_float;
typedef void GL_void;

typedef struct {
    GL_float x, y, z, w;
} GL_coord;

typedef struct {
    GL_float s, t, r, q;
} GL_texcoord;

typedef struct {
    GL_float x, y, z;
} GL_vector;

typedef struct {
    GL_float r, g, b, a;
} GL_color;

typedef struct {
    GL_color a, d, s, e;
    GL_float sh;
} GL_material;

typedef struct {
    GL_color a, d, s;
    GL_coord p;
    GL_vector sd;
    GL_float se, sc;
    GL_float ac, al, aq;
} GL_light;

typedef struct {
    GL_int x, y, w, h;
} GL_rect;

typedef struct {
    GL_float m00, m01, m02, m03;
    GL_float m10, m11, m12, m13;
    GL_float m20, m21, m22, m23;
    GL_float m30, m31, m32, m33;
} GL_matrix;

typedef struct {
    GL_bool valid;
    GL_int width;
    GL_int height;
    GL_int components;
    GL_int border;
    GL_int format;
    GL_int type;
    GL_int n_pixels;
    GL_void *pixels;
    GL_bool swap_bytes;
    GL_bool lsb_first;
} GL_texlevel;

#define NUM_TEX_LEVELS 12

typedef struct {
    GL_int min_filter;
    GL_int mag_filter;
    GL_int wrap_s;
    GL_int wrap_t;
    GL_color border_color;
    GL_texlevel levels[NUM_TEX_LEVELS];
} GL_texture;

#define NUM_CLIP_PLANES 6
#define NUM_LIGHTS 8
#define MODELVIEW_STACK_DEPTH 64
#define PROJECTION_STACK_DEPTH 8
#define TEXTURE_STACK_DEPTH 8

enum {
    ALPHA_TEST_ENABLE = 0,
    BLEND_ENABLE,
    CLIP_PLANE0_ENABLE,
    CLIP_PLANE1_ENABLE,
    CLIP_PLANE2_ENABLE,
    CLIP_PLANE3_ENABLE,
    CLIP_PLANE4_ENABLE,
    CLIP_PLANE5_ENABLE,
    COLOR_MATERIAL_ENABLE,
    CULL_FACE_ENABLE,
    DEPTH_TEST_ENABLE,
    DITHER_ENABLE,
    FOG_ENABLE,
    LIGHT0_ENABLE,
    LIGHT1_ENABLE,
    LIGHT2_ENABLE,
    LIGHT3_ENABLE,
    LIGHT4_ENABLE,
    LIGHT5_ENABLE,
    LIGHT6_ENABLE,
    LIGHT7_ENABLE,
    LIGHTING_ENABLE,
    LINE_SMOOTH_ENABLE,
    LINE_STIPPLE_ENABLE,
    LOGIC_OP_ENABLE,
    NORMALIZE_ENABLE,
    POLYGON_OFFSET_EXT_ENABLE,
    POINT_SMOOTH_ENABLE,
    POLYGON_STIPPLE_ENABLE,
    SCISSOR_TEST_ENABLE,
    STENCIL_TEST_ENABLE,
    TEXTURE_1D_ENABLE,
    TEXTURE_2D_ENABLE,
    NUM_ENABLES
};

typedef struct {
    GL_color color;
    GL_vector normal;
    GL_texcoord texcoord;
    GL_material front_material;
    GL_material back_material;
    GL_bool edge_flag;
    GL_coord raster_pos;
    GL_float index;

    GL_color clear_accum;
    GL_color clear_color;
    GL_float clear_depth; /* Hack? This should be a double */
    GL_float clear_index;
    GL_int clear_stencil;

    GL_int color_material_face;
    GL_int color_material_mode;

    GL_int alpha_func;
    GL_float alpha_ref;
    GL_int blend_src;
    GL_int blend_dst;
    GL_int depth_func;
    GL_int stencil_func;
    GL_int stencil_ref;
    GL_uint stencil_mask;
    GL_int stencil_op_fail;
    GL_int stencil_op_zfail;
    GL_int stencil_op_zpass;

    GL_coord clip_planes[NUM_CLIP_PLANES]; /* Hack! These should use doubles */

    GL_rect viewport;
    GL_float depth_range_near;
    GL_float depth_range_far;

    GL_rect scissor;

    GL_float point_size;
    GL_float line_width;

    GL_int cull_face;
    GL_int front_face;
    GL_int shade_model;
    GL_int draw_buffer;
    GL_int front_polygon_mode;
    GL_int back_polygon_mode;
    GL_float polygon_offset_factor;
    GL_float polygon_offset_bias;

    GL_int fog_mode;
    GL_float fog_density;
    GL_float fog_start;
    GL_float fog_end;
    GL_float fog_index;
    GL_color fog_color;

    GL_bool color_write_r;
    GL_bool color_write_g;
    GL_bool color_write_b;
    GL_bool color_write_a;
    GL_bool depth_write;
    GL_uint index_mask;
    GL_uint stencil_wmask;

    GL_int matrix_mode;
    GL_int modelview_stack_index;
    GL_matrix modelview_stack[MODELVIEW_STACK_DEPTH];
    GL_int projection_stack_index;
    GL_matrix projection_stack[PROJECTION_STACK_DEPTH];
    GL_int texture_stack_index;
    GL_matrix texture_stack[TEXTURE_STACK_DEPTH];

    GL_color light_ambient_color;
    GL_bool light_local_viewer;
    GL_bool light_two_side;
    GL_light lights[NUM_LIGHTS];

    GL_float pixel_zoom_x;
    GL_float pixel_zoom_y;

    GL_bool pack_swap_bytes;
    GL_bool pack_lsb_first;
    GL_int pack_row_length;
    GL_int pack_skip_pixels;
    GL_int pack_skip_rows;
    GL_int pack_alignment;

    GL_bool unpack_swap_bytes;
    GL_bool unpack_lsb_first;
    GL_int unpack_row_length;
    GL_int unpack_skip_pixels;
    GL_int unpack_skip_rows;
    GL_int unpack_alignment;

    GL_bool transfer_map_color;
    GL_bool transfer_map_stencil;
    GL_int transfer_index_shift;
    GL_int transfer_index_offset;
    GL_float transfer_red_scale;
    GL_float transfer_red_bias;
    GL_float transfer_green_scale;
    GL_float transfer_green_bias;
    GL_float transfer_blue_scale;
    GL_float transfer_blue_bias;
    GL_float transfer_alpha_scale;
    GL_float transfer_alpha_bias;
    GL_float transfer_depth_scale;
    GL_float transfer_depth_bias;

    GL_bool enable_flags[NUM_ENABLES];

    GL_int fog_hint;
    GL_int line_smooth_hint;
    GL_int perspective_correction_hint;
    GL_int point_smooth_hint;
    GL_int polygon_smooth_hint;

    GL_uint list_base;

    GL_int tex_env_mode;
    GL_color tex_env_color;

    GL_int n_textures;
    GL_texture **textures;
    GL_int texture;
} GL_context;

extern void gs_init(GL_context *ctx);
extern int gs_update(GL_context *ctx, GLT_opcode op, GLT_data *data);
extern void gs_put(GLT_trace *trace, GL_context *ctx);

#endif /* STATE_H_DEFINED */
