/*
 *  state.c
 *
 *  Routine to track GL state given GLT data.
 *
 *  Kekoa Proudfoot
 *  2/17/99
 */

/* Hack? Maybe change some glt_warns to set rval=0 */

#include <math.h>
#include <string.h>

/* For GL defines */
#ifdef WIN32
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif
#include <GL/gl.h>

#include <glt.h>

#include "state.h"

/* Conversion macros */

#define VINT(x)      ((GL_float)(x))
#define VSHORT(x)    ((GL_float)(x))
#define VFLOAT(x)    ((GL_float)(x))
#define VDOUBLE(x)   ((GL_float)(x))

#define TSHORT(x)    ((GL_float)(x))
#define TINT(x)      ((GL_float)(x))
#define TFLOAT(x)    ((GL_float)(x))
#define TDOUBLE(x)   ((GL_float)(x))

#define NBYTE(x)     ((((GL_float)(x)) + 0.5f) * (1.0f / 127.5f))
#define NSHORT(x)    ((((GL_float)(x)) + 0.5f) * (1.0f / 32767.5f))
#define NINT(x)      ((((GL_float)(x)) + 0.5f) * (1.0f / 2.14748365e+09f))
#define NFLOAT(x)    ((GL_float)(x))
#define NDOUBLE(x)   ((GL_float)(x))

#define CBYTE(x)     ((((GL_float)(x)) + 0.5f) * (1.0f / 127.5f))
#define CSHORT(x)    ((((GL_float)(x)) + 0.5f) * (1.0f / 32767.5f))
#define CINT(x)      ((((GL_float)(x)) + 0.5f) * (1.0f / 2.14748365e+09f))
#define CUBYTE(x)    (((GL_float)(x)) * (1.0f / 255.0f))
#define CUSHORT(x)   (((GL_float)(x)) * (1.0f / 65535.0f))
#define CUINT(x)     (((GL_float)(x)) * (1.0f / 4.29496729e+09f))
#define CFLOAT(x)    ((GL_float)(x))
#define CDOUBLE(x)   ((GL_float)(x))

/* Other macros */

#ifndef NDEBUG
#define ONCE(cmd) \
    do { \
        static int once = 0; \
        if (!once && ++once) cmd ; \
    } while (0)
#else
#define ONCE(cmd)
#endif

/* Global variables */

static char *updatef = "gs_update";

/* Ignore function */

static void
ignore(char *loc, char *func)
{
    glt_warn("%s: ignoring %s", loc, func);
}

/* Matrix ops */

#define FLOAT_ONE  1.0f
#define FLOAT_ZERO 0.0f

#ifdef IRIX
#define Sin(x) fsin(x)
#define Cos(x) fcos(x)
#define Sqrt(x) fsqrt(x)
#else
#define Sin(x) sin(x)
#define Cos(x) cos(x)
#define Sqrt(x) sqrt(x)
#endif
#define Len(x,y,z) Sqrt((x)*(x)+(y)*(y)+(z)*(z))

#define DEG_TO_RAD_MULT 0.0174532925f

static GL_matrix m_identity = {
    1.0, 0.0, 0.0, 0.0,
    0.0, 1.0, 0.0, 0.0,
    0.0, 0.0, 1.0, 0.0,
    0.0, 0.0, 0.0, 1.0
};

static int
m_is_identity(GL_matrix *m)
{
    return (memcmp(m, &m_identity, sizeof(m_identity)) == 0);
}

static __inline GL_matrix *
m_cur_matrix(GL_context *ctx)
{
    static GL_matrix temp;
    switch (ctx->matrix_mode) {
    case GL_MODELVIEW:
        return &ctx->modelview_stack[ctx->modelview_stack_index];
    case GL_PROJECTION:
        return &ctx->projection_stack[ctx->projection_stack_index];
    case GL_TEXTURE:
        return &ctx->texture_stack[ctx->texture_stack_index];
    default:
        /* error already reported in MATRIX_MODE */
        return &temp;
    }
}

static __inline GL_int *
m_cur_matrix_index(GL_context *ctx)
{
    static GL_int temp;
    switch (ctx->matrix_mode) {
    case GL_MODELVIEW:
        return &ctx->modelview_stack_index;
    case GL_PROJECTION:
        return &ctx->projection_stack_index;
    case GL_TEXTURE:
        return &ctx->texture_stack_index;
    default:
        /* error already reported in MATRIX_MODE */
        return &temp;
    }
}

static __inline GL_int
m_cur_matrix_max_index(GL_context *ctx)
{
    switch (ctx->matrix_mode) {
    case GL_MODELVIEW:
        return MODELVIEW_STACK_DEPTH;
    case GL_PROJECTION:
        return PROJECTION_STACK_DEPTH;
    case GL_TEXTURE:
        return TEXTURE_STACK_DEPTH;
    default:
        /* error already reported in MATRIX_MODE */
        return 4096; /* allow something ridiculous */
    }
}

#define m_dot(l,r,i,j) \
    ((l)->m##i##0 * (r)->m0##j + (l)->m##i##1 * (r)->m1##j + \
     (l)->m##i##2 * (r)->m2##j + (l)->m##i##3 * (r)->m3##j)

static void
m_mult(GL_matrix *p, GL_matrix *l, GL_matrix *r)
{
    p->m00 = m_dot(l, r, 0, 0);
    p->m01 = m_dot(l, r, 0, 1);
    p->m02 = m_dot(l, r, 0, 2);
    p->m03 = m_dot(l, r, 0, 3);
    p->m10 = m_dot(l, r, 1, 0);
    p->m11 = m_dot(l, r, 1, 1);
    p->m12 = m_dot(l, r, 1, 2);
    p->m13 = m_dot(l, r, 1, 3);
    p->m20 = m_dot(l, r, 2, 0);
    p->m21 = m_dot(l, r, 2, 1);
    p->m22 = m_dot(l, r, 2, 2);
    p->m23 = m_dot(l, r, 2, 3);
    p->m30 = m_dot(l, r, 3, 0);
    p->m31 = m_dot(l, r, 3, 1);
    p->m32 = m_dot(l, r, 3, 2);
    p->m33 = m_dot(l, r, 3, 3);
}

static void
m_multiply(GL_matrix *m1, GL_matrix *m2)
{
    GL_matrix left = *m1;
    m_mult(m1, &left, m2);
}

static void
m_rotate(GL_matrix *m, GL_float angle, GL_float x, GL_float y, GL_float z)
{
    GL_matrix left = *m;
    GL_matrix right;
    GL_float invlen, ang, s, c;

    if (x == FLOAT_ZERO && y == FLOAT_ZERO && z == FLOAT_ZERO)
        z = FLOAT_ONE;

    invlen = FLOAT_ONE / Len(x, y, z);
    ang = angle * DEG_TO_RAD_MULT;
    s = Sin(ang);
    c = Cos(ang);

    x *= invlen;
    y *= invlen;
    z *= invlen;

    right.m00 = x * x * (FLOAT_ONE - c) + c;
    right.m01 = y * x * (FLOAT_ONE - c) - z * s;
    right.m02 = z * x * (FLOAT_ONE - c) + y * s;
    right.m03 = FLOAT_ZERO;

    right.m10 = x * y * (FLOAT_ONE - c) + z * s;
    right.m11 = y * y * (FLOAT_ONE - c) + c;
    right.m12 = z * y * (FLOAT_ONE - c) - x * s;
    right.m13 = FLOAT_ZERO;

    right.m20 = x * z * (FLOAT_ONE - c) - y * s;
    right.m21 = y * z * (FLOAT_ONE - c) + x * s;
    right.m22 = z * z * (FLOAT_ONE - c) + c;
    right.m23 = FLOAT_ZERO;

    right.m30 = FLOAT_ZERO;
    right.m31 = FLOAT_ZERO;
    right.m32 = FLOAT_ZERO;
    right.m33 = FLOAT_ONE;

    m_mult(m, &left, &right);
}

static void
m_translate(GL_matrix *m, GL_float x, GL_float y, GL_float z)
{
    m->m03 += m->m00 * x + m->m01 * y + m->m02 * z;
    m->m13 += m->m10 * x + m->m11 * y + m->m12 * z;
    m->m23 += m->m20 * x + m->m21 * y + m->m22 * z;
    m->m33 += m->m30 * x + m->m31 * y + m->m32 * z;
}

static void
m_scale(GL_matrix *m, GL_float x, GL_float y, GL_float z)
{
    m->m00 *= x; m->m01 *= y; m->m02 *= z;
    m->m10 *= x; m->m11 *= y; m->m12 *= z;
    m->m20 *= x; m->m21 *= y; m->m22 *= z;
    m->m30 *= x; m->m31 *= y; m->m32 *= z;
}

static void
m_frustum(GL_matrix *m, GL_float l, GL_float r, GL_float b, GL_float t,
          GL_float n, GL_float f)
{
    GL_matrix left = *m;
    GL_matrix right;
    GL_float invdx, invdy, invdz, twon;

    invdx = (l == r) ? FLOAT_ZERO : FLOAT_ONE / (r - l);
    invdy = (b == t) ? FLOAT_ZERO : FLOAT_ONE / (t - b);
    invdz = (n == f) ? FLOAT_ZERO : FLOAT_ONE / (n - f);
    twon = n + n;

    right.m00 = twon * invdx;
    right.m01 = FLOAT_ZERO;
    right.m02 = (r + l) * invdx;
    right.m03 = FLOAT_ZERO;

    right.m10 = FLOAT_ZERO;
    right.m11 = twon * invdy;
    right.m12 = (t + b) * invdy;
    right.m13 = FLOAT_ZERO;

    right.m20 = FLOAT_ZERO;
    right.m21 = FLOAT_ZERO;
    right.m22 = (f + n) * invdz;
    right.m23 = twon * f * invdz;

    right.m30 = FLOAT_ZERO;
    right.m31 = FLOAT_ZERO;
    right.m32 = -FLOAT_ONE;
    right.m33 = FLOAT_ZERO;

    m_mult(m, &left, &right);
}

static void
m_ortho(GL_matrix *m, GL_float l, GL_float r, GL_float b, GL_float t,
        GL_float n, GL_float f)
{
    GL_matrix left = *m;
    GL_matrix right;
    GL_float invdx, invdy, invdz;

    invdx = (l == r) ? FLOAT_ZERO : FLOAT_ONE / (r - l);
    invdy = (b == t) ? FLOAT_ZERO : FLOAT_ONE / (t - b);
    invdz = (n == f) ? FLOAT_ZERO : FLOAT_ONE / (n - f);

    right.m00 = - (invdx + invdx);
    right.m01 = FLOAT_ZERO;
    right.m02 = FLOAT_ZERO;
    right.m03 = (l + r) * invdx;

    right.m10 = FLOAT_ZERO;
    right.m11 = - (invdy + invdy);
    right.m12 = FLOAT_ZERO;
    right.m13 = (b + t) * invdy;

    right.m20 = FLOAT_ZERO;
    right.m21 = FLOAT_ZERO;
    right.m22 = invdz + invdz;
    right.m23 = (f + n) * invdz;

    right.m30 = FLOAT_ZERO;
    right.m31 = FLOAT_ZERO;
    right.m32 = FLOAT_ZERO;
    right.m33 = FLOAT_ONE;

    m_mult(m, &left, &right);
}

/* Texture functions */

static void
t_bind(GL_context *ctx, GL_int tex)
{
    int i;
    if (ctx->n_textures <= tex) {
        void *p = glt_malloc((tex + 1) * sizeof(GL_texture *));
        if (ctx->n_textures) {
            memcpy(p, ctx->textures, ctx->n_textures * sizeof(GL_texture *));
            glt_free(ctx->textures);
        }
        ctx->textures = (GL_texture **)p;
        for (i = ctx->n_textures; i <= tex; i++)
            ctx->textures[i] = NULL;
        ctx->n_textures = tex + 1;
    }
    if (!ctx->textures[tex]) {
        GL_texture *tp = (GL_texture *)glt_malloc(sizeof(GL_texture));
        tp->min_filter = GL_NEAREST_MIPMAP_LINEAR;
        tp->mag_filter = GL_LINEAR;
        tp->wrap_s = GL_REPEAT;
        tp->wrap_t = GL_REPEAT;
        tp->border_color.r = 0.0;
        tp->border_color.g = 0.0;
        tp->border_color.b = 0.0;
        tp->border_color.a = 0.0;
        for (i = 0; i < NUM_TEX_LEVELS; i++)
            tp->levels[i].valid = GL_FALSE;
        ctx->textures[tex] = tp;
    }
    ctx->texture = tex;
}

static void
t_del(GL_context *ctx, GL_int tex)
{
    if (tex < ctx->n_textures) {
        if (ctx->textures[tex]) {
            GL_texture *tp = ctx->textures[tex];
            int i;
            for (i = 0; i < NUM_TEX_LEVELS; i++) {
                if (tp->levels[i].valid && tp->levels[i].pixels)
                    glt_free(tp->levels[i].pixels);
            }
            glt_free(tp);
            ctx->textures[tex] = NULL;
        }
    }
    if (ctx->texture == tex)
        ctx->texture = 0;
}

static void
t_init(GL_context *ctx)
{
    ctx->n_textures = 0;
    ctx->textures = NULL;
    ctx->texture = 0;
    t_bind(ctx, 0);
}

#if 0

/* uncalled */

static void
t_shutdown(GL_context *ctx)
{
    int i;
    for (i = 0; i < ctx->n_textures; i++) {
        if (ctx->textures[i])
            t_del(ctx, i);
    }
    glt_free(ctx->textures);
}

#endif

static __inline GL_texture *
t_curtex(GL_context *ctx)
{
    GL_texture *texture;
    if (ctx->n_textures <= ctx->texture)
        glt_fatal("%s: internal error: invalid texture index", updatef);
    texture = ctx->textures[ctx->texture];
    if (!texture)
        glt_fatal("%s: internal error: null texture", updatef);
    return texture;
}

static __inline GL_texlevel *
t_curtexgetlev(GL_context *ctx, GLT_data *data)
{
    GL_texture *texture = t_curtex(ctx);
    if (data->tex_image_2d.level >= NUM_TEX_LEVELS)
        glt_fatal("%s: internal error: not enough texture levels", updatef);
    return &texture->levels[data->tex_image_2d.level];
}

/* Init function */

void
gs_init(GL_context *ctx) {
    int i;

    ctx->color.r = 1.0;
    ctx->color.g = 1.0;
    ctx->color.b = 1.0;
    ctx->color.a = 1.0;

    ctx->normal.x = 0.0;
    ctx->normal.y = 0.0;
    ctx->normal.z = 1.0;

    ctx->texcoord.s = 0.0;
    ctx->texcoord.t = 0.0;
    ctx->texcoord.r = 0.0;
    ctx->texcoord.q = 1.0;

    ctx->front_material.a.r = 0.2;
    ctx->front_material.a.g = 0.2;
    ctx->front_material.a.b = 0.2;
    ctx->front_material.a.a = 1.0;

    ctx->front_material.d.r = 0.8;
    ctx->front_material.d.g = 0.8;
    ctx->front_material.d.b = 0.8;
    ctx->front_material.d.a = 1.0;

    ctx->front_material.s.r = 0.0;
    ctx->front_material.s.g = 0.0;
    ctx->front_material.s.b = 0.0;
    ctx->front_material.s.a = 1.0;

    ctx->front_material.e.r = 0.0;
    ctx->front_material.e.g = 0.0;
    ctx->front_material.e.b = 0.0;
    ctx->front_material.e.a = 1.0;

    ctx->front_material.sh = 0.0;

    ctx->back_material = ctx->front_material;

    ctx->edge_flag = GL_TRUE;

    ctx->raster_pos.x = 0.0;
    ctx->raster_pos.y = 0.0;
    ctx->raster_pos.z = 0.0;
    ctx->raster_pos.w = 1.0;

    ctx->index = 1.0;

    ctx->clear_accum.r = 0.0;
    ctx->clear_accum.g = 0.0;
    ctx->clear_accum.b = 0.0;
    ctx->clear_accum.a = 0.0;

    ctx->clear_color.r = 0.0;
    ctx->clear_color.g = 0.0;
    ctx->clear_color.b = 0.0;
    ctx->clear_color.a = 0.0;

    ctx->clear_depth = 1.0;

    ctx->clear_index = 0.0;

    ctx->clear_stencil = 0;

    ctx->color_material_face = GL_FRONT_AND_BACK;
    ctx->color_material_mode = GL_AMBIENT_AND_DIFFUSE;

    ctx->alpha_func = GL_ALWAYS;
    ctx->alpha_ref = 0.0;

    ctx->blend_src = GL_ONE;
    ctx->blend_dst = GL_ZERO;

    ctx->depth_func = GL_LESS;

    ctx->stencil_func = GL_ALWAYS;
    ctx->stencil_ref = 0;
    ctx->stencil_mask = (GL_uint)(-1);
    ctx->stencil_op_fail = GL_KEEP;
    ctx->stencil_op_zfail = GL_KEEP;
    ctx->stencil_op_zpass = GL_KEEP;

    for (i = 0; i < NUM_CLIP_PLANES; i++) {
        ctx->clip_planes[i].x = 0.0;
        ctx->clip_planes[i].y = 0.0;
        ctx->clip_planes[i].z = 0.0;
        ctx->clip_planes[i].w = 0.0;
    }

    ctx->viewport.x = 0;
    ctx->viewport.y = 0;
    ctx->viewport.w = -1; /* Hack! But we never emit this value */
    ctx->viewport.h = -1; /* Hack! But we never emit this value */
    ctx->depth_range_near = 0.0;
    ctx->depth_range_far = 1.0;

    ctx->scissor.x = 0;
    ctx->scissor.y = 0;
    ctx->scissor.w = -1; /* Hack! But we never emit this value */
    ctx->scissor.h = -1; /* Hack! But we never emit this value */

    ctx->point_size = 1.0;
    ctx->line_width = 1.0;

    ctx->cull_face = GL_BACK;
    ctx->front_face = GL_CCW;
    ctx->shade_model = GL_SMOOTH;
    ctx->draw_buffer = GL_BACK; /* Hack? Only GL_BACK for double buffered */
    ctx->front_polygon_mode = GL_FILL;
    ctx->back_polygon_mode = GL_FILL;

    ctx->polygon_offset_factor = 0.0;
    ctx->polygon_offset_bias = 0.0;

    ctx->fog_mode = GL_EXP;
    ctx->fog_density = 1.0;
    ctx->fog_start = 0.0;
    ctx->fog_end = 1.0;
    ctx->fog_index = 0.0;
    ctx->fog_color.r = 0.0;
    ctx->fog_color.g = 0.0;
    ctx->fog_color.b = 0.0;
    ctx->fog_color.a = 0.0;

    ctx->color_write_r = GL_TRUE;
    ctx->color_write_g = GL_TRUE;
    ctx->color_write_b = GL_TRUE;
    ctx->color_write_a = GL_TRUE;
    ctx->depth_write = GL_TRUE;
    ctx->index_mask = (GL_uint)(-1);
    ctx->stencil_wmask = (GL_uint)(-1);

    ctx->matrix_mode = GL_MODELVIEW;
    ctx->modelview_stack_index = 0;
    ctx->modelview_stack[0] = m_identity;
    ctx->projection_stack_index = 0;
    ctx->projection_stack[0] = m_identity;
    ctx->texture_stack_index = 0;
    ctx->texture_stack[0] = m_identity;

    ctx->light_ambient_color.r = 0.2;
    ctx->light_ambient_color.g = 0.2;
    ctx->light_ambient_color.b = 0.2;
    ctx->light_ambient_color.a = 1.0;

    ctx->light_local_viewer = GL_FALSE;
    ctx->light_two_side = GL_FALSE;

    for (i = 0; i < NUM_LIGHTS; i++) {
        ctx->lights[i].a.r = 0.0;
        ctx->lights[i].a.g = 0.0;
        ctx->lights[i].a.b = 0.0;
        ctx->lights[i].a.a = 1.0;

        ctx->lights[i].d.r = 1.0;
        ctx->lights[i].d.g = 1.0;
        ctx->lights[i].d.b = 1.0;
        ctx->lights[i].d.a = 1.0;

        ctx->lights[i].s.r = 1.0;
        ctx->lights[i].s.g = 1.0;
        ctx->lights[i].s.b = 1.0;
        ctx->lights[i].s.a = 1.0;

        ctx->lights[i].p.x = 0.0;
        ctx->lights[i].p.y = 0.0;
        ctx->lights[i].p.z = 1.0;
        ctx->lights[i].p.w = 0.0;

        ctx->lights[i].sd.x = 0.0;
        ctx->lights[i].sd.y = 0.0;
        ctx->lights[i].sd.z = -1.0;

        ctx->lights[i].se = 0.0;
        ctx->lights[i].sc = 180.0;
        ctx->lights[i].ac = 1.0;
        ctx->lights[i].al = 0.0;
        ctx->lights[i].aq = 0.0;
    }

    ctx->pixel_zoom_x = 1.0;
    ctx->pixel_zoom_y = 1.0;

    ctx->pack_swap_bytes = GL_FALSE;
    ctx->pack_lsb_first = GL_FALSE;
    ctx->pack_row_length = 0;
    ctx->pack_skip_pixels = 0;
    ctx->pack_skip_rows = 0;
    ctx->pack_alignment = 4;

    ctx->unpack_swap_bytes = GL_FALSE;
    ctx->unpack_lsb_first = GL_FALSE;
    ctx->unpack_row_length = 0;
    ctx->unpack_skip_pixels = 0;
    ctx->unpack_skip_rows = 0;
    ctx->unpack_alignment = 4;

    ctx->transfer_map_color = GL_FALSE;
    ctx->transfer_map_stencil = GL_FALSE;
    ctx->transfer_index_shift = 0;
    ctx->transfer_index_offset = 0;
    ctx->transfer_red_scale = 1.0;
    ctx->transfer_red_bias = 0.0;
    ctx->transfer_green_scale = 1.0;
    ctx->transfer_green_bias = 0.0;
    ctx->transfer_blue_scale = 1.0;
    ctx->transfer_blue_bias = 0.0;
    ctx->transfer_alpha_scale = 1.0;
    ctx->transfer_alpha_bias = 0.0;
    ctx->transfer_depth_scale = 1.0;
    ctx->transfer_depth_bias = 0.0;

    for (i = 0; i < NUM_ENABLES; i++)
        ctx->enable_flags[i] = GL_FALSE;
    ctx->enable_flags[DITHER_ENABLE] = GL_TRUE;

    ctx->fog_hint = GL_DONT_CARE;
    ctx->line_smooth_hint = GL_DONT_CARE;
    ctx->perspective_correction_hint = GL_DONT_CARE;
    ctx->point_smooth_hint = GL_DONT_CARE;
    ctx->polygon_smooth_hint = GL_DONT_CARE;

    ctx->list_base = 0;

    ctx->tex_env_mode = GL_MODULATE;

    ctx->tex_env_color.r = 0.0;
    ctx->tex_env_color.g = 0.0;
    ctx->tex_env_color.b = 0.0;
    ctx->tex_env_color.a = 0.0;

    t_init(ctx);
}

/* Update inlines and macros */

static void
UpdateColorMaterial(GL_context *ctx)
{
    /* Assume color material is enabled... */
    if (ctx->color_material_face == GL_FRONT ||
        ctx->color_material_face == GL_FRONT_AND_BACK) {
        switch (ctx->color_material_mode) {
        case GL_EMISSION:
            ctx->front_material.e = ctx->color;
            break;
        case GL_AMBIENT:
            ctx->front_material.a = ctx->color;
            break;
        case GL_DIFFUSE:
            ctx->front_material.d = ctx->color;
            break;
        case GL_SPECULAR:
            ctx->front_material.s = ctx->color;
            break;
        case GL_AMBIENT_AND_DIFFUSE:
            ctx->front_material.a = ctx->color;
            ctx->front_material.d = ctx->color;
            break;
        default:
            /* Hack? Print an error here? */
            break;
        }
    }
    if (ctx->color_material_face == GL_BACK ||
        ctx->color_material_face == GL_FRONT_AND_BACK) {
        switch (ctx->color_material_mode) {
        case GL_EMISSION:
            ctx->back_material.e = ctx->color;
            break;
        case GL_AMBIENT:
            ctx->back_material.a = ctx->color;
            break;
        case GL_DIFFUSE:
            ctx->back_material.d = ctx->color;
            break;
        case GL_SPECULAR:
            ctx->back_material.s = ctx->color;
            break;
        case GL_AMBIENT_AND_DIFFUSE:
            ctx->back_material.a = ctx->color;
            ctx->back_material.d = ctx->color;
            break;
        default:
            /* Hack? Print an error here? */
            break;
        }
    }
}

static __inline void
Color(GL_context *ctx, GL_float r, GL_float g, GL_float b, GL_float a)
{
    ctx->color.r = r;
    ctx->color.g = g;
    ctx->color.b = b;
    ctx->color.a = a;
    if (ctx->enable_flags[COLOR_MATERIAL_ENABLE])
        UpdateColorMaterial(ctx);
}

#define Color3(c,r,g,b) Color(c, r, g, b, 1.0)
#define Color4(c,r,g,b,a) Color(c, r, g, b, a)

static __inline void
Normal(GL_context *ctx, GL_float nx, GL_float ny, GL_float nz)
{
    ctx->normal.x = nx;
    ctx->normal.y = ny;
    ctx->normal.z = nz;
}

static __inline void
TexCoord(GL_context *ctx, GL_float s, GL_float t, GL_float r, GL_float q)
{
    ctx->texcoord.s = s;
    ctx->texcoord.t = t;
    ctx->texcoord.r = r;
    ctx->texcoord.q = q;
}

#define TexCoord1(c,s) TexCoord(c, s, 0.0, 0.0, 1.0)
#define TexCoord2(c,s,t) TexCoord(c, s, t, 0.0, 1.0)
#define TexCoord3(c,s,t,r) TexCoord(c, s, t, r, 1.0)
#define TexCoord4(c,s,t,r,q) TexCoord(c, s, t, r, q)

/* Hack? Material face looks ugly because of ColorMaterial */
static void
MaterialFace(GL_context *ctx, GLenum face, GLT_materialfv *matfv)
{
    GL_material *material = &ctx->front_material;
    int maskface;

    if (face != GL_FRONT)
        material = &ctx->back_material;

    maskface = (ctx->enable_flags[COLOR_MATERIAL_ENABLE] == GL_TRUE &&
                (ctx->color_material_face == face ||
                 ctx->color_material_face == GL_FRONT_AND_BACK));

    switch (matfv->pname) {
    case GL_AMBIENT:
        if (maskface)
            if (ctx->color_material_mode == GL_AMBIENT ||
                ctx->color_material_mode == GL_AMBIENT_AND_DIFFUSE)
                break;
        material->a.r = matfv->params[0];
        material->a.g = matfv->params[1];
        material->a.b = matfv->params[2];
        material->a.a = matfv->params[3];
        break;
    case GL_DIFFUSE:
        if (maskface)
            if (ctx->color_material_mode == GL_DIFFUSE ||
                ctx->color_material_mode == GL_AMBIENT_AND_DIFFUSE)
                break;
        material->d.r = matfv->params[0];
        material->d.g = matfv->params[1];
        material->d.b = matfv->params[2];
        material->d.a = matfv->params[3];
        break;
    case GL_SPECULAR:
        if (maskface)
            if (ctx->color_material_mode == GL_SPECULAR)
                break;
        material->s.r = matfv->params[0];
        material->s.g = matfv->params[1];
        material->s.b = matfv->params[2];
        material->s.a = matfv->params[3];
        break;
    case GL_EMISSION:
        if (maskface)
            if (ctx->color_material_mode == GL_EMISSION)
                break;
        material->e.r = matfv->params[0];
        material->e.g = matfv->params[1];
        material->e.b = matfv->params[2];
        material->e.a = matfv->params[3];
        break;
    case GL_AMBIENT_AND_DIFFUSE:
        material->a.r = matfv->params[0];
        material->a.g = matfv->params[1];
        material->a.b = matfv->params[2];
        material->a.a = matfv->params[3];
        material->d.r = matfv->params[0];
        material->d.g = matfv->params[1];
        material->d.b = matfv->params[2];
        material->d.a = matfv->params[3];
        break;
    case GL_SHININESS:
        material->sh = matfv->params[0];
        break;
    default:
        /* If you add a case here, also add one to MATERIALIV in gs_update */
        ONCE(glt_warn("%s: unknown material pname (0x%04x)", updatef,
                      matfv->pname));
        break;
    }
}

static void
Material(GL_context *ctx, GLT_materialfv *mat)
{
    switch (mat->face) {
    case GL_FRONT:
        MaterialFace(ctx, GL_FRONT, mat);
        break;
    case GL_BACK:
        MaterialFace(ctx, GL_BACK, mat);
        break;
    case GL_FRONT_AND_BACK:
        MaterialFace(ctx, GL_FRONT, mat);
        MaterialFace(ctx, GL_BACK, mat);
        break;
    default:
        ONCE(glt_warn("%s: invalid material face (0x%04x)", updatef,
                      mat->face));
        break;
    }
}

static __inline void
RasterPos(GL_context *ctx, GL_float x, GL_float y, GL_float z, GL_float w)
{
    ctx->raster_pos.x = x;
    ctx->raster_pos.y = y;
    ctx->raster_pos.z = z;
    ctx->raster_pos.w = w;
}

#define RasterPos2(c,x,y) RasterPos(c, x, y, 0.0, 1.0);
#define RasterPos3(c,x,y,z) RasterPos(c, x, y, z, 1.0);
#define RasterPos4(c,x,y,z,w) RasterPos(c, x, y, z, w);

static void
Fog(GL_context *ctx, GLT_fogfv *fog)
{
    switch (fog->pname) {
    case GL_FOG_MODE:
        ctx->fog_mode = fog->params[0];
        break;
    case GL_FOG_DENSITY:
        ctx->fog_density = fog->params[0];
        break;
    case GL_FOG_START:
        ctx->fog_start = fog->params[0];
        break;
    case GL_FOG_END:
        ctx->fog_end = fog->params[0];
        break;
    case GL_FOG_INDEX:
        ctx->fog_index = fog->params[0];
        break;
    case GL_FOG_COLOR:
        ctx->fog_color.r = fog->params[0];
        ctx->fog_color.g = fog->params[1];
        ctx->fog_color.b = fog->params[2];
        ctx->fog_color.a = fog->params[3];
        break;
    default:
        /* If you add a case here, also add one to FOGIV in gs_update */
        ONCE(glt_warn("%s: unknown fog pname (0x%04x)", updatef,
                      fog->pname));
        break;
    }
}

static void
LightModel(GL_context *ctx, GLT_light_modelfv *model)
{
    switch (model->pname) {
    case GL_LIGHT_MODEL_AMBIENT:
        ctx->light_ambient_color.r = model->params[0];
        ctx->light_ambient_color.g = model->params[1];
        ctx->light_ambient_color.b = model->params[2];
        ctx->light_ambient_color.a = model->params[3];
        break;
    case GL_LIGHT_MODEL_LOCAL_VIEWER:
        ctx->light_local_viewer = model->params[0];
        break;
    case GL_LIGHT_MODEL_TWO_SIDE:
        ctx->light_two_side = model->params[0];
        break;
    default:
        /* If you add a case here, add one to LIGHT_MODELIV in gs_update too */
        ONCE(glt_warn("%s: unknown light model pname (0x%04x)", updatef,
                      model->pname));
        break;
    }
}

static void
Light(GL_context *ctx, GLT_lightfv *lightfv)
{
    GL_light *light;
    switch (lightfv->light) {
    case GL_LIGHT0:
    case GL_LIGHT1:
    case GL_LIGHT2:
    case GL_LIGHT3:
    case GL_LIGHT4:
    case GL_LIGHT5:
    case GL_LIGHT6:
    case GL_LIGHT7:
        light = &ctx->lights[lightfv->light - GL_LIGHT0];
        switch (lightfv->pname) {
        case GL_AMBIENT:
            light->a.r = lightfv->params[0];
            light->a.g = lightfv->params[1];
            light->a.b = lightfv->params[2];
            light->a.a = lightfv->params[3];
            break;
        case GL_DIFFUSE:
            light->d.r = lightfv->params[0];
            light->d.g = lightfv->params[1];
            light->d.b = lightfv->params[2];
            light->d.a = lightfv->params[3];
            break;
        case GL_SPECULAR:
            light->s.r = lightfv->params[0];
            light->s.g = lightfv->params[1];
            light->s.b = lightfv->params[2];
            light->s.a = lightfv->params[3];
            break;
        case GL_POSITION:
            light->p.x = lightfv->params[0];
            light->p.y = lightfv->params[1];
            light->p.z = lightfv->params[2];
            light->p.w = lightfv->params[3];
            break;
        case GL_SPOT_DIRECTION:
            light->sd.x = lightfv->params[0];
            light->sd.y = lightfv->params[1];
            light->sd.z = lightfv->params[2];
            break;
        case GL_SPOT_EXPONENT:
            light->se = lightfv->params[0];
            break;
        case GL_SPOT_CUTOFF:
            light->sc = lightfv->params[0];
            break;
        case GL_CONSTANT_ATTENUATION:
            light->ac = lightfv->params[0];
            break;
        case GL_LINEAR_ATTENUATION:
            light->al = lightfv->params[0];
            break;
        case GL_QUADRATIC_ATTENUATION:
            light->aq = lightfv->params[0];
            break;
        default:
            /* If you add a case here, also add one to LIGHTIV in gs_update */
            ONCE(glt_warn("%s: invalid light pname (0x%04x)", updatef,
                          lightfv->pname));
            break;
        }
        break;
    default:
        ONCE(glt_warn("%s: invalid light (0x%04x)", updatef, lightfv->light));
        break;
    }
}

static void
PixelStore(GL_context *ctx, GLT_pixel_storef *pstore)
{
    switch (pstore->pname) {
    case GL_PACK_SWAP_BYTES:
        ctx->pack_swap_bytes = pstore->param;
        break;
    case GL_PACK_LSB_FIRST:
        ctx->pack_lsb_first = pstore->param;
        break;
    case GL_PACK_ROW_LENGTH:
        ctx->pack_row_length = pstore->param;
        break;
    case GL_PACK_SKIP_PIXELS:
        ctx->pack_skip_pixels = pstore->param;
        break;
    case GL_PACK_SKIP_ROWS:
        ctx->pack_skip_rows = pstore->param;
        break;
    case GL_PACK_ALIGNMENT:
        ctx->pack_alignment = pstore->param;
        break;
    case GL_UNPACK_SWAP_BYTES:
        ctx->unpack_swap_bytes = pstore->param;
        break;
    case GL_UNPACK_LSB_FIRST:
        ctx->unpack_lsb_first = pstore->param;
        break;
    case GL_UNPACK_ROW_LENGTH:
        ctx->unpack_row_length = pstore->param;
        break;
    case GL_UNPACK_SKIP_PIXELS:
        ctx->unpack_skip_pixels = pstore->param;
        break;
    case GL_UNPACK_SKIP_ROWS:
        ctx->unpack_skip_rows = pstore->param;
        break;
    case GL_UNPACK_ALIGNMENT:
        ctx->unpack_alignment = pstore->param;
        break;
    default:
        /* If you add a case here, add one to PIXEL_STOREIV in gs_update too */
        ONCE(glt_warn("%s: invalid pixel store pname (0x%04x)", updatef,
                      pstore->pname));
        break;
    }
}

static void
PixelTransfer(GL_context *ctx, GLT_pixel_transferf *ptransfer)
{
    switch (ptransfer->pname) {
    case GL_MAP_COLOR:
        ctx->transfer_map_color = ptransfer->param;
        break;
    case GL_MAP_STENCIL:
        ctx->transfer_map_stencil = ptransfer->param;
        break;
    case GL_INDEX_SHIFT:
        ctx->transfer_index_shift = ptransfer->param;
        break;
    case GL_INDEX_OFFSET:
        ctx->transfer_index_offset = ptransfer->param;
        break;
    case GL_RED_SCALE:
        ctx->transfer_red_scale = ptransfer->param;
        break;
    case GL_RED_BIAS:
        ctx->transfer_red_bias = ptransfer->param;
        break;
    case GL_GREEN_SCALE:
        ctx->transfer_green_scale = ptransfer->param;
        break;
    case GL_GREEN_BIAS:
        ctx->transfer_green_bias = ptransfer->param;
        break;
    case GL_BLUE_SCALE:
        ctx->transfer_blue_scale = ptransfer->param;
        break;
    case GL_BLUE_BIAS:
        ctx->transfer_blue_bias = ptransfer->param;
        break;
    case GL_ALPHA_SCALE:
        ctx->transfer_alpha_scale = ptransfer->param;
        break;
    case GL_ALPHA_BIAS:
        ctx->transfer_alpha_bias = ptransfer->param;
        break;
    case GL_DEPTH_SCALE:
        ctx->transfer_depth_scale = ptransfer->param;
        break;
    case GL_DEPTH_BIAS:
        ctx->transfer_depth_bias = ptransfer->param;
        break;
    default:
        /* If you add a case here, add one to PIXEL_TRANSFERIV in gs_update */
        ONCE(glt_warn("%s: invalid pixel transfer pname (0x%04x)", updatef,
                      ptransfer->pname));
        break;
    }
}

static void
TexEnv(GL_context *ctx, GLT_tex_envfv *texenv)
{
    switch (texenv->target) {
    case GL_TEXTURE_ENV:
        switch (texenv->pname) {
        case GL_TEXTURE_ENV_MODE:
            ctx->tex_env_mode = texenv->params[0];
            break;
        case GL_TEXTURE_ENV_COLOR:
            ctx->tex_env_color.r = texenv->params[0];
            ctx->tex_env_color.g = texenv->params[1];
            ctx->tex_env_color.b = texenv->params[2];
            ctx->tex_env_color.a = texenv->params[3];
            break;
        default:
            /* If a case is added here, add a case to TEX_ENVIV in gs_update */
            ONCE(glt_warn("%s: invalid tex env pname (0x%04x)", updatef,
                          texenv->pname));
            break;
        }
        break;
    default:
        /* If a case is added here, add a case to TEX_ENVIV in gs_update */
        ONCE(glt_warn("%s: invalid tex env target (0x%04x)", updatef,
                      texenv->target));
        break;
    }
}

static void
TexParameter(GL_context *ctx, GLT_tex_parameterfv *texparam)
{
    switch (texparam->target) {
    case GL_TEXTURE_2D:
    {
        GL_texture *texture = t_curtex(ctx);
        switch (texparam->pname) {
        case GL_TEXTURE_MIN_FILTER:
            texture->min_filter = texparam->params[0];
            break;
        case GL_TEXTURE_MAG_FILTER:
            texture->mag_filter = texparam->params[0];
            break;
        case GL_TEXTURE_WRAP_S:
            texture->wrap_s = texparam->params[0];
            break;
        case GL_TEXTURE_WRAP_T:
            texture->wrap_t = texparam->params[0];
            break;
        case GL_TEXTURE_BORDER_COLOR:
            texture->border_color.r = texparam->params[0];
            texture->border_color.g = texparam->params[1];
            texture->border_color.b = texparam->params[2];
            texture->border_color.a = texparam->params[3];
            break;
        default:
            /* If case added here, add case to TEX_PARAMETERIV in gs_update */
            ONCE(glt_warn("%s: invalid tex parameter pname (0x%04x)", updatef,
                          texparam->pname));
            break;
        }
        break;
    }
    default:
        /* If case added here, add case to TEX_PARAMETERIV in gs_update */
        ONCE(glt_warn("%s: invalid tex parameter target (0x%04x)", updatef,
                      texparam->target));
        break;
    }
}

static __inline void
SetFlag(GL_context *ctx, GL_int cap, GL_int state)
{
    switch(cap) {
    case GL_ALPHA_TEST:
        ctx->enable_flags[ALPHA_TEST_ENABLE] = state;
        break;
    case GL_BLEND:
        ctx->enable_flags[BLEND_ENABLE] = state;
        break;
    case GL_CLIP_PLANE0:
        ctx->enable_flags[CLIP_PLANE0_ENABLE] = state;
        break;
    case GL_CLIP_PLANE1:
        ctx->enable_flags[CLIP_PLANE1_ENABLE] = state;
        break;
    case GL_CLIP_PLANE2:
        ctx->enable_flags[CLIP_PLANE2_ENABLE] = state;
        break;
    case GL_CLIP_PLANE3:
        ctx->enable_flags[CLIP_PLANE3_ENABLE] = state;
        break;
    case GL_CLIP_PLANE4:
        ctx->enable_flags[CLIP_PLANE4_ENABLE] = state;
        break;
    case GL_CLIP_PLANE5:
        ctx->enable_flags[CLIP_PLANE5_ENABLE] = state;
        break;
    case GL_COLOR_MATERIAL:
        ctx->enable_flags[COLOR_MATERIAL_ENABLE] = state;
        if (state)
            UpdateColorMaterial(ctx);
        break;
    case GL_CULL_FACE:
        ctx->enable_flags[CULL_FACE_ENABLE] = state;
        break;
    case GL_DEPTH_TEST:
        ctx->enable_flags[DEPTH_TEST_ENABLE] = state;
        break;
    case GL_DITHER:
        ctx->enable_flags[DITHER_ENABLE] = state;
        break;
    case GL_FOG:
        ctx->enable_flags[FOG_ENABLE] = state;
        break;
    case GL_LIGHT0:
        ctx->enable_flags[LIGHT0_ENABLE] = state;
        break;
    case GL_LIGHT1:
        ctx->enable_flags[LIGHT1_ENABLE] = state;
        break;
    case GL_LIGHT2:
        ctx->enable_flags[LIGHT2_ENABLE] = state;
        break;
    case GL_LIGHT3:
        ctx->enable_flags[LIGHT3_ENABLE] = state;
        break;
    case GL_LIGHT4:
        ctx->enable_flags[LIGHT4_ENABLE] = state;
        break;
    case GL_LIGHT5:
        ctx->enable_flags[LIGHT5_ENABLE] = state;
        break;
    case GL_LIGHT6:
        ctx->enable_flags[LIGHT6_ENABLE] = state;
        break;
    case GL_LIGHT7:
        ctx->enable_flags[LIGHT7_ENABLE] = state;
        break;
    case GL_LIGHTING:
        ctx->enable_flags[LIGHTING_ENABLE] = state;
        break;
    case GL_LINE_SMOOTH:
        ctx->enable_flags[LINE_SMOOTH_ENABLE] = state;
        break;
    case GL_LINE_STIPPLE:
        ctx->enable_flags[LINE_STIPPLE_ENABLE] = state;
        break;
    case GL_LOGIC_OP:
        ctx->enable_flags[LOGIC_OP_ENABLE] = state;
        break;
    case GL_NORMALIZE:
        ctx->enable_flags[NORMALIZE_ENABLE] = state;
        break;
#ifdef GL_POLYGON_OFFSET_EXT
    case GL_POLYGON_OFFSET_EXT:
        ctx->enable_flags[POLYGON_OFFSET_EXT_ENABLE] = state;
        break;
#endif
    case GL_POINT_SMOOTH:
        ctx->enable_flags[POINT_SMOOTH_ENABLE] = state;
        break;
    case GL_POLYGON_STIPPLE:
        ctx->enable_flags[POLYGON_STIPPLE_ENABLE] = state;
        break;
    case GL_SCISSOR_TEST:
        ctx->enable_flags[SCISSOR_TEST_ENABLE] = state;
        break;
    case GL_STENCIL_TEST:
        ctx->enable_flags[STENCIL_TEST_ENABLE] = state;
        break;
    case GL_TEXTURE_1D:
        ctx->enable_flags[TEXTURE_1D_ENABLE] = state;
        break;
    case GL_TEXTURE_2D:
        ctx->enable_flags[TEXTURE_2D_ENABLE] = state;
        break;
    default:
        ONCE(glt_warn("%s: invalid enable (0x%04x)", updatef, cap));
    }
}

/* Update function */

int
gs_update(GL_context *ctx, GLT_opcode op, GLT_data *data)
{
    char *loc = updatef;
    int rval = 1;

    switch (op) {
    case GLT_OP_BAD:
    case GLT_OP_HEADER:
    case GLT_OP_PAD:
    case GLT_OP_PROCESS_EVENTS:
        /* Ignore GLT specific opcodes */
        break;
    case GLT_OP_CREATE_CONTEXT:
    case GLT_OP_DESTROY_CONTEXT:
    case GLT_OP_MAKE_CURRENT:
        /* Hack. Copies these opcodes to output */
        rval = 0;
        break;
    case GLT_OP_SWAP_BUFFERS:
        break;
    case GLT_OP_ACCUM:
        break;
    case GLT_OP_ALPHA_FUNC:
        ctx->alpha_func = data->alpha_func.func;
        ctx->alpha_ref = data->alpha_func.ref;
        break;
    case GLT_OP_BEGIN:
        break;
    case GLT_OP_BIND_TEXTURE:
    case GLT_OP_BIND_TEXTURE_EXT:
        switch (data->bind_texture.target) {
        case GL_TEXTURE_2D:
            t_bind(ctx, data->bind_texture.texture);
            break;
        default:
            ONCE(glt_warn("%s: unknown bind texture target (0x%04x)", loc,
                          data->bind_texture.target));
            break;
        }
        break;
    case GLT_OP_BLEND_FUNC:
        ctx->blend_src = data->blend_func.sfactor;
        ctx->blend_dst = data->blend_func.dfactor;
        break;
    case GLT_OP_CALL_LIST:
        /* Ugh, have to unwind display lists to get this */
        ONCE(ignore(loc, "glCallList"));
        rval = 0;
        break;
    case GLT_OP_CALL_LISTS:
        /* Ugh, have to unwind display lists to get this */
        ONCE(ignore(loc, "glCallLists"));
        rval = 0;
        break;
    case GLT_OP_CLEAR:
        break;
    case GLT_OP_CLEAR_ACCUM:
        ctx->clear_accum.r = data->clear_accum.red;
        ctx->clear_accum.g = data->clear_accum.green;
        ctx->clear_accum.b = data->clear_accum.blue;
        ctx->clear_accum.a = data->clear_accum.alpha;
        break;
    case GLT_OP_CLEAR_COLOR:
        ctx->clear_color.r = data->clear_color.red;
        ctx->clear_color.g = data->clear_color.green;
        ctx->clear_color.b = data->clear_color.blue;
        ctx->clear_color.a = data->clear_color.alpha;
        break;
    case GLT_OP_CLEAR_DEPTH:
        ctx->clear_depth = data->clear_depth.depth;
        break;
    case GLT_OP_CLEAR_INDEX:
        ctx->clear_index = data->clear_index.c;
        break;
    case GLT_OP_CLEAR_STENCIL:
        ctx->clear_stencil = data->clear_stencil.s;
        break;
    case GLT_OP_CLIP_PLANE:
    {
        GL_coord temp;
        GL_coord *eqn = &temp;
        switch (data->clip_plane.plane) {
        case GL_CLIP_PLANE0:
            eqn = &ctx->clip_planes[0];
            break;
        case GL_CLIP_PLANE1:
            eqn = &ctx->clip_planes[1];
            break;
        case GL_CLIP_PLANE2:
            eqn = &ctx->clip_planes[2];
            break;
        case GL_CLIP_PLANE3:
            eqn = &ctx->clip_planes[3];
            break;
        case GL_CLIP_PLANE4:
            eqn = &ctx->clip_planes[4];
            break;
        case GL_CLIP_PLANE5:
            eqn = &ctx->clip_planes[5];
            break;
        default:
            ONCE(glt_warn("%s: invalid clip plane (0x%04x)", loc,
                          data->clip_plane.plane));
            break;
        }
        eqn->x = data->clip_plane.equation[0];
        eqn->y = data->clip_plane.equation[1];
        eqn->z = data->clip_plane.equation[2];
        eqn->w = data->clip_plane.equation[3];
        break;
    }
    case GLT_OP_COLOR_3B:
    case GLT_OP_COLOR_3BV:
        Color3(ctx, CBYTE(data->color_3b.red),
               CBYTE(data->color_3b.green),
               CBYTE(data->color_3b.blue));
        break;
    case GLT_OP_COLOR_3D:
    case GLT_OP_COLOR_3DV:
        Color3(ctx, CDOUBLE(data->color_3d.red),
               CDOUBLE(data->color_3d.green),
               CDOUBLE(data->color_3d.blue));
        break;
    case GLT_OP_COLOR_3F:
    case GLT_OP_COLOR_3FV:
        Color3(ctx, CFLOAT(data->color_3f.red),
               CFLOAT(data->color_3f.green),
               CFLOAT(data->color_3f.blue));
        break;
    case GLT_OP_COLOR_3I:
    case GLT_OP_COLOR_3IV:
        Color3(ctx, CINT(data->color_3i.red),
               CINT(data->color_3i.green),
               CINT(data->color_3i.blue));
        break;
    case GLT_OP_COLOR_3S:
    case GLT_OP_COLOR_3SV:
        Color3(ctx, CSHORT(data->color_3s.red),
               CSHORT(data->color_3s.green),
               CSHORT(data->color_3s.blue));
        break;
    case GLT_OP_COLOR_3UB:
    case GLT_OP_COLOR_3UBV:
        Color3(ctx, CUBYTE(data->color_3ub.red),
               CUBYTE(data->color_3ub.green),
               CUBYTE(data->color_3ub.blue));
        break;
    case GLT_OP_COLOR_3UI:
    case GLT_OP_COLOR_3UIV:
        Color3(ctx, CUINT(data->color_3ui.red),
               CUINT(data->color_3ui.green),
               CUINT(data->color_3ui.blue));
        break;
    case GLT_OP_COLOR_3US:
    case GLT_OP_COLOR_3USV:
        Color3(ctx, CUSHORT(data->color_3us.red),
               CUSHORT(data->color_3us.green),
               CUSHORT(data->color_3us.blue));
        break;
    case GLT_OP_COLOR_4B:
    case GLT_OP_COLOR_4BV:
        Color4(ctx,
               CBYTE(data->color_4b.red),
               CBYTE(data->color_4b.green),
               CBYTE(data->color_4b.blue),
               CBYTE(data->color_4b.alpha));
        break;
    case GLT_OP_COLOR_4D:
    case GLT_OP_COLOR_4DV:
        Color4(ctx,
               CDOUBLE(data->color_4d.red),
               CDOUBLE(data->color_4d.green),
               CDOUBLE(data->color_4d.blue),
               CDOUBLE(data->color_4d.alpha));
        break;
    case GLT_OP_COLOR_4F:
    case GLT_OP_COLOR_4FV:
        Color4(ctx,
               CFLOAT(data->color_4f.red),
               CFLOAT(data->color_4f.green),
               CFLOAT(data->color_4f.blue),
               CFLOAT(data->color_4f.alpha));
        break;
    case GLT_OP_COLOR_4I:
    case GLT_OP_COLOR_4IV:
        Color4(ctx,
               CINT(data->color_4i.red),
               CINT(data->color_4i.green),
               CINT(data->color_4i.blue),
               CINT(data->color_4i.alpha));
        break;
    case GLT_OP_COLOR_4S:
    case GLT_OP_COLOR_4SV:
        Color4(ctx,
               CSHORT(data->color_4s.red),
               CSHORT(data->color_4s.green),
               CSHORT(data->color_4s.blue),
               CSHORT(data->color_4s.alpha));
        break;
    case GLT_OP_COLOR_4UB:
    case GLT_OP_COLOR_4UBV:
        Color4(ctx,
               CUBYTE(data->color_4ub.red),
               CUBYTE(data->color_4ub.green),
               CUBYTE(data->color_4ub.blue),
               CUBYTE(data->color_4ub.alpha));
        break;
    case GLT_OP_COLOR_4UI:
    case GLT_OP_COLOR_4UIV:
        Color4(ctx,
               CUINT(data->color_4ui.red),
               CUINT(data->color_4ui.green),
               CUINT(data->color_4ui.blue),
               CUINT(data->color_4ui.alpha));
        break;
    case GLT_OP_COLOR_4US:
    case GLT_OP_COLOR_4USV:
        Color4(ctx,
               CUSHORT(data->color_4us.red),
               CUSHORT(data->color_4us.green),
               CUSHORT(data->color_4us.blue),
               CUSHORT(data->color_4us.alpha));
        break;
    case GLT_OP_COLOR_MASK:
        ctx->color_write_r = data->color_mask.red;
        ctx->color_write_g = data->color_mask.green;
        ctx->color_write_b = data->color_mask.blue;
        ctx->color_write_a = data->color_mask.alpha;
        break;
    case GLT_OP_COLOR_MATERIAL:
        ctx->color_material_face = data->color_material.face;
        ctx->color_material_mode = data->color_material.mode;
        if (ctx->enable_flags[COLOR_MATERIAL_ENABLE])
            UpdateColorMaterial(ctx);
        break;
    case GLT_OP_COPY_PIXELS:
        break;
    case GLT_OP_CULL_FACE:
        ctx->cull_face = data->cull_face.mode;
        break;
    case GLT_OP_DELETE_LISTS:
        /* Display lists not stored, do nothing */
        break;
    case GLT_OP_DELETE_TEXTURES:
    case GLT_OP_DELETE_TEXTURES_EXT:
    {
        int i;
        for (i = 0; i < data->delete_textures.n; i++) {
            if (data->delete_textures.textures[i])
                t_del(ctx, data->delete_textures.textures[i]);
        }
        break;
    }
    case GLT_OP_DEPTH_FUNC:
        ctx->depth_func = data->depth_func.func;
        break;
    case GLT_OP_DEPTH_MASK:
        ctx->depth_write = data->depth_mask.flag;
        break;
    case GLT_OP_DEPTH_RANGE:
        ctx->depth_range_near = data->depth_range.znear;
        ctx->depth_range_far = data->depth_range.zfar;
        break;
    case GLT_OP_DISABLE:
        SetFlag(ctx, data->enable.cap, 0);
        break;
    case GLT_OP_DRAW_BUFFER:
        ctx->draw_buffer = data->draw_buffer.mode;
        break;
    case GLT_OP_DRAW_PIXELS:
        break;
    case GLT_OP_EDGE_FLAG:
    case GLT_OP_EDGE_FLAGV:
        ctx->edge_flag = data->edge_flag.flag;
        break;
    case GLT_OP_ENABLE:
        SetFlag(ctx, data->enable.cap, 1);
        break;
    case GLT_OP_END:
        break;
    case GLT_OP_END_LIST:
        break;
    case GLT_OP_FINISH:
        break;
    case GLT_OP_FLUSH:
        break;
    case GLT_OP_FOGF:
    case GLT_OP_FOGFV:
        Fog(ctx, &data->fogfv);
        break;
    case GLT_OP_FOGI:
    case GLT_OP_FOGIV:
    {
        GLT_fogfv ffog;
        /* Convert to fogfv */
        ffog.pname = data->fogiv.pname;
        switch (data->fogiv.pname) {
        case GL_FOG_MODE:
        case GL_FOG_DENSITY:
        case GL_FOG_START:
        case GL_FOG_END:
        case GL_FOG_INDEX:
            ffog.params[0] = data->fogiv.params[0];
            break;
        case GL_FOG_COLOR:
            ffog.params[0] = CINT(data->fogiv.params[0]);
            ffog.params[1] = CINT(data->fogiv.params[1]);
            ffog.params[2] = CINT(data->fogiv.params[2]);
            ffog.params[3] = CINT(data->fogiv.params[3]);
            break;
        default:
            /* do nothing, will handle later */
            break;
        }
        Fog(ctx, &ffog);
        break;
    }
    case GLT_OP_FRONT_FACE:
        ctx->front_face = data->front_face.mode;
        break;
    case GLT_OP_FRUSTUM:
        m_frustum(m_cur_matrix(ctx), data->frustum.left,
                  data->frustum.right, data->frustum.bottom,
                  data->frustum.top, data->frustum.znear,
                  data->frustum.zfar);
        break;
    case GLT_OP_HINT:
        switch (data->hint.target) {
        case GL_FOG_HINT:
            ctx->fog_hint = data->hint.mode;
            break;
        case GL_LINE_SMOOTH_HINT:
            ctx->line_smooth_hint = data->hint.mode;
            break;
        case GL_PERSPECTIVE_CORRECTION_HINT:
            ctx->perspective_correction_hint = data->hint.mode;
            break;
        case GL_POINT_SMOOTH_HINT:
            ctx->point_smooth_hint = data->hint.mode;
            break;
        case GL_POLYGON_SMOOTH_HINT:
            ctx->polygon_smooth_hint = data->hint.mode;
            break;
        default:
            ONCE(glt_warn("%s: invalid hint target (0x%04x)", loc,
                          data->hint.target));
            break;
        }
        break;
    case GLT_OP_INDEX_MASK:
        ctx->index_mask = data->index_mask.mask;
        break;
    case GLT_OP_INDEXD:
    case GLT_OP_INDEXDV:
        ctx->index = (GL_float)data->indexd.c;
        break;
    case GLT_OP_INDEXF:
    case GLT_OP_INDEXFV:
        ctx->index = (GL_float)data->indexf.c;
        break;
    case GLT_OP_INDEXI:
    case GLT_OP_INDEXIV:
        ctx->index = (GL_float)data->indexi.c;
        break;
    case GLT_OP_INDEXS:
    case GLT_OP_INDEXSV:
        ctx->index = (GL_float)data->indexs.c;
        break;
    case GLT_OP_INDEXUB:
    case GLT_OP_INDEXUBV:
        ctx->index = (GL_float)data->indexub.c;
        break;
    case GLT_OP_LIGHT_MODELF:
    case GLT_OP_LIGHT_MODELFV:
        LightModel(ctx, &data->light_modelfv);
        break;
    case GLT_OP_LIGHT_MODELI:
    case GLT_OP_LIGHT_MODELIV:
    {
        GLT_light_modelfv lightmodelfv;
        /* Convert to light_modelfv */
        lightmodelfv.pname = data->light_modeliv.pname;
        switch (data->light_modeliv.pname) {
        case GL_LIGHT_MODEL_AMBIENT:
            lightmodelfv.params[0] = CINT(data->light_modeliv.params[0]);
            lightmodelfv.params[1] = CINT(data->light_modeliv.params[1]);
            lightmodelfv.params[2] = CINT(data->light_modeliv.params[2]);
            lightmodelfv.params[3] = CINT(data->light_modeliv.params[3]);
            break;
        case GL_LIGHT_MODEL_LOCAL_VIEWER:
        case GL_LIGHT_MODEL_TWO_SIDE:
            lightmodelfv.params[0] = data->light_modeliv.params[0];
            break;
        default:
            /* do nothing, will handle later */
            break;
        }
        LightModel(ctx, &lightmodelfv);
        break;
    }
    case GLT_OP_LIGHTF:
    case GLT_OP_LIGHTFV:
        Light(ctx, &data->lightfv);
        break;
    case GLT_OP_LIGHTI:
    case GLT_OP_LIGHTIV:
    {
        GLT_lightfv lightfv;
        /* Convert to lightfv */
        lightfv.light = data->lightiv.light;
        lightfv.pname = data->lightiv.pname;
        switch (data->lightiv.pname) {
        case GL_SPOT_EXPONENT:
        case GL_SPOT_CUTOFF:
        case GL_CONSTANT_ATTENUATION:
        case GL_LINEAR_ATTENUATION:
        case GL_QUADRATIC_ATTENUATION:
            lightfv.params[0] = data->lightiv.params[0];
            break;
        case GL_AMBIENT:
        case GL_DIFFUSE:
        case GL_SPECULAR:
            lightfv.params[0] = CINT(data->lightiv.params[0]);
            lightfv.params[1] = CINT(data->lightiv.params[1]);
            lightfv.params[2] = CINT(data->lightiv.params[2]);
            lightfv.params[3] = CINT(data->lightiv.params[3]);
            break;
        case GL_POSITION:
            lightfv.params[0] = data->lightiv.params[0];
            lightfv.params[1] = data->lightiv.params[1];
            lightfv.params[2] = data->lightiv.params[2];
            lightfv.params[3] = data->lightiv.params[3];
            break;
        case GL_SPOT_DIRECTION:
            lightfv.params[0] = data->lightiv.params[0];
            lightfv.params[1] = data->lightiv.params[1];
            lightfv.params[2] = data->lightiv.params[2];
            break;
        default:
            /* do nothing, will handle later */
            break;
        }
        Light(ctx, &lightfv);
        break;
    }
    case GLT_OP_LINE_WIDTH:
        ctx->line_width = data->line_width.width;
        break;
    case GLT_OP_LIST_BASE:
        ctx->list_base = data->list_base.base;
        break;
    case GLT_OP_LOAD_IDENTITY:
        *m_cur_matrix(ctx) = m_identity;
        break;
    case GLT_OP_LOAD_MATRIXD:
    {
        GL_matrix *m = m_cur_matrix(ctx);
        m->m00 = (GL_float)data->load_matrixd.m[0];
        m->m01 = (GL_float)data->load_matrixd.m[1];
        m->m02 = (GL_float)data->load_matrixd.m[2];
        m->m03 = (GL_float)data->load_matrixd.m[3];
        m->m10 = (GL_float)data->load_matrixd.m[4];
        m->m11 = (GL_float)data->load_matrixd.m[5];
        m->m12 = (GL_float)data->load_matrixd.m[6];
        m->m13 = (GL_float)data->load_matrixd.m[7];
        m->m20 = (GL_float)data->load_matrixd.m[8];
        m->m21 = (GL_float)data->load_matrixd.m[9];
        m->m22 = (GL_float)data->load_matrixd.m[10];
        m->m23 = (GL_float)data->load_matrixd.m[11];
        m->m30 = (GL_float)data->load_matrixd.m[12];
        m->m31 = (GL_float)data->load_matrixd.m[13];
        m->m32 = (GL_float)data->load_matrixd.m[14];
        m->m33 = (GL_float)data->load_matrixd.m[15];
        break;
    }
    case GLT_OP_LOAD_MATRIXF:
    {
        GL_matrix *m = m_cur_matrix(ctx);
        m->m00 = (GL_float)data->load_matrixf.m[0];
        m->m01 = (GL_float)data->load_matrixf.m[1];
        m->m02 = (GL_float)data->load_matrixf.m[2];
        m->m03 = (GL_float)data->load_matrixf.m[3];
        m->m10 = (GL_float)data->load_matrixf.m[4];
        m->m11 = (GL_float)data->load_matrixf.m[5];
        m->m12 = (GL_float)data->load_matrixf.m[6];
        m->m13 = (GL_float)data->load_matrixf.m[7];
        m->m20 = (GL_float)data->load_matrixf.m[8];
        m->m21 = (GL_float)data->load_matrixf.m[9];
        m->m22 = (GL_float)data->load_matrixf.m[10];
        m->m23 = (GL_float)data->load_matrixf.m[11];
        m->m30 = (GL_float)data->load_matrixf.m[12];
        m->m31 = (GL_float)data->load_matrixf.m[13];
        m->m32 = (GL_float)data->load_matrixf.m[14];
        m->m33 = (GL_float)data->load_matrixf.m[15];
        break;
    }
    case GLT_OP_MATERIALF:
    case GLT_OP_MATERIALFV:
        Material(ctx, &data->materialfv);
        break;
    case GLT_OP_MATERIALI:
    case GLT_OP_MATERIALIV:
    {
        GLT_materialfv fmaterial;
        /* Convert to Materialfv */
        fmaterial.face = data->materialiv.face;
        fmaterial.pname = data->materialiv.pname;
        switch (data->materialiv.pname) {
        case GL_AMBIENT:
        case GL_DIFFUSE:
        case GL_SPECULAR:
        case GL_EMISSION:
        case GL_AMBIENT_AND_DIFFUSE:
            /* convert params as colors */
            fmaterial.params[0] = CINT(data->materialiv.params[0]);
            fmaterial.params[1] = CINT(data->materialiv.params[1]);
            fmaterial.params[2] = CINT(data->materialiv.params[2]);
            fmaterial.params[3] = CINT(data->materialiv.params[3]);
            break;
        case GL_SHININESS:
            /* no conversion */
            fmaterial.params[0] = data->materialiv.params[0];
            break;
        case GL_COLOR_INDEXES:
            /* no conversion */
            fmaterial.params[0] = data->materialiv.params[0];
            fmaterial.params[1] = data->materialiv.params[1];
            fmaterial.params[2] = data->materialiv.params[2];
            fmaterial.params[3] = data->materialiv.params[3];
            break;
        default:
            /* do nothing, will handle later */
            break;
        }
        Material(ctx, &fmaterial);
        break;
    }
    case GLT_OP_MATRIX_MODE:
        switch(data->matrix_mode.mode) {
        case GL_MODELVIEW:
        case GL_PROJECTION:
        case GL_TEXTURE:
            ctx->matrix_mode = data->matrix_mode.mode;
            break;
        default:
            ONCE(glt_warn("%s: invalid matrix mode (0x%04x)", loc,
                          data->matrix_mode.mode));
            break;
        }
        break;
    case GLT_OP_MULT_MATRIXD:
    {
        GL_matrix *m = m_cur_matrix(ctx);
        m->m00 = (GL_float)data->load_matrixd.m[0];
        m->m01 = (GL_float)data->load_matrixd.m[1];
        m->m02 = (GL_float)data->load_matrixd.m[2];
        m->m03 = (GL_float)data->load_matrixd.m[3];
        m->m10 = (GL_float)data->load_matrixd.m[4];
        m->m11 = (GL_float)data->load_matrixd.m[5];
        m->m12 = (GL_float)data->load_matrixd.m[6];
        m->m13 = (GL_float)data->load_matrixd.m[7];
        m->m20 = (GL_float)data->load_matrixd.m[8];
        m->m21 = (GL_float)data->load_matrixd.m[9];
        m->m22 = (GL_float)data->load_matrixd.m[10];
        m->m23 = (GL_float)data->load_matrixd.m[11];
        m->m30 = (GL_float)data->load_matrixd.m[12];
        m->m31 = (GL_float)data->load_matrixd.m[13];
        m->m32 = (GL_float)data->load_matrixd.m[14];
        m->m33 = (GL_float)data->load_matrixd.m[15];
        m_multiply(m_cur_matrix(ctx), m);
        break;
    }
    case GLT_OP_MULT_MATRIXF:
    {
        GL_matrix *m = m_cur_matrix(ctx);
        m->m00 = (GL_float)data->load_matrixf.m[0];
        m->m01 = (GL_float)data->load_matrixf.m[1];
        m->m02 = (GL_float)data->load_matrixf.m[2];
        m->m03 = (GL_float)data->load_matrixf.m[3];
        m->m10 = (GL_float)data->load_matrixf.m[4];
        m->m11 = (GL_float)data->load_matrixf.m[5];
        m->m12 = (GL_float)data->load_matrixf.m[6];
        m->m13 = (GL_float)data->load_matrixf.m[7];
        m->m20 = (GL_float)data->load_matrixf.m[8];
        m->m21 = (GL_float)data->load_matrixf.m[9];
        m->m22 = (GL_float)data->load_matrixf.m[10];
        m->m23 = (GL_float)data->load_matrixf.m[11];
        m->m30 = (GL_float)data->load_matrixf.m[12];
        m->m31 = (GL_float)data->load_matrixf.m[13];
        m->m32 = (GL_float)data->load_matrixf.m[14];
        m->m33 = (GL_float)data->load_matrixf.m[15];
        m_multiply(m_cur_matrix(ctx), m);
        break;
    }
    case GLT_OP_NEW_LIST:
        break;
    case GLT_OP_NORMAL_3B:
    case GLT_OP_NORMAL_3BV:
        Normal(ctx, NBYTE(data->normal_3b.nx),
               NBYTE(data->normal_3b.ny),
               NBYTE(data->normal_3b.nz));
        break;
    case GLT_OP_NORMAL_3D:
    case GLT_OP_NORMAL_3DV:
        Normal(ctx, NDOUBLE(data->normal_3d.nx),
               NDOUBLE(data->normal_3d.ny),
               NDOUBLE(data->normal_3d.nz));
        break;
    case GLT_OP_NORMAL_3F:
    case GLT_OP_NORMAL_3FV:
        Normal(ctx, NFLOAT(data->normal_3f.nx),
               NFLOAT(data->normal_3f.ny),
               NFLOAT(data->normal_3f.nz));
        break;
    case GLT_OP_NORMAL_3I:
    case GLT_OP_NORMAL_3IV:
        Normal(ctx, NINT(data->normal_3i.nx),
               NINT(data->normal_3i.ny),
               NINT(data->normal_3i.nz));
        break;
    case GLT_OP_NORMAL_3S:
    case GLT_OP_NORMAL_3SV:
        Normal(ctx, NSHORT(data->normal_3s.nx),
               NSHORT(data->normal_3s.ny),
               NSHORT(data->normal_3s.nz));
        break;
    case GLT_OP_ORTHO:
        m_ortho(m_cur_matrix(ctx), data->ortho.left, data->ortho.right,
                data->ortho.bottom, data->ortho.top, data->ortho.znear,
                data->ortho.zfar);
        break;
    case GLT_OP_PIXEL_STOREF:
        PixelStore(ctx, &data->pixel_storef);
        break;
    case GLT_OP_PIXEL_STOREI:
    {
        GLT_pixel_storef pixel_storef;
        pixel_storef.pname = data->pixel_storei.pname;
        pixel_storef.param = data->pixel_storei.param;
        PixelStore(ctx, &pixel_storef);
        break;
    }
    case GLT_OP_PIXEL_TRANSFERF:
        PixelTransfer(ctx, &data->pixel_transferf);
        /* Hack! Pixel transfer has no effect on texture downloads! */
        ONCE(glt_warn("%s: glPixelTransfer will have no effect on textures",
                      loc));
        break;
    case GLT_OP_PIXEL_TRANSFERI:
    {
        GLT_pixel_transferf pixel_transferf;
        pixel_transferf.pname = data->pixel_transferi.pname;
        pixel_transferf.param = data->pixel_transferi.param;
        PixelTransfer(ctx, &pixel_transferf);
        /* Hack! Pixel transfer has no effect on texture downloads! */
        ONCE(glt_warn("%s: glPixelTransfer will have no effect on textures",
                      loc));
        break;
    }
    case GLT_OP_PIXEL_ZOOM:
        ctx->pixel_zoom_x = data->pixel_zoom.xfactor;
        ctx->pixel_zoom_y = data->pixel_zoom.yfactor;
        break;
    case GLT_OP_POINT_SIZE:
        ctx->point_size = data->point_size.size;
        break;
    case GLT_OP_POLYGON_MODE:
        switch (data->polygon_mode.face) {
        case GL_FRONT:
            ctx->front_polygon_mode = data->polygon_mode.mode;
            break;
        case GL_BACK:
            ctx->back_polygon_mode = data->polygon_mode.mode;
            break;
        case GL_FRONT_AND_BACK:
            ctx->front_polygon_mode = data->polygon_mode.mode;
            ctx->back_polygon_mode = data->polygon_mode.mode;
            break;
        default:
            ONCE(glt_warn("%s: invalid polygon mode face (0x%04x)", loc,
                          data->polygon_mode.face));
        }
        break;
    case GLT_OP_POLYGON_OFFSET:
    case GLT_OP_POLYGON_OFFSET_EXT:
        ctx->polygon_offset_factor = data->polygon_offset.factor;
        ctx->polygon_offset_bias = data->polygon_offset.units;
        break;
    case GLT_OP_POP_ATTRIB:
        ONCE(ignore(loc, "glPopAttrib"));
        rval = 0;
        break;
    case GLT_OP_POP_MATRIX:
        if (--(*m_cur_matrix_index(ctx)) < 0) {
            (*m_cur_matrix_index(ctx)) = 0;
            glt_warn("%s: popped empty matrix stack (matrix mode 0x%04x)",
                     loc, ctx->matrix_mode);
        }
        break;
    case GLT_OP_PRIORITIZE_TEXTURES:
        ONCE(ignore(loc, "glPrioritizeTextures"));
        rval = 0;
        break;
    case GLT_OP_PRIORITIZE_TEXTURES_EXT:
        ONCE(ignore(loc, "glPrioritizeTexturesEXT"));
        rval = 0;
        break;
    case GLT_OP_PUSH_ATTRIB:
        ONCE(ignore(loc, "glPushAttrib"));
        rval = 0;
        break;
    case GLT_OP_PUSH_MATRIX:
        if (++(*m_cur_matrix_index(ctx)) >= m_cur_matrix_max_index(ctx)) {
            (*m_cur_matrix_index(ctx)) = m_cur_matrix_max_index(ctx) - 1;
            glt_warn("%s: pushed full matrix stack (matrix mode 0x%04x)",
                     loc, ctx->matrix_mode);
        }
        break;
    case GLT_OP_RASTER_POS_2D:
    case GLT_OP_RASTER_POS_2DV:
        RasterPos2(ctx, VDOUBLE(data->raster_pos_2d.x),
                   VDOUBLE(data->raster_pos_2d.y));
        break;
    case GLT_OP_RASTER_POS_2F:
    case GLT_OP_RASTER_POS_2FV:
        RasterPos2(ctx, VFLOAT(data->raster_pos_2f.x),
                   VFLOAT(data->raster_pos_2f.y));
        break;
    case GLT_OP_RASTER_POS_2I:
    case GLT_OP_RASTER_POS_2IV:
        RasterPos2(ctx, VINT(data->raster_pos_2i.x),
                   VINT(data->raster_pos_2i.y));
        break;
    case GLT_OP_RASTER_POS_2S:
    case GLT_OP_RASTER_POS_2SV:
        RasterPos2(ctx, VSHORT(data->raster_pos_2s.x),
                   VSHORT(data->raster_pos_2s.y));
        break;
    case GLT_OP_RASTER_POS_3D:
    case GLT_OP_RASTER_POS_3DV:
        RasterPos3(ctx, VDOUBLE(data->raster_pos_3d.x),
                   VDOUBLE(data->raster_pos_3d.y),
                   VDOUBLE(data->raster_pos_3d.z));
        break;
    case GLT_OP_RASTER_POS_3F:
    case GLT_OP_RASTER_POS_3FV:
        RasterPos3(ctx, VFLOAT(data->raster_pos_3f.x),
                   VFLOAT(data->raster_pos_3f.y),
                   VFLOAT(data->raster_pos_3f.z));
        break;
    case GLT_OP_RASTER_POS_3I:
    case GLT_OP_RASTER_POS_3IV:
        RasterPos3(ctx, VINT(data->raster_pos_3i.x),
                   VINT(data->raster_pos_3i.y),
                   VINT(data->raster_pos_3i.z));
        break;
    case GLT_OP_RASTER_POS_3S:
    case GLT_OP_RASTER_POS_3SV:
        RasterPos3(ctx, VSHORT(data->raster_pos_3s.x),
                   VSHORT(data->raster_pos_3s.y),
                   VSHORT(data->raster_pos_3s.z));
        break;
    case GLT_OP_RASTER_POS_4D:
    case GLT_OP_RASTER_POS_4DV:
        RasterPos4(ctx, VDOUBLE(data->raster_pos_4d.x),
                   VDOUBLE(data->raster_pos_4d.y),
                   VDOUBLE(data->raster_pos_4d.z),
                   VDOUBLE(data->raster_pos_4d.w));
        break;
    case GLT_OP_RASTER_POS_4F:
    case GLT_OP_RASTER_POS_4FV:
        RasterPos4(ctx, VFLOAT(data->raster_pos_4f.x),
                   VFLOAT(data->raster_pos_4f.y),
                   VFLOAT(data->raster_pos_4f.z),
                   VFLOAT(data->raster_pos_4f.w));
        break;
    case GLT_OP_RASTER_POS_4I:
    case GLT_OP_RASTER_POS_4IV:
        RasterPos4(ctx, VINT(data->raster_pos_4i.x),
                   VINT(data->raster_pos_4i.y),
                   VINT(data->raster_pos_4i.z),
                   VINT(data->raster_pos_4i.w));
        break;
    case GLT_OP_RASTER_POS_4S:
    case GLT_OP_RASTER_POS_4SV:
        RasterPos4(ctx, VSHORT(data->raster_pos_4s.x),
                   VSHORT(data->raster_pos_4s.y),
                   VSHORT(data->raster_pos_4s.z),
                   VSHORT(data->raster_pos_4s.w));
        break;
    case GLT_OP_RECTD:
    case GLT_OP_RECTDV:
    case GLT_OP_RECTF:
    case GLT_OP_RECTFV:
    case GLT_OP_RECTI:
    case GLT_OP_RECTIV:
    case GLT_OP_RECTS:
    case GLT_OP_RECTSV:
        break;
    case GLT_OP_ROTATED:
        m_rotate(m_cur_matrix(ctx), data->rotated.angle, data->rotated.x,
                 data->rotated.y, data->rotated.z);
        break;
    case GLT_OP_ROTATEF:
        m_rotate(m_cur_matrix(ctx), data->rotatef.angle, data->rotatef.x,
                 data->rotatef.y, data->rotatef.z);
        break;
    case GLT_OP_SCALED:
        m_scale(m_cur_matrix(ctx), data->scaled.x, data->scaled.y,
                 data->scaled.z);
        break;
    case GLT_OP_SCALEF:
        m_scale(m_cur_matrix(ctx), data->scalef.x, data->scalef.y,
                 data->scalef.z);
        break;
    case GLT_OP_SCISSOR:
        ctx->scissor.x = data->scissor.x;
        ctx->scissor.y = data->scissor.y;
        ctx->scissor.w = data->scissor.width;
        ctx->scissor.h = data->scissor.height;
        break;
    case GLT_OP_SHADE_MODEL:
        ctx->shade_model = data->shade_model.mode;
        break;
    case GLT_OP_STENCIL_FUNC:
        ctx->stencil_func = data->stencil_func.func;
        ctx->stencil_ref = data->stencil_func.ref;
        ctx->stencil_mask = data->stencil_func.mask;
        break;
    case GLT_OP_STENCIL_MASK:
        ctx->stencil_wmask = data->stencil_mask.mask;
        break;
    case GLT_OP_STENCIL_OP:
        ctx->stencil_op_fail = data->stencil_op.fail;
        ctx->stencil_op_zfail = data->stencil_op.zfail;
        ctx->stencil_op_zpass = data->stencil_op.zpass;
        break;
    case GLT_OP_TEX_COORD_1D:
    case GLT_OP_TEX_COORD_1DV:
        TexCoord1(ctx, TDOUBLE(data->tex_coord_1d.s));
        break;
    case GLT_OP_TEX_COORD_1F:
    case GLT_OP_TEX_COORD_1FV:
        TexCoord1(ctx, TFLOAT(data->tex_coord_1f.s));
        break;
    case GLT_OP_TEX_COORD_1I:
    case GLT_OP_TEX_COORD_1IV:
        TexCoord1(ctx, TINT(data->tex_coord_1i.s));
        break;
    case GLT_OP_TEX_COORD_1S:
    case GLT_OP_TEX_COORD_1SV:
        TexCoord1(ctx, TSHORT(data->tex_coord_1s.s));
        break;
    case GLT_OP_TEX_COORD_2D:
    case GLT_OP_TEX_COORD_2DV:
        TexCoord2(ctx, TDOUBLE(data->tex_coord_2d.s),
                  TDOUBLE(data->tex_coord_2d.t));
        break;
    case GLT_OP_TEX_COORD_2F:
    case GLT_OP_TEX_COORD_2FV:
        TexCoord2(ctx, TFLOAT(data->tex_coord_2f.s),
                  TFLOAT(data->tex_coord_2f.t));
        break;
    case GLT_OP_TEX_COORD_2I:
    case GLT_OP_TEX_COORD_2IV:
        TexCoord2(ctx, TINT(data->tex_coord_2i.s),
                  TINT(data->tex_coord_2i.t));
        break;
    case GLT_OP_TEX_COORD_2S:
    case GLT_OP_TEX_COORD_2SV:
        TexCoord2(ctx, TSHORT(data->tex_coord_2s.s),
                  TSHORT(data->tex_coord_2s.t));
        break;
    case GLT_OP_TEX_COORD_3D:
    case GLT_OP_TEX_COORD_3DV:
        TexCoord3(ctx, TDOUBLE(data->tex_coord_3d.s),
                  TDOUBLE(data->tex_coord_3d.t),
                  TDOUBLE(data->tex_coord_3d.r));
        break;
    case GLT_OP_TEX_COORD_3F:
    case GLT_OP_TEX_COORD_3FV:
        TexCoord3(ctx, TFLOAT(data->tex_coord_3f.s),
                  TFLOAT(data->tex_coord_3f.t),
                  TFLOAT(data->tex_coord_3f.r));
        break;
    case GLT_OP_TEX_COORD_3I:
    case GLT_OP_TEX_COORD_3IV:
        TexCoord3(ctx, TINT(data->tex_coord_3i.s),
                  TINT(data->tex_coord_3i.t),
                  TINT(data->tex_coord_3i.r));
        break;
    case GLT_OP_TEX_COORD_3S:
    case GLT_OP_TEX_COORD_3SV:
        TexCoord3(ctx, TSHORT(data->tex_coord_3s.s),
                  TSHORT(data->tex_coord_3s.t),
                  TSHORT(data->tex_coord_3s.r));
        break;
    case GLT_OP_TEX_COORD_4D:
    case GLT_OP_TEX_COORD_4DV:
        TexCoord4(ctx, TDOUBLE(data->tex_coord_4d.s),
                  TDOUBLE(data->tex_coord_4d.t),
                  TDOUBLE(data->tex_coord_4d.r),
                  TDOUBLE(data->tex_coord_4d.q));
        break;
    case GLT_OP_TEX_COORD_4F:
    case GLT_OP_TEX_COORD_4FV:
        TexCoord4(ctx, TFLOAT(data->tex_coord_4f.s),
                  TFLOAT(data->tex_coord_4f.t),
                  TFLOAT(data->tex_coord_4f.r),
                  TFLOAT(data->tex_coord_4f.q));
        break;
    case GLT_OP_TEX_COORD_4I:
    case GLT_OP_TEX_COORD_4IV:
        TexCoord4(ctx, TINT(data->tex_coord_4i.s),
                  TINT(data->tex_coord_4i.t),
                  TINT(data->tex_coord_4i.r),
                  TINT(data->tex_coord_4i.q));
        break;
    case GLT_OP_TEX_COORD_4S:
    case GLT_OP_TEX_COORD_4SV:
        TexCoord4(ctx, TSHORT(data->tex_coord_4s.s),
                  TSHORT(data->tex_coord_4s.t),
                  TSHORT(data->tex_coord_4s.r),
                  TSHORT(data->tex_coord_4s.q));
        break;
    case GLT_OP_TEX_ENVF:
    case GLT_OP_TEX_ENVFV:
        TexEnv(ctx, &data->tex_envfv);
        break;
    case GLT_OP_TEX_ENVI:
    case GLT_OP_TEX_ENVIV:
    {
        GLT_tex_envfv texenvfv;
        /* Convert to tex_envfv */
        texenvfv.target = data->tex_enviv.target;
        texenvfv.pname = data->tex_enviv.pname;
        switch (data->tex_enviv.pname) {
        case GL_TEXTURE_ENV_MODE:
            texenvfv.params[0] = data->tex_parameteriv.params[0];
            break;
        case GL_TEXTURE_ENV_COLOR:
            texenvfv.params[0] = CINT(data->tex_enviv.params[0]);
            texenvfv.params[1] = CINT(data->tex_enviv.params[1]);
            texenvfv.params[2] = CINT(data->tex_enviv.params[2]);
            texenvfv.params[3] = CINT(data->tex_enviv.params[3]);
            break;
        default:
            /* do nothing, will handle later */
            break;
        }
        TexEnv(ctx, &texenvfv);
        break;
    }
    case GLT_OP_TEX_IMAGE_2D:
        switch (data->tex_image_2d.target) {
        case GL_TEXTURE_2D:
        {
            GL_texlevel *level = t_curtexgetlev(ctx, data);

            if (level->valid && level->pixels)
                glt_free(level->pixels);

            level->valid = 1;
            level->width = data->tex_image_2d.width;
            level->height = data->tex_image_2d.height;
            level->components = data->tex_image_2d.components;
            level->border = data->tex_image_2d.border;
            level->format = data->tex_image_2d.format;
            level->type = data->tex_image_2d.type;
            level->n_pixels = data->tex_image_2d.n_pixels;
            if (data->tex_image_2d.pixels) {
                level->pixels = glt_malloc(data->tex_image_2d.n_pixels);
                memcpy(level->pixels, data->tex_image_2d.pixels,
                       data->tex_image_2d.n_pixels);
            }
            else {
                level->pixels = NULL;
            }
            level->swap_bytes = ctx->unpack_swap_bytes;
            level->lsb_first = ctx->unpack_lsb_first;
            break;
        }
        default:
            ONCE(glt_warn("%s: unknown tex image texture target (0x%04x)",
                          loc, data->bind_texture.target));
            break;
        }
        break;
    case GLT_OP_TEX_PARAMETERF:
    case GLT_OP_TEX_PARAMETERFV:
        TexParameter(ctx, &data->tex_parameterfv);
        break;
    case GLT_OP_TEX_PARAMETERI:
    case GLT_OP_TEX_PARAMETERIV:
    {
        GLT_tex_parameterfv texparameterfv;
        /* Convert to tex_parameterfv */
        texparameterfv.target = data->tex_parameteriv.target;
        texparameterfv.pname = data->tex_parameteriv.pname;
        switch (data->tex_parameteriv.pname) {
        case GL_TEXTURE_MIN_FILTER:
        case GL_TEXTURE_MAG_FILTER:
        case GL_TEXTURE_WRAP_S:
        case GL_TEXTURE_WRAP_T:
            texparameterfv.params[0] = data->tex_parameteriv.params[0];
            break;
        case GL_TEXTURE_BORDER_COLOR:
            texparameterfv.params[0] = CINT(data->tex_parameteriv.params[0]);
            texparameterfv.params[1] = CINT(data->tex_parameteriv.params[1]);
            texparameterfv.params[2] = CINT(data->tex_parameteriv.params[2]);
            texparameterfv.params[3] = CINT(data->tex_parameteriv.params[3]);
            break;
        default:
            /* do nothing, will handle later */
            break;
        }
        TexParameter(ctx, &texparameterfv);
        break;
    }
    case GLT_OP_TEX_SUB_IMAGE_2D:
    case GLT_OP_TEX_SUB_IMAGE_2D_EXT:
        switch (data->tex_image_2d.target) {
        case GL_TEXTURE_2D:
            ONCE(glt_warn("%s: glTexSubImage2D not supported", loc));
            ONCE(glt_warn("%s: some textures will not be correct", loc));
            break;
        default:
            ONCE(glt_warn("%s: unknown tex sub image texture target (0x%04x)",
                          loc, data->bind_texture.target));
            break;
        }
        break;
    case GLT_OP_TRANSLATED:
        m_translate(m_cur_matrix(ctx), data->translated.x, data->translated.y,
                    data->translated.z);
        break;
    case GLT_OP_TRANSLATEF:
        m_translate(m_cur_matrix(ctx), data->translatef.x, data->translatef.y,
                    data->translatef.z);
        break;
    case GLT_OP_VERTEX_2D:
    case GLT_OP_VERTEX_2DV:
    case GLT_OP_VERTEX_2F:
    case GLT_OP_VERTEX_2FV:
    case GLT_OP_VERTEX_2I:
    case GLT_OP_VERTEX_2IV:
    case GLT_OP_VERTEX_2S:
    case GLT_OP_VERTEX_2SV:
    case GLT_OP_VERTEX_3D:
    case GLT_OP_VERTEX_3DV:
    case GLT_OP_VERTEX_3F:
    case GLT_OP_VERTEX_3FV:
    case GLT_OP_VERTEX_3I:
    case GLT_OP_VERTEX_3IV:
    case GLT_OP_VERTEX_3S:
    case GLT_OP_VERTEX_3SV:
    case GLT_OP_VERTEX_4D:
    case GLT_OP_VERTEX_4DV:
    case GLT_OP_VERTEX_4F:
    case GLT_OP_VERTEX_4FV:
    case GLT_OP_VERTEX_4I:
    case GLT_OP_VERTEX_4IV:
    case GLT_OP_VERTEX_4S:
    case GLT_OP_VERTEX_4SV:
        break;
    case GLT_OP_VIEWPORT:
        ctx->viewport.x = data->viewport.x;
        ctx->viewport.y = data->viewport.y;
        ctx->viewport.w = data->viewport.width;
        ctx->viewport.h = data->viewport.height;
        break;
    default:
        glt_fatal("%s: invalid opcode %d", loc, op);
        break;
    }

    return rval;
}

/* Put function */

static __inline int
match_coord(GL_coord *c, GL_float x, GL_float y, GL_float z, GL_float w)
{
    return (c->x == x && c->y == y && c->z == z && c->w == w);
}

static __inline int
match_color(GL_color *c, GL_float r, GL_float g, GL_float b, GL_float a)
{
    return (c->r == r && c->g == g && c->b == b && c->a == a);
}

static __inline int
match_vector(GL_vector *v, GL_float x, GL_float y, GL_float z)
{
    return (v->x == x && v->y == y && v->z == z);
}

static __inline int
match_texcoord(GL_texcoord *tc, GL_float s, GL_float t, GL_float r, GL_float q)
{
    return (tc->s == s && tc->t == t && tc->r == r && tc->q == q);
}

static __inline int
match_rect(GL_rect *r, GL_int x, GL_int y, GL_int w, GL_int h)
{
    return (r->x == x && r->y == y && r->w == w && r->h == h);
}

void
gs_put(GLT_trace *trace, GL_context *ctx)
{
    int i;
    GLT_data data;
    GL_int matrix_mode = GL_MODELVIEW;
    GL_int cur_texture = 0;
    GL_bool swap_bytes = GL_FALSE;
    GL_bool lsb_first = GL_FALSE;

    /* Current color */
    if (!match_color(&ctx->color, 1.0, 1.0, 1.0, 1.0)) {
        data.color_4f.red = ctx->color.r;
        data.color_4f.green = ctx->color.g;
        data.color_4f.blue = ctx->color.b;
        data.color_4f.alpha = ctx->color.a;
        glt_put(trace, GLT_OP_COLOR_4F, &data);
    }

    /* Current normal */
    if (!match_vector(&ctx->normal, 0.0, 0.0, 1.0)) {
        data.normal_3f.nx = ctx->normal.x;
        data.normal_3f.ny = ctx->normal.y;
        data.normal_3f.nz = ctx->normal.z;
        glt_put(trace, GLT_OP_NORMAL_3F, &data);
    }

    /* Current texture coordinates */
    if (!match_texcoord(&ctx->texcoord, 0.0, 0.0, 0.0, 1.0)) {
        data.tex_coord_4f.s = ctx->texcoord.s;
        data.tex_coord_4f.t = ctx->texcoord.t;
        data.tex_coord_4f.r = ctx->texcoord.r;
        data.tex_coord_4f.q = ctx->texcoord.q;
        glt_put(trace, GLT_OP_TEX_COORD_4F, &data);
    }

    /* Current front material ambient */
    if (!match_color(&ctx->front_material.a, 0.2, 0.2, 0.2, 1.0)) {
        data.materialfv.face = GL_FRONT;
        data.materialfv.pname = GL_AMBIENT;
        data.materialfv.params[0] = ctx->front_material.a.r;
        data.materialfv.params[1] = ctx->front_material.a.g;
        data.materialfv.params[2] = ctx->front_material.a.b;
        data.materialfv.params[3] = ctx->front_material.a.a;
        glt_put(trace, GLT_OP_MATERIALFV, &data);
    }

    /* Current front material diffuse */
    if (!match_color(&ctx->front_material.d, 0.8, 0.8, 0.8, 1.0)) {
        data.materialfv.face = GL_FRONT;
        data.materialfv.pname = GL_DIFFUSE;
        data.materialfv.params[0] = ctx->front_material.d.r;
        data.materialfv.params[1] = ctx->front_material.d.g;
        data.materialfv.params[2] = ctx->front_material.d.b;
        data.materialfv.params[3] = ctx->front_material.d.a;
        glt_put(trace, GLT_OP_MATERIALFV, &data);
    }

    /* Current front material specular */
    if (!match_color(&ctx->front_material.s, 0.0, 0.0, 0.0, 1.0)) {
        data.materialfv.face = GL_FRONT;
        data.materialfv.pname = GL_SPECULAR;
        data.materialfv.params[0] = ctx->front_material.s.r;
        data.materialfv.params[1] = ctx->front_material.s.g;
        data.materialfv.params[2] = ctx->front_material.s.b;
        data.materialfv.params[3] = ctx->front_material.s.a;
        glt_put(trace, GLT_OP_MATERIALFV, &data);
    }

    /* Current front material emission */
    if (!match_color(&ctx->front_material.e, 0.0, 0.0, 0.0, 1.0)) {
        data.materialfv.face = GL_FRONT;
        data.materialfv.pname = GL_EMISSION;
        data.materialfv.params[0] = ctx->front_material.e.r;
        data.materialfv.params[1] = ctx->front_material.e.g;
        data.materialfv.params[2] = ctx->front_material.e.b;
        data.materialfv.params[3] = ctx->front_material.e.a;
        glt_put(trace, GLT_OP_MATERIALFV, &data);
    }

    /* Current front material shininess */
    if (ctx->front_material.sh != 0.0) {
        data.materialf.face = GL_FRONT;
        data.materialf.pname = GL_SHININESS;
        data.materialf.param = ctx->front_material.sh;
        glt_put(trace, GLT_OP_MATERIALF, &data);
    }

    /* Current back material ambient */
    if (!match_color(&ctx->back_material.a, 0.2, 0.2, 0.2, 1.0)) {
        data.materialfv.face = GL_BACK;
        data.materialfv.pname = GL_AMBIENT;
        data.materialfv.params[0] = ctx->back_material.a.r;
        data.materialfv.params[1] = ctx->back_material.a.g;
        data.materialfv.params[2] = ctx->back_material.a.b;
        data.materialfv.params[3] = ctx->back_material.a.a;
        glt_put(trace, GLT_OP_MATERIALFV, &data);
    }

    /* Current back material diffuse */
    if (!match_color(&ctx->back_material.d, 0.8, 0.8, 0.8, 1.0)) {
        data.materialfv.face = GL_BACK;
        data.materialfv.pname = GL_DIFFUSE;
        data.materialfv.params[0] = ctx->back_material.d.r;
        data.materialfv.params[1] = ctx->back_material.d.g;
        data.materialfv.params[2] = ctx->back_material.d.b;
        data.materialfv.params[3] = ctx->back_material.d.a;
        glt_put(trace, GLT_OP_MATERIALFV, &data);
    }

    /* Current back material specular */
    if (!match_color(&ctx->back_material.s, 0.0, 0.0, 0.0, 1.0)) {
        data.materialfv.face = GL_BACK;
        data.materialfv.pname = GL_SPECULAR;
        data.materialfv.params[0] = ctx->back_material.s.r;
        data.materialfv.params[1] = ctx->back_material.s.g;
        data.materialfv.params[2] = ctx->back_material.s.b;
        data.materialfv.params[3] = ctx->back_material.s.a;
        glt_put(trace, GLT_OP_MATERIALFV, &data);
    }

    /* Current back material emission */
    if (!match_color(&ctx->back_material.e, 0.0, 0.0, 0.0, 1.0)) {
        data.materialfv.face = GL_BACK;
        data.materialfv.pname = GL_EMISSION;
        data.materialfv.params[0] = ctx->back_material.e.r;
        data.materialfv.params[1] = ctx->back_material.e.g;
        data.materialfv.params[2] = ctx->back_material.e.b;
        data.materialfv.params[3] = ctx->back_material.e.a;
        glt_put(trace, GLT_OP_MATERIALFV, &data);
    }

    /* Current back material shininess */
    if (ctx->back_material.sh != 0.0) {
        data.materialf.face = GL_BACK;
        data.materialf.pname = GL_SHININESS;
        data.materialf.param = ctx->back_material.sh;
        glt_put(trace, GLT_OP_MATERIALF, &data);
    }

    /* Current edge flag */
    if (ctx->edge_flag != GL_TRUE) {
        data.edge_flag.flag = ctx->edge_flag;
        glt_put(trace, GLT_OP_EDGE_FLAG, &data);
    }

    /* Current raster position */
    if (!match_coord(&ctx->raster_pos, 0.0, 0.0, 0.0, 1.0)) {
        data.raster_pos_4f.x = ctx->raster_pos.x;
        data.raster_pos_4f.y = ctx->raster_pos.y;
        data.raster_pos_4f.z = ctx->raster_pos.z;
        data.raster_pos_4f.w = ctx->raster_pos.w;
        glt_put(trace, GLT_OP_RASTER_POS_4F, &data);
    }

    /* Current color index */
    if (ctx->index != FLOAT_ONE) {
        data.indexf.c = ctx->index;
        glt_put(trace, GLT_OP_INDEXF, &data);
    }

    /* Clear accum */
    if (!match_color(&ctx->clear_accum, 0.0, 0.0, 0.0, 0.0)) {
        data.clear_accum.red = ctx->clear_accum.r;
        data.clear_accum.green = ctx->clear_accum.g;
        data.clear_accum.blue = ctx->clear_accum.b;
        data.clear_accum.alpha = ctx->clear_accum.a;
        glt_put(trace, GLT_OP_CLEAR_ACCUM, &data);
    }

    /* Clear color */
    if (!match_color(&ctx->clear_color, 0.0, 0.0, 0.0, 0.0)) {
        data.clear_color.red = ctx->clear_color.r;
        data.clear_color.green = ctx->clear_color.g;
        data.clear_color.blue = ctx->clear_color.b;
        data.clear_color.alpha = ctx->clear_color.a;
        glt_put(trace, GLT_OP_CLEAR_COLOR, &data);
    }

    /* Clear depth */
    if (ctx->clear_depth != 1.0) {
        data.clear_depth.depth = ctx->clear_depth;
        glt_put(trace, GLT_OP_CLEAR_DEPTH, &data);
    }

    /* Clear index */
    if (ctx->clear_index != FLOAT_ZERO) {
        data.clear_index.c = ctx->clear_index;
        glt_put(trace, GLT_OP_CLEAR_INDEX, &data);
    }

    /* Clear stencil */
    if (ctx->clear_stencil != 0) {
        data.clear_stencil.s = ctx->clear_stencil;
        glt_put(trace, GLT_OP_CLEAR_STENCIL, &data);
    }

    /* Color material */
    if (ctx->color_material_face != GL_FRONT_AND_BACK ||
        ctx->color_material_mode != GL_AMBIENT_AND_DIFFUSE) {
        data.color_material.face = ctx->color_material_face;
        data.color_material.mode = ctx->color_material_mode;
        glt_put(trace, GLT_OP_COLOR_MATERIAL, &data);
    }

    /* Alpha func */
    if (ctx->alpha_func != GL_ALWAYS || ctx->alpha_ref != 0.0) {
        data.alpha_func.func = ctx->alpha_func;
        data.alpha_func.ref = ctx->alpha_ref;
        glt_put(trace, GLT_OP_ALPHA_FUNC, &data);
    }

    /* Blend func */
    if (ctx->blend_src != GL_ONE || ctx->blend_dst != GL_ZERO) {
        data.blend_func.sfactor = ctx->blend_src;
        data.blend_func.dfactor = ctx->blend_dst;
        glt_put(trace, GLT_OP_BLEND_FUNC, &data);
    }

    /* Depth func */
    if (ctx->depth_func != GL_LESS) {
        data.depth_func.func = ctx->depth_func;
        glt_put(trace, GLT_OP_DEPTH_FUNC, &data);
    }

    /* Stencil func */
    if (ctx->stencil_func != GL_ALWAYS || ctx->stencil_ref != 0 ||
        ctx->stencil_mask != (GL_uint)(-1)) {
        data.stencil_func.func = ctx->stencil_func;
        data.stencil_func.ref = ctx->stencil_ref;
        data.stencil_func.mask = ctx->stencil_mask;
        glt_put(trace, GLT_OP_STENCIL_FUNC, &data);
    }

    /* Stencil op */
    if (ctx->stencil_op_fail != GL_KEEP || ctx->stencil_op_zfail != GL_KEEP ||
        ctx->stencil_op_zpass != GL_KEEP) {
        data.stencil_op.fail = ctx->stencil_op_fail;
        data.stencil_op.zfail = ctx->stencil_op_zfail;
        data.stencil_op.zpass = ctx->stencil_op_zpass;
        glt_put(trace, GLT_OP_STENCIL_OP, &data);
    }

    /* Clip planes */
    for (i = 0; i < NUM_CLIP_PLANES; i++) {
        if (!match_coord(&ctx->clip_planes[i], 0.0, 0.0, 0.0, 0.0)) {
            data.clip_plane.plane = GL_CLIP_PLANE0 + i;
            data.clip_plane.equation[0] = ctx->clip_planes[i].x;
            data.clip_plane.equation[1] = ctx->clip_planes[i].y;
            data.clip_plane.equation[2] = ctx->clip_planes[i].z;
            data.clip_plane.equation[3] = ctx->clip_planes[i].w;
            glt_put(trace, GLT_OP_CLIP_PLANE, &data);
        }
    }

    /* Viewport */
    if (!match_rect(&ctx->viewport, 0, 0, -1, -1)) {
        data.viewport.x = ctx->viewport.x;
        data.viewport.y = ctx->viewport.y;
        data.viewport.width = ctx->viewport.w;
        data.viewport.height = ctx->viewport.h;
        glt_put(trace, GLT_OP_VIEWPORT, &data);
    }

    /* Depth range */
    if (ctx->depth_range_near != FLOAT_ZERO ||
        ctx->depth_range_far != FLOAT_ONE) {
        data.depth_range.znear = ctx->depth_range_near;
        data.depth_range.zfar = ctx->depth_range_far;
        glt_put(trace, GLT_OP_DEPTH_RANGE, &data);
    }

    /* Scissor rectangle */
    if (!match_rect(&ctx->scissor, 0, 0, -1, -1)) {
        data.scissor.x = ctx->scissor.x;
        data.scissor.y = ctx->scissor.y;
        data.scissor.width = ctx->scissor.w;
        data.scissor.height = ctx->scissor.h;
        glt_put(trace, GLT_OP_SCISSOR, &data);
    }

    /* Point size */
    if (ctx->point_size != FLOAT_ONE) {
        data.point_size.size = ctx->point_size;
        glt_put(trace, GLT_OP_POINT_SIZE, &data);
    }

    /* Line width */
    if (ctx->line_width != FLOAT_ONE) {
        data.line_width.width = ctx->line_width;
        glt_put(trace, GLT_OP_LINE_WIDTH, &data);
    }

    /* Cull face */
    if (ctx->cull_face != GL_BACK) {
        data.cull_face.mode = ctx->cull_face;
        glt_put(trace, GLT_OP_CULL_FACE, &data);
    }

    /* Front face */
    if (ctx->front_face != GL_CCW) {
        data.front_face.mode = ctx->front_face;
        glt_put(trace, GLT_OP_FRONT_FACE, &data);
    }

    /* Shade model */
    if (ctx->shade_model != GL_SMOOTH) {
        data.shade_model.mode = ctx->shade_model;
        glt_put(trace, GLT_OP_SHADE_MODEL, &data);
    }

    /* Draw buffer */
    if (ctx->draw_buffer != GL_BACK) {
        data.draw_buffer.mode = ctx->draw_buffer;
        glt_put(trace, GLT_OP_DRAW_BUFFER, &data);
    }

    /* Front polygon mode */
    if (ctx->front_polygon_mode != GL_FILL) {
        data.polygon_mode.face = GL_FRONT;
        data.polygon_mode.mode = ctx->front_polygon_mode;
        glt_put(trace, GLT_OP_POLYGON_MODE, &data);
    }

    /* Back polygon mode */
    if (ctx->back_polygon_mode != GL_FILL) {
        data.polygon_mode.face = GL_BACK;
        data.polygon_mode.mode = ctx->back_polygon_mode;
        glt_put(trace, GLT_OP_POLYGON_MODE, &data);
    }

    /* Polygon offset */
    if (ctx->polygon_offset_factor != FLOAT_ZERO ||
        ctx->polygon_offset_bias != FLOAT_ZERO) {
        data.polygon_offset.factor = ctx->polygon_offset_factor;
        data.polygon_offset.units = ctx->polygon_offset_bias;
        glt_put(trace, GLT_OP_POLYGON_OFFSET, &data);
    }

    /* Fog mode */
    if (ctx->fog_mode != GL_EXP) {
        data.fogi.pname = GL_FOG_MODE;
        data.fogi.param = ctx->fog_mode;
        glt_put(trace, GLT_OP_FOGI, &data);
    }

    /* Fog density */
    if (ctx->fog_density != FLOAT_ONE) {
        data.fogf.pname = GL_FOG_DENSITY;
        data.fogf.param = ctx->fog_density;
        glt_put(trace, GLT_OP_FOGF, &data);
    }

    /* Fog start */
    if (ctx->fog_start != FLOAT_ZERO) {
        data.fogf.pname = GL_FOG_START;
        data.fogf.param = ctx->fog_start;
        glt_put(trace, GLT_OP_FOGF, &data);
    }

    /* Fog end */
    if (ctx->fog_end != FLOAT_ONE) {
        data.fogf.pname = GL_FOG_END;
        data.fogf.param = ctx->fog_end;
        glt_put(trace, GLT_OP_FOGF, &data);
    }

    /* Fog index */
    if (ctx->fog_index != FLOAT_ZERO) {
        data.fogf.pname = GL_FOG_INDEX;
        data.fogf.param = ctx->fog_index;
        glt_put(trace, GLT_OP_FOGF, &data);
    }

    /* Fog color */
    if (!match_color(&ctx->fog_color, 0.0, 0.0, 0.0, 0.0)) {
        data.fogfv.pname = GL_FOG_COLOR;
        data.fogfv.params[0] = ctx->fog_color.r;
        data.fogfv.params[1] = ctx->fog_color.g;
        data.fogfv.params[2] = ctx->fog_color.b;
        data.fogfv.params[3] = ctx->fog_color.a;
        glt_put(trace, GLT_OP_FOGFV, &data);
    }

    /* Color mask */
    if (ctx->color_write_r != GL_TRUE || ctx->color_write_g != GL_TRUE ||
        ctx->color_write_b != GL_TRUE || ctx->color_write_a != GL_TRUE) {
        data.color_mask.red = ctx->color_write_r;
        data.color_mask.green = ctx->color_write_g;
        data.color_mask.blue = ctx->color_write_b;
        data.color_mask.alpha = ctx->color_write_a;
        glt_put(trace, GLT_OP_COLOR_MASK, &data);
    }

    /* Depth mask */
    if (ctx->depth_write != GL_TRUE) {
        data.depth_mask.flag = ctx->depth_write;
        glt_put(trace, GLT_OP_DEPTH_MASK, &data);
    }

    /* Index mask */
    if (ctx->index_mask != (GL_uint)(-1)) {
        data.index_mask.mask = ctx->index_mask;
        glt_put(trace, GLT_OP_INDEX_MASK, &data);
    }

    /* Stencil mask */
    if (ctx->stencil_wmask != (GL_uint)(-1)) {
        data.stencil_mask.mask = ctx->stencil_wmask;
        glt_put(trace, GLT_OP_STENCIL_MASK, &data);
    }

    /* Modelview matrix stack */
    if (ctx->modelview_stack_index != 0 ||
        !m_is_identity(&ctx->modelview_stack[0])) {

        data.matrix_mode.mode = GL_MODELVIEW;
        glt_put(trace, GLT_OP_MATRIX_MODE, &data);

        for (i = 0; i <= ctx->modelview_stack_index; i++) {
            if (m_is_identity(&ctx->modelview_stack[i])) {
                glt_put(trace, GLT_OP_LOAD_IDENTITY, NULL);
            }
            else {
                data.load_matrixf.m[0] = ctx->modelview_stack[i].m00;
                data.load_matrixf.m[1] = ctx->modelview_stack[i].m01;
                data.load_matrixf.m[2] = ctx->modelview_stack[i].m02;
                data.load_matrixf.m[3] = ctx->modelview_stack[i].m03;
                data.load_matrixf.m[4] = ctx->modelview_stack[i].m10;
                data.load_matrixf.m[5] = ctx->modelview_stack[i].m11;
                data.load_matrixf.m[6] = ctx->modelview_stack[i].m12;
                data.load_matrixf.m[7] = ctx->modelview_stack[i].m13;
                data.load_matrixf.m[8] = ctx->modelview_stack[i].m20;
                data.load_matrixf.m[9] = ctx->modelview_stack[i].m21;
                data.load_matrixf.m[10] = ctx->modelview_stack[i].m22;
                data.load_matrixf.m[11] = ctx->modelview_stack[i].m23;
                data.load_matrixf.m[12] = ctx->modelview_stack[i].m30;
                data.load_matrixf.m[13] = ctx->modelview_stack[i].m31;
                data.load_matrixf.m[14] = ctx->modelview_stack[i].m32;
                data.load_matrixf.m[15] = ctx->modelview_stack[i].m33;
                glt_put(trace, GLT_OP_LOAD_MATRIXF, &data);
            }
            if (i < ctx->modelview_stack_index)
                glt_put(trace, GLT_OP_PUSH_MATRIX, NULL);
        }

        matrix_mode = GL_MODELVIEW;
    }

    /* Projection matrix stack */
    if (ctx->projection_stack_index != 0 ||
        !m_is_identity(&ctx->projection_stack[0])) {
        int i;

        data.matrix_mode.mode = GL_PROJECTION;
        glt_put(trace, GLT_OP_MATRIX_MODE, &data);

        for (i = 0; i <= ctx->projection_stack_index; i++) {
            if (m_is_identity(&ctx->projection_stack[i])) {
                glt_put(trace, GLT_OP_LOAD_IDENTITY, NULL);
            }
            else {
                data.load_matrixf.m[0] = ctx->projection_stack[i].m00;
                data.load_matrixf.m[1] = ctx->projection_stack[i].m01;
                data.load_matrixf.m[2] = ctx->projection_stack[i].m02;
                data.load_matrixf.m[3] = ctx->projection_stack[i].m03;
                data.load_matrixf.m[4] = ctx->projection_stack[i].m10;
                data.load_matrixf.m[5] = ctx->projection_stack[i].m11;
                data.load_matrixf.m[6] = ctx->projection_stack[i].m12;
                data.load_matrixf.m[7] = ctx->projection_stack[i].m13;
                data.load_matrixf.m[8] = ctx->projection_stack[i].m20;
                data.load_matrixf.m[9] = ctx->projection_stack[i].m21;
                data.load_matrixf.m[10] = ctx->projection_stack[i].m22;
                data.load_matrixf.m[11] = ctx->projection_stack[i].m23;
                data.load_matrixf.m[12] = ctx->projection_stack[i].m30;
                data.load_matrixf.m[13] = ctx->projection_stack[i].m31;
                data.load_matrixf.m[14] = ctx->projection_stack[i].m32;
                data.load_matrixf.m[15] = ctx->projection_stack[i].m33;
                glt_put(trace, GLT_OP_LOAD_MATRIXF, &data);
            }
            if (i < ctx->projection_stack_index)
                glt_put(trace, GLT_OP_PUSH_MATRIX, NULL);
        }

        matrix_mode = GL_PROJECTION;
    }

    /* Texture matrix stack */
    if (ctx->texture_stack_index != 0 ||
        !m_is_identity(&ctx->texture_stack[0])) {
        int i;

        data.matrix_mode.mode = GL_TEXTURE;
        glt_put(trace, GLT_OP_MATRIX_MODE, &data);

        for (i = 0; i <= ctx->texture_stack_index; i++) {
            if (m_is_identity(&ctx->texture_stack[i])) {
                glt_put(trace, GLT_OP_LOAD_IDENTITY, NULL);
            }
            else {
                data.load_matrixf.m[0] = ctx->texture_stack[i].m00;
                data.load_matrixf.m[1] = ctx->texture_stack[i].m01;
                data.load_matrixf.m[2] = ctx->texture_stack[i].m02;
                data.load_matrixf.m[3] = ctx->texture_stack[i].m03;
                data.load_matrixf.m[4] = ctx->texture_stack[i].m10;
                data.load_matrixf.m[5] = ctx->texture_stack[i].m11;
                data.load_matrixf.m[6] = ctx->texture_stack[i].m12;
                data.load_matrixf.m[7] = ctx->texture_stack[i].m13;
                data.load_matrixf.m[8] = ctx->texture_stack[i].m20;
                data.load_matrixf.m[9] = ctx->texture_stack[i].m21;
                data.load_matrixf.m[10] = ctx->texture_stack[i].m22;
                data.load_matrixf.m[11] = ctx->texture_stack[i].m23;
                data.load_matrixf.m[12] = ctx->texture_stack[i].m30;
                data.load_matrixf.m[13] = ctx->texture_stack[i].m31;
                data.load_matrixf.m[14] = ctx->texture_stack[i].m32;
                data.load_matrixf.m[15] = ctx->texture_stack[i].m33;
                glt_put(trace, GLT_OP_LOAD_MATRIXF, &data);
            }
            if (i < ctx->texture_stack_index)
                glt_put(trace, GLT_OP_PUSH_MATRIX, NULL);
        }

        matrix_mode = GL_TEXTURE;
    }

    /* Matrix mode */
    if (ctx->matrix_mode != matrix_mode) {
        data.matrix_mode.mode = ctx->matrix_mode;
        glt_put(trace, GLT_OP_MATRIX_MODE, &data);
    }

    /* Light model ambient color */
    if (!match_color(&ctx->light_ambient_color, 0.2, 0.2, 0.2, 1.0)) {
        data.light_modelfv.pname = GL_LIGHT_MODEL_AMBIENT;
        data.light_modelfv.params[0] = ctx->light_ambient_color.r;
        data.light_modelfv.params[1] = ctx->light_ambient_color.g;
        data.light_modelfv.params[2] = ctx->light_ambient_color.b;
        data.light_modelfv.params[3] = ctx->light_ambient_color.a;
        glt_put(trace, GLT_OP_LIGHT_MODELFV, &data);
    }

    /* Light model local viewer */
    if (ctx->light_local_viewer != GL_FALSE) {
        data.light_modeli.pname = GL_LIGHT_MODEL_LOCAL_VIEWER;
        data.light_modeli.param = ctx->light_local_viewer;
        glt_put(trace, GLT_OP_LIGHT_MODELI, &data);
    }

    /* Light model two side */
    if (ctx->light_two_side != GL_FALSE) {
        data.light_modeli.pname = GL_LIGHT_MODEL_TWO_SIDE;
        data.light_modeli.param = ctx->light_two_side;
        glt_put(trace, GLT_OP_LIGHT_MODELI, &data);
    }

    /* Lights */
    for (i = 0; i < NUM_LIGHTS; i++) {
        /* Ambient color */
        if (!match_color(&ctx->lights[i].a, 0.0, 0.0, 0.0, 1.0)) {
            data.lightfv.light = GL_LIGHT0 + i;
            data.lightfv.pname = GL_AMBIENT;
            data.lightfv.params[0] = ctx->lights[i].a.r;
            data.lightfv.params[1] = ctx->lights[i].a.g;
            data.lightfv.params[2] = ctx->lights[i].a.b;
            data.lightfv.params[3] = ctx->lights[i].a.a;
            glt_put(trace, GLT_OP_LIGHTFV, &data);
        }

        /* Diffuse color */
        if (!match_color(&ctx->lights[i].d, 1.0, 1.0, 1.0, 1.0)) {
            data.lightfv.light = GL_LIGHT0 + i;
            data.lightfv.pname = GL_DIFFUSE;
            data.lightfv.params[0] = ctx->lights[i].d.r;
            data.lightfv.params[1] = ctx->lights[i].d.g;
            data.lightfv.params[2] = ctx->lights[i].d.b;
            data.lightfv.params[3] = ctx->lights[i].d.a;
            glt_put(trace, GLT_OP_LIGHTFV, &data);
        }

        /* Specular color */
        if (!match_color(&ctx->lights[i].s, 1.0, 1.0, 1.0, 1.0)) {
            data.lightfv.light = GL_LIGHT0 + i;
            data.lightfv.pname = GL_SPECULAR;
            data.lightfv.params[0] = ctx->lights[i].s.r;
            data.lightfv.params[1] = ctx->lights[i].s.g;
            data.lightfv.params[2] = ctx->lights[i].s.b;
            data.lightfv.params[3] = ctx->lights[i].s.a;
            glt_put(trace, GLT_OP_LIGHTFV, &data);
        }

        /* Position */
        if (!match_coord(&ctx->lights[i].p, 0.0, 0.0, 1.0, 0.0)) {
            data.lightfv.light = GL_LIGHT0 + i;
            data.lightfv.pname = GL_POSITION;
            data.lightfv.params[0] = ctx->lights[i].p.x;
            data.lightfv.params[1] = ctx->lights[i].p.y;
            data.lightfv.params[2] = ctx->lights[i].p.z;
            data.lightfv.params[3] = ctx->lights[i].p.w;
            glt_put(trace, GLT_OP_LIGHTFV, &data);
        }

        /* Spot direction */
        if (!match_vector(&ctx->lights[i].sd, 0.0, 0.0, -1.0)) {
            data.lightfv.light = GL_LIGHT0 + i;
            data.lightfv.pname = GL_SPOT_DIRECTION;
            data.lightfv.params[0] = ctx->lights[i].sd.x;
            data.lightfv.params[1] = ctx->lights[i].sd.y;
            data.lightfv.params[2] = ctx->lights[i].sd.z;
            glt_put(trace, GLT_OP_LIGHTFV, &data);
        }

        /* Spot exponent */
        if (ctx->lights[i].se != 0.0) {
            data.lightf.light = GL_LIGHT0 + i;
            data.lightf.pname = GL_SPOT_EXPONENT;
            data.lightf.param = ctx->lights[i].se;
            glt_put(trace, GLT_OP_LIGHTF, &data);
        }

        /* Spot cutoff */
        if (ctx->lights[i].sc != 180.0) {
            data.lightf.light = GL_LIGHT0 + i;
            data.lightf.pname = GL_SPOT_CUTOFF;
            data.lightf.param = ctx->lights[i].sc;
            glt_put(trace, GLT_OP_LIGHTF, &data);
        }

        /* Constant attenuation */
        if (ctx->lights[i].ac != 1.0) {
            data.lightf.light = GL_LIGHT0 + i;
            data.lightf.pname = GL_CONSTANT_ATTENUATION;
            data.lightf.param = ctx->lights[i].ac;
            glt_put(trace, GLT_OP_LIGHTF, &data);
        }

        /* Linear attenuation */
        if (ctx->lights[i].al != 0.0) {
            data.lightf.light = GL_LIGHT0 + i;
            data.lightf.pname = GL_LINEAR_ATTENUATION;
            data.lightf.param = ctx->lights[i].al;
            glt_put(trace, GLT_OP_LIGHTF, &data);
        }

        /* Quadratic attenuation */
        if (ctx->lights[i].aq != 0.0) {
            data.lightf.light = GL_LIGHT0 + i;
            data.lightf.pname = GL_QUADRATIC_ATTENUATION;
            data.lightf.param = ctx->lights[i].aq;
            glt_put(trace, GLT_OP_LIGHTF, &data);
        }
    }

    /* Pixel zoom */
    if (ctx->pixel_zoom_x != FLOAT_ONE || ctx->pixel_zoom_y != FLOAT_ONE) {
        data.pixel_zoom.xfactor = ctx->pixel_zoom_x;
        data.pixel_zoom.yfactor = ctx->pixel_zoom_y;
        glt_put(trace, GLT_OP_PIXEL_ZOOM, &data);
    }

    /* Pack swap bytes */
    if (ctx->pack_swap_bytes != GL_FALSE) {
        data.pixel_storei.pname = GL_PACK_SWAP_BYTES;
        data.pixel_storei.param = ctx->pack_swap_bytes;
        glt_put(trace, GLT_OP_PIXEL_STOREI, &data);
    }

    /* Pack lsb first */
    if (ctx->pack_lsb_first != GL_FALSE) {
        data.pixel_storei.pname = GL_PACK_LSB_FIRST;
        data.pixel_storei.param = ctx->pack_lsb_first;
        glt_put(trace, GLT_OP_PIXEL_STOREI, &data);
    }

    /* Pack row length */
    if (ctx->pack_row_length != 0) {
        data.pixel_storei.pname = GL_PACK_ROW_LENGTH;
        data.pixel_storei.param = ctx->pack_row_length;
        glt_put(trace, GLT_OP_PIXEL_STOREI, &data);
    }

    /* Pack skip pixels */
    if (ctx->pack_skip_pixels != 0) {
        data.pixel_storei.pname = GL_PACK_SKIP_PIXELS;
        data.pixel_storei.param = ctx->pack_skip_pixels;
        glt_put(trace, GLT_OP_PIXEL_STOREI, &data);
    }

    /* Pack skip rows */
    if (ctx->pack_skip_rows != 0) {
        data.pixel_storei.pname = GL_PACK_SKIP_ROWS;
        data.pixel_storei.param = ctx->pack_skip_rows;
        glt_put(trace, GLT_OP_PIXEL_STOREI, &data);
    }

    /* Pack alignment */
    if (ctx->pack_alignment != 4) {
        data.pixel_storei.pname = GL_PACK_ALIGNMENT;
        data.pixel_storei.param = ctx->pack_alignment;
        glt_put(trace, GLT_OP_PIXEL_STOREI, &data);
    }

    /* Unpack swap bytes is after texture download */

    /* Unpack lsb first is after texture download */

    /* Unpack row length */
    if (ctx->unpack_row_length != 0) {
        data.pixel_storei.pname = GL_UNPACK_ROW_LENGTH;
        data.pixel_storei.param = ctx->unpack_row_length;
        glt_put(trace, GLT_OP_PIXEL_STOREI, &data);
    }

    /* Unpack skip pixels */
    if (ctx->unpack_skip_pixels != 0) {
        data.pixel_storei.pname = GL_UNPACK_SKIP_PIXELS;
        data.pixel_storei.param = ctx->unpack_skip_pixels;
        glt_put(trace, GLT_OP_PIXEL_STOREI, &data);
    }

    /* Unpack skip rows */
    if (ctx->unpack_skip_rows != 0) {
        data.pixel_storei.pname = GL_UNPACK_SKIP_ROWS;
        data.pixel_storei.param = ctx->unpack_skip_rows;
        glt_put(trace, GLT_OP_PIXEL_STOREI, &data);
    }

    /* Unpack alignment */
    if (ctx->unpack_alignment != 4) {
        data.pixel_storei.pname = GL_UNPACK_ALIGNMENT;
        data.pixel_storei.param = ctx->unpack_alignment;
        glt_put(trace, GLT_OP_PIXEL_STOREI, &data);
    }

    /* Pixel transfer map color */
    if (ctx->transfer_map_color != GL_FALSE) {
        data.pixel_transferi.pname = GL_MAP_COLOR;
        data.pixel_transferi.param = ctx->transfer_map_color;
        glt_put(trace, GLT_OP_PIXEL_TRANSFERI, &data);
    }

    /* Pixel transfer map stencil */
    if (ctx->transfer_map_stencil != GL_FALSE) {
        data.pixel_transferi.pname = GL_MAP_STENCIL;
        data.pixel_transferi.param = ctx->transfer_map_stencil;
        glt_put(trace, GLT_OP_PIXEL_TRANSFERI, &data);
    }

    /* Pixel transfer index shift */
    if (ctx->transfer_index_shift != 0) {
        data.pixel_transferi.pname = GL_INDEX_SHIFT;
        data.pixel_transferi.param = ctx->transfer_index_shift;
        glt_put(trace, GLT_OP_PIXEL_TRANSFERI, &data);
    }

    /* Pixel transfer index offset */
    if (ctx->transfer_index_offset != 0) {
        data.pixel_transferi.pname = GL_INDEX_OFFSET;
        data.pixel_transferi.param = ctx->transfer_index_offset;
        glt_put(trace, GLT_OP_PIXEL_TRANSFERI, &data);
    }

    /* Pixel transfer red scale */
    if (ctx->transfer_red_scale != FLOAT_ONE) {
        data.pixel_transferf.pname = GL_RED_SCALE;
        data.pixel_transferf.param = ctx->transfer_red_scale;
        glt_put(trace, GLT_OP_PIXEL_TRANSFERF, &data);
    }

    /* Pixel transfer red bias */
    if (ctx->transfer_red_bias != FLOAT_ZERO) {
        data.pixel_transferf.pname = GL_RED_BIAS;
        data.pixel_transferf.param = ctx->transfer_red_bias;
        glt_put(trace, GLT_OP_PIXEL_TRANSFERF, &data);
    }

    /* Pixel transfer green scale */
    if (ctx->transfer_green_scale != FLOAT_ONE) {
        data.pixel_transferf.pname = GL_GREEN_SCALE;
        data.pixel_transferf.param = ctx->transfer_green_scale;
        glt_put(trace, GLT_OP_PIXEL_TRANSFERF, &data);
    }

    /* Pixel transfer green bias */
    if (ctx->transfer_green_bias != FLOAT_ZERO) {
        data.pixel_transferf.pname = GL_GREEN_BIAS;
        data.pixel_transferf.param = ctx->transfer_green_bias;
        glt_put(trace, GLT_OP_PIXEL_TRANSFERF, &data);
    }

    /* Pixel transfer blue scale */
    if (ctx->transfer_blue_scale != FLOAT_ONE) {
        data.pixel_transferf.pname = GL_BLUE_SCALE;
        data.pixel_transferf.param = ctx->transfer_blue_scale;
        glt_put(trace, GLT_OP_PIXEL_TRANSFERF, &data);
    }

    /* Pixel transfer blue bias */
    if (ctx->transfer_blue_bias != FLOAT_ZERO) {
        data.pixel_transferf.pname = GL_BLUE_BIAS;
        data.pixel_transferf.param = ctx->transfer_blue_bias;
        glt_put(trace, GLT_OP_PIXEL_TRANSFERF, &data);
    }

    /* Pixel transfer alpha scale */
    if (ctx->transfer_alpha_scale != FLOAT_ONE) {
        data.pixel_transferf.pname = GL_ALPHA_SCALE;
        data.pixel_transferf.param = ctx->transfer_alpha_scale;
        glt_put(trace, GLT_OP_PIXEL_TRANSFERF, &data);
    }

    /* Pixel transfer alpha bias */
    if (ctx->transfer_alpha_bias != FLOAT_ZERO) {
        data.pixel_transferf.pname = GL_ALPHA_BIAS;
        data.pixel_transferf.param = ctx->transfer_alpha_bias;
        glt_put(trace, GLT_OP_PIXEL_TRANSFERF, &data);
    }

    /* Pixel transfer depth scale */
    if (ctx->transfer_depth_scale != FLOAT_ONE) {
        data.pixel_transferf.pname = GL_DEPTH_SCALE;
        data.pixel_transferf.param = ctx->transfer_depth_scale;
        glt_put(trace, GLT_OP_PIXEL_TRANSFERF, &data);
    }

    /* Pixel transfer depth bias */
    if (ctx->transfer_depth_bias != FLOAT_ZERO) {
        data.pixel_transferf.pname = GL_DEPTH_BIAS;
        data.pixel_transferf.param = ctx->transfer_depth_bias;
        glt_put(trace, GLT_OP_PIXEL_TRANSFERF, &data);
    }

    /* Enable flags */
    if (ctx->enable_flags[ALPHA_TEST_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_ALPHA_TEST;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[BLEND_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_BLEND;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[CLIP_PLANE0_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_CLIP_PLANE0;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[CLIP_PLANE1_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_CLIP_PLANE1;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[CLIP_PLANE2_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_CLIP_PLANE2;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[CLIP_PLANE3_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_CLIP_PLANE3;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[CLIP_PLANE4_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_CLIP_PLANE4;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[CLIP_PLANE5_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_CLIP_PLANE5;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[COLOR_MATERIAL_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_COLOR_MATERIAL;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[CULL_FACE_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_CULL_FACE;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[DEPTH_TEST_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_DEPTH_TEST;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[DITHER_ENABLE] != GL_TRUE) {
        data.disable.cap = GL_DITHER;
        glt_put(trace, GLT_OP_DISABLE, &data);
    }
    if (ctx->enable_flags[FOG_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_FOG;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[LIGHT0_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_LIGHT0;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[LIGHT1_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_LIGHT1;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[LIGHT2_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_LIGHT2;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[LIGHT3_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_LIGHT3;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[LIGHT4_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_LIGHT4;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[LIGHT5_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_LIGHT5;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[LIGHT6_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_LIGHT6;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[LIGHT7_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_LIGHT7;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[LIGHTING_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_LIGHTING;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[LINE_SMOOTH_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_LINE_SMOOTH;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[LINE_STIPPLE_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_LINE_STIPPLE;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[LOGIC_OP_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_LOGIC_OP;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[NORMALIZE_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_NORMALIZE;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
#ifdef GL_POLYGON_OFFSET_EXT
    if (ctx->enable_flags[POLYGON_OFFSET_EXT_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_POLYGON_OFFSET_EXT;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
#endif
    if (ctx->enable_flags[POINT_SMOOTH_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_POINT_SMOOTH;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[POLYGON_STIPPLE_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_POLYGON_STIPPLE;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[SCISSOR_TEST_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_SCISSOR_TEST;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[STENCIL_TEST_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_STENCIL_TEST;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[TEXTURE_1D_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_TEXTURE_1D;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }
    if (ctx->enable_flags[TEXTURE_2D_ENABLE] != GL_FALSE) {
        data.enable.cap = GL_TEXTURE_2D;
        glt_put(trace, GLT_OP_ENABLE, &data);
    }

    /* Fog hint */
    if (ctx->fog_hint != GL_DONT_CARE) {
        data.hint.target = GL_FOG_HINT;
        data.hint.mode = ctx->fog_hint;
        glt_put(trace, GLT_OP_HINT, &data);
    }

    /* Line smooth hint */
    if (ctx->line_smooth_hint != GL_DONT_CARE) {
        data.hint.target = GL_LINE_SMOOTH_HINT;
        data.hint.mode = ctx->line_smooth_hint;
        glt_put(trace, GLT_OP_HINT, &data);
    }

    /* Perspective correction hint */
    if (ctx->perspective_correction_hint != GL_DONT_CARE) {
        data.hint.target = GL_PERSPECTIVE_CORRECTION_HINT;
        data.hint.mode = ctx->perspective_correction_hint;
        glt_put(trace, GLT_OP_HINT, &data);
    }

    /* Point smooth hint */
    if (ctx->point_smooth_hint != GL_DONT_CARE) {
        data.hint.target = GL_POINT_SMOOTH_HINT;
        data.hint.mode = ctx->point_smooth_hint;
        glt_put(trace, GLT_OP_HINT, &data);
    }

    /* Polygon_Smooth hint */
    if (ctx->polygon_smooth_hint != GL_DONT_CARE) {
        data.hint.target = GL_POLYGON_SMOOTH_HINT;
        data.hint.mode = ctx->polygon_smooth_hint;
        glt_put(trace, GLT_OP_HINT, &data);
    }

    /* List base */
    if (ctx->list_base != 0) {
        data.list_base.base = ctx->list_base;
        glt_put(trace, GLT_OP_LIST_BASE, &data);
    }

    /* Texture environment mode */
    if (ctx->tex_env_mode != GL_MODULATE) {
        data.tex_envi.target = GL_TEXTURE_ENV;
        data.tex_envi.pname = GL_TEXTURE_ENV_MODE;
        data.tex_envi.param = ctx->tex_env_mode;
        glt_put(trace, GLT_OP_TEX_ENVI, &data);
    }

    /* Texture environment color */
    if (!match_color(&ctx->tex_env_color, 0.0, 0.0, 0.0, 0.0)) {
        data.tex_envfv.target = GL_TEXTURE_ENV;
        data.tex_envfv.pname = GL_TEXTURE_ENV_COLOR;
        data.tex_envfv.params[0] = ctx->tex_env_color.r;
        data.tex_envfv.params[1] = ctx->tex_env_color.g;
        data.tex_envfv.params[2] = ctx->tex_env_color.b;
        data.tex_envfv.params[3] = ctx->tex_env_color.a;
        glt_put(trace, GLT_OP_TEX_ENVFV, &data);
    }

    /* Textures */
    for (i = 0; i < ctx->n_textures; i++) {
        GL_texture *texture = ctx->textures[i];
        if (texture) {
            int j;

            /* Bind texture */
            if (cur_texture != i) {
                data.bind_texture.target = GL_TEXTURE_2D;
                data.bind_texture.texture = i;
                glt_put(trace, GLT_OP_BIND_TEXTURE, &data);
                cur_texture = i;
            }

            /* Texture min filter */
            if (texture->min_filter != GL_NEAREST_MIPMAP_LINEAR) {
                data.tex_parameteri.target = GL_TEXTURE_2D;
                data.tex_parameteri.pname = GL_TEXTURE_MIN_FILTER;
                data.tex_parameteri.param = texture->min_filter;
                glt_put(trace, GLT_OP_TEX_PARAMETERI, &data);
            }

            /* Texture mag filter */
            if (texture->mag_filter != GL_LINEAR) {
                data.tex_parameteri.target = GL_TEXTURE_2D;
                data.tex_parameteri.pname = GL_TEXTURE_MAG_FILTER;
                data.tex_parameteri.param = texture->mag_filter;
                glt_put(trace, GLT_OP_TEX_PARAMETERI, &data);
            }

            /* Texture wrap s */
            if (texture->wrap_s != GL_REPEAT) {
                data.tex_parameteri.target = GL_TEXTURE_2D;
                data.tex_parameteri.pname = GL_TEXTURE_WRAP_S;
                data.tex_parameteri.param = texture->wrap_s;
                glt_put(trace, GLT_OP_TEX_PARAMETERI, &data);
            }

            /* Texture wrap t */
            if (texture->wrap_t != GL_REPEAT) {
                data.tex_parameteri.target = GL_TEXTURE_2D;
                data.tex_parameteri.pname = GL_TEXTURE_WRAP_T;
                data.tex_parameteri.param = texture->wrap_t;
                glt_put(trace, GLT_OP_TEX_PARAMETERI, &data);
            }

            /* Texture border color */
            if (!match_color(&texture->border_color, 0.0, 0.0, 0.0, 0.0)) {
                data.tex_parameterfv.target = GL_TEXTURE_2D;
                data.tex_parameterfv.pname = GL_TEXTURE_BORDER_COLOR;
                data.tex_parameterfv.params[0] = texture->border_color.r;
                data.tex_parameterfv.params[1] = texture->border_color.g;
                data.tex_parameterfv.params[2] = texture->border_color.b;
                data.tex_parameterfv.params[3] = texture->border_color.a;
                glt_put(trace, GLT_OP_TEX_PARAMETERFV, &data);
            }

            /* Texture images */
            for (j = 0; j < NUM_TEX_LEVELS; j++) {
                GL_texlevel *level = &texture->levels[j];
                if (level->valid) {
                    /* Level swap bytes */
                    if (level->swap_bytes != swap_bytes) {
                        data.pixel_storei.pname = GL_UNPACK_SWAP_BYTES;
                        data.pixel_storei.param = level->swap_bytes;
                        glt_put(trace, GLT_OP_PIXEL_STOREI, &data);
                        swap_bytes = level->swap_bytes;
                    }

                    /* Level lsb first */
                    if (level->lsb_first != lsb_first) {
                        data.pixel_storei.pname = GL_UNPACK_LSB_FIRST;
                        data.pixel_storei.param = level->lsb_first;
                        glt_put(trace, GLT_OP_PIXEL_STOREI, &data);
                        lsb_first = level->lsb_first;
                    }

                    /* Texture data */
                    data.tex_image_2d.target = GL_TEXTURE_2D;;
                    data.tex_image_2d.level = j;
                    data.tex_image_2d.components = level->components;
                    data.tex_image_2d.width = level->width;
                    data.tex_image_2d.height = level->height;
                    data.tex_image_2d.border = level->border;
                    data.tex_image_2d.format = level->format;
                    data.tex_image_2d.type = level->type;
                    data.tex_image_2d.n_pixels = level->n_pixels;
                    data.tex_image_2d.pixels = level->pixels;
                    glt_put(trace, GLT_OP_TEX_IMAGE_2D, &data);
                }
            }
        }
    }

    /* Bind texture */
    if (cur_texture != ctx->texture) {
        data.bind_texture.target = GL_TEXTURE_2D;
        data.bind_texture.texture = ctx->texture;
        glt_put(trace, GLT_OP_BIND_TEXTURE, &data);
    }

    /* Unpack swap bytes */
    if (swap_bytes != ctx->unpack_swap_bytes) {
        data.pixel_storei.pname = GL_UNPACK_SWAP_BYTES;
        data.pixel_storei.param = ctx->unpack_swap_bytes;
        glt_put(trace, GLT_OP_PIXEL_STOREI, &data);
    }

    /* Unpack lsb first */
    if (lsb_first != ctx->unpack_lsb_first) {
        data.pixel_storei.pname = GL_UNPACK_LSB_FIRST;
        data.pixel_storei.param = ctx->unpack_lsb_first;
        glt_put(trace, GLT_OP_PIXEL_STOREI, &data);
    }
}
