/*
 *  skip.c
 *
 *  A complete hack to shorten a glt trace by skipping frames at the start
 *  of the trace.
 *
 *  Kekoa Proudfoot
 *  2/17/99
 */

/* Include files */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* For GL defines */
#ifdef WIN32
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif
#include <GL/gl.h>

#include <glt.h>

#include "clparse.h"
#include "state.h"

const char *
basename( const char *path )
{
    char *last;
    last = strrchr( path, '/' );
    if ( !last )
        return path;
    else
        return last + 1;
}

enum {
    OPT_SWAP_AFTER_PUT,
    OPT_CONTEXT,
    OPT_FIRST_FRAME,
    OPT_LAST_FRAME,
    OPT_COUNT,
    OPT_UNWIND,
    OPT_OUT_NAME
};

opt_t opts[] = {
    { "swap",    OPT_SWAP_AFTER_PUT },
    { "ctx",     OPT_CONTEXT        },
    { "context", OPT_CONTEXT        },
    { "f",       OPT_FIRST_FRAME    },
    { "first",   OPT_FIRST_FRAME    },
    { "n",       OPT_COUNT          },
    { "count",   OPT_COUNT          },
    { "unwind",  OPT_UNWIND         },
    { "o",       OPT_OUT_NAME       }
};

struct {
    const char  *progname;
    int          first_frame;
    int          num_frames;
    int          swap_after_put;
    int          ctx_number;
    int          unwind;
    char        *in_name;
    char        *out_name;
} param;

GLT_trace    in;
GLT_trace    out;
GL_context   ctx;
int          num_contexts;
int          frame_num;

void
handle_context( GLT_opcode op, GLT_data *data, int emit )
{
    while ( op ) {

        switch ( op ) {

          case GLT_OP_CREATE_CONTEXT:
            num_contexts++;
            if ( !param.ctx_number && num_contexts > 1 )
                glt_warn( "glt: multiple contexts being tracked" );
            if ( !param.ctx_number ||
                 data->create_context.ctx == param.ctx_number ) {
                glt_put( &out, op, data );
                return;
            }
            break;

          case GLT_OP_MAKE_CURRENT:
            if ( !param.ctx_number ||
                 data->make_current.ctx == param.ctx_number ) {
                glt_put( &out, op, data );
                return;
            }
            break;

          case GLT_OP_DESTROY_CONTEXT:
            if ( !param.ctx_number ||
                 data->destroy_context.ctx == param.ctx_number ) {
                glt_put( &out, op, data );
                return;
            }
            break;

          case GLT_OP_NEW_LIST:
            data->new_list.mode = GL_COMPILE;
            glt_put( &out, op, data );
            while ( op && op != GLT_OP_END_LIST ) {
                op = glt_get( &in, data );
                glt_put( &out, op, data );
            }
            break;

          case GLT_OP_END_LIST:
            glt_fatal( "%s: glEndList unexpected", param.progname );
            break;

          case GLT_OP_DELETE_LISTS:
            glt_put( &out, op, data );
            break;

          case GLT_OP_BIND_TEXTURE:
          case GLT_OP_BIND_TEXTURE_EXT:
          case GLT_OP_DELETE_TEXTURES:
          case GLT_OP_DELETE_TEXTURES_EXT:
          case GLT_OP_PRIORITIZE_TEXTURES:
          case GLT_OP_PRIORITIZE_TEXTURES_EXT:
          case GLT_OP_TEX_IMAGE_2D:
          case GLT_OP_TEX_PARAMETERF:
          case GLT_OP_TEX_PARAMETERFV:
          case GLT_OP_TEX_PARAMETERI:
          case GLT_OP_TEX_PARAMETERIV:
          case GLT_OP_TEX_SUB_IMAGE_2D:
          case GLT_OP_TEX_SUB_IMAGE_2D_EXT:
            if ( emit || !gs_update( &ctx, op, data ) )
                glt_put( &out, op, data );
            break;

          case GLT_OP_SWAP_BUFFERS:
            frame_num++;
            break;
        }

        op = glt_get( &in, data );
    }
}

void
skip_frames( int count )
{
    GLT_opcode op;
    GLT_data   data;
    int        last_frame = frame_num + count;

    if ( frame_num >= last_frame )
        return;

    while ( (op = glt_get( &in, &data )) ) {

        switch ( op ) {

          case GLT_OP_CREATE_CONTEXT:
          case GLT_OP_MAKE_CURRENT:
          case GLT_OP_DESTROY_CONTEXT:
            handle_context( op, &data, 0 );
            if ( frame_num >= last_frame )
                return;
            break;

          case GLT_OP_PAD:
            break;

          case GLT_OP_NEW_LIST:
            {
                int update = ( data.new_list.mode == GL_COMPILE_AND_EXECUTE );
                glt_put( &out, op, &data );
                do {
                    op = glt_get( &in, &data );
                    if ( update )
                        gs_update( &ctx, op, &data );
                    glt_put( &out, op, &data );
                } while ( op && op != GLT_OP_END_LIST );
            }
            break;

          case GLT_OP_END_LIST:
            glt_fatal( "%s: didn't expect glEndList", param.progname );
            break;

          case GLT_OP_DELETE_LISTS:
            glt_put( &out, op, &data );
            break;

          case GLT_OP_SWAP_BUFFERS:
            frame_num++;
            if ( frame_num >= last_frame )
                return;
            break;

          case GLT_OP_PUSH_ATTRIB:
          case GLT_OP_POP_ATTRIB:
            /* assume these are balanced within a frame, and just
             * pitch them, since we don't track them in the ctx */
            break;

          default:
            if ( !gs_update( &ctx, op, &data ) )
                glt_put( &out, op, &data );

        } /* switch ( op ) */
    }
}

void
play_frames( int count )
{
    GLT_opcode op;
    GLT_data   data;
    int        last_frame = frame_num + count;

    if ( frame_num >= last_frame )
        return;

    while ( (op = glt_get( &in, &data )) ) {

        switch ( op ) {

          case GLT_OP_CREATE_CONTEXT:
          case GLT_OP_MAKE_CURRENT:
          case GLT_OP_DESTROY_CONTEXT:
            handle_context( op, &data, 1 );
            if ( frame_num >= last_frame )
                return;
            break;

          case GLT_OP_PAD:
            break;

          case GLT_OP_SWAP_BUFFERS:
            glt_put( &out, op, &data );
            frame_num++;
            if ( frame_num >= last_frame )
                return;
            break;

          default:
            glt_put( &out, op, &data );

        } /* switch ( op ) */
    }
}

int
main( int argc, char **argv )
{
    int          option;

    param.progname       = basename( argv[0] );
    param.first_frame    = -1;
    param.num_frames     = 100000;
    param.swap_after_put = 0;
    param.ctx_number     = 0;
    param.unwind         = 0;
    param.in_name        = NULL;
    param.out_name       = NULL;

    /* Parse command line */

    clp_init( argc, argv, opts, sizeof(opts)/sizeof(opts[0]) );

    while ( !clp_geterror() && ( option = clp_getopt() ) >= 0 ) {

        switch ( option ) {

          case OPT_SWAP_AFTER_PUT:
            param.swap_after_put = 1;
            break;

          case OPT_CONTEXT:
            param.ctx_number = clp_getint( );
            break;

          case OPT_FIRST_FRAME:
            param.first_frame = clp_getint( );
            if ( param.first_frame < 0 )
                glt_fatal( "%s: first frame should be >= 0\n", argv[0] );
            break;

          case OPT_COUNT:
            param.num_frames = clp_getint( );
            if ( param.num_frames <= 0 )
                glt_fatal( "%s: count should be > 0\n", argv[0] );
            break;

          case OPT_UNWIND:
            param.unwind = 1;
            break;

          case OPT_OUT_NAME:
            param.out_name = clp_getstring( );
            break;
        }
    }

    argv += clp_getpos();
    argc -= clp_getpos();

    if ( argc == 1 )
        param.in_name = clp_getstring( );

    if ( clp_geterror() || argc < 0 || argc > 1 ) {
        glt_warn( "usage: %s [options] [infile]", param.progname );
        glt_warn( "options:" );
        glt_warn( "  -swap          : insert a swap after state rollup" );
        glt_warn( "  -first <n>     : first frame to emit" );
        glt_warn( "  -count <n>     : number of frames to emit" );
        glt_warn( "  -o <file>      : output file" );
        glt_warn( "  -ctx <n>       : only emit this context" );
        glt_warn( "  -unwind        : unwind display lists" );
        exit( 1 );
    }

    if ( param.first_frame < 0 )
        glt_fatal("%s: first frame unspecified\n", argv[0] );

    /* Open traces */

    if ( param.unwind )
        glt_open( &in, param.in_name, ( GLT_RDONLY | GLT_UNWIND ) );
    else
        glt_open( &in, param.in_name, GLT_RDONLY );
    glt_open( &out, param.out_name, GLT_WRONLY );

    /* Rename file names for stdio */

    if ( !param.in_name )
        param.in_name = "stdin";
    if ( !param.out_name )
        param.out_name = "stdout";

    /* Init the context */

    gs_init( &ctx );

    /* Read data and copy to output */

    num_contexts = 0;
    frame_num = 0;

    if ( param.first_frame > 0 )
        skip_frames( param.first_frame );

    /* Write deltas to output */

    gs_put( &out, &ctx );

    if ( param.swap_after_put )
        glt_put( &out, GLT_OP_SWAP_BUFFERS, NULL );

    /* Copy remainder of input to output */

    play_frames( param.num_frames );

    /* Check for error condition */

    if ( glt_err( &in ) ) {
        glt_warn( "%s: Parse error near file offset 0x%08x!",
                  param.in_name, glt_tell( &in ) );
    }

    /* Close traces */

    glt_close( &in );
    glt_close( &out );

    return 0;
}
