/*
 * mipmap.c
 *
 *   generate nicely filter mipmapped textures
 *
 *  Matthew Eldridge
 *  29-Oct-1999
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <glt.h>

#ifdef WIN32
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif

#include <GL/gl.h>

#include "clparse.h"

static const char *progname = "texfix";
static int quiet   = 0;
static int verbose = 0;

void
convert_ushort_to_ubyte( int n, GLushort *src, GLubyte *dst )
{
    int i;

    for ( i = 0; i < n; i++ ) {
        GLushort val = src[i];
        dst[i] = val >> 8;
    }
}

void
crunch_image( int width, int height, int components,
              GLubyte *src, GLubyte *dst )
{
    int      i, x, y, src_x_stride, src_y_stride;
    GLubyte *src0, *src1, *src2, *src3;

    src0 = src;
    if ( width > 1 ) {
        src1 = src0 + components;
        src_x_stride = components * 2;
    } else {
        src1 = src0;
        src_x_stride = 0;
    }

    if ( height > 1 ) {
        src2 = src0 + width * components;
        src3 = src1 + width * components;
        src_y_stride = width * components;
    } else {
        src2 = src0;
        src3 = src1;
        src_y_stride = 0;
    }

    for ( y = 0; y < height; y += 2 ) {

        for ( x = 0; x < width; x += 2 ) {

            for ( i = 0; i < components; i++ ) {
                int val = src0[i] + src1[i] + src2[i] + src3[i];
                dst[i] = ( val + 2 ) >> 2;
            }

            dst  += components;
            src0 += src_x_stride;
            src1 += src_x_stride;
            src2 += src_x_stride;
            src3 += src_x_stride;
        }

        src0 += src_y_stride;
        src1 += src_y_stride;
        src2 += src_y_stride;
        src3 += src_y_stride;
    }
}

int
convert_type( int npixels, GLenum format, GLenum type, GLvoid *data )
{
    int components = 0;

    switch ( format ) {

      case 1:
      case GL_RED:
      case GL_GREEN:
      case GL_BLUE:
      case GL_ALPHA:
      case GL_LUMINANCE:
        components = 1;
        break;

      case 2:
      case GL_LUMINANCE_ALPHA:
        components = 2;
        break;

      case 3:
      case GL_RGB:
        components = 3;
        break;

      case 4:
      case GL_RGBA:
        components = 4;
        break;

      default:
        if ( !quiet ) {
            glt_warn( "%s: unsupported format 0x%x", progname, format );
        }
        return 0;
    }

    switch ( type ) {

      case GL_UNSIGNED_BYTE:
        break;

      case GL_UNSIGNED_SHORT:
        if ( verbose ) {
            static int count = 0;
            if ( !count )
                glt_warn( "%s: converting UNSIGNED_SHORT -> UNSIGNED_BYTE",
                          progname );
            count++;
        }
        convert_ushort_to_ubyte( npixels * components, data, data );
        break;

      default:
        if ( !quiet ) {
            glt_warn( "%s: unsupported type 0x%x\n", progname, type );
        }
        return 0;
    }

    return components;
}

void
set_tex_filters( GLT_trace *out, GLenum magFilter, GLenum minFilter )
{
    GLT_data data;

    data.tex_parameteri.target = GL_TEXTURE_2D;
    data.tex_parameteri.pname  = GL_TEXTURE_MAG_FILTER;
    data.tex_parameteri.param  = magFilter;

    glt_put( out, GLT_OP_TEX_PARAMETERI, &data );

    data.tex_parameteri.target = GL_TEXTURE_2D;
    data.tex_parameteri.pname  = GL_TEXTURE_MIN_FILTER;
    data.tex_parameteri.param  = minFilter;

    glt_put( out, GLT_OP_TEX_PARAMETERI, &data );
}

void
handle_tex_image_2d( GLT_trace *out, GLT_data *data )
{
    GLT_tex_image_2d *tex = &data->tex_image_2d;

    if ( tex->target != GL_TEXTURE_2D ) {

        if ( !quiet ) {
            glt_warn( "%s: unsupported target=0x%x\n", tex->target );
        }

        glt_put( out, GLT_OP_TEX_IMAGE_2D, data );
        return;
    }

    if ( tex->border != 0 ) {

        if ( !quiet ) {
            glt_warn( "%s: unsupported border=%d\n", tex->border );
        }

        glt_put( out, GLT_OP_TEX_IMAGE_2D, data );
        return;
    }

    if ( tex->level == 0 ) {

        int components;

        components = convert_type( tex->width * tex->height, tex->format,
                                   tex->type, tex->pixels );

        /* if we can't convert it, puke */
        if ( components < 1 ) {

            set_tex_filters( out, GL_LINEAR, GL_LINEAR );
            glt_put( out, GLT_OP_TEX_IMAGE_2D, data );
            return;
        }

        assert( tex->n_pixels >= tex->width * tex->height * components );

        tex->type = GL_UNSIGNED_BYTE;
        tex->n_pixels = tex->width * tex->height * components;

        glt_put( out, GLT_OP_TEX_IMAGE_2D, data );

        while ( tex->width > 1 || tex->height > 1 ) {

            crunch_image( tex->width, tex->height, components,
                          tex->pixels, tex->pixels );

            if ( tex->width > 1 )
                tex->width >>= 1;

            if ( tex->height > 1 )
                tex->height >>= 1;

            tex->level++;
            tex->n_pixels = tex->width * tex->height * components;

            glt_put( out, GLT_OP_TEX_IMAGE_2D, data );

        }

        set_tex_filters( out, GL_LINEAR, GL_LINEAR_MIPMAP_LINEAR );
    }
}

void
handle_tex_parameter( GLT_trace *out, GLT_opcode op, GLT_data *data )
{
    GLenum   target;
    GLenum   pname;
    GLint    param;

    switch ( op ) {

      case GLT_OP_TEX_PARAMETERI:
        target = data->tex_parameteri.target;
        pname  = data->tex_parameteri.pname;
        param  = data->tex_parameteri.param;
        break;

      case GLT_OP_TEX_PARAMETERIV:
        target = data->tex_parameteriv.target;
        pname  = data->tex_parameteriv.pname;
        param  = data->tex_parameteriv.params[0];
        break;

      case GLT_OP_TEX_PARAMETERF:
        target = data->tex_parameterf.target;
        pname  = data->tex_parameterf.pname;
        param  = (GLint) data->tex_parameterf.param;
        break;

      case GLT_OP_TEX_PARAMETERFV:
        target = data->tex_parameterf.target;
        pname  = data->tex_parameterfv.pname;
        param  = (GLint) data->tex_parameterfv.params[0];
        break;

      default:
        return;
    }

    /* unused */
    param = param;

    /* the user is a puny worm and may not specify the texture filter */
    if ( target == GL_TEXTURE_2D &&
         ( pname  == GL_TEXTURE_MAG_FILTER ||
           pname == GL_TEXTURE_MIN_FILTER ) )
        return;

    glt_put( out, op, data );
}

const char *
basename( const char *path )
{
    char *last;
    last = strrchr(path, '/');
    if (!last)
        return path;
    else
        return last + 1;
}

enum {
    OPT_VERBOSE,
    OPT_QUIET,
    OPT_HELP
};

opt_t clopts[] = {
    { "verbose", OPT_VERBOSE },
    { "v",       OPT_VERBOSE },
    { "quiet",   OPT_QUIET   },
    { "q",       OPT_QUIET   },
    { "help",    OPT_HELP    },
    { "h",       OPT_HELP    }
};

void
usage( void )
{
    glt_warn( "usage: %s [options] [infile [outfile]]", progname );
    glt_warn( "options:" );
    glt_warn( "  -verbose" );
    glt_warn( "  -quiet (suppresses some bad news)" );
    glt_warn( "  -help" );
    glt_warn( "Converts all textures to be mip-mapped.  On each glTexImage2d( level=0 )" );
    glt_warn( "synthesizes the remaining texture levels.  Forces the texture filter" );
    glt_warn( "to be GL_LINEAR/GL_LINEAR_MIPMAP_LINEAR.  All textures converted to" );
    glt_warn( "be GL_UNSIGNED_BYTE." );
}

int
main( int argc, char **argv )
{
    GLT_trace    in;
    GLT_trace    out;
    GLT_data     data;
    GLT_opcode   op;
    char        *inname;
    char        *outname;
    int          option;

    quiet   = 0;
    verbose = 0;

    progname = basename( argv[0] );

    clp_init( argc, argv, clopts, sizeof(clopts)/sizeof(clopts[0]) );

    while ( !clp_geterror() && ( option = clp_getopt() ) >= 0 ) {

        switch ( option ) {

          case OPT_QUIET:
            quiet   = 1;
            verbose = 0;
            break;

          case OPT_VERBOSE:
            quiet   = 0;
            verbose = 1;
            break;

          case OPT_HELP:
            usage( );
            exit( 0 );
            break;
        }
    }

    argv += clp_getpos();
    argc -= clp_getpos();

    if ( clp_geterror() || argc < 0 || argc > 2 ) {
        usage( );
        exit(1);
    }

    inname  = ( argc > 0 ) ? argv[0] : NULL;
    outname = ( argc > 1 ) ? argv[1] : NULL;

    glt_open( &in, inname, GLT_RDONLY );

    glt_open( &out, outname, GLT_WRONLY );

    if ( !inname )
        inname = "stdin";
    if ( !outname )
        outname = "stdout";

    while ( ( op = glt_get( &in, &data ) ) ) {

        switch ( op ) {

          case GLT_OP_TEX_PARAMETERI:
          case GLT_OP_TEX_PARAMETERIV:
          case GLT_OP_TEX_PARAMETERF:
          case GLT_OP_TEX_PARAMETERFV:
            handle_tex_parameter( &out, op, &data );
            break;

          case GLT_OP_TEX_IMAGE_2D:
            handle_tex_image_2d( &out, &data );
            break;

          default:
            glt_put( &out, op, &data );
        }
    }

    /* Check for error condition */

    if ( glt_err( &in ) )
        glt_warn( "%s: Parse error near file offset 0x%08x!", inname,
                  glt_tell( &in ) );

    /* Close traces */

    glt_close( &in  );
    glt_close( &out );

    return 0;
}
