/*
 *  glttofgls.c
 *
 *  A first crack at a program to convert from glt to fgls
 *
 *  Kekoa Proudfoot
 *  5/14/98
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <glt.h>

#include <fgls.h>
#include <GL/gl.h>
#include <FGL/fgl.h>

#include "timer.h"

#if 1
#define PRINTUNIMPL
#endif

#define PROGRESS_INTERVAL 30.0f

/* Macros */

#ifdef PRINTUNIMPL
#if 1
#define IGNORE(op,data) \
    do { \
        static int once = 0; \
        if (!once && ++once) \
            glt_warn("%s: ignoring %s", progname, glt_name(op)); \
    } while (0)
#else
#define IGNORE(op,data) \
    glt_warn("%s: ignoring %s", progname, glt_text(op, data))
#endif
#else /* PRINTUNIMPL */
#define IGNORE(op,data)
#endif /* PRINTUNIMPL */

/* Globals */

static FGLSint stream;
static int _hack_light_model_ambient = 0;
static char *progname;
static int show_progress = 1;

/*****************************************************************************
 * Miscellaneous functions
 *****************************************************************************/

void ConvertTexImage2D(int l, int w, int h, int c, int f, const void *p) {
    if (c == 1 && f == FGL_LUMINANCE)
        fglsAdd(stream, FGLS_OP_TEX_IMAGE_2D, l, w, h, f, p, w * h);
    else if (c == 2 && f == FGL_LUMINANCE_ALPHA)
        fglsAdd(stream, FGLS_OP_TEX_IMAGE_2D, l, w, h, f, p, w * h * 2);
    else if (c == 3 && f == FGL_RGB)
        fglsAdd(stream, FGLS_OP_TEX_IMAGE_2D, l, w, h, f, p, w * h * 3);
    else if (c == 3 && f == FGL_RGBA) {
        unsigned int  *tmp;
        unsigned int  *tp;
        unsigned char *pp;
        int            i;
        tmp = (unsigned int *)glt_malloc(w * h * sizeof(unsigned int));
        for (i = w * h, tp = tmp, pp = (unsigned char *)p; i; i--) {
            unsigned char r, g, b, a;
            r = *pp++; g = *pp++; b = *pp++; a = *pp++;
            /* why is alpha not being used here? */
            a = a;
            *tp++ = (r << 24) | (g << 16) | (b << 8) | (0xff << 0);
        }
        fglsAdd(stream, FGLS_OP_TEX_IMAGE_2D, l, w, h, f, tmp, w * h * 4);
        glt_free(tmp);
    }
    else if (c == 4 && f == FGL_RGBA)
        fglsAdd(stream, FGLS_OP_TEX_IMAGE_2D, l, w, h, f, p, w * h * 4);
}

/* Execute functions */

void ExecuteSwapBuffers (void) {
    fglsAdd(stream, FGLS_OP_SWAP_BUFFERS);
}
void ExecuteNewList (GLuint list, GLenum mode) {
    switch (mode) {
    case GL_COMPILE:              mode = FGLS_COMPILE;              break;
    case GL_COMPILE_AND_EXECUTE:  mode = FGLS_COMPILE_AND_EXECUTE;  break;
    default:                      return;
    }
    fglsAdd(stream, FGLS_OP_NEW_LIST, list, mode);
}
void ExecuteEndList (void) {
    fglsAdd(stream, FGLS_OP_END_LIST);
}
void ExecuteCallList (GLuint list) {
    fglsAdd(stream, FGLS_OP_CALL_LIST, list);
}
void ExecuteBegin (GLenum mode) {
    fglsAdd(stream, FGLS_OP_BEGIN, mode);
}
void ExecuteColor3bv (const GLbyte *v) {
    fglsAdd(stream, FGLS_OP_COLOR_3BV, v);
}
void ExecuteColor3dv (const GLdouble *v) {
    fglsAdd(stream, FGLS_OP_COLOR_3DV, v);
}
void ExecuteColor3fv (const GLfloat *v) {
    fglsAdd(stream, FGLS_OP_COLOR_3FV, v);
}
void ExecuteColor3iv (const GLint *v) {
    fglsAdd(stream, FGLS_OP_COLOR_3IV, v);
}
void ExecuteColor3sv (const GLshort *v) {
    fglsAdd(stream, FGLS_OP_COLOR_3SV, v);
}
void ExecuteColor3ubv (const GLubyte *v) {
    fglsAdd(stream, FGLS_OP_COLOR_3UBV, v);
}
void ExecuteColor3uiv (const GLuint *v) {
    fglsAdd(stream, FGLS_OP_COLOR_3UIV, v);
}
void ExecuteColor3usv (const GLushort *v) {
    fglsAdd(stream, FGLS_OP_COLOR_3USV, v);
}
void ExecuteColor4bv (const GLbyte *v) {
    fglsAdd(stream, FGLS_OP_COLOR_4BV, v);
}
void ExecuteColor4dv (const GLdouble *v) {
    fglsAdd(stream, FGLS_OP_COLOR_4DV, v);
}
void ExecuteColor4fv (const GLfloat *v) {
    fglsAdd(stream, FGLS_OP_COLOR_4FV, v);
}
void ExecuteColor4iv (const GLint *v) {
    fglsAdd(stream, FGLS_OP_COLOR_4IV, v);
}
void ExecuteColor4sv (const GLshort *v) {
    fglsAdd(stream, FGLS_OP_COLOR_4SV, v);
}
void ExecuteColor4ubv (const GLubyte *v) {
    fglsAdd(stream, FGLS_OP_COLOR_4UBV, v);
}
void ExecuteColor4uiv (const GLuint *v) {
    fglsAdd(stream, FGLS_OP_COLOR_4UIV, v);
}
void ExecuteColor4usv (const GLushort *v) {
    fglsAdd(stream, FGLS_OP_COLOR_4USV, v);
}
void ExecuteEnd (void) {
    fglsAdd(stream, FGLS_OP_END);
}
void ExecuteNormal3bv (const GLbyte *v) {
    fglsAdd(stream, FGLS_OP_NORMAL_3BV, v);
}
void ExecuteNormal3dv (const GLdouble *v) {
    fglsAdd(stream, FGLS_OP_NORMAL_3DV, v);
}
void ExecuteNormal3fv (const GLfloat *v) {
    fglsAdd(stream, FGLS_OP_NORMAL_3FV, v);
}
void ExecuteNormal3iv (const GLint *v) {
    fglsAdd(stream, FGLS_OP_NORMAL_3IV, v);
}
void ExecuteNormal3sv (const GLshort *v) {
    fglsAdd(stream, FGLS_OP_NORMAL_3SV, v);
}
void ExecuteTexCoord1dv (const GLdouble *v) {
    fglsAdd(stream, FGLS_OP_TEX_COORD_1DV, v);
}
void ExecuteTexCoord1fv (const GLfloat *v) {
    fglsAdd(stream, FGLS_OP_TEX_COORD_1FV, v);
}
void ExecuteTexCoord1iv (const GLint *v) {
    fglsAdd(stream, FGLS_OP_TEX_COORD_1IV, v);
}
void ExecuteTexCoord1sv (const GLshort *v) {
    fglsAdd(stream, FGLS_OP_TEX_COORD_1SV, v);
}
void ExecuteTexCoord2dv (const GLdouble *v) {
    fglsAdd(stream, FGLS_OP_TEX_COORD_2DV, v);
}
void ExecuteTexCoord2fv (const GLfloat *v) {
    fglsAdd(stream, FGLS_OP_TEX_COORD_2FV, v);
}
void ExecuteTexCoord2iv (const GLint *v) {
    fglsAdd(stream, FGLS_OP_TEX_COORD_2IV, v);
}
void ExecuteTexCoord2sv (const GLshort *v) {
    fglsAdd(stream, FGLS_OP_TEX_COORD_2SV, v);
}
void ExecuteTexCoord3dv (const GLdouble *v) {
    fglsAdd(stream, FGLS_OP_TEX_COORD_3DV, v);
}
void ExecuteTexCoord3fv (const GLfloat *v) {
    fglsAdd(stream, FGLS_OP_TEX_COORD_3FV, v);
}
void ExecuteTexCoord3iv (const GLint *v) {
    fglsAdd(stream, FGLS_OP_TEX_COORD_3IV, v);
}
void ExecuteTexCoord3sv (const GLshort *v) {
    fglsAdd(stream, FGLS_OP_TEX_COORD_3SV, v);
}
void ExecuteTexCoord4dv (const GLdouble *v) {
    fglsAdd(stream, FGLS_OP_TEX_COORD_4DV, v);
}
void ExecuteTexCoord4fv (const GLfloat *v) {
    fglsAdd(stream, FGLS_OP_TEX_COORD_4FV, v);
}
void ExecuteTexCoord4iv (const GLint *v) {
    fglsAdd(stream, FGLS_OP_TEX_COORD_4IV, v);
}
void ExecuteTexCoord4sv (const GLshort *v) {
    fglsAdd(stream, FGLS_OP_TEX_COORD_4SV, v);
}
void ExecuteVertex2dv (const GLdouble *v) {
    fglsAdd(stream, FGLS_OP_VERTEX_2DV, v);
}
void ExecuteVertex2fv (const GLfloat *v) {
    fglsAdd(stream, FGLS_OP_VERTEX_2FV, v);
}
void ExecuteVertex2iv (const GLint *v) {
    fglsAdd(stream, FGLS_OP_VERTEX_2IV, v);
}
void ExecuteVertex2sv (const GLshort *v) {
    fglsAdd(stream, FGLS_OP_VERTEX_2SV, v);
}
void ExecuteVertex3dv (const GLdouble *v) {
    fglsAdd(stream, FGLS_OP_VERTEX_3DV, v);
}
void ExecuteVertex3fv (const GLfloat *v) {
    fglsAdd(stream, FGLS_OP_VERTEX_3FV, v);
}
void ExecuteVertex3iv (const GLint *v) {
    fglsAdd(stream, FGLS_OP_VERTEX_3IV, v);
}
void ExecuteVertex3sv (const GLshort *v) {
    fglsAdd(stream, FGLS_OP_VERTEX_3SV, v);
}
void ExecuteVertex4dv (const GLdouble *v) {
    fglsAdd(stream, FGLS_OP_VERTEX_4DV, v);
}
void ExecuteVertex4fv (const GLfloat *v) {
    fglsAdd(stream, FGLS_OP_VERTEX_4FV, v);
}
void ExecuteVertex4iv (const GLint *v) {
    fglsAdd(stream, FGLS_OP_VERTEX_4IV, v);
}
void ExecuteVertex4sv (const GLshort *v) {
    fglsAdd(stream, FGLS_OP_VERTEX_4SV, v);
}
void ExecuteColorMaterial (GLenum face, GLenum mode) {
    if (face == GL_BACK)
        return;
    switch (mode) {
    case GL_EMISSION:             mode = FGL_EMISSION;  break;
    case GL_DIFFUSE:              mode = FGL_DIFFUSE;   break;
    case GL_AMBIENT_AND_DIFFUSE:  mode = FGL_DIFFUSE;   break;
    case GL_SPECULAR:             mode = FGL_SPECULAR;  break;
    case GL_AMBIENT:              return;
    default:                      return;
    }
    fglsAdd(stream, FGLS_OP_COLOR_MATERIAL, mode);
}
void ExecuteCullFace (GLenum mode) {
    switch (mode) {
    case GL_FRONT:           mode = FGL_FRONT;  break;
    case GL_BACK:            mode = FGL_BACK;   break;
    case GL_FRONT_AND_BACK:  return;
    default:                 return;
    }
    fglsAdd(stream, FGLS_OP_CULL_FACE, mode);
}
void ExecuteFrontFace (GLenum mode) {
    switch (mode) {
    case GL_CW:   mode = FGL_CW;   break;
    case GL_CCW:  mode = FGL_CCW;  break;
    default:      return;
    }
    fglsAdd(stream, FGLS_OP_FRONT_FACE, mode);
}
void ExecuteLightf (GLenum light, GLenum pname, GLfloat param) {
    switch (light) {
    case GL_LIGHT0:  light = FGL_LIGHT0;  break;
    case GL_LIGHT1:  light = FGL_LIGHT1;  break;
    case GL_LIGHT2:  light = FGL_LIGHT2;  break;
    case GL_LIGHT3:  light = FGL_LIGHT3;  break;
    case GL_LIGHT4:  light = FGL_LIGHT4;  break;
    case GL_LIGHT5:  light = FGL_LIGHT5;  break;
    case GL_LIGHT6:  light = FGL_LIGHT6;  break;
    case GL_LIGHT7:  light = FGL_LIGHT7;  break;
    default:         return;
    }
    switch (pname) {
    case GL_SPOT_EXPONENT:       pname = FGL_SPOT_EXPONENT;      break;
    case GL_SPOT_CUTOFF:         pname = FGL_SPOT_CUTOFF;        break;
    case GL_LINEAR_ATTENUATION:  pname = FGL_ATTENUATION;        break;
    default:                     return;
    }
    fglsAdd(stream, FGLS_OP_LIGHT_1F, light, pname, param);
    if (pname == FGL_SPOT_EXPONENT || pname == FGL_SPOT_CUTOFF)
        fglsAdd(stream, FGLS_OP_LIGHT_TYPE, light, FGL_SPOT_LIGHT);
}
void ExecuteLightfv (GLenum light, GLenum pname, const GLfloat *params) {
    switch (pname) {
    case GL_SPOT_EXPONENT:
    case GL_SPOT_CUTOFF:
    case GL_CONSTANT_ATTENUATION:
    case GL_LINEAR_ATTENUATION:
    case GL_QUADRATIC_ATTENUATION:
        ExecuteLightf(light, pname, params[0]);
        return;
    }
    switch (light) {
    case GL_LIGHT0:  light = FGL_LIGHT0;  break;
    case GL_LIGHT1:  light = FGL_LIGHT1;  break;
    case GL_LIGHT2:  light = FGL_LIGHT2;  break;
    case GL_LIGHT3:  light = FGL_LIGHT3;  break;
    case GL_LIGHT4:  light = FGL_LIGHT4;  break;
    case GL_LIGHT5:  light = FGL_LIGHT5;  break;
    case GL_LIGHT6:  light = FGL_LIGHT6;  break;
    case GL_LIGHT7:  light = FGL_LIGHT7;  break;
    default:         return;
    }
    switch (pname) {
    case GL_DIFFUSE:
        fglsAdd(stream, FGLS_OP_LIGHT_3F, light, FGL_INTENSITY,
                params[0], params[1], params[2]);
        break;
    case GL_POSITION:
        if (params[3] == 0.0) {
            fglsAdd(stream, FGLS_OP_LIGHT_TYPE, light, FGL_DISTANT_LIGHT);
            fglsAdd(stream, FGLS_OP_LIGHT_3F, light, FGL_DIRECTION,
                    -params[0], -params[1], -params[2]);
        }
        else {
            fglsAdd(stream, FGLS_OP_LIGHT_TYPE, light, FGL_POINT_LIGHT);
            fglsAdd(stream, FGLS_OP_LIGHT_3F, light, FGL_POSITION,
                    params[0] / params[3], params[1] / params[3],
                    params[2] / params[3]);
        }
        break;
    case GL_SPOT_DIRECTION:
        fglsAdd(stream, FGLS_OP_LIGHT_3F, light, FGL_POSITION,
                params[0], params[1], params[2]);
        break;
    case GL_SPECULAR:
        break;
    }
}
void ExecuteLighti (GLenum light, GLenum pname, GLint param) {
    ExecuteLightf(light, pname, param);
}
void ExecuteLightiv (GLenum light, GLenum pname, const GLint *params) {
    GLfloat floatparams[4];

    floatparams[0] = params[0];
    floatparams[1] = params[1];
    floatparams[2] = params[2];
    floatparams[3] = params[3];

    ExecuteLightfv(light, pname, floatparams);
}
void ExecuteLightModelfv (GLenum pname, const GLfloat *params) {
    /* Hack! */
    if (_hack_light_model_ambient) {
        /* Big hack! */
        if (pname != GL_LIGHT_MODEL_AMBIENT)
            return;
        fglsAdd(stream, FGLS_OP_ENABLE, FGL_LIGHT7);
        fglsAdd(stream, FGLS_OP_LIGHT_TYPE, FGL_LIGHT7, FGL_AMBIENT_LIGHT);
        fglsAdd(stream, FGLS_OP_LIGHT_3F, FGL_LIGHT7, FGL_INTENSITY,
                0.5, 0.5, 0.5);
    }
}
void ExecuteMaterialf (GLenum face, GLenum pname, GLfloat param) {
    if (face == GL_BACK)
        return;
    if (pname == GL_SHININESS)
        fglsAdd(stream, FGLS_OP_MATERIAL_1F, FGL_SHININESS, param);
}
void ExecuteMaterialfv (GLenum face, GLenum pname, const GLfloat *params) {
    int specular;
    if (face == GL_BACK)
        return;
    switch (pname) {
    case GL_SHININESS:
        ExecuteMaterialf(face, pname, params[0]);
        return;
    }
    switch (pname) {
    case GL_DIFFUSE:   pname = FGL_DIFFUSE_COLOR;   break;
    case GL_SPECULAR:  pname = FGL_SPECULAR_COLOR;  break;
    case GL_EMISSION:  pname = FGL_EMISSION_COLOR;  break;
    default:           return;
    }
    specular = (pname == FGL_SPECULAR_COLOR && params[0] != 0.0f &&
                params[1] != 0.0f && params[2] != 0.0f);
    fglsAdd(stream, FGLS_OP_MATERIAL_3F, pname,
            params[0], params[1], params[2]);
    if (specular) fglsAdd(stream, FGLS_OP_MATERIAL_TYPE, FGL_PHONG_MATERIAL);
}
void ExecuteMateriali (GLenum face, GLenum pname, GLint param) {
    ExecuteMaterialf(face, pname, param);
}
void ExecuteMaterialiv (GLenum face, GLenum pname, const GLint *params) {
    GLfloat floatparams[4];

    floatparams[0] = params[0];
    floatparams[1] = params[1];
    floatparams[2] = params[2];
    floatparams[3] = params[3];

    ExecuteMaterialfv(face, pname, floatparams);
}
void ExecuteScissor (GLint x, GLint y, GLsizei width, GLsizei height) {
    fglsAdd(stream, FGLS_OP_SCISSOR, x, y, width, height);
}
void ExecuteShadeModel (GLenum mode) {
    switch (mode) {
    case GL_FLAT:    mode = FGL_FLAT;    break;
    case GL_SMOOTH:  mode = FGL_SMOOTH;  break;
    default:         return;
    }
    fglsAdd(stream, FGLS_OP_SHADE_MODEL, mode);
}
void ExecuteTexParameteri (GLenum target, GLenum pname, GLint param) {
    if (target != GL_TEXTURE_2D)
        return;
    switch (pname) {
    case GL_TEXTURE_MIN_FILTER:     pname = FGL_TEXTURE_MIN_FILTER;      break;
    case GL_TEXTURE_MAG_FILTER:     pname = FGL_TEXTURE_MAG_FILTER;      break;
    case GL_TEXTURE_WRAP_S:         pname = FGL_TEXTURE_WRAP_S;          break;
    case GL_TEXTURE_WRAP_T:         pname = FGL_TEXTURE_WRAP_T;          break;
    default:                        return;
    }
    switch (param) {
    case GL_NEAREST:                param = FGL_NEAREST;                 break;
    case GL_LINEAR:                 param = FGL_LINEAR;                  break;
    case GL_NEAREST_MIPMAP_NEAREST: param = FGL_NEAREST_MIPMAP_NEAREST;  break;
    case GL_NEAREST_MIPMAP_LINEAR:  param = FGL_NEAREST_MIPMAP_LINEAR;   break;
    case GL_LINEAR_MIPMAP_NEAREST:  param = FGL_LINEAR_MIPMAP_NEAREST;   break;
    case GL_LINEAR_MIPMAP_LINEAR:   param = FGL_LINEAR_MIPMAP_LINEAR;    break;
    case GL_REPEAT:                 param = FGL_REPEAT;                  break;
    case GL_CLAMP:                  param = FGL_CLAMP;                   break;
    default:                        return;
    }
    fglsAdd(stream, FGLS_OP_TEX_PARAMETER, pname, param);
}
void ExecuteTexParameteriv (GLenum target, GLenum pname, const GLint *params) {
    ExecuteTexParameteri(target, pname, *params);
}
void ExecuteTexParameterf (GLenum target, GLenum pname, GLfloat param) {
    ExecuteTexParameteri(target, pname, param);
}
void ExecuteTexParameterfv (GLenum target, GLenum pname, const GLfloat *params) {
    ExecuteTexParameteri(target, pname, *params);
}
/* Hack! For power of two check */
#define IsPow2(a)      (((a) & ((a) - 1)) == 0)
void ExecuteTexImage2D (GLenum target, GLint level, GLint components, GLsizei width, GLsizei height, GLint border, GLenum format, GLenum type, const GLvoid *pixels) {
    void *_pixels = NULL;
    void *_pixels2 = NULL;
    if (!pixels)
        return;
    if (target != GL_TEXTURE_2D || border != 0)
        return;
    switch (type) {
    case GL_UNSIGNED_BYTE:
    case GL_UNSIGNED_SHORT:
    case GL_UNSIGNED_SHORT_4_4_4_4_EXT:
        break;
    default:
        return;
    }
    /* Hack! So Argus doesn't puke */
    if (!IsPow2(width) || !IsPow2(height))
        return;
    switch (components) {
    case 1:
    case GL_LUMINANCE:
    case GL_LUMINANCE8_EXT:
        components = 1; break;
    case 2:
    case GL_LUMINANCE_ALPHA:
    case GL_LUMINANCE8_ALPHA8_EXT:
    case GL_LUMINANCE12_ALPHA4_EXT:
        components = 2; break;
    case 3:
    case GL_RGB:
    case GL_RGB5_EXT:
        components = 3; break;
    case 4:
    case GL_RGBA:
    case GL_RGBA4_EXT:
    case GL_RGB5_A1_EXT:
        components = 4; break;
    default:
        return;
    }
    if (type == GL_UNSIGNED_SHORT) {
        int count;
        const unsigned short *src;
        unsigned char *dst;
        int i;
        switch (format) {
        case GL_LUMINANCE:
            count = width * height;
            break;
        case GL_LUMINANCE_ALPHA:
            count = 2 * width * height;
            break;
        case GL_RGB:
            count = 3 * width * height;
            break;
        case GL_RGBA:
        case GL_ABGR_EXT:
            count = 4 * width * height;
            break;
        default:
            count = components * width * height;
            break;
        }
        src = (const unsigned short *)pixels;
        dst = _pixels = (unsigned char *)glt_malloc(count);
        for (i = 0; i < count; i++)
            *dst++ = (*src++) >> 8;
        pixels = _pixels;
    }
    if (type == GL_UNSIGNED_SHORT_4_4_4_4_EXT) {
        int count = width * height;
        const unsigned short *src = (const unsigned short *)pixels;
        unsigned char *dst = _pixels = (unsigned char *)glt_malloc(4*count);
        int i;
        for (i = 0; i < count; i++) {
            *dst++ = (*src >> 8) & 0xf0;
            *dst++ = (*src >> 4) & 0xf0;
            *dst++ = (*src >> 0) & 0xf0;
            *dst++ = (*src << 4) & 0xf0;
            src++;
        }
        pixels = _pixels;
    }
    if (format == GL_ABGR_EXT) {
        int count = width * height;
        const unsigned char *src = (const unsigned char *)pixels;
        unsigned char *dst = _pixels2 = (unsigned char *)glt_malloc(4*count);
        int i;
        for (i = 0; i < count; i++) {
            *dst++ = src[3];
            *dst++ = src[2];
            *dst++ = src[1];
            *dst++ = src[0];
            src += 4;
        }
        pixels = _pixels2;
        format = GL_RGBA;
    }
    if (components == 1 && format != GL_LUMINANCE)
        return;
    if (components == 2 && format != GL_LUMINANCE_ALPHA)
        return;
    if (components == 3 && format != GL_RGB && format != GL_RGBA)
        return;
    if (components == 4 && format != GL_RGBA)
        return;
    switch (format) {
    case GL_LUMINANCE:       format = FGL_LUMINANCE;       break;
    case GL_LUMINANCE_ALPHA: format = FGL_LUMINANCE_ALPHA; break;
    case GL_RGB:             format = FGL_RGB;             break;
    case GL_RGBA:            format = FGL_RGBA;            break;
    default:                 return;
    }
    /* Does not yet handle textures in display lists, but it should ... */
    ConvertTexImage2D(level, width, height, components, format, pixels);
    if (_pixels)
        glt_free(_pixels);
    if (_pixels2)
        glt_free(_pixels2);
}
void ExecuteTexEnvi (GLenum target, GLenum pname, GLint param) {
    if (target == GL_TEXTURE_ENV && pname == GL_TEXTURE_ENV_MODE) {
        switch (param) {
        case GL_REPLACE_EXT: param = FGL_REPLACE;  break;
        case GL_REPLACE:     param = FGL_REPLACE;  break;
        case GL_DECAL:       param = FGL_DECAL;    break;
        case GL_MODULATE:    param = FGL_MODULATE; break;
        default:
            return;
        }
        fglsAdd(stream, FGLS_OP_TEXTURE_FUNC, param);
    }
}
void ExecuteTexEnviv (GLenum target, GLenum pname, const GLint *params) {
    ExecuteTexEnvi(target, pname, params[0]);
}
void ExecuteTexEnvf (GLenum target, GLenum pname, GLfloat param) {
    ExecuteTexEnvi(target, pname, param);
}
void ExecuteTexEnvfv (GLenum target, GLenum pname, const GLfloat *params) {
    ExecuteTexEnvi(target, pname, params[0]);
}
void ExecuteDrawBuffer (GLenum mode) {
    switch (mode) {
    case GL_FRONT:  mode = FGL_FRONT;  break;
    case GL_BACK:   mode = FGL_BACK;   break;
    default:        return;
    }
#if 0
    fglsAdd(stream, FGLS_OP_DRAW_BUFFER, mode);
#endif
}
void ExecuteClear (GLbitfield mask) {
    int fglmask = 0;

    if (mask & GL_COLOR_BUFFER_BIT)
        fglmask |= FGL_COLOR_BUFFER_BIT;
    if (mask & GL_DEPTH_BUFFER_BIT)
        fglmask |= FGL_DEPTH_BUFFER_BIT;

    if (fglmask)
        fglsAdd(stream, FGLS_OP_CLEAR, fglmask);
}
void ExecuteClearColor (GLclampf red, GLclampf green, GLclampf blue, GLclampf alpha) {
    fglsAdd(stream, FGLS_OP_CLEAR_COLOR, red, green, blue, alpha);
}
void ExecuteClearDepth (GLclampd depth) {
    fglsAdd(stream, FGLS_OP_CLEAR_DEPTH, depth);
}
void ExecuteDisable (GLenum cap) {
    switch (cap) {
    case GL_NORMALIZE:       cap = FGL_NORMALIZE;         break;
    case GL_LIGHTING:        cap = FGL_LIGHTING;          break;
    case GL_LIGHT0:          cap = FGL_LIGHT0;            break;
    case GL_LIGHT1:          cap = FGL_LIGHT1;            break;
    case GL_LIGHT2:          cap = FGL_LIGHT2;            break;
    case GL_LIGHT3:          cap = FGL_LIGHT3;            break;
    case GL_LIGHT4:          cap = FGL_LIGHT4;            break;
    case GL_LIGHT5:          cap = FGL_LIGHT5;            break;
    case GL_LIGHT6:          cap = FGL_LIGHT6;            break;
    case GL_LIGHT7:          cap = FGL_LIGHT7;            break;
    case GL_CULL_FACE:       cap = FGL_CULL_FACE;         break;
    case GL_TEXTURE_2D:      cap = FGL_TEXTURE_FRAGMENT;  break;
    case GL_SCISSOR_TEST:    cap = FGL_SCISSOR_TEST;      break;
    case GL_DEPTH_TEST:      cap = FGL_DEPTH_TEST;        break;
    case GL_BLEND:           cap = FGL_BLEND;             break;
    case GL_COLOR_MATERIAL:  cap = FGL_COLOR_MATERIAL;    break;
    default:                 return;
    }
    fglsAdd(stream, FGLS_OP_DISABLE, cap);
}
void ExecuteEnable (GLenum cap) {
    switch (cap) {
    case GL_NORMALIZE:       cap = FGL_NORMALIZE;         break;
    case GL_LIGHTING:        cap = FGL_LIGHTING;          break;
    case GL_LIGHT0:          cap = FGL_LIGHT0;            break;
    case GL_LIGHT1:          cap = FGL_LIGHT1;            break;
    case GL_LIGHT2:          cap = FGL_LIGHT2;            break;
    case GL_LIGHT3:          cap = FGL_LIGHT3;            break;
    case GL_LIGHT4:          cap = FGL_LIGHT4;            break;
    case GL_LIGHT5:          cap = FGL_LIGHT5;            break;
    case GL_LIGHT6:          cap = FGL_LIGHT6;            break;
    case GL_LIGHT7:          cap = FGL_LIGHT7;            break;
    case GL_CULL_FACE:       cap = FGL_CULL_FACE;         break;
    case GL_TEXTURE_2D:      cap = FGL_TEXTURE_FRAGMENT;  break;
    case GL_SCISSOR_TEST:    cap = FGL_SCISSOR_TEST;      break;
    case GL_DEPTH_TEST:      cap = FGL_DEPTH_TEST;        break;
    case GL_BLEND:           cap = FGL_BLEND;             break;
    case GL_COLOR_MATERIAL:  cap = FGL_COLOR_MATERIAL;    break;
    default:                 return;
    }
    fglsAdd(stream, FGLS_OP_ENABLE, cap);
}
void ExecuteFinish (void) {
    fglsAdd(stream, FGLS_OP_FINISH);
}
void ExecuteFlush (void) {
    fglsAdd(stream, FGLS_OP_FLUSH);
}
void ExecuteBlendFunc (GLenum sfactor, GLenum dfactor) {
    switch (sfactor) {
    case GL_ZERO:                 sfactor = FGL_ZERO;                 break;
    case GL_ONE:                  sfactor = FGL_ONE;                  break;
    case GL_DST_COLOR:            sfactor = FGL_DST_COLOR;            break;
    case GL_ONE_MINUS_DST_COLOR:  sfactor = FGL_ONE_MINUS_DST_COLOR;  break;
    case GL_SRC_ALPHA:            sfactor = FGL_SRC_ALPHA;            break;
    case GL_ONE_MINUS_SRC_ALPHA:  sfactor = FGL_ONE_MINUS_SRC_ALPHA;  break;
    case GL_DST_ALPHA:            sfactor = FGL_DST_ALPHA;            break;
    case GL_ONE_MINUS_DST_ALPHA:  sfactor = FGL_ONE_MINUS_DST_ALPHA;  break;
    case GL_SRC_ALPHA_SATURATE:   sfactor = FGL_SRC_ALPHA_SATURATE;   break;
    default:                      return;
    }
    switch (dfactor) {
    case GL_ZERO:                 dfactor = FGL_ZERO;                 break;
    case GL_ONE:                  dfactor = FGL_ONE;                  break;
    case GL_SRC_COLOR:            dfactor = FGL_SRC_COLOR;            break;
    case GL_ONE_MINUS_SRC_COLOR:  dfactor = FGL_ONE_MINUS_SRC_COLOR;  break;
    case GL_SRC_ALPHA:            dfactor = FGL_SRC_ALPHA;            break;
    case GL_ONE_MINUS_SRC_ALPHA:  dfactor = FGL_ONE_MINUS_SRC_ALPHA;  break;
    case GL_DST_ALPHA:            dfactor = FGL_DST_ALPHA;            break;
    case GL_ONE_MINUS_DST_ALPHA:  dfactor = FGL_ONE_MINUS_DST_ALPHA;  break;
    default:                      return;
    }
    fglsAdd(stream, FGLS_OP_BLEND_FUNC, sfactor, dfactor);
}
void ExecuteDepthFunc (GLenum func) {
    switch (func) {
    case GL_NEVER:     func = FGL_NEVER;     break;
    case GL_LESS:      func = FGL_LESS;      break;
    case GL_EQUAL:     func = FGL_EQUAL;     break;
    case GL_LEQUAL:    func = FGL_LEQUAL;    break;
    case GL_GREATER:   func = FGL_GREATER;   break;
    case GL_NOTEQUAL:  func = FGL_NOTEQUAL;  break;
    case GL_GEQUAL:    func = FGL_GEQUAL;    break;
    case GL_ALWAYS:    func = FGL_ALWAYS;    break;
    default:           return;
    }
    fglsAdd(stream, FGLS_OP_DEPTH_FUNC, func);
}
void ExecuteFrustum (GLdouble left, GLdouble right, GLdouble bottom, GLdouble top, GLdouble near, GLdouble zfar) {
    fglsAdd(stream, FGLS_OP_FRUSTUM, left, right, bottom, top, near, zfar);
}
void ExecuteLoadIdentity (void) {
    fglsAdd(stream, FGLS_OP_LOAD_IDENTITY);
}
void ExecuteLoadMatrixf (const GLfloat *m) {
    fglsAdd(stream, FGLS_OP_LOAD_MATRIX, m);
}
void ExecuteLoadMatrixd (const GLdouble *m) {
    GLfloat mf[16];

    mf[ 0] = m[ 0]; mf[ 1] = m[ 1]; mf[ 2] = m[ 2]; mf[ 3] = m[ 3];
    mf[ 4] = m[ 4]; mf[ 5] = m[ 5]; mf[ 6] = m[ 6]; mf[ 7] = m[ 7];
    mf[ 8] = m[ 8]; mf[ 9] = m[ 9]; mf[10] = m[10]; mf[11] = m[11];
    mf[12] = m[12]; mf[13] = m[13]; mf[14] = m[14]; mf[15] = m[15];

    ExecuteLoadMatrixf(mf);
}
void ExecuteMatrixMode (GLenum mode) {
    switch (mode) {
    case GL_MODELVIEW:   mode = FGL_MODELVIEW;   break;
    case GL_PROJECTION:  mode = FGL_PROJECTION;  break;
    case GL_TEXTURE:     mode = FGL_TEXTURE;     break;
    default:             return;
    }
    fglsAdd(stream, FGLS_OP_MATRIX_MODE, mode);
}
void ExecuteMultMatrixf (const GLfloat *m) {
    fglsAdd(stream, FGLS_OP_MULT_MATRIX, m);
}
void ExecuteMultMatrixd (const GLdouble *m) {
    GLfloat mf[16];

    mf[ 0] = m[ 0]; mf[ 1] = m[ 1]; mf[ 2] = m[ 2]; mf[ 3] = m[ 3];
    mf[ 4] = m[ 4]; mf[ 5] = m[ 5]; mf[ 6] = m[ 6]; mf[ 7] = m[ 7];
    mf[ 8] = m[ 8]; mf[ 9] = m[ 9]; mf[10] = m[10]; mf[11] = m[11];
    mf[12] = m[12]; mf[13] = m[13]; mf[14] = m[14]; mf[15] = m[15];

    ExecuteMultMatrixf(mf);
}
void ExecuteOrtho (GLdouble left, GLdouble right, GLdouble bottom, GLdouble top, GLdouble znear, GLdouble zfar) {
    fglsAdd(stream, FGLS_OP_ORTHO, left, right, bottom, top, znear, zfar);
}
void ExecutePopMatrix (void) {
    fglsAdd(stream, FGLS_OP_POP_MATRIX);
}
void ExecutePushMatrix (void) {
    fglsAdd(stream, FGLS_OP_PUSH_MATRIX);
}
void ExecuteRotated (GLdouble angle, GLdouble x, GLdouble y, GLdouble z) {
    fglsAdd(stream, FGLS_OP_ROTATE, angle, x, y, z);
}
void ExecuteRotatef (GLfloat angle, GLfloat x, GLfloat y, GLfloat z) {
    fglsAdd(stream, FGLS_OP_ROTATE, angle, x, y, z);
}
void ExecuteScaled (GLdouble x, GLdouble y, GLdouble z) {
    fglsAdd(stream, FGLS_OP_SCALE, x, y, z);
}
void ExecuteScalef (GLfloat x, GLfloat y, GLfloat z) {
    fglsAdd(stream, FGLS_OP_SCALE, x, y, z);
}
void ExecuteTranslated (GLdouble x, GLdouble y, GLdouble z) {
    fglsAdd(stream, FGLS_OP_TRANSLATE, x, y, z);
}
void ExecuteTranslatef (GLfloat x, GLfloat y, GLfloat z) {
    fglsAdd(stream, FGLS_OP_TRANSLATE, x, y, z);
}
void ExecuteViewport (GLint x, GLint y, GLsizei width, GLsizei height) {
    fglsAdd(stream, FGLS_OP_VIEWPORT, x, y, width, height);
}
void ExecuteBindTexture (GLenum target, GLuint texture) {
    if (target != GL_TEXTURE_2D)
        return;
    fglsAdd(stream, FGLS_OP_BIND_TEXTURE, texture);
}
void ExecuteDeleteTextures (GLsizei n, const GLuint *textures) {
    while (n--)
        fglsAdd(stream, FGLS_OP_DELETE_TEXTURE, textures[n]);
}

/* Dispatch function */

int
dispatch_all(GLT_trace *trace)
{
    GLT_data data;
    GLT_opcode op;

    timeval_t timer;
    int frame = 0;

    if (show_progress)
        timer_reset(&timer);

    while (op = glt_get(trace, &data)) {
        switch (op) {
        case GLT_OP_HEADER:
        case GLT_OP_PAD:
            break;
        case GLT_OP_CREATE_CONTEXT:
            IGNORE(op, &data);
            break;
        case GLT_OP_DESTROY_CONTEXT:
            IGNORE(op, &data);
            break;
        case GLT_OP_MAKE_CURRENT:
        {
            GLT_make_current *d = &data.make_current;
            fglsAdd(stream, FGLS_OP_BUFFER_SIZE, d->width, d->height);
            break;
        }
        case GLT_OP_SWAP_BUFFERS:
            ExecuteSwapBuffers();

            frame++;
            if (show_progress && timer_read(&timer) > PROGRESS_INTERVAL) {
                glt_warn("%s: %d frames processed...", progname, frame);
                timer_reset(&timer);
            }

            break;
        case GLT_OP_ACCUM:
            IGNORE(op, &data);
            break;
        case GLT_OP_ALPHA_FUNC:
            IGNORE(op, &data);
            break;
        case GLT_OP_BEGIN:
        {
            GLT_begin *d = &data.begin;
            ExecuteBegin(d->mode);
            break;
        }
        case GLT_OP_BIND_TEXTURE:
        case GLT_OP_BIND_TEXTURE_EXT:
        {
            GLT_bind_texture *d = &data.bind_texture;
            ExecuteBindTexture(d->target, d->texture);
            break;
        }
        case GLT_OP_BLEND_FUNC:
        {
            GLT_blend_func *d = &data.blend_func;
            ExecuteBlendFunc(d->sfactor, d->dfactor);
            break;
        }
        case GLT_OP_CALL_LIST:
        {
            GLT_call_list *d = &data.call_list;
            ExecuteCallList(d->list);
            break;
        }
        case GLT_OP_CALL_LISTS:
            IGNORE(op, &data);
            break;
        case GLT_OP_CLEAR:
        {
            GLT_clear *d = &data.clear;
            ExecuteClear(d->mask);
            break;
        }
        case GLT_OP_CLEAR_ACCUM:
            IGNORE(op, &data);
            break;
        case GLT_OP_CLEAR_COLOR:
        {
            GLT_clear_color *d = &data.clear_color;
            ExecuteClearColor(d->red, d->green, d->blue, d->alpha);
            break;
        }
        case GLT_OP_CLEAR_DEPTH:
        {
            GLT_clear_depth *d = &data.clear_depth;
            ExecuteClearDepth(d->depth);
            break;
        }
        case GLT_OP_CLEAR_INDEX:
            IGNORE(op, &data);
            break;
        case GLT_OP_CLEAR_STENCIL:
            IGNORE(op, &data);
            break;
        case GLT_OP_CLIP_PLANE:
            IGNORE(op, &data);
            break;
        case GLT_OP_COLOR_3B:
        case GLT_OP_COLOR_3BV:
        {
            GLT_color_3bv *d = &data.color_3bv;
            ExecuteColor3bv(d->v);
            break;
        }
        case GLT_OP_COLOR_3D:
        case GLT_OP_COLOR_3DV:
        {
            GLT_color_3dv *d = &data.color_3dv;
            ExecuteColor3dv(d->v);
            break;
        }
        case GLT_OP_COLOR_3F:
        case GLT_OP_COLOR_3FV:
        {
            GLT_color_3fv *d = &data.color_3fv;
            ExecuteColor3fv(d->v);
            break;
        }
        case GLT_OP_COLOR_3I:
        case GLT_OP_COLOR_3IV:
        {
            GLT_color_3iv *d = &data.color_3iv;
            ExecuteColor3iv(d->v);
            break;
        }
        case GLT_OP_COLOR_3S:
        case GLT_OP_COLOR_3SV:
        {
            GLT_color_3sv *d = &data.color_3sv;
            ExecuteColor3sv(d->v);
            break;
        }
        case GLT_OP_COLOR_3UB:
        case GLT_OP_COLOR_3UBV:
        {
            GLT_color_3ubv *d = &data.color_3ubv;
            ExecuteColor3ubv(d->v);
            break;
        }
        case GLT_OP_COLOR_3UI:
        case GLT_OP_COLOR_3UIV:
        {
            GLT_color_3uiv *d = &data.color_3uiv;
            ExecuteColor3uiv(d->v);
            break;
        }
        case GLT_OP_COLOR_3US:
        case GLT_OP_COLOR_3USV:
        {
            GLT_color_3usv *d = &data.color_3usv;
            ExecuteColor3usv(d->v);
            break;
        }
        case GLT_OP_COLOR_4B:
        case GLT_OP_COLOR_4BV:
        {
            GLT_color_4bv *d = &data.color_4bv;
            ExecuteColor4bv(d->v);
            break;
        }
        case GLT_OP_COLOR_4D:
        case GLT_OP_COLOR_4DV:
        {
            GLT_color_4dv *d = &data.color_4dv;
            ExecuteColor4dv(d->v);
            break;
        }
        case GLT_OP_COLOR_4F:
        case GLT_OP_COLOR_4FV:
        {
            GLT_color_4fv *d = &data.color_4fv;
            ExecuteColor4fv(d->v);
            break;
        }
        case GLT_OP_COLOR_4I:
        case GLT_OP_COLOR_4IV:
        {
            GLT_color_4iv *d = &data.color_4iv;
            ExecuteColor4iv(d->v);
            break;
        }
        case GLT_OP_COLOR_4S:
        case GLT_OP_COLOR_4SV:
        {
            GLT_color_4sv *d = &data.color_4sv;
            ExecuteColor4sv(d->v);
            break;
        }
        case GLT_OP_COLOR_4UB:
        case GLT_OP_COLOR_4UBV:
        {
            GLT_color_4ubv *d = &data.color_4ubv;
            ExecuteColor4ubv(d->v);
            break;
        }
        case GLT_OP_COLOR_4UI:
        case GLT_OP_COLOR_4UIV:
        {
            GLT_color_4uiv *d = &data.color_4uiv;
            ExecuteColor4uiv(d->v);
            break;
        }
        case GLT_OP_COLOR_4US:
        case GLT_OP_COLOR_4USV:
        {
            GLT_color_4usv *d = &data.color_4usv;
            ExecuteColor4usv(d->v);
            break;
        }
        case GLT_OP_COLOR_MASK:
            IGNORE(op, &data);
            break;
        case GLT_OP_COLOR_MATERIAL:
        {
            GLT_color_material *d = &data.color_material;
            ExecuteColorMaterial(d->face, d->mode);
            break;
        }
        case GLT_OP_COPY_PIXELS:
            IGNORE(op, &data);
            break;
        case GLT_OP_CULL_FACE:
        {
            GLT_cull_face *d = &data.cull_face;
            ExecuteCullFace(d->mode);
            break;
        }
        case GLT_OP_DELETE_LISTS:
            IGNORE(op, &data);
            break;
        case GLT_OP_DELETE_TEXTURES:
        {
            GLT_delete_textures *d = &data.delete_textures;
            ExecuteDeleteTextures(d->n, d->textures);
            break;
        }
        case GLT_OP_DELETE_TEXTURES_EXT:
            IGNORE(op, &data);
            break;
        case GLT_OP_DEPTH_FUNC:
        {
            GLT_depth_func *d = &data.depth_func;
            ExecuteDepthFunc(d->func);
            break;
        }
        case GLT_OP_DEPTH_MASK:
            IGNORE(op, &data);
            break;
        case GLT_OP_DEPTH_RANGE:
            IGNORE(op, &data);
            break;
        case GLT_OP_DISABLE:
        {
            GLT_disable *d = &data.disable;
            ExecuteDisable(d->cap);
            break;
        }
        case GLT_OP_DRAW_BUFFER:
        {
            GLT_draw_buffer *d = &data.draw_buffer;
            ExecuteDrawBuffer(d->mode);
            break;
        }
        case GLT_OP_DRAW_PIXELS:
            IGNORE(op, &data);
            break;
        case GLT_OP_EDGE_FLAG:
            IGNORE(op, &data);
            break;
        case GLT_OP_EDGE_FLAGV:
            IGNORE(op, &data);
            break;
        case GLT_OP_ENABLE:
        {
            GLT_enable *d = &data.enable;
            ExecuteEnable(d->cap);
            break;
        }
        case GLT_OP_END:
            ExecuteEnd();
            break;
        case GLT_OP_END_LIST:
            ExecuteEndList();
            break;
        case GLT_OP_FINISH:
            ExecuteFinish();
            break;
        case GLT_OP_FLUSH:
            ExecuteFlush();
            break;
        case GLT_OP_FOGF:
            IGNORE(op, &data);
            break;
        case GLT_OP_FOGFV:
            IGNORE(op, &data);
            break;
        case GLT_OP_FOGI:
            IGNORE(op, &data);
            break;
        case GLT_OP_FOGIV:
            IGNORE(op, &data);
            break;
        case GLT_OP_FRONT_FACE:
        {
            GLT_front_face *d = &data.front_face;
            ExecuteFrontFace(d->mode);
            break;
        }
        case GLT_OP_FRUSTUM:
        {
            GLT_frustum *d = &data.frustum;
            ExecuteFrustum(d->left, d->right, d->bottom, d->top, d->znear, d->zfar);
            break;
        }
        case GLT_OP_HINT:
            IGNORE(op, &data);
            break;
        case GLT_OP_INDEX_MASK:
            IGNORE(op, &data);
            break;
        case GLT_OP_INDEXD:
            IGNORE(op, &data);
            break;
        case GLT_OP_INDEXDV:
            IGNORE(op, &data);
            break;
        case GLT_OP_INDEXF:
            IGNORE(op, &data);
            break;
        case GLT_OP_INDEXFV:
            IGNORE(op, &data);
            break;
        case GLT_OP_INDEXI:
            IGNORE(op, &data);
            break;
        case GLT_OP_INDEXIV:
            IGNORE(op, &data);
            break;
        case GLT_OP_INDEXS:
            IGNORE(op, &data);
            break;
        case GLT_OP_INDEXSV:
            IGNORE(op, &data);
            break;
        case GLT_OP_INDEXUB:
            IGNORE(op, &data);
            break;
        case GLT_OP_INDEXUBV:
            IGNORE(op, &data);
            break;
        case GLT_OP_LIGHT_MODELF:
                IGNORE(op, &data);
                break;
        case GLT_OP_LIGHT_MODELFV:
        {
            GLT_light_modelfv *d = &data.light_modelfv;
            ExecuteLightModelfv(d->pname, d->params);
            break;
        }
        case GLT_OP_LIGHT_MODELI:
            IGNORE(op, &data);
            break;
        case GLT_OP_LIGHT_MODELIV:
            IGNORE(op, &data);
            break;
        case GLT_OP_LIGHTF:
        {
            GLT_lightf *d = &data.lightf;
            ExecuteLightf(d->light, d->pname, d->param);
            break;
        }
        case GLT_OP_LIGHTFV:
        {
            GLT_lightfv *d = &data.lightfv;
            ExecuteLightfv(d->light, d->pname, d->params);
            break;
        }
        case GLT_OP_LIGHTI:
        {
            GLT_lighti *d = &data.lighti;
            ExecuteLighti(d->light, d->pname, d->param);
            break;
        }
        case GLT_OP_LIGHTIV:
        {
            GLT_lightiv *d = &data.lightiv;
            ExecuteLightiv(d->light, d->pname, d->params);
            break;
        }
        case GLT_OP_LINE_WIDTH:
            IGNORE(op, &data);
            break;
        case GLT_OP_LIST_BASE:
            IGNORE(op, &data);
            break;
        case GLT_OP_LOAD_IDENTITY:
            ExecuteLoadIdentity();
            break;
        case GLT_OP_LOAD_MATRIXD:
        {
            GLT_load_matrixd *d = &data.load_matrixd;
            ExecuteLoadMatrixd(d->m);
            break;
        }
        case GLT_OP_LOAD_MATRIXF:
        {
            GLT_load_matrixf *d = &data.load_matrixf;
            ExecuteLoadMatrixf(d->m);
            break;
        }
        case GLT_OP_MATERIALF:
        {
            GLT_materialf *d = &data.materialf;
            ExecuteMaterialf(d->face, d->pname, d->param);
            break;
        }
        case GLT_OP_MATERIALFV:
        {
            GLT_materialfv *d = &data.materialfv;
            ExecuteMaterialfv(d->face, d->pname, d->params);
            break;
        }
        case GLT_OP_MATERIALI:
        {
            GLT_materiali *d = &data.materiali;
            ExecuteMateriali(d->face, d->pname, d->param);
            break;
        }
        case GLT_OP_MATERIALIV:
        {
            GLT_materialiv *d = &data.materialiv;
            ExecuteMaterialiv(d->face, d->pname, d->params);
            break;
        }
        case GLT_OP_MATRIX_MODE:
        {
            GLT_matrix_mode *d = &data.matrix_mode;
            ExecuteMatrixMode(d->mode);
            break;
        }
        case GLT_OP_MULT_MATRIXD:
        {
            GLT_mult_matrixd *d = &data.mult_matrixd;
            ExecuteMultMatrixd(d->m);
            break;
        }
        case GLT_OP_MULT_MATRIXF:
        {
            GLT_mult_matrixf *d = &data.mult_matrixf;
            ExecuteMultMatrixf(d->m);
            break;
        }
        case GLT_OP_NEW_LIST:
        {
            GLT_new_list *d = &data.new_list;
            ExecuteNewList(d->list, d->mode);
            break;
        }
        case GLT_OP_NORMAL_3B:
        case GLT_OP_NORMAL_3BV:
        {
            GLT_normal_3bv *d = &data.normal_3bv;
            ExecuteNormal3bv(d->v);
            break;
        }
        case GLT_OP_NORMAL_3D:
        case GLT_OP_NORMAL_3DV:
        {
            GLT_normal_3dv *d = &data.normal_3dv;
            ExecuteNormal3dv(d->v);
            break;
        }
        case GLT_OP_NORMAL_3F:
        case GLT_OP_NORMAL_3FV:
        {
            GLT_normal_3fv *d = &data.normal_3fv;
            ExecuteNormal3fv(d->v);
            break;
        }
        case GLT_OP_NORMAL_3I:
        case GLT_OP_NORMAL_3IV:
        {
            GLT_normal_3iv *d = &data.normal_3iv;
            ExecuteNormal3iv(d->v);
            break;
        }
        case GLT_OP_NORMAL_3S:
        case GLT_OP_NORMAL_3SV:
        {
            GLT_normal_3sv *d = &data.normal_3sv;
            ExecuteNormal3sv(d->v);
            break;
        }
        case GLT_OP_ORTHO:
        {
            GLT_ortho *d = &data.ortho;
            ExecuteOrtho(d->left, d->right, d->bottom, d->top, d->znear, d->zfar);
            break;
        }
        case GLT_OP_PIXEL_STOREF:
            IGNORE(op, &data);
            break;
        case GLT_OP_PIXEL_STOREI:
            IGNORE(op, &data);
            break;
        case GLT_OP_PIXEL_TRANSFERF:
            IGNORE(op, &data);
            break;
        case GLT_OP_PIXEL_TRANSFERI:
            IGNORE(op, &data);
            break;
        case GLT_OP_PIXEL_ZOOM:
            IGNORE(op, &data);
            break;
        case GLT_OP_POINT_SIZE:
            IGNORE(op, &data);
            break;
        case GLT_OP_POLYGON_MODE:
            IGNORE(op, &data);
            break;
        case GLT_OP_POLYGON_OFFSET:
            IGNORE(op, &data);
            break;
        case GLT_OP_POLYGON_OFFSET_EXT:
            IGNORE(op, &data);
            break;
        case GLT_OP_POP_ATTRIB:
            IGNORE(op, &data);
            break;
        case GLT_OP_POP_MATRIX:
            ExecutePopMatrix();
            break;
        case GLT_OP_PRIORITIZE_TEXTURES:
            IGNORE(op, &data);
            break;
        case GLT_OP_PRIORITIZE_TEXTURES_EXT:
            IGNORE(op, &data);
            break;
        case GLT_OP_PUSH_ATTRIB:
            IGNORE(op, &data);
            break;
        case GLT_OP_PUSH_MATRIX:
            ExecutePushMatrix();
            break;
        case GLT_OP_RASTER_POS_2D:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_2DV:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_2F:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_2FV:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_2I:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_2IV:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_2S:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_2SV:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_3D:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_3DV:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_3F:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_3FV:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_3I:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_3IV:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_3S:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_3SV:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_4D:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_4DV:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_4F:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_4FV:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_4I:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_4IV:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_4S:
            IGNORE(op, &data);
            break;
        case GLT_OP_RASTER_POS_4SV:
            IGNORE(op, &data);
            break;
        case GLT_OP_RECTD:
            IGNORE(op, &data);
            break;
        case GLT_OP_RECTDV:
            IGNORE(op, &data);
            break;
        case GLT_OP_RECTF:
            IGNORE(op, &data);
            break;
        case GLT_OP_RECTFV:
            IGNORE(op, &data);
            break;
        case GLT_OP_RECTI:
            IGNORE(op, &data);
            break;
        case GLT_OP_RECTIV:
            IGNORE(op, &data);
            break;
        case GLT_OP_RECTS:
            IGNORE(op, &data);
            break;
        case GLT_OP_RECTSV:
            IGNORE(op, &data);
            break;
        case GLT_OP_ROTATED:
        {
            GLT_rotated *d = &data.rotated;
            ExecuteRotated(d->angle, d->x, d->y, d->z);
            break;
        }
        case GLT_OP_ROTATEF:
        {
            GLT_rotatef *d = &data.rotatef;
            ExecuteRotatef(d->angle, d->x, d->y, d->z);
            break;
        }
        case GLT_OP_SCALED:
        {
            GLT_scaled *d = &data.scaled;
            ExecuteScaled(d->x, d->y, d->z);
            break;
        }
        case GLT_OP_SCALEF:
        {
            GLT_scalef *d = &data.scalef;
            ExecuteScalef(d->x, d->y, d->z);
            break;
        }
        case GLT_OP_SCISSOR:
        {
            GLT_scissor *d = &data.scissor;
            ExecuteScissor(d->x, d->y, d->width, d->height);
            break;
        }
        case GLT_OP_SHADE_MODEL:
        {
            GLT_shade_model *d = &data.shade_model;
            ExecuteShadeModel(d->mode);
            break;
        }
        case GLT_OP_STENCIL_FUNC:
            IGNORE(op, &data);
            break;
        case GLT_OP_STENCIL_MASK:
            IGNORE(op, &data);
            break;
        case GLT_OP_STENCIL_OP:
            IGNORE(op, &data);
            break;
        case GLT_OP_TEX_COORD_1D:
        case GLT_OP_TEX_COORD_1DV:
        {
            GLT_tex_coord_1dv *d = &data.tex_coord_1dv;
            ExecuteTexCoord1dv(d->v);
            break;
        }
        case GLT_OP_TEX_COORD_1F:
        case GLT_OP_TEX_COORD_1FV:
        {
            GLT_tex_coord_1fv *d = &data.tex_coord_1fv;
            ExecuteTexCoord1fv(d->v);
            break;
        }
        case GLT_OP_TEX_COORD_1I:
        case GLT_OP_TEX_COORD_1IV:
        {
            GLT_tex_coord_1iv *d = &data.tex_coord_1iv;
            ExecuteTexCoord1iv(d->v);
            break;
        }
        case GLT_OP_TEX_COORD_1S:
        case GLT_OP_TEX_COORD_1SV:
        {
            GLT_tex_coord_1sv *d = &data.tex_coord_1sv;
            ExecuteTexCoord1sv(d->v);
            break;
        }
        case GLT_OP_TEX_COORD_2D:
        case GLT_OP_TEX_COORD_2DV:
        {
            GLT_tex_coord_2dv *d = &data.tex_coord_2dv;
            ExecuteTexCoord2dv(d->v);
            break;
        }
        case GLT_OP_TEX_COORD_2F:
        case GLT_OP_TEX_COORD_2FV:
        {
            GLT_tex_coord_2fv *d = &data.tex_coord_2fv;
            ExecuteTexCoord2fv(d->v);
            break;
        }
        case GLT_OP_TEX_COORD_2I:
        case GLT_OP_TEX_COORD_2IV:
        {
            GLT_tex_coord_2iv *d = &data.tex_coord_2iv;
            ExecuteTexCoord2iv(d->v);
            break;
        }
        case GLT_OP_TEX_COORD_2S:
        case GLT_OP_TEX_COORD_2SV:
        {
            GLT_tex_coord_2sv *d = &data.tex_coord_2sv;
            ExecuteTexCoord2sv(d->v);
            break;
        }
        case GLT_OP_TEX_COORD_3D:
        case GLT_OP_TEX_COORD_3DV:
        {
            GLT_tex_coord_3dv *d = &data.tex_coord_3dv;
            ExecuteTexCoord3dv(d->v);
            break;
        }
        case GLT_OP_TEX_COORD_3F:
        case GLT_OP_TEX_COORD_3FV:
        {
            GLT_tex_coord_3fv *d = &data.tex_coord_3fv;
            ExecuteTexCoord3fv(d->v);
            break;
        }
        case GLT_OP_TEX_COORD_3I:
        case GLT_OP_TEX_COORD_3IV:
        {
            GLT_tex_coord_3iv *d = &data.tex_coord_3iv;
            ExecuteTexCoord3iv(d->v);
            break;
        }
        case GLT_OP_TEX_COORD_3S:
        case GLT_OP_TEX_COORD_3SV:
        {
            GLT_tex_coord_3sv *d = &data.tex_coord_3sv;
            ExecuteTexCoord3sv(d->v);
            break;
        }
        case GLT_OP_TEX_COORD_4D:
        case GLT_OP_TEX_COORD_4DV:
        {
            GLT_tex_coord_4dv *d = &data.tex_coord_4dv;
            ExecuteTexCoord4dv(d->v);
            break;
        }
        case GLT_OP_TEX_COORD_4F:
        case GLT_OP_TEX_COORD_4FV:
        {
            GLT_tex_coord_4fv *d = &data.tex_coord_4fv;
            ExecuteTexCoord4fv(d->v);
            break;
        }
        case GLT_OP_TEX_COORD_4I:
        case GLT_OP_TEX_COORD_4IV:
        {
            GLT_tex_coord_4iv *d = &data.tex_coord_4iv;
            ExecuteTexCoord4iv(d->v);
            break;
        }
        case GLT_OP_TEX_COORD_4S:
        case GLT_OP_TEX_COORD_4SV:
        {
            GLT_tex_coord_4sv *d = &data.tex_coord_4sv;
            ExecuteTexCoord4sv(d->v);
            break;
        }
        case GLT_OP_TEX_ENVF:
        {
            GLT_tex_envf *d = &data.tex_envf;
            ExecuteTexEnvf(d->target, d->pname, d->param);
            break;
        }
        case GLT_OP_TEX_ENVFV:
        {
            GLT_tex_envfv *d = &data.tex_envfv;
            ExecuteTexEnvfv(d->target, d->pname, d->params);
            break;
        }
        case GLT_OP_TEX_ENVI:
        {
            GLT_tex_envi *d = &data.tex_envi;
            ExecuteTexEnvi(d->target, d->pname, d->param);
            break;
        }
        case GLT_OP_TEX_ENVIV:
        {
            GLT_tex_enviv *d = &data.tex_enviv;
            ExecuteTexEnviv(d->target, d->pname, d->params);
            break;
        }
        case GLT_OP_TEX_IMAGE_2D:
        {
            GLT_tex_image_2d *d = &data.tex_image_2d;
            ExecuteTexImage2D(d->target, d->level, d->components, d->width, d->height, d->border, d->format, d->type, d->pixels);
            break;
        }
        case GLT_OP_TEX_PARAMETERF:
        {
            GLT_tex_parameterf *d = &data.tex_parameterf;
            ExecuteTexParameterf(d->target, d->pname, d->param);
            break;
        }
        case GLT_OP_TEX_PARAMETERFV:
        {
            GLT_tex_parameterfv *d = &data.tex_parameterfv;
            ExecuteTexParameterfv(d->target, d->pname, d->params);
            break;
        }
        case GLT_OP_TEX_PARAMETERI:
        {
            GLT_tex_parameteri *d = &data.tex_parameteri;
            ExecuteTexParameteri(d->target, d->pname, d->param);
            break;
        }
        case GLT_OP_TEX_PARAMETERIV:
        {
            GLT_tex_parameteriv *d = &data.tex_parameteriv;
            ExecuteTexParameteriv(d->target, d->pname, d->params);
            break;
        }
        case GLT_OP_TRANSLATED:
        {
            GLT_translated *d = &data.translated;
            ExecuteTranslated(d->x, d->y, d->z);
            break;
        }
        case GLT_OP_TRANSLATEF:
        {
            GLT_translatef *d = &data.translatef;
            ExecuteTranslatef(d->x, d->y, d->z);
            break;
        }
        case GLT_OP_VERTEX_2D:
        case GLT_OP_VERTEX_2DV:
        {
            GLT_vertex_2dv *d = &data.vertex_2dv;
            ExecuteVertex2dv(d->v);
            break;
        }
        case GLT_OP_VERTEX_2F:
        case GLT_OP_VERTEX_2FV:
        {
            GLT_vertex_2fv *d = &data.vertex_2fv;
            ExecuteVertex2fv(d->v);
            break;
        }
        case GLT_OP_VERTEX_2I:
        case GLT_OP_VERTEX_2IV:
        {
            GLT_vertex_2iv *d = &data.vertex_2iv;
            ExecuteVertex2iv(d->v);
            break;
        }
        case GLT_OP_VERTEX_2S:
        case GLT_OP_VERTEX_2SV:
        {
            GLT_vertex_2sv *d = &data.vertex_2sv;
            ExecuteVertex2sv(d->v);
            break;
        }
        case GLT_OP_VERTEX_3D:
        case GLT_OP_VERTEX_3DV:
        {
            GLT_vertex_3dv *d = &data.vertex_3dv;
            ExecuteVertex3dv(d->v);
            break;
        }
        case GLT_OP_VERTEX_3F:
        case GLT_OP_VERTEX_3FV:
        {
            GLT_vertex_3fv *d = &data.vertex_3fv;
            ExecuteVertex3fv(d->v);
            break;
        }
        case GLT_OP_VERTEX_3I:
        case GLT_OP_VERTEX_3IV:
        {
            GLT_vertex_3iv *d = &data.vertex_3iv;
            ExecuteVertex3iv(d->v);
            break;
        }
        case GLT_OP_VERTEX_3S:
        case GLT_OP_VERTEX_3SV:
        {
            GLT_vertex_3sv *d = &data.vertex_3sv;
            ExecuteVertex3sv(d->v);
            break;
        }
        case GLT_OP_VERTEX_4D:
        case GLT_OP_VERTEX_4DV:
        {
            GLT_vertex_4dv *d = &data.vertex_4dv;
            ExecuteVertex4dv(d->v);
            break;
        }
        case GLT_OP_VERTEX_4F:
        case GLT_OP_VERTEX_4FV:
        {
            GLT_vertex_4fv *d = &data.vertex_4fv;
            ExecuteVertex4fv(d->v);
            break;
        }
        case GLT_OP_VERTEX_4I:
        case GLT_OP_VERTEX_4IV:
        {
            GLT_vertex_4iv *d = &data.vertex_4iv;
            ExecuteVertex4iv(d->v);
            break;
        }
        case GLT_OP_VERTEX_4S:
        case GLT_OP_VERTEX_4SV:
        {
            GLT_vertex_4sv *d = &data.vertex_4sv;
            ExecuteVertex4sv(d->v);
            break;
        }
        case GLT_OP_VIEWPORT:
        {
            GLT_viewport *d = &data.viewport;
            ExecuteViewport(d->x, d->y, d->width, d->height);
            break;
        }
        default:
            glt_fatal("%s: invalid opcode %d", progname, op);
            break;
        }
    }
    return !glt_err(trace);
}

/* Utility functions */

char *
basename(char *path)
{
    char *last;
    last = strrchr(path, '/');
    if (!last)
        return path;
    else
        return last + 1;
}

/* Command line parsing */

#include "clparse.h"

enum {
    HACK_AMBIENT,
    NO_SHOW_PROGRESS
};

opt_t opts[] = {
    "amb",          HACK_AMBIENT,
    "ambient",      HACK_AMBIENT,
    "hackambient",  HACK_AMBIENT,
    "noprogress",   NO_SHOW_PROGRESS,
    "np",           NO_SHOW_PROGRESS,
};

int
main(int argc, char **argv)
{
    GLT_trace trace;
    char *input;
    int opt;

    /* Set progname */

    progname = basename(argv[0]);

    /* Parse the command line */

    clp_init(argc, argv, opts, sizeof(opts) / sizeof(opt_t));

    while (!clp_geterror() && (opt = clp_getopt()) >= 0) {
        switch (opt) {
        case HACK_AMBIENT:
            _hack_light_model_ambient = 1;
            break;
        case NO_SHOW_PROGRESS:
            show_progress = 0;
        }
    }

    if (clp_geterror() || argc - clp_getpos() > 1) {
        fprintf(stderr, "usage: %s [-amb] [-noprogress] [gltfile]\n",
                argv[0]);
        exit(1);
    }

    /* Open input and output streams */

    input = clp_maybegetstring(NULL);

    stream = fglsOpen("", FGLS_WRITE);

    glt_open(&trace, input, GLT_RDONLY);

    /* Convert file */

    if (!dispatch_all(&trace)) {
        glt_warn("Parse error!");
        glt_warn("Error near file offset 0x%08x", glt_tell(&trace));
    }

    return 0;
}
