/*
 *  glplay.c
 *
 *  A glt player.
 *  Compile with -DPROMPT to enable the prompting command line option -p.
 *  Compile with -DNOGLTWIN to use non-glt buffer routines.
 *
 *  Kekoa Proudfoot
 *  5/9/98
 */

/* Include files */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef WIN32
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#else
#include <unistd.h>
#endif

#include <GL/gl.h>

#include <glt.h>

#include "clparse.h"
#include "timer.h"

typedef struct {
    char ident;
    int stepsize;
    int count;
} prompt_t;

/* Global variables */

struct {
    float win_scale;
    int   win_single;
    int   trace_memory;
    int   trace_unwind;
    int   auto_clear;
    int   only_context;
    int   first_frame;
    int   num_frames;
    int   time_frames;
    int   glt_debug;
    int   no_exit;
    char *filename;
} param;

char     *progname;
GLT_trace trace;
int       swap_pending = 0;
int       indent       = 0;
int       frame_count  = 0;
int       show_prompt  = 0;
int show_funcs = 0;
int show_info = 0;
int show_prims = 1;
int wire_frame = 0, in_begin = 0;
int auto_flush = 0;
int disable_texture = 0, texture_enabled = 0;
int disable_lights  = 0, lights_enabled  = 0;
GLenum frontpmode = GL_FILL, backpmode = GL_FILL;
int promptcount = 0;
prompt_t cmdprompt = { 'c', 1, 0 };
prompt_t endprompt = { 'p', 1, 0 };
prompt_t swapprompt = { 'f', 1, 0 };
prompt_t *curprompt = NULL;

/* Hack! */

extern int __glt_text_enums;

/* Macros */

#define Round(n)           ((int)((n)+0.5f)-((n)<0.0f))

/* Utility functions */

void
prompt(void)
{
    static char buf[128] = "";
    static char *p = buf;
    if (++promptcount < curprompt->stepsize)
        return;
#if 0
    if ( auto_flush && !in_begin )
        glt_callgl(GLT_OP_FLUSH, NULL);
#else
    if ( auto_flush && !in_begin )
        glt_callgl(GLT_OP_FINISH, NULL);
#endif
    for ( ; ; ) {
        while (*p && *p != '\n') {
            if (isdigit(*p)) {
                int value = 0;
                while (isdigit(*p))
                    value = value * 10 + (*p++ - '0');
                if (value > 0) {
                    if (curprompt->stepsize != value)
                        printf("step=%d\n", value);
                    curprompt->stepsize = value;
                }
                promptcount = 0;
                return;
            }
            switch(*p) {
            case 'c':
                curprompt = &cmdprompt;
                printf("command mode set: step=%d\n", curprompt->stepsize);
                break;
            case 'p':
                curprompt = &endprompt;
                printf("primitive mode set: step=%d\n", curprompt->stepsize);
                break;
            case 'f':
                curprompt = &swapprompt;
                printf("frame mode set: step=%d\n", curprompt->stepsize);
                break;
            case 'i':
                printf("command=%d\n", cmdprompt.count);
                printf("primitive=%d\n", endprompt.count);
                printf("frame=%d\n", swapprompt.count);
                printf("showfuncs=%d\n", show_funcs);
                /* Hack? */
                printf("showenums=%d\n", __glt_text_enums);
                printf("showprims=%d\n", show_prims);
                printf("showinfo=%d\n", show_info);
                printf("autoflush=%d\n", auto_flush);
                printf("autoclear=%d\n", param.auto_clear);
                printf("wireframe=%d\n", wire_frame);
                printf("disable_texture=%d\n", disable_texture);
                printf("disable_lights=%d\n", disable_lights);
                break;
            case 'I':
                show_info = (!show_info);
                if (!show_info)
                    printf("showinfo=%d\n", show_info);
                break;
            case 'S':
                show_funcs = (!show_funcs);
                printf("showfuncs=%d\n", show_funcs);
                break;
            case 'E':
                __glt_text_enums = !__glt_text_enums;
                /* Hack? */
                printf("showenums=%d\n", __glt_text_enums);
                break;
            case 'P':
                show_prims = !show_prims;
                printf("showprims=%d\n", show_prims);
                break;
            case 'F':
                auto_flush = !auto_flush;
                printf("autoflush=%d\n", auto_flush);
                break;
            case 'C':
              param.auto_clear = !param.auto_clear;
                printf("autoclear=%d\n", param.auto_clear);
                break;
            case 'W':
                if (in_begin)
                    printf("wireframe not changed (inside begin/end)\n");
                else {
                    wire_frame = (!wire_frame);
                    printf("wireframe=%d\n", wire_frame);
                    if (wire_frame) {
                        GLT_data pm;
                        pm.polygon_mode.face = GL_FRONT_AND_BACK;
                        pm.polygon_mode.mode = GL_LINE;
                        glt_callgl( GLT_OP_POLYGON_MODE, &pm );
                    }
                    else {
                        GLT_data pm;
                        pm.polygon_mode.face = GL_FRONT;
                        pm.polygon_mode.mode = frontpmode;
                        glt_callgl( GLT_OP_POLYGON_MODE, &pm );
                        pm.polygon_mode.face = GL_BACK;
                        pm.polygon_mode.mode = backpmode;
                        glt_callgl( GLT_OP_POLYGON_MODE, &pm );
                    }
                }
                break;
            case 'T':
                if (in_begin)
                    printf("disable_texture not changed (inside begin/end)\n");
                else {
                    disable_texture = !disable_texture;
                    printf("disable_texture=%d\n", disable_texture);
                    if (disable_texture || !texture_enabled) {
                        GLT_data data;
                        data.disable.cap = GL_TEXTURE_2D;
                        glt_callgl(GLT_OP_DISABLE,&data);
                    } else {
                        GLT_data data;
                        data.enable.cap = GL_TEXTURE_2D;
                        glt_callgl(GLT_OP_ENABLE,&data);
                    }
                }
                break;
            case 'L':
                if (in_begin)
                    printf("disable_lights not changed (inside begin/end)\n");
                else {
                    disable_lights = !disable_lights;
                    printf("disable_lights=%d\n", disable_lights);
                    if (disable_lights || !lights_enabled) {
                        GLT_data data;
                        data.disable.cap = GL_LIGHTING;
                        glt_callgl(GLT_OP_DISABLE,&data);
                    } else {
                        GLT_data data;
                        data.enable.cap = GL_LIGHTING;
                        glt_callgl(GLT_OP_ENABLE,&data);
                    }
                }
                break;
            case 'h':
            case '?':
                printf("valid commands:\n");
                printf("c    set command mode\n");
                printf("p    set primitive mode\n");
                printf("f    set frame mode\n");
                printf("i    show info\n");
                printf("I    toggle showinfo\n");
                printf("S    toggle showfuncs\n");
                printf("E    toggle showenums\n");
                printf("P    toggle showprims\n");
                printf("F    toggle autoflush\n");
                printf("C    toggle autoclear\n");
                printf("W    toggle wireframe\n");
                printf("T    toggle texture\n");
                printf("L    toggle lighting\n");
                printf("#    set step size and run\n");
                printf("h,?  help\n");
                printf("q    quit\n");
                break;
            case 'q':
                exit(0);
                break;
            default:
                if (!isspace(*p) && isprint(*p))
                    printf("invalid command %c\n", *p);
                break;
            }
            p++;
        }
        if (show_info) {
            printf("command=%d\n", cmdprompt.count);
            printf("primitive=%d\n", endprompt.count);
            printf("frame=%d\n", swapprompt.count);
            printf("showfuncs=%d\n", show_funcs);
            printf("showinfo=%d\n", show_info);
            printf("wireframe=%d\n", wire_frame);
            printf("disable_texture=%d\n", disable_texture);
            printf("disable_lights=%d\n", disable_lights);
        }
        printf("%d%c# ", curprompt->count, curprompt->ident);
        fflush(stdout);
        fgets(buf,sizeof(buf),stdin);
        p = buf;
        if (!*p) {
            promptcount = 0;
            return;
        }
    }
}

char *
basename(char *path)
{
    char *last;
    last = strrchr(path, '/');
    if (!last)
        return path;
    else
        return last + 1;
}

void
handle_context( GLT_opcode *op_ptr, GLT_data *data )
{
    GLT_opcode op = *op_ptr;

    do {

        switch ( op ) {

          case GLT_OP_CREATE_CONTEXT:
            if ( param.win_single )
                data->create_context.doublebuffer = 0;
            if ( param.only_context <= 0 ||
                 param.only_context == data->create_context.ctx ) {
                glt_callgl( op, data );
                *op_ptr = op;
                return;
            }
            break;

          case GLT_OP_DESTROY_CONTEXT:
            if ( param.only_context <= 0 ||
                 param.only_context == data->destroy_context.ctx ) {
                glt_callgl( op, data );
                *op_ptr = op;
                return;
            }
            break;

          case GLT_OP_MAKE_CURRENT:
            if ( param.only_context <= 0 ||
                 param.only_context == data->make_current.ctx ) {

                data->make_current.width  *= param.win_scale;
                data->make_current.height *= param.win_scale;
                glt_callgl( op, data );
                *op_ptr = op;

#ifdef PROMPT
                if ( wire_frame ) {
                    GLT_data data;
                    data.polygon_mode.face = GL_FRONT_AND_BACK;
                    data.polygon_mode.mode = GL_LINE;
                    glt_callgl( GLT_OP_POLYGON_MODE, &data );
                }
#endif
                return;
            }
            break;

          default:
            break;
        }

        op = glt_get( &trace, data );

    } while ( op );

}

void
handle_special( GLT_opcode op, GLT_data *data )
{

    switch ( op ) {

      case GLT_OP_ENABLE:
        {
            switch ( data->enable.cap ) {

              case GL_TEXTURE_2D:
                texture_enabled = 1;
                if ( !disable_texture )
                    glt_callgl( op, data );
                break;

              case GL_LIGHTING:
                lights_enabled = 1;
                if ( !disable_lights )
                    glt_callgl( op, data );
                break;

              default:
                glt_callgl( op, data );
            }
        }
        break;

      case GLT_OP_DISABLE:
        {
            switch ( data->disable.cap ) {

              case GL_TEXTURE_2D:
                texture_enabled = 0;
                glt_callgl( op, data );
                break;

              case GL_LIGHTING:
                lights_enabled = 0;
                glt_callgl( op, data );
                break;

              default:
                glt_callgl( op, data );
            }
        }
        break;

      case GLT_OP_VIEWPORT:
      case GLT_OP_SCISSOR:
        {
            /* note that GLT_scissor and GLT_viewport structs look alike */
            data->viewport.x      = Round( data->viewport.x      * param.win_scale);
            data->viewport.y      = Round( data->viewport.y      * param.win_scale);
            data->viewport.width  = Round( data->viewport.width  * param.win_scale);
            data->viewport.height = Round( data->viewport.height * param.win_scale);
            glt_callgl( op, data );
        }
        break;

      case GLT_OP_POLYGON_MODE:
        {
            if ( !wire_frame )
                glt_callgl( op, data );

            switch ( data->polygon_mode.face ) {

              case GL_FRONT:
                frontpmode = data->polygon_mode.mode;
                break;
              case GL_BACK:
                backpmode = data->polygon_mode.mode;
                break;
              case GL_FRONT_AND_BACK:
                frontpmode = data->polygon_mode.mode;
                backpmode = data->polygon_mode.mode;
                break;
            }
        }
        break;

      default:
        abort( );
    }
}

void
skip_frames( int count )
{
    GLT_opcode op;
    GLT_data   data;

    op = glt_get( &trace, &data );
    while ( op ) {

        switch ( op ) {

          case GLT_OP_CREATE_CONTEXT:
          case GLT_OP_MAKE_CURRENT:
          case GLT_OP_DESTROY_CONTEXT:
            handle_context( &op, &data );
            break;

          case GLT_OP_NEW_LIST:
            data.new_list.mode = GL_COMPILE;
            glt_callgl( op, &data );
            do {
                op = glt_get( &trace, &data );
                glt_callgl( op, &data );
            } while ( op != GLT_OP_END_LIST );
            break;

          case GLT_OP_CALL_LIST:
          case GLT_OP_CALL_LISTS:
            break;

          case GLT_OP_END_LIST:
            glt_warn( "glt: where'd this end-list come from?" );
            break;

          case GLT_OP_ENABLE:
          case GLT_OP_DISABLE:
          case GLT_OP_VIEWPORT:
          case GLT_OP_SCISSOR:
          case GLT_OP_POLYGON_MODE:
            handle_special( op, &data );
            break;

          case GLT_OP_CLEAR:
          case GLT_OP_VERTEX_2D:
          case GLT_OP_VERTEX_2DV:
          case GLT_OP_VERTEX_2F:
          case GLT_OP_VERTEX_2FV:
          case GLT_OP_VERTEX_2I:
          case GLT_OP_VERTEX_2IV:
          case GLT_OP_VERTEX_2S:
          case GLT_OP_VERTEX_2SV:
          case GLT_OP_VERTEX_3D:
          case GLT_OP_VERTEX_3DV:
          case GLT_OP_VERTEX_3F:
          case GLT_OP_VERTEX_3FV:
          case GLT_OP_VERTEX_3I:
          case GLT_OP_VERTEX_3IV:
          case GLT_OP_VERTEX_3S:
          case GLT_OP_VERTEX_3SV:
          case GLT_OP_VERTEX_4D:
          case GLT_OP_VERTEX_4DV:
          case GLT_OP_VERTEX_4F:
          case GLT_OP_VERTEX_4FV:
          case GLT_OP_VERTEX_4I:
          case GLT_OP_VERTEX_4IV:
          case GLT_OP_VERTEX_4S:
          case GLT_OP_VERTEX_4SV:
          case GLT_OP_DRAW_PIXELS:
          case GLT_OP_RECTD:
          case GLT_OP_RECTDV:
          case GLT_OP_RECTF:
          case GLT_OP_RECTFV:
          case GLT_OP_RECTI:
          case GLT_OP_RECTIV:
          case GLT_OP_RECTS:
          case GLT_OP_RECTSV:
            break;

          case GLT_OP_SWAP_BUFFERS:
            count--;
            if ( count < 0 )
                return;
            break;

          default:
            glt_callgl( op, &data );
        }

        op = glt_get( &trace, &data );
    }
}

void
emit_func( GLT_opcode op, GLT_data *data )
{
    int i;
    for ( i = 0; i < indent; i++ )
        fputc( ' ', stdout );
    fputs( glt_text(op, data), stdout );
    fputc( '\n', stdout );
}

/* Option table for command line parser */

enum {
    OPT_FIRST,
    OPT_COUNT,
    OPT_CONTEXT,
    OPT_SCALE,
    OPT_MEMORY,
    OPT_TIMES,
    OPT_SINGLE,
    OPT_UNWIND,
    OPT_AUTOCLEAR,
    OPT_PROMPT,
    OPT_DEBUG,
    OPT_NOEXIT
};

opt_t opts[] = {
    { "f",           OPT_FIRST     },
    { "first",       OPT_FIRST     },
    { "c",           OPT_COUNT     },
    { "count",       OPT_COUNT     },
    { "s",           OPT_SCALE     },
    { "scale",       OPT_SCALE     },
    { "C",           OPT_CONTEXT   },
    { "ct",          OPT_CONTEXT   },
    { "ctx",         OPT_CONTEXT   },
    { "context",     OPT_CONTEXT   },
    { "o",           OPT_CONTEXT   },
    { "only",        OPT_CONTEXT   },
    { "m",           OPT_MEMORY    },
    { "mem",         OPT_MEMORY    },
    { "memory",      OPT_MEMORY    },
    { "t",           OPT_TIMES     },
    { "time",        OPT_TIMES     },
    { "times",       OPT_TIMES     },
    { "sb",          OPT_SINGLE    },
    { "single",      OPT_SINGLE    },
    { "u",           OPT_UNWIND    },
    { "unwind",      OPT_UNWIND    },
    { "clear",       OPT_AUTOCLEAR },
    { "autoclear",   OPT_AUTOCLEAR },
#ifdef PROMPT
    { "p",           OPT_PROMPT    },
    { "prompt",      OPT_PROMPT    },
#endif
    { "d",           OPT_DEBUG     },
    { "debug",       OPT_DEBUG     },
    { "noexit",      OPT_NOEXIT    }
};

/* Main */

int
main( int argc, char **argv )
{
    GLT_opcode   op;
    GLT_data     data;
    int          option;
    timeval_t    timer;

    /* Set progname */

    progname = basename( argv[0] );

    /* Parse command line */

    param.win_scale    = 1.0f;
    param.win_single   = 0;
    param.trace_memory = 0;
    param.trace_unwind = 0;
    param.auto_clear   = 0;
    param.only_context = -1;
    param.first_frame  = 0;
    param.num_frames   = 100000;
    param.time_frames  = 0;
    param.glt_debug    = 0;
    param.no_exit      = 0;
    param.filename     = NULL;

    clp_init( argc, argv, opts, sizeof(opts) / sizeof(opt_t) );

    while (!clp_geterror() && (option = clp_getopt()) >= 0) {

        switch (option) {

          case OPT_FIRST:
            param.first_frame = clp_getint( );
            break;

          case OPT_COUNT:
            param.num_frames = clp_getint( );
            break;

          case OPT_SCALE:
            param.win_scale = clp_getfloat( );
            if ( param.win_scale < 0.1f || param.win_scale > 4.0f )
                clp_fail( "win_scale out of range" );
            break;

          case OPT_CONTEXT:
            param.only_context = clp_getint( );
            if ( param.only_context <= 0 )
                clp_fail( "context out of range" );
            break;

          case OPT_MEMORY:
            param.trace_memory = 1;
            break;

          case OPT_TIMES:
            param.time_frames = 1;
            break;

          case OPT_SINGLE:
            param.win_single = 1;
            break;

          case OPT_UNWIND:
            param.trace_unwind = 1;
            break;

          case OPT_AUTOCLEAR:
            param.auto_clear = 1;
            break;

#ifdef PROMPT
          case OPT_PROMPT:
            curprompt = &swapprompt;
            break;
#endif
          case OPT_DEBUG:
            param.glt_debug = 1;
            break;

          case OPT_NOEXIT:
            param.no_exit = 1;
            break;
        }
    }

    argv += clp_getpos( );
    argc -= clp_getpos( );

    /* Print usage on error */

    if ( clp_geterror() || argc < 0 || argc > 1 ) {
        glt_warn("usage: %s [options] [filename]", progname);
        glt_warn("options:");
        glt_warn("    -count   <frames>");
        glt_warn("    -first   <frame>");
        glt_warn("    -scale   <scale>");
        glt_warn("    -context <context>");
        glt_warn("    -memory");
        glt_warn("    -times");
        glt_warn("    -single");
        glt_warn("    -unwind");
        glt_warn("    -clear");
        glt_warn("    -noexit");
#ifdef PROMPT
        glt_warn("    -prompt");
#endif
        glt_warn("    -debug");
        exit(1);
    }

    /* Initialize some variables */

    if ( param.glt_debug )
        __glt_debug |= GLT_DEBUG;

    if ( param.num_frames <= 0 )
        exit( 0 );

    /* Check for prompt and input on stdin */

    if ( !argc && curprompt )
        glt_fatal( "%s: prompt mode not allowed when input is stdin",
                   progname );

    /* Open the trace file */

    if ( argc )
        param.filename = argv[0];

    if ( param.trace_unwind )
        glt_open( &trace, param.filename, GLT_UNWIND | GLT_RETLIST );
    else if ( param.trace_memory )
        glt_open( &trace, param.filename, GLT_RDONLY | GLT_MEM );
    else
        glt_open( &trace, param.filename, GLT_RDONLY );

    if ( param.filename )
        param.filename = basename( param.filename );
    else
        param.filename = "stdin";

    /* skip any initial frames */

    if ( param.first_frame > 0 )
        skip_frames( param.first_frame );

    /* Prompt if we have one */

#ifdef PROMPT
    if ( curprompt )
        prompt( );
#endif

    if ( param.time_frames )
        timer_reset( &timer );

    /* Fetch commands and run them */

    while (( op = glt_get( &trace, &data ) )) {

        switch ( op ) {

          case GLT_OP_CREATE_CONTEXT:
          case GLT_OP_DESTROY_CONTEXT:
          case GLT_OP_MAKE_CURRENT:
            handle_context( &op, &data );
            break;
            /* continue; */

          case GLT_OP_SWAP_BUFFERS:
            glt_callgl( op, &data );
            swap_pending = 1;
            if ( param.time_frames ) {
                float time;
                frame_count++;
                glt_callgl( GLT_OP_FINISH, NULL );
                time = timer_read( &timer );
                if ( time > 1.0f ) {
                    float fps =  (float) frame_count / time;
                    glt_warn( "%s: %6.3f fps", param.filename, fps );
                    timer_reset( &timer );
                    frame_count = 0;
                }
            }
#ifdef PROMPT
            if ( curprompt ) {
                swapprompt.count++;
                show_prompt = ( curprompt == &swapprompt );
            }
#endif
            if ( --param.num_frames == 0 ) {
                glt_callgl( GLT_OP_FINISH, NULL );
                goto exit;
            }
            glt_callgl( GLT_OP_PROCESS_EVENTS, NULL );
            break;

          case GLT_OP_PAD:
            /* ignore pad opcodes */
            break;

#ifdef PROMPT
          case GLT_OP_ENABLE:
          case GLT_OP_DISABLE:
          case GLT_OP_SCISSOR:
          case GLT_OP_VIEWPORT:
          case GLT_OP_POLYGON_MODE:
            handle_special( op, &data );
            break;

          case GLT_OP_BEGIN:
            in_begin = 1;
            glt_callgl(op, &data);
            break;

          case GLT_OP_END:
            in_begin = 0;
            glt_callgl( op, &data );
            if ( curprompt ) {
                endprompt.count++;
                show_prompt = ( curprompt == &endprompt );
            }
            break;
#endif

          case GLT_OP_NEW_LIST:

            if ( param.trace_unwind ) {

                if ( data.new_list.mode == GL_COMPILE ) {

                    while ( op != GLT_OP_END_LIST ) {
#ifdef PROMPT
                        if ( show_funcs && ( !in_begin || show_prims ) )
                            emit_func( op, &data );
#endif
                        if ( op == GLT_OP_NEW_LIST )
                            indent += 2;
                        op = glt_get( &trace, &data );
                    }
                }

            } else {
                glt_callgl( op, &data );
            }
            break;

          case GLT_OP_CALL_LIST:
          case GLT_OP_CALL_LISTS:
          case GLT_OP_END_LIST:
          case GLT_OP_DELETE_LISTS:
            if ( !param.trace_unwind )
                glt_callgl( op, &data );
            break;

          default:
            glt_callgl( op, &data );
        } /* switch ( op ) */

#ifdef PROMPT
        if ( curprompt ) {
            cmdprompt.count++;
            if ( curprompt == &cmdprompt )
                show_prompt = 1;
            if ( param.trace_unwind ) {
                if ( op == GLT_OP_END_LIST )
                    indent -= 2;
            }
            if ( show_funcs && ( !in_begin || show_prims ) )
                emit_func( op, &data );
            if ( param.trace_unwind ) {
                if ( op == GLT_OP_NEW_LIST || op == GLT_OP_CALL_LIST )
                    indent += 2;
            }

            if ( show_prompt ) {
                prompt( );
                show_prompt = 0;
            }
        }
#endif

        if ( swap_pending && param.auto_clear ) {
            /* so we don't damage the actual data field */
            GLT_data data;
            data.clear.mask = GL_COLOR_BUFFER_BIT;
            glt_callgl( GLT_OP_CLEAR, &data );
        }
        swap_pending = 0;

    } /* while ( op ... ) */

 exit:

    /* Check for error condition */

    if ( glt_err( &trace ) ) {
        glt_warn( "Parse error!" );
        glt_warn( "Error near file offset 0x%08x", glt_tell( &trace ) );
    }

    glt_close(&trace);

    if ( param.no_exit ) {
        for ( ; ; ) {
#ifndef WIN32
            sleep(1);
#else
            ;
#endif
        }
    }

    return 0;
}
