/*
 *  filter.c
 *
 *  a filter for glt (tlg) traces that can perform numerous operations
 *
 *  Matthew Eldridge
 *  11-Oct-1999
 */

/* Include files */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <glt.h>

#ifdef WIN32
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif

#include <GL/gl.h>

#ifndef GL_BGRA
#define GL_BGRA           0x80E1
#endif

#ifndef GL_CLAMP_TO_EDGE
#define GL_CLAMP_TO_EDGE  0x812F
#endif

#ifndef GL_TEXTURE_WRAP_R
#define GL_TEXTURE_WRAP_R 0x8072
#endif

#include "clparse.h"

const char *progname = "filter";

int fix_trilinear = 0;
int change_clamp_to_clamp_to_edge = 0;

void
fix_tex_parameter( GLT_opcode *op, GLT_data *data )
{
    GLenum target;
    GLenum pname;
    GLint  param;

    switch ( *op ) {

      case GLT_OP_TEX_PARAMETERI:
        target = data->tex_parameteri.target;
        pname  = data->tex_parameteri.pname;
        param  = data->tex_parameteri.param;
        break;

      case GLT_OP_TEX_PARAMETERIV:
        target = data->tex_parameteriv.target;
        pname  = data->tex_parameteriv.pname;
        param  = data->tex_parameteriv.params[0];
        break;

      case GLT_OP_TEX_PARAMETERF:
        target = data->tex_parameterf.target;
        pname  = data->tex_parameterf.pname;
        param  = (GLint) data->tex_parameterf.param;
        break;

      case GLT_OP_TEX_PARAMETERFV:
        target = data->tex_parameterf.target;
        pname  = data->tex_parameterfv.pname;
        param  = (GLint) data->tex_parameterfv.params[0];
        break;

      default:
        return;
    }

    if ( fix_trilinear ) {

        if ( pname == GL_TEXTURE_MAG_FILTER ||
             pname == GL_TEXTURE_MIN_FILTER ) {

            switch ( param ) {

              case GL_NEAREST:
              case GL_LINEAR:
                param = GL_LINEAR;
                break;

              case GL_NEAREST_MIPMAP_NEAREST:
              case GL_NEAREST_MIPMAP_LINEAR:
              case GL_LINEAR_MIPMAP_NEAREST:
              case GL_LINEAR_MIPMAP_LINEAR:
                param = GL_LINEAR_MIPMAP_LINEAR;
                break;

              default:
                /* unrecognized filter */
                return;
            }

            *op = GLT_OP_TEX_PARAMETERI;
            data->tex_parameteri.target = target;
            data->tex_parameteri.pname  = pname;
            data->tex_parameteri.param  = param;
        }
    }

    if ( change_clamp_to_clamp_to_edge ) {
        if ( pname == GL_TEXTURE_WRAP_S ||
             pname == GL_TEXTURE_WRAP_T ||
             pname == GL_TEXTURE_WRAP_R ) {
            if ( param == GL_CLAMP && change_clamp_to_clamp_to_edge ) {
                *op = GLT_OP_TEX_PARAMETERI;
                param = GL_CLAMP_TO_EDGE;
                data->tex_parameteri.target = target;
                data->tex_parameteri.pname  = pname;
                data->tex_parameteri.param  = param;
            }
        }
    }
}

void
swap_bgra_pixels( int n, GLubyte *src )
{
    GLubyte red, blue;

    while ( n > 0 ) {

        blue = src[0];
        red  = src[2];

        src[0] = red;
        src[2] = blue;

        src += 4;
        n--;
    }
}

const char *
basename( const char *path )
{
    char *last;
    last = strrchr(path, '/');
    if (!last)
        return path;
    else
        return last + 1;
}

enum {
    OPT_TRILINEAR,
    OPT_CLAMP,
    OPT_SPLIT,
    OPT_SWAP_BGRA,
    OPT_NO_PROXY,
    OPT_NO_FOG,
    OPT_NO_POLYGON_MODE,
    OPT_CONTEXT_RGBA8,
    OPT_SWAP,
    OPT_TLG,
    OPT_GLT,
    OPT_NO_SWAP,
    OPT_UNWIND
};

opt_t opts[] = {
    { "trilinear",       OPT_TRILINEAR       },
    { "clamp",           OPT_CLAMP           },
    { "split",           OPT_SPLIT           },
    { "swapbgra",        OPT_SWAP_BGRA       },
    { "swap-bgra",       OPT_SWAP_BGRA       },
    { "no-proxy",        OPT_NO_PROXY        },
    { "no-fog",          OPT_NO_FOG          },
    { "no-polygon-mode", OPT_NO_POLYGON_MODE },
    { "no-poly-mode",    OPT_NO_POLYGON_MODE },
    { "ctxrgba8",        OPT_CONTEXT_RGBA8   },
    { "ctx-rgba8",       OPT_CONTEXT_RGBA8   },
    { "swap",            OPT_SWAP            },
    { "tlg",             OPT_TLG             },
    { "glt",             OPT_GLT             },
    { "no-swap",         OPT_NO_SWAP         },
    { "unwind",          OPT_UNWIND          }
};

static int
native_trace_byte_order( void )
{
    unsigned int   x = 0x01020304;
    unsigned char *c = (unsigned char *) &x;

    return (( c[0] == 0x01 ) ? OPT_GLT : OPT_TLG );
}

int
main( int argc, char **argv )
{
    GLT_trace    in;
    GLT_trace    out;
    GLT_data     data;
    GLT_opcode   op;
    char        *inname;
    char        *outname;

    int  split               = 0;
    int  swap_bgra           = 0;
    int  no_proxy_texture    = 0;
    int  no_fog              = 0;
    int  no_polygon_mode     = 0;
    int  force_rgba8_context = 0;
    int  output_byte_order   = OPT_NO_SWAP;
    int  unwind              = 0;

    int    in_begin     = 0;
    GLenum begin_mode   = GL_POINTS;
    int    vertex_count = 0;
    int    need_begin   = 0;

    int    option;
    int    out_flags;

    /* Set progname */

    progname = basename( argv[0] );

    /* Parse command line */

    clp_init(argc, argv, opts, sizeof(opts)/sizeof(opts[0]) );

    while ( !clp_geterror() && ( option = clp_getopt() ) >= 0 ) {

        switch ( option ) {

          case OPT_TRILINEAR:
            fix_trilinear = 1;
            break;

          case OPT_CLAMP:
            change_clamp_to_clamp_to_edge = 1;
            break;

          case OPT_SPLIT:
            split = 1;
            break;

          case OPT_SWAP_BGRA:
            swap_bgra = 1;
            break;

          case OPT_NO_PROXY:
            no_proxy_texture = 1;
            break;

          case OPT_NO_FOG:
            no_fog = 1;
            break;

          case OPT_NO_POLYGON_MODE:
            no_polygon_mode = 1;
            break;

          case OPT_CONTEXT_RGBA8:
            force_rgba8_context = 1;
            break;

          case OPT_SWAP:
            output_byte_order = OPT_SWAP;
            break;

          case OPT_TLG:
            output_byte_order = OPT_TLG;
            break;

          case OPT_GLT:
            output_byte_order = OPT_GLT;
            break;

          case OPT_NO_SWAP:
            output_byte_order = OPT_NO_SWAP;
            break;

          case OPT_UNWIND:
            unwind = 1;
            break;
        }
    }

    argv += clp_getpos();
    argc -= clp_getpos();

    if ( clp_geterror() || argc < 0 || argc > 2 ) {
        glt_warn("usage: %s [options] [infile [outfile]]", progname );
        glt_warn("options:");
        glt_warn("  -trilinear       : force trilinear (bilinear) filtering");
        glt_warn("  -clamp           : GL_CLAMP -> GL_CLAMP_TO_EDGE");
        glt_warn("  -split           : split primitive blocks");
        glt_warn("  -swapbgra        : translate bgra to rgba textures");
        glt_warn("  -no-proxy        : remove proxy textures");
        glt_warn("  -no-fog          : remove fog commands");
        glt_warn("  -no-polygon-mode : remove polygon mode commands");
        glt_warn("  -ctxrgba8        : force rgba 8/8/8/8 context");
        glt_warn("  -swap            : flip byte order");
        glt_warn("  -no-swap         : maintain byte order (default)");
        glt_warn("  -glt             : force GLT output");
        glt_warn("  -tlg             : force TLG output");
        glt_warn("  -unwind          : unwind display lists");
        exit(1);
    }

    inname  = ( argc > 0 ) ? argv[0] : NULL;
    outname = ( argc > 1 ) ? argv[1] : NULL;


    /* Open traces */

    glt_open( &in, inname, ( unwind ? GLT_UNWIND : 0 ) | GLT_RDONLY );

    out_flags = 0;
    switch ( output_byte_order ) {

      case OPT_SWAP:
        out_flags = ( in.flags & GLT_RSWAP ) ? 0 : GLT_WSWAP;
        break;

      case OPT_TLG:
        out_flags = ( native_trace_byte_order( ) == OPT_GLT ) ? GLT_WSWAP : 0;
        break;

      case OPT_GLT:
        out_flags = ( native_trace_byte_order( ) == OPT_GLT ) ? 0 : GLT_WSWAP;
        break;

      default:
        /* by default leave the byte order alone */
        out_flags = ( in.flags & GLT_RSWAP ) ? GLT_WSWAP : 0;
        break;
    }

    glt_open( &out, outname, GLT_WRONLY | out_flags );

    /* Rename file names for stdio */

    if ( !inname )
        inname = "stdin";
    if ( !outname )
        outname = "stdout";

    /* Read data and copy to output */

    while ( ( op = glt_get( &in, &data ) ) ) {

        /* this is clumsy, but it lets us avoid a glBegin()
           immediately followed by a glEnd() when we are splitting
           primitives */
        if ( need_begin ) {

            if ( op != GLT_OP_END ) {
                GLT_data data;
                data.begin.mode = begin_mode;
                glt_put( &out, GLT_OP_BEGIN, &data );
            } else {
                /* shouldn't generate the end, for obvious reasons */
                op = GLT_OP_PAD;
            }

            need_begin = 0;
        }

        switch ( op ) {

          case GLT_OP_TEX_PARAMETERI:
          case GLT_OP_TEX_PARAMETERIV:
          case GLT_OP_TEX_PARAMETERF:
          case GLT_OP_TEX_PARAMETERFV:
            fix_tex_parameter( &op, &data );
            glt_put( &out, op, &data );
            break;

          case GLT_OP_CREATE_CONTEXT:
            if ( force_rgba8_context ) {
                data.create_context.red   = 8;
                data.create_context.green = 8;
                data.create_context.blue  = 8;
                data.create_context.alpha = 8;
            }
            glt_put( &out, op, &data );
            break;

          case GLT_OP_TEX_IMAGE_2D:
            if ( no_proxy_texture &&
                 data.tex_image_2d.target == GL_PROXY_TEXTURE_2D ) {
                op = GLT_OP_PAD;
            }
            if ( swap_bgra &&
                 data.tex_image_2d.target == GL_TEXTURE_2D &&
                 data.tex_image_2d.format == GL_BGRA &&
                 data.tex_image_2d.type   == GL_UNSIGNED_BYTE ) {
                int n = data.tex_image_2d.width * data.tex_image_2d.height;
                swap_bgra_pixels( n, data.tex_image_2d.pixels );
            }
            if ( op != GLT_OP_PAD )
                glt_put( &out, op, &data );
            break;

          case GLT_OP_TEX_SUB_IMAGE_2D:
          case GLT_OP_TEX_SUB_IMAGE_2D_EXT:
            if ( swap_bgra &&
                 data.tex_sub_image_2d.target == GL_TEXTURE_2D &&
                 data.tex_sub_image_2d.format == GL_BGRA &&
                 data.tex_sub_image_2d.type   == GL_UNSIGNED_BYTE ) {
                int n = ( data.tex_sub_image_2d.width *
                          data.tex_sub_image_2d.height );
                swap_bgra_pixels( n, data.tex_sub_image_2d.pixels );
            }
            glt_put( &out, op, &data );
            break;

          case GLT_OP_FOGF:
          case GLT_OP_FOGFV:
          case GLT_OP_FOGI:
          case GLT_OP_FOGIV:
            if ( !no_fog)
                glt_put( &out, op, &data );
            break;

          case GLT_OP_ENABLE:
            /* tests for things that shouldn't be enabled */
            if ( no_fog && data.enable.cap == GL_FOG )
                op = GLT_OP_PAD;

            if ( op != GLT_OP_PAD )
                glt_put( &out, op, &data );
            break;

          case GLT_OP_POLYGON_MODE:
            if ( !no_polygon_mode )
                glt_put( &out, op, &data );
            break;

          case GLT_OP_BEGIN:
            in_begin     = 1;
            begin_mode   = data.begin.mode;
            vertex_count = 0;
            glt_put( &out, op, &data );
            break;

          case GLT_OP_END:
            in_begin = 0;
            glt_put( &out, op, &data );
            break;

          case GLT_OP_VERTEX_2D:
          case GLT_OP_VERTEX_2DV:
          case GLT_OP_VERTEX_2F:
          case GLT_OP_VERTEX_2FV:
          case GLT_OP_VERTEX_2I:
          case GLT_OP_VERTEX_2IV:
          case GLT_OP_VERTEX_2S:
          case GLT_OP_VERTEX_2SV:
          case GLT_OP_VERTEX_3D:
          case GLT_OP_VERTEX_3DV:
          case GLT_OP_VERTEX_3F:
          case GLT_OP_VERTEX_3FV:
          case GLT_OP_VERTEX_3I:
          case GLT_OP_VERTEX_3IV:
          case GLT_OP_VERTEX_3S:
          case GLT_OP_VERTEX_3SV:
          case GLT_OP_VERTEX_4D:
          case GLT_OP_VERTEX_4DV:
          case GLT_OP_VERTEX_4F:
          case GLT_OP_VERTEX_4FV:
          case GLT_OP_VERTEX_4I:
          case GLT_OP_VERTEX_4IV:
          case GLT_OP_VERTEX_4S:
          case GLT_OP_VERTEX_4SV:

            if ( split && in_begin ) {

                /* the actual vertex */
                glt_put( &out, op, &data );

                vertex_count++;

                switch ( begin_mode ) {

                  case GL_POINTS:
                    /* if ( vertex_count == 1 ) */
                    vertex_count = 0;
                    break;

                  case GL_LINES:
                    if ( vertex_count == 2 )
                        vertex_count = 0;
                    break;

                  case GL_TRIANGLES:
                    if ( vertex_count == 3 )
                        vertex_count = 0;
                    break;

                  case GL_QUADS:
                    if ( vertex_count == 4 )
                        vertex_count = 0;
                    break;

                  default:
                    break;
                }

                /* is it the last vertex in a block? */
                if ( vertex_count == 0 ) {
                    glt_put( &out, GLT_OP_END, NULL );
                    /* we should output a glBegin() here, but the next
                     * command might be a glEnd(), so just remember that
                     * we might need a begin */
                    need_begin = 1;
                }

            } else {

                glt_put( &out, op, &data );

            }
            break;

          default:
            glt_put( &out, op, &data );
            break;
        }
    }

    /* Check for error condition */

    if ( glt_err( &in ) )
        glt_warn( "%s: Parse error near file offset 0x%08x!", inname,
                  glt_tell( &in ) );

    /* Close traces */

    glt_close( &in  );
    glt_close( &out );

    return 0;
}
