/*
 *  dcview.c
 *
 *  A glt player for depth complexity visualization.
 *
 *  Matthew Eldridge
 *  30-Oct-1999
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef WIN32
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#else
#include <unistd.h>
#endif

#include <GL/gl.h>

#include <glt.h>

#include "clparse.h"
#include "timer.h"

typedef struct {
    GLubyte r, g, b;
} RGB8;

static RGB8 palette[] = {
    { 202, 225, 255 },
    { 203, 225, 255 },
    { 204, 226, 255 },
    { 205, 227, 255 },
    { 207, 227, 255 },
    { 208, 228, 255 },
    { 209, 229, 255 },
    { 210, 230, 255 },
    { 211, 230, 255 },
    { 213, 231, 255 },
    { 214, 232, 255 },
    { 215, 232, 255 },
    { 216, 233, 255 },
    { 218, 234, 255 },
    { 219, 234, 255 },
    { 220, 235, 255 },
    { 221, 236, 255 },
    { 222, 236, 255 },
    { 223, 237, 255 },
    { 225, 238, 255 },
    { 226, 238, 255 },
    { 227, 239, 255 },
    { 228, 240, 255 },
    { 230, 240, 255 },
    { 231, 241, 255 },
    { 232, 242, 255 },
    { 233, 243, 255 },
    { 234, 243, 255 },
    { 236, 244, 255 },
    { 237, 245, 255 },
    { 238, 245, 255 },
    { 239, 246, 255 },
    { 240, 247, 255 },
    { 242, 247, 255 },
    { 243, 248, 255 },
    { 244, 249, 255 },
    { 245, 249, 255 },
    { 247, 250, 255 },
    { 248, 251, 255 },
    { 249, 251, 255 },
    { 250, 252, 255 },
    { 251, 253, 255 },
    { 252, 253, 255 },
    { 254, 254, 255 },
    { 254, 255, 254 },
    { 253, 255, 253 },
    { 252, 255, 252 },
    { 250, 255, 250 },
    { 249, 255, 249 },
    { 248, 255, 248 },
    { 247, 255, 247 },
    { 246, 255, 246 },
    { 244, 255, 244 },
    { 243, 255, 243 },
    { 242, 255, 242 },
    { 241, 255, 241 },
    { 239, 255, 239 },
    { 238, 255, 238 },
    { 237, 255, 237 },
    { 236, 255, 236 },
    { 235, 255, 235 },
    { 233, 255, 233 },
    { 232, 255, 232 },
    { 231, 255, 231 },
    { 230, 255, 230 },
    { 228, 255, 228 },
    { 227, 255, 227 },
    { 226, 255, 226 },
    { 225, 255, 225 },
    { 224, 255, 224 },
    { 223, 255, 223 },
    { 221, 255, 221 },
    { 220, 255, 220 },
    { 219, 255, 219 },
    { 218, 255, 218 },
    { 217, 255, 217 },
    { 215, 255, 215 },
    { 214, 255, 214 },
    { 213, 255, 213 },
    { 212, 255, 212 },
    { 210, 255, 210 },
    { 209, 255, 209 },
    { 208, 255, 208 },
    { 207, 255, 207 },
    { 206, 255, 206 },
    { 204, 255, 204 },
    { 203, 255, 203 },
    { 202, 255, 202 },
    { 201, 255, 201 },
    { 199, 255, 199 },
    { 198, 255, 198 },
    { 197, 255, 197 },
    { 196, 255, 196 },
    { 195, 255, 195 },
    { 193, 255, 193 },
    { 192, 255, 192 },
    { 191, 255, 191 },
    { 190, 255, 190 },
    { 188, 255, 188 },
    { 187, 255, 187 },
    { 186, 255, 186 },
    { 185, 255, 185 },
    { 184, 255, 184 },
    { 183, 255, 183 },
    { 181, 255, 181 },
    { 180, 255, 180 },
    { 179, 255, 179 },
    { 178, 255, 178 },
    { 176, 255, 176 },
    { 175, 255, 175 },
    { 174, 255, 174 },
    { 173, 255, 173 },
    { 172, 255, 172 },
    { 170, 255, 170 },
    { 169, 255, 169 },
    { 168, 255, 168 },
    { 167, 255, 167 },
    { 166, 255, 166 },
    { 164, 255, 164 },
    { 163, 255, 163 },
    { 162, 255, 162 },
    { 161, 255, 161 },
    { 159, 255, 159 },
    { 158, 255, 158 },
    { 157, 255, 157 },
    { 156, 255, 156 },
    { 155, 255, 155 },
    { 154, 255, 154 },
    { 152, 255, 152 },
    { 151, 255, 151 },
    { 150, 255, 150 },
    { 149, 255, 149 },
    { 147, 255, 147 },
    { 146, 255, 146 },
    { 145, 255, 145 },
    { 144, 255, 144 },
    { 143, 255, 143 },
    { 141, 255, 141 },
    { 140, 255, 140 },
    { 139, 255, 139 },
    { 138, 255, 138 },
    { 136, 255, 136 },
    { 135, 255, 135 },
    { 134, 255, 134 },
    { 133, 255, 133 },
    { 132, 255, 132 },
    { 130, 255, 130 },
    { 129, 255, 129 },
    { 128, 255, 128 },
    { 127, 255, 127 },
    { 125, 255, 125 },
    { 124, 255, 124 },
    { 123, 255, 123 },
    { 122, 255, 122 },
    { 121, 255, 121 },
    { 120, 255, 120 },
    { 118, 255, 118 },
    { 117, 255, 117 },
    { 116, 255, 116 },
    { 115, 255, 115 },
    { 113, 255, 113 },
    { 112, 255, 112 },
    { 111, 255, 111 },
    { 110, 255, 110 },
    { 109, 255, 109 },
    { 107, 255, 107 },
    { 106, 255, 106 },
    { 105, 255, 105 },
    { 104, 255, 104 },
    { 103, 255, 103 },
    { 101, 255, 101 },
    { 100, 255, 100 },
    {  99, 255,  99 },
    {  98, 255,  98 },
    {  96, 255,  96 },
    {  95, 255,  95 },
    {  94, 255,  94 },
    {  93, 255,  93 },
    {  92, 255,  92 },
    {  90, 255,  90 },
    {  89, 255,  89 },
    {  88, 255,  88 },
    {  87, 255,  87 },
    {  85, 255,  85 },
    {  84, 255,  84 },
    {  83, 255,  83 },
    {  82, 255,  82 },
    {  81, 255,  81 },
    {  80, 255,  80 },
    {  78, 255,  78 },
    {  77, 255,  77 },
    {  76, 255,  76 },
    {  75, 255,  75 },
    {  73, 255,  73 },
    {  72, 255,  72 },
    {  71, 255,  71 },
    {  70, 255,  70 },
    {  69, 255,  69 },
    {  67, 255,  67 },
    {  66, 255,  66 },
    {  65, 255,  65 },
    {  64, 255,  64 },
    {  62, 255,  62 },
    {  61, 255,  61 },
    {  60, 255,  60 },
    {  59, 255,  59 },
    {  58, 255,  58 },
    {  56, 255,  56 },
    {  55, 255,  55 },
    {  54, 255,  54 },
    {  53, 255,  53 },
    {  52, 255,  52 },
    {  51, 255,  51 },
    {  49, 255,  49 },
    {  48, 255,  48 },
    {  47, 255,  47 },
    {  46, 255,  46 },
    {  44, 255,  44 },
    {  43, 255,  43 },
    {  42, 255,  42 },
    {  41, 255,  41 },
    {  40, 255,  40 },
    {  38, 255,  38 },
    {  37, 255,  37 },
    {  36, 255,  36 },
    {  35, 255,  35 },
    {  33, 255,  33 },
    {  32, 255,  32 },
    {  31, 255,  31 },
    {  30, 255,  30 },
    {  29, 255,  29 },
    {  27, 255,  27 },
    {  26, 255,  26 },
    {  25, 255,  25 },
    {  24, 255,  24 },
    {  22, 255,  22 },
    {  21, 255,  21 },
    {  20, 255,  20 },
    {  19, 255,  19 },
    {  18, 255,  18 },
    {  17, 255,  17 },
    {  15, 255,  15 },
    {  14, 255,  14 },
    {  13, 255,  13 },
    {  12, 255,  12 },
    {  10, 255,  10 },
    {   9, 255,   9 },
    {   8, 255,   8 },
    {   7, 255,   7 },
    {   6, 255,   6 },
    {   4, 255,   4 },
    {   3, 255,   3 },
    {   2, 255,   2 },
    {   1, 255,   1 },
    {   0, 255,   0 },
};

struct {
    int   ctx;
    float scale;
    int   single;
    struct {
        int w;
        int h;
    } win;
} opt;

const char *progname = "dcview";

#define Round(n)           ((int)((n)+0.5f)-((n)<0.0f))

char *
basename( char *path )
{
    char *last;
    last = strrchr( path, '/' );
    if ( !last )
        return path;
    else
        return last + 1;
}

enum {
    OPT_CONTEXT,
    OPT_SCALE,
    OPT_SINGLE,
    OPT_HELP
};

opt_t clopts[] = {
    { "ctx",     OPT_CONTEXT },
    { "context", OPT_CONTEXT },
    { "s",       OPT_SCALE   },
    { "scale",   OPT_SCALE   },
    { "single",  OPT_SINGLE  },
    { "h",       OPT_HELP    },
    { "help",    OPT_HELP    }
};

int
skip_till_context( GLT_trace *trace, GLT_opcode op, GLT_data *data )
{
    /* Fetch commands and run them */

    do {

        switch ( op ) {

          case GLT_OP_CREATE_CONTEXT:
            if ( data->create_context.ctx == opt.ctx ) {

                GLT_create_context *ctx = &data->create_context;

                ctx->share        = 0;
                ctx->rgba         = 1;
                ctx->red          = 1;
                ctx->green        = 1;
                ctx->blue         = 1;
                ctx->alpha        = 0;
                ctx->buffersize   = 0;
                ctx->doublebuffer = opt.single ? 0 : 1;
                ctx->depth        = 0;
                ctx->stencil      = 8;
                ctx->accumred     = 0;
                ctx->accumgreen   = 0;
                ctx->accumblue    = 0;
                ctx->accumalpha   = 0;
                data->create_context.share = 0;

                glt_callgl( op, data );
            }
            break;

          case GLT_OP_DESTROY_CONTEXT:
            if ( data->destroy_context.ctx == opt.ctx ) {
                glt_callgl( op, data );
            }
            break;

          case GLT_OP_MAKE_CURRENT:
            if ( data->make_current.ctx == opt.ctx ) {

                opt.win.w = data->make_current.width  *= opt.scale;
                opt.win.h = data->make_current.height *= opt.scale;
                glt_callgl( op, data );
                return op;
            }
            break;
        }

        op = glt_get( trace, data );

    } while ( op );

    return op;
}

void
cook_scissor_or_viewport( GLT_opcode op, GLT_data *data )
{
    switch ( op ) {

      case GLT_OP_VIEWPORT:
        data->viewport.x      = Round( data->viewport.x      * opt.scale );
        data->viewport.y      = Round( data->viewport.y      * opt.scale );
        data->viewport.width  = Round( data->viewport.width  * opt.scale );
        data->viewport.height = Round( data->viewport.height * opt.scale );
        break;

      case GLT_OP_SCISSOR:
        data->scissor.x       = Round( data->scissor.x       * opt.scale );
        data->scissor.y       = Round( data->scissor.y       * opt.scale );
        data->scissor.width   = Round( data->scissor.width   * opt.scale );
        data->scissor.height  = Round( data->scissor.height  * opt.scale );
        break;

      default:
        break;
    }
}

void
setup_depth_complexity( void )
{
    glStencilFunc( GL_ALWAYS, 0, ~0 );
    glStencilOp( GL_INCR, GL_INCR, GL_INCR );
    glEnable( GL_STENCIL_TEST );
    glClear( GL_STENCIL_BUFFER_BIT );
    glColorMask( 0, 0, 0, 0 );
}

void
display_depth_complexity( void )
{
    static int      n_pixels = 0;
    static GLubyte *stencil  = NULL;
    static RGB8    *color    = NULL;
    static RGB8     temp_palette[256];
    int i, n, counts[256], save_matrix_mode;
    GLubyte max;

    n = opt.win.w * opt.win.h;
    if ( n > n_pixels ) {
        if ( stencil )
            free( stencil );
        if ( color )
            free( color );
        n_pixels = n;
        stencil = (GLubyte *) malloc( n_pixels );
        if ( stencil == NULL )
            glt_fatal( "%s: couldn't allocate %d bytes", progname, n_pixels );
        color = (RGB8 *) malloc( n_pixels * 3 );
        if ( color == NULL )
            glt_fatal( "%s: couldn't allocate %d bytes",
                       progname, n_pixels * 3 );
    }

    glReadPixels( 0, 0, opt.win.w, opt.win.h, GL_STENCIL_INDEX,
                  GL_UNSIGNED_BYTE, stencil );

    for ( i = 0; i < 256; i++ )
        counts[i] = 0;

    max = stencil[0];
    for ( i = 0; i < n; i++ ) {
        if ( stencil[i] > max )
            max = stencil[i];
        counts[ stencil[i] ] += 1;
    }

    temp_palette[0].r = 0;
    temp_palette[0].g = 0;
    temp_palette[0].b = 0;

    if ( max ) {
        max--;
        for ( i = 0; i <= max; i++ )
            temp_palette[i+1] = palette[ ( i * 254 ) / max ];
    }

    for ( i = 0; i < n; i++ )
        color[i] = temp_palette[ stencil[i] ];

    glColorMask( 1, 1, 1, 1 );

    glDrawPixels( opt.win.w, opt.win.h, GL_RGB,
                  GL_UNSIGNED_BYTE, color );

    glGetIntegerv( GL_MATRIX_MODE, &save_matrix_mode );

    glMatrixMode( GL_PROJECTION );
    glPushMatrix( );
    glLoadIdentity( );
    glOrtho( 0.0, opt.win.w, 0.0, opt.win.h, 0.0, 100.0 );
    glTranslatef( 0.375f, 0.375f, 0.0f );

    glMatrixMode( GL_MODELVIEW );
    glPushMatrix( );
    glLoadIdentity( );
    glTranslatef( 5, 5, 0 );

    glBegin( GL_LINES );

    glColor3f( 1.0f, 1.0f, 0.0f );

    glVertex2i( 0, 0 );
    glVertex2i( 256, 0 );

    glColor3f( 1.0f, 0.0f, 0.0f );

    for ( i = 0; i < 256; i++ ) {
        if ( counts[i] > 0 ) {
            float y = 100.0f * (float) counts[i] / (float) n;
            glVertex2f( i, 0 );
            glVertex2f( i, y );
        }
    }

    glEnd( );

    glPopMatrix( );
    glMatrixMode( GL_PROJECTION );
    glPopMatrix( );
    glMatrixMode( save_matrix_mode );

    glColorMask( 0, 0, 0, 0 );

    glClear( GL_STENCIL_BUFFER_BIT );
}

void
play_trace( GLT_trace *trace )
{
    GLT_opcode op;
    GLT_data   data;

    op = glt_get( trace, &data );
    skip_till_context( trace, op, &data );

    setup_depth_complexity( );

    while (( op = glt_get( trace, &data ) )) {

        switch( op ) {

          case GLT_OP_CREATE_CONTEXT:
          case GLT_OP_DESTROY_CONTEXT:
          case GLT_OP_MAKE_CURRENT:
            skip_till_context( trace, op, &data );
            break;

          case GLT_OP_VIEWPORT:
          case GLT_OP_SCISSOR:
            cook_scissor_or_viewport( op, &data );
            glt_callgl( op, &data );
            break;

          case GLT_OP_SWAP_BUFFERS:
            display_depth_complexity( );
            glt_callgl( op, &data );
            break;

          case GLT_OP_ENABLE:

            switch ( data.enable.cap ) {

              case GL_TEXTURE_2D:
              case GL_DEPTH_TEST:
              case GL_STENCIL_TEST:
                break;

              default:
                glt_callgl( op, &data );
            }
            break;

          case GLT_OP_DISABLE:

            switch ( data.disable.cap ) {

              case GL_STENCIL_TEST:
                break;

              default:
                glt_callgl( op, &data );
            }
            break;

          case GLT_OP_BEGIN:
          case GLT_OP_CALL_LIST:
          case GLT_OP_CALL_LISTS:
          case GLT_OP_CLIP_PLANE:
          case GLT_OP_CULL_FACE:
          case GLT_OP_DELETE_LISTS:
          case GLT_OP_EDGE_FLAG:
          case GLT_OP_EDGE_FLAGV:
          case GLT_OP_END:
          case GLT_OP_END_LIST:
          case GLT_OP_FRONT_FACE:
          case GLT_OP_FRUSTUM:
          case GLT_OP_LINE_WIDTH:
          case GLT_OP_LIST_BASE:
          case GLT_OP_LOAD_IDENTITY:
          case GLT_OP_LOAD_MATRIXD:
          case GLT_OP_LOAD_MATRIXF:
          case GLT_OP_MATRIX_MODE:
          case GLT_OP_MULT_MATRIXD:
          case GLT_OP_MULT_MATRIXF:
          case GLT_OP_NEW_LIST:
          case GLT_OP_ORTHO:
          case GLT_OP_POINT_SIZE:
          case GLT_OP_POLYGON_MODE:
          case GLT_OP_POLYGON_OFFSET:
          case GLT_OP_POLYGON_OFFSET_EXT:
          case GLT_OP_POP_ATTRIB:
          case GLT_OP_POP_MATRIX:
          case GLT_OP_PUSH_ATTRIB:
          case GLT_OP_PUSH_MATRIX:
          case GLT_OP_RECTD:
          case GLT_OP_RECTDV:
          case GLT_OP_RECTF:
          case GLT_OP_RECTFV:
          case GLT_OP_RECTI:
          case GLT_OP_RECTIV:
          case GLT_OP_RECTS:
          case GLT_OP_RECTSV:
          case GLT_OP_ROTATED:
          case GLT_OP_ROTATEF:
          case GLT_OP_SCALED:
          case GLT_OP_SCALEF:
          case GLT_OP_TRANSLATED:
          case GLT_OP_TRANSLATEF:
          case GLT_OP_VERTEX_2D:
          case GLT_OP_VERTEX_2DV:
          case GLT_OP_VERTEX_2F:
          case GLT_OP_VERTEX_2FV:
          case GLT_OP_VERTEX_2I:
          case GLT_OP_VERTEX_2IV:
          case GLT_OP_VERTEX_2S:
          case GLT_OP_VERTEX_2SV:
          case GLT_OP_VERTEX_3D:
          case GLT_OP_VERTEX_3DV:
          case GLT_OP_VERTEX_3F:
          case GLT_OP_VERTEX_3FV:
          case GLT_OP_VERTEX_3I:
          case GLT_OP_VERTEX_3IV:
          case GLT_OP_VERTEX_3S:
          case GLT_OP_VERTEX_3SV:
          case GLT_OP_VERTEX_4D:
          case GLT_OP_VERTEX_4DV:
          case GLT_OP_VERTEX_4F:
          case GLT_OP_VERTEX_4FV:
          case GLT_OP_VERTEX_4I:
          case GLT_OP_VERTEX_4IV:
          case GLT_OP_VERTEX_4S:
          case GLT_OP_VERTEX_4SV:
            glt_callgl( op, &data );
        }
    }
}

void
usage( void )
{
    glt_warn( "usage: %s [options] [infile]", progname );
    glt_warn( "options:" );
    glt_warn( "  -scale <mult>" );
    glt_warn( "  -single" );
    glt_warn( "  -help" );
}

int
main( int argc, char *argv[] )
{
    GLT_trace  trace;
    char      *filename = NULL;
    int        option;

    progname = basename( argv[0] );

    opt.ctx    = 1;
    opt.scale  = 1.0f;
    opt.single = 0;

    clp_init( argc, argv, clopts, sizeof(clopts) / sizeof(clopts[0]) );

    while ( !clp_geterror( ) && ( option = clp_getopt( ) ) >= 0 ) {

        switch ( option ) {

          case OPT_SCALE:
            opt.scale = clp_getfloat( );
            break;

          case OPT_CONTEXT:
            opt.ctx = clp_getint( );
            break;

          case OPT_SINGLE:
            opt.single = 1;
            break;

          case OPT_HELP:
            usage( );
            exit( 0 );
            break;
        }
    }

    if ( opt.scale > 1.0f )
        glt_fatal( "%s: scale=%f means magnification!", progname, opt.scale );

    if ( opt.scale <= 0.0f )
        glt_fatal( "%s: scale=%f?", progname, opt.scale );

    argv += clp_getpos();
    argc -= clp_getpos();

    /* Print usage on error */

    if ( clp_geterror( ) || argc < 0 || argc > 1 ) {
        usage( );
        exit( 1 );
    }
    if ( argc )
        filename = argv[0];

    glt_open( &trace, filename, GLT_RDONLY );

    if ( filename )
        filename = basename( filename );
    else
        filename = "stdin";

    play_trace( &trace );

    if ( glt_err( &trace ) ) {
        glt_warn( "%s: Parse error!", filename );
        glt_warn( "%s: Error near file offset 0x%08x",
                  filename, glt_tell( &trace ) );
    }

    glt_close( &trace );

    return 0;
}
