/*
 *  win.c
 *
 *  GLT window routines, specialized for X Windows
 *
 *  Kekoa Proudfoot
 *  5/9/98, 6/21/98
 */

/* Include files */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

#include <GL/gl.h>
#include <GL/glx.h>

#include "glt.h"
#include "win.h"

/* Type definitions */

#define MAX_CONTEXTS 32
#define MAX_WINDOWS 32

typedef struct {
    int          id;
    XVisualInfo *visualinfo;
    GLXContext   context;
} GLT_context;

typedef struct {
    int          id;
    int          width;
    int          height;
    Colormap     colormap;
    Window       window;
    /* Hack? */
    int          context_list[MAX_CONTEXTS + 1];
} GLT_window;

/* Global variables */

static char *windownamepattern = NULL;
static Display *display = NULL;
static int screen = 0;
static GLT_context contexts[MAX_CONTEXTS + 1];
static GLT_window windows[MAX_WINDOWS + 1];
static int curcontext;
static int curwindow;

/* Utility functions */

static void
evaluate_window_name( GLT_window *win, char *buf )
{
    char *src, *dst;

    if ( windownamepattern )
        src = windownamepattern;
    else
        src = "glplay %w [ctx %c]";

    dst = buf;

    while ( *src ) {
        if ( *src == '%' ) {
            src++;
            switch ( *src ) {

              case 'w':
                dst += sprintf( dst, "%d", win->id );
                break;

              case 'c':
                {
                    int i, n;
                    n = 0;
                    for ( i = 0; i <= MAX_CONTEXTS; i++ ) {
                        if ( win->context_list[i] ) {
                            if ( n )
                                *dst++ = ' ';
                            dst += sprintf( dst, "%d", i );
                            n++;
                        }
                    }
                }
                break;

              case '%':
                *dst++ = '%';
                break;

              default:
                *dst++ = '%';
                *dst++ = *src;
            }
            src++;
        } else {
            *dst++ = *src++;
        }
    }

    *dst = 0;
}

static void
set_window_name( GLT_window *win, char *name )
{
    XStoreName( display, win->window, name );
    XSetIconName( display, win->window, name );
    XFlush( display );
}

static void
set_all_window_names( void )
{
    int  i;
    char buf[256];

    for ( i = 0; i <= MAX_WINDOWS; i++ ) {
        if ( windows[i].id ) {
            evaluate_window_name( &windows[i], buf );
            set_window_name( &windows[i], buf );
        }
    }
}

static int
error_handler(Display *display)
{
    display = display; /* unused */
    exit(1);
    return 0;
}

static XVisualInfo *
get_visual(GLT_create_context *data)
{
    int list[32], *p;

    p = list;
    if (data->doublebuffer)
        *p++ = GLX_DOUBLEBUFFER;
    if (data->rgba) {
        *p++ = GLX_RGBA;
        *p++ = GLX_RED_SIZE;
        *p++ = data->red;
        *p++ = GLX_GREEN_SIZE;
        *p++ = data->green;
        *p++ = GLX_BLUE_SIZE;
        *p++ = data->blue;
        if (data->alpha) {
            *p++ = GLX_ALPHA_SIZE;
            *p++ = data->alpha;
        }
        if (data->accumred || data->accumgreen || data->accumblue ||
            data->accumalpha) {
            *p++ = GLX_ACCUM_RED_SIZE;
            *p++ = data->accumred;
            *p++ = GLX_ACCUM_GREEN_SIZE;
            *p++ = data->accumgreen;
            *p++ = GLX_ACCUM_BLUE_SIZE;
            *p++ = data->accumblue;
            *p++ = GLX_ACCUM_ALPHA_SIZE;
            *p++ = data->accumalpha;
        }
    }
    else {
        *p++ = GLX_BUFFER_SIZE;
        *p++ = data->buffersize;
    }
    if (data->depth) {
        *p++ = GLX_DEPTH_SIZE;
        *p++ = data->depth;
    }
    if (data->stencil) {
        *p++ = GLX_STENCIL_SIZE;
        *p++ = data->stencil;
    }
    *p++ = (int)None;

    return glXChooseVisual(display, screen, list);
}

static void
describe_visual(XVisualInfo *vi, GLT_create_context *compare)
{
    int tmp;
    char *fmt = "    %-15s\t%d\t[%d]";

    glXGetConfig(display, vi, GLX_RGBA, &tmp);
    glt_warn(fmt, "rgba", tmp, compare->rgba);
    glXGetConfig(display, vi, GLX_RED_SIZE, &tmp);
    glt_warn(fmt, "red", tmp, compare->red);
    glXGetConfig(display, vi, GLX_GREEN_SIZE, &tmp);
    glt_warn(fmt, "green", tmp, compare->green);
    glXGetConfig(display, vi, GLX_BLUE_SIZE, &tmp);
    glt_warn(fmt, "blue", tmp, compare->blue);
    glXGetConfig(display, vi, GLX_ALPHA_SIZE, &tmp);
    glt_warn(fmt, "alpha", tmp, compare->alpha);
    glXGetConfig(display, vi, GLX_BUFFER_SIZE, &tmp);
    glt_warn(fmt, "buffersize", tmp, compare->buffersize);
    glXGetConfig(display, vi, GLX_DOUBLEBUFFER, &tmp);
    glt_warn(fmt, "doublebuffer", tmp, compare->doublebuffer);
    glXGetConfig(display, vi, GLX_DEPTH_SIZE, &tmp);
    glt_warn(fmt, "depth", tmp, compare->depth);
    glXGetConfig(display, vi, GLX_STENCIL_SIZE, &tmp);
    glt_warn(fmt, "stencil", tmp, compare->stencil);
    glXGetConfig(display, vi, GLX_ACCUM_RED_SIZE, &tmp);
    glt_warn(fmt, "accumred", tmp, compare->accumred);
    glXGetConfig(display, vi, GLX_ACCUM_GREEN_SIZE, &tmp);
    glt_warn(fmt, "accumgreen", tmp, compare->accumgreen);
    glXGetConfig(display, vi, GLX_ACCUM_BLUE_SIZE, &tmp);
    glt_warn(fmt, "accumblue", tmp, compare->accumblue);
    glXGetConfig(display, vi, GLX_ACCUM_ALPHA_SIZE, &tmp);
    glt_warn(fmt, "accumalpha", tmp, compare->accumalpha);
}

static XVisualInfo *
find_visual(GLT_create_context *data)
{
    GLT_create_context tmp;
    XVisualInfo *vi;

    vi = get_visual(data);

    if (!vi) {
        /* try smaller sizes */
        memcpy(&tmp, data, sizeof(GLT_create_context));
        if (tmp.red) tmp.red = 1;
        if (tmp.green) tmp.green = 1;
        if (tmp.blue) tmp.blue = 1;
        if (tmp.alpha) tmp.alpha = 1;
        if (tmp.accumred) tmp.accumred = 1;
        if (tmp.accumgreen) tmp.accumgreen = 1;
        if (tmp.accumblue) tmp.accumblue = 1;
        if (tmp.accumalpha) tmp.accumalpha = 1;
        if (tmp.buffersize) tmp.buffersize = 1;
        if (tmp.depth) tmp.depth = 1;
        if (tmp.stencil) tmp.stencil = 1;
        vi = get_visual(&tmp);

        if (!vi) {
            /* try no accumulation, no alpha */
            tmp.alpha = 0;
            tmp.accumred = 0;
            tmp.accumgreen = 0;
            tmp.accumblue = 0;
            tmp.accumalpha = 0;

            vi = get_visual(&tmp);
        }

        if (!vi) {
            /* try no stencil too */
            tmp.stencil = 0;

            vi = get_visual(&tmp);
        }

        if (vi && (__glt_debug & GLT_DEBUG)) {
            glt_warn("Found replacement visual for context %d:", data->ctx);
            describe_visual(vi, data);
        }
    }
    else {
        if (__glt_debug & GLT_DEBUG) {
            glt_warn("Found visual for context %d:", data->ctx);
            describe_visual(vi, data);
        }
    }

    return vi;
}

static void
create_context(GLT_context *ctx, GLT_create_context *data)
{
    XVisualInfo *visualinfo;
    GLXContext context;

    if (!data->ctx)
        glt_fatal("Zero create context index!");

    if (!display) {
        display = XOpenDisplay(0);
        if (!display)
            glt_fatal("Couldn't open display!");
        if (!glXQueryExtension(display, NULL, NULL))
            glt_fatal("No glx extension!");
        screen = DefaultScreen(display);
        XSetIOErrorHandler(error_handler);
    }

    if (!(visualinfo = find_visual(data)))
        glt_fatal("Couldn't find visual!");
    if (!(context = glXCreateContext(display, visualinfo, None, GL_TRUE)))
        glt_fatal("Couldn't create context!");

    /* Currently, no sharing done, but that would be done here if anywhere */

    ctx->id = data->ctx;
    ctx->visualinfo = visualinfo;
    ctx->context = context;
}

static void
destroy_context(GLT_context *ctx, GLT_destroy_context *data)
{
    data = data;     /* unused */

    if (!display)
        glt_fatal("Destroy context with no display!");
    ctx->id = 0;
    if (ctx->visualinfo) {
        XFree(ctx->visualinfo);
        ctx->visualinfo = 0;
    }
    if (ctx->context) {
        glXDestroyContext(display, ctx->context);
        ctx->context = 0;
    }
}

static int
wait_for_map_notify(Display *display, XEvent *event, char *data)
{
    GLT_window *win = (GLT_window *)data;

    display = display;    /* unused */

    if (event->type == MapNotify)
        if (event->xmap.window == win->window)
            return 1;
    return 0;
}

static int
wait_for_configure_notify(Display *display, XEvent *event, char *data)
{
    GLT_window *win = (GLT_window *)data;

    display = display;     /* unused */

    if (event->type == ConfigureNotify)
        if (event->xconfigure.window == win->window)
            /* also check that width and height match? */
            return 1;
    return 0;
}

static void
create_window(GLT_window *win, XVisualInfo *visualinfo, GLT_make_current *data)
{
    XSetWindowAttributes wa;
    Colormap colormap;
    Window window;
    unsigned long mask;
    char buf[256];
    XEvent event;
    XSizeHints hints;

    if (!data->win)
        glt_fatal("Zero create window index!");
    if (!data->width)
        glt_fatal("Zero create window width!");
    if (!data->height)
        glt_fatal("Zero create window height!");

    /* We never even capture a color map, but we should, shouldn't we? */

    colormap = XCreateColormap(display, RootWindow(display, screen),
                               visualinfo->visual, AllocNone);

    wa.colormap = colormap;
    wa.background_pixel = 0;
    wa.border_pixel = 0;
    wa.event_mask = StructureNotifyMask | KeyPressMask;

    mask = CWColormap | CWBackPixel | CWBorderPixel | CWEventMask;

    window = XCreateWindow(display, RootWindow(display, screen), 0, 0,
                           data->width, data->height, 0, visualinfo->depth,
                           InputOutput, visualinfo->visual, mask, &wa);

    win->id = data->win;
    win->width = data->width;
    win->height = data->height;
    win->colormap = colormap;
    win->window = window;
    memset(win->context_list, 0, sizeof(win->context_list));

    evaluate_window_name( win, buf );

    hints.flags = PMinSize | PMaxSize;
    hints.min_width = hints.max_width = data->width;
    hints.min_height = hints.max_height = data->height;

    XSetStandardProperties(display, window, buf, buf, None, 0, 0, &hints);

    XMapWindow(display, window);
    XIfEvent(display, &event, wait_for_map_notify, (XPointer)win);
}

static void
resize_window(GLT_window *win, GLT_make_current *data)
{
    XSizeHints hints;
    XEvent event;

    if (!data->width)
        glt_fatal("Zero resize window width!");
    if (!data->height)
        glt_fatal("Zero resize window height!");

    hints.flags = PMinSize | PMaxSize;
    hints.min_width = hints.max_width = data->width;
    hints.min_height = hints.max_height = data->height;

    XSetNormalHints(display, win->window, &hints);

    win->width = data->width;
    win->height = data->height;

    XResizeWindow(display, win->window, data->width, data->height);
    XIfEvent(display, &event, wait_for_configure_notify, (XPointer)win);
}

static void
make_current(GLT_window *win, GLT_context *ctx, GLT_make_current *data)
{
    if ( data->fail ) {
        glt_warn("Make current has failed flag set, skipping call!");
        return;
    }
    if (!display)
        glt_fatal("Make current with no display!");
    /* Hack? */
    if (win->id && ctx->id) {
        char buf[256];
        win->context_list[ctx->id] = 1;
        evaluate_window_name( win, buf );
        set_window_name( win, buf );
    }
    if (!glXMakeCurrent(display, win->window, ctx->context)) {
        glt_warn("Make current failed!");
    }
}

static void
swap_buffers(GLT_window *win)
{
    if (!display)
        glt_fatal("Swap buffers with no display!");
    glXSwapBuffers(display, win->window);
}

#include <unistd.h>

static void
process_events(void)
{
    XEvent event;
    char   buf[1024];
    KeySym ks;

    while (XPending(display)) {
        XNextEvent(display, &event);
        switch (event.type) {
        case KeyPress:
            XLookupString(&event.xkey, buf, sizeof(buf), &ks, 0);
            switch (ks) {
            case XK_Q:
            case XK_q:
            case XK_Escape:
                exit(0);
                break;
            default:
                break;
            }
            break;
        default:
            break;
        }
    }
}

/* Interface functions */

void
glt_set_window_name_pattern( const char *pattern )
{
    if ( !windownamepattern || strcmp( pattern, windownamepattern ) ) {
        if ( windownamepattern )
            glt_free( windownamepattern );
        windownamepattern = glt_malloc( strlen(pattern) + 1 );
        strcpy( windownamepattern, pattern );
    }

    set_all_window_names( );
}

void
__glt_create_context(GLT_create_context *data)
{
    int ctxid = data->ctx;
    GLT_context *ctx;
    if (ctxid > MAX_CONTEXTS)
        glt_fatal("Context index exceeded %d", MAX_CONTEXTS);
    ctx = &contexts[ctxid];
    if (ctx->id)
        glt_warn("Context %d already created!", ctx->id);
    /* Create context */
    create_context(ctx, data);
}

void
__glt_destroy_context(GLT_destroy_context *data)
{
    int ctxid = data->ctx;
    GLT_context *ctx;
    if (ctxid > MAX_CONTEXTS)
        glt_fatal("Context index exceeded %d", MAX_CONTEXTS);
    ctx = &contexts[ctxid];
    if (ctxid == 0 || ctx->id != ctxid)
        glt_fatal("Can't destroy invalid context %d", ctxid);
    /* Destroy context */
    destroy_context(ctx, data);
}

void
__glt_make_current(GLT_make_current *data)
{
    int winid = data->win;
    int ctxid = data->ctx;
    int winwidth = data->width;
    int winheight = data->height;
    GLT_window *win;
    GLT_context *ctx;
    if (winid > MAX_WINDOWS)
        glt_fatal("Window index exceeded %d", MAX_WINDOWS);
    if (ctxid > MAX_CONTEXTS)
        glt_fatal("Context index exceeded %d", MAX_CONTEXTS);
    win = &windows[winid];
    ctx = &contexts[ctxid];
    if (winid && !win->id)
        /* Create window */
        create_window(win, ctx->visualinfo, data);
    if (win->id != winid)
        glt_fatal("Make current with invalid window %d!", winid);
    if (ctx->id != ctxid)
        glt_fatal("Make current with invalid context %d!", ctxid);
    if (winid && (win->width != winwidth || win->height != winheight)) {
        /* Resize window */
        resize_window(win, data);
    }
    if (curwindow != winid || curcontext != ctxid) {
        /* Make current */
        glt_mesg_va(GLT_DEBUG, ("Making context %d current to window %d",
                                ctxid, winid));
        make_current(win, ctx, data);
    }
    curwindow = winid;
    curcontext = ctxid;
}

void
__glt_swap_buffers(void)
{
    GLT_window *win = &windows[curwindow];
    if (curwindow == 0 || win->id != curwindow)
        glt_fatal("Swap buffers with invalid current window!");
    /* Swap buffers */
    swap_buffers(win);
}

void
__glt_process_events(void)
{
    /* Process events */
    process_events();
}
