/*
 *  text.c
 *
 *  Text functions for glt library
 *
 *  Kekoa Proudfoot
 *  5/6/98, 5/20/98
 */

/* Includes */

#include <stdio.h>
#include <string.h>
#include <assert.h>

#ifdef WIN32
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif
#include <GL/gl.h>

#include "glt.h"
#include "optab.h"
#include "enums.h"

/* Macros */

#define align(p,type) (-(int)(p)&(sizeof(type)-1))

#ifndef GL_TEXTURE_WRAP_R
#define GL_TEXTURE_WRAP_R     0x8072
#endif

/* Functions */

static int
print_begin_mode( char *dst, GLTenum mode )
{
    static const char *mode_names[] = {
        "GL_POINTS",
        "GL_LINES",
        "GL_LINE_LOOP",
        "GL_LINE_STRIP",
        "GL_TRIANGLES",
        "GL_TRIANGLE_STRIP",
        "GL_TRIANGLE_FAN",
        "GL_QUADS",
        "GL_QUAD_STRIP",
        "GL_POLYGON"
    };

    if ( mode <= 9 ) {
        strcpy( dst, mode_names[ mode ] );
        return strlen( mode_names[ mode ] );
    }

    return sprintf( dst, "%u", mode );
}

static int
print_clear_mask( char *buf, GLTenum mask )
{
    char *bufp = buf;

    if (mask & GL_COLOR_BUFFER_BIT)
        bufp += sprintf(bufp, "GL_COLOR_BUFFER_BIT|");

    if (mask & GL_DEPTH_BUFFER_BIT)
        bufp += sprintf(bufp, "GL_DEPTH_BUFFER_BIT|");

    if (mask & GL_ACCUM_BUFFER_BIT)
        bufp += sprintf(bufp, "GL_ACCUM_BUFFER_BIT|");

    if (mask & GL_STENCIL_BUFFER_BIT)
        bufp += sprintf(bufp, "GL_STENCIL_BUFFER_BIT|");

    mask &= ~( GL_COLOR_BUFFER_BIT |
               GL_DEPTH_BUFFER_BIT |
               GL_ACCUM_BUFFER_BIT |
               GL_STENCIL_BUFFER_BIT );

    if (mask)
        bufp += sprintf(bufp, "0x%x", mask);
    else
        bufp -= 1;

    return bufp - buf;
}

static int
fix_tex_parameter( GLT_opcode op, GLT_data *src, GLT_data *dst )
{
    GLTenum target;
    GLTenum pname;
    GLTint  param;

    switch ( op ) {

      case GLT_OP_TEX_PARAMETERI:
        target = src->tex_parameteri.target;
        pname  = src->tex_parameteri.pname;
        param  = src->tex_parameteri.param;
        break;

      case GLT_OP_TEX_PARAMETERIV:
        target = src->tex_parameteriv.target;
        pname  = src->tex_parameteriv.pname;
        param  = src->tex_parameteriv.params[0];
        break;

      case GLT_OP_TEX_PARAMETERF:
        target = src->tex_parameterf.target;
        pname  = src->tex_parameterf.pname;
        param  = (GLTint) src->tex_parameterf.param;
        break;

      case GLT_OP_TEX_PARAMETERFV:
        target = src->tex_parameterf.target;
        pname  = src->tex_parameterfv.pname;
        param  = (GLTint) src->tex_parameterfv.params[0];
        break;

      default:
        return 0;
    }

    switch ( pname ) {

      case GL_TEXTURE_MAG_FILTER:
      case GL_TEXTURE_MIN_FILTER:
      case GL_TEXTURE_WRAP_S:
      case GL_TEXTURE_WRAP_T:
      case GL_TEXTURE_WRAP_R:
        dst->tex_parameteri.target = target;
        dst->tex_parameteri.pname  = pname;
        dst->tex_parameteri.param  = param;
        return 1;
    }

    return 0;
}

static int
fix_tex_env( GLT_opcode op, GLT_data *src, GLT_data *dst )
{
    GLTenum target;
    GLTenum pname;
    GLTint  param;

    switch ( op ) {

      case GLT_OP_TEX_ENVI:
        target = src->tex_envi.target;
        pname  = src->tex_envi.pname;
        param  = src->tex_envi.param;
        break;

      case GLT_OP_TEX_ENVIV:
        target = src->tex_enviv.target;
        pname  = src->tex_enviv.pname;
        param  = src->tex_enviv.params[0];
        break;

      case GLT_OP_TEX_ENVF:
        target = src->tex_envf.target;
        pname  = src->tex_envf.pname;
        param  = (GLTint) src->tex_envf.param;
        break;

      case GLT_OP_TEX_ENVFV:
        target = src->tex_envf.target;
        pname  = src->tex_envfv.pname;
        param  = (GLTint) src->tex_envfv.params[0];
        break;

      default:
        return 0;
    }

    switch ( pname ) {

      case GL_TEXTURE_ENV_MODE:
        dst->tex_envi.target = target;
        dst->tex_envi.pname  = pname;
        dst->tex_envi.param  = param;
        return 1;
    }

    return 0;
}

char *
glt_label(GLT_opcode op)
{
    if (op > GLT_NUM_OPS)
        op = GLT_OP_BAD;
    return ot_getstrent(op)->opname;
}

char *
glt_name(GLT_opcode op)
{
    if (op > GLT_NUM_OPS)
        op = GLT_OP_BAD;
    return ot_getstrent(op)->funcname;
}

char *
glt_text(GLT_opcode op, GLT_data *data)
{
    static char buf[2048];

    char *format;
    char *datap = (char *)data;
    char *bufp = buf;
    char *tmp;
    int offset = 0;
    GLT_data hack;

    if (op > GLT_NUM_OPS)
        op = GLT_OP_BAD;

    bufp += sprintf(bufp, "%s(", ot_getstrent(op)->funcname);

    format = ot_getstrent(op)->format;

    if ( __glt_text_enums ) {
        switch ( op ) {

          case GLT_OP_BEGIN:
            bufp += print_begin_mode(bufp, data->begin.mode);
            format = "";
            break;

          case GLT_OP_CLEAR:
            bufp += print_clear_mask(bufp, data->clear.mask);
            format = "";
            break;

          case GLT_OP_TEX_PARAMETERI:
          case GLT_OP_TEX_PARAMETERF:
          case GLT_OP_TEX_PARAMETERIV:
          case GLT_OP_TEX_PARAMETERFV:
            if (fix_tex_parameter(op, data, &hack)) {
                data = &hack;
                datap = (char *)data;
                format = "eee";
            }
            break;

          case GLT_OP_TEX_ENVI:
          case GLT_OP_TEX_ENVF:
          case GLT_OP_TEX_ENVIV:
          case GLT_OP_TEX_ENVFV:
            if (fix_tex_env(op, data, &hack)) {
                data = &hack;
                datap = (char *)data;
                format = "eee";
            }
            break;
        }
    }

    while (format[0]) {
        switch (format[0]) {
        case OT_BITFIELD:
            offset += align(offset, int);
            bufp += sprintf(bufp, "0x%x", *(int *)&datap[offset]);
            offset += sizeof(int);
            break;
        case OT_BOOLEAN:
            offset += align(offset, char);
            bufp += sprintf(bufp, "%d", *(unsigned char *)&datap[offset]);
            offset += sizeof(char);
            break;
        case OT_BYTE:
            offset += align(offset, char);
            bufp += sprintf(bufp, "%d", *(signed char *)&datap[offset]);
            offset += sizeof(char);
            break;
        case OT_DOUBLE:
            offset += align(offset, double);
            bufp += sprintf(bufp, "%.5f", *(double *)&datap[offset]);
            offset += sizeof(double);
            break;
        case OT_ENUM:
            offset += align(offset, int);
            tmp = __glt_enum_name(*(unsigned int *)&datap[offset]);
            bufp += sprintf(bufp, "%s", tmp);
            offset += sizeof(unsigned int);
            break;
        case OT_FLOAT:
            offset += align(offset, float);
            bufp += sprintf(bufp, "%.5f", *(float *)&datap[offset]);
            offset += sizeof(float);
            break;
        case OT_INT:
            offset += align(offset, int);
            bufp += sprintf(bufp, "%d", *(int *)&datap[offset]);
            offset += sizeof(unsigned int);
            break;
        case OT_SHORT:
            offset += align(offset, short);
            bufp += sprintf(bufp, "%d", *(short *)&datap[offset]);
            offset += sizeof(short);
            break;
        case OT_UBYTE:
            offset += align(offset, char);
            bufp += sprintf(bufp, "%u", *(unsigned char *)&datap[offset]);
            offset += sizeof(char);
            break;
        case OT_UINT:
            offset += align(offset, int);
            bufp += sprintf(bufp, "%u", *(unsigned int *)&datap[offset]);
            offset += sizeof(int);
            break;
        case OT_USHORT:
            offset += align(offset, short);
            bufp += sprintf(bufp, "%u", *(unsigned short *)&datap[offset]);
            offset += sizeof(short);
            break;
        case OT_VOIDP:
            offset += align(offset, int);
            offset += sizeof(int);
            bufp += sprintf(bufp, "0x%08x", *(unsigned int *)&datap[offset]);
            offset += sizeof(int);
            break;
        case OT_ASTART:
            *bufp++ = '{';
            break;
        case OT_AFINISH:
            *bufp++ = '}';
            break;
        default:
            assert(format[0] != format[0]);
            break;
        }
        if (format[0] != OT_ASTART && format[1] && format[1] != OT_AFINISH) {
            *bufp++ = ',';
            *bufp++ = ' ';
        }
        format++;
    }
    *bufp++ = ')';
    *bufp++ = ';';
    *bufp++ = '\0';

    return buf;
}
