#!/usr/common/bin/perl
#
# optab.c.pl
#
# Kekoa Proudfoot 3/31/98
#

use lib "../perl/modules";

use Includes;

use Opcodes qw(: @ops optofn fntolb lbtoop fntoop);
use Flags   qw(: %ignore %noencode);
use Headers qw();
use Encode  qw(fnpstypes fnpisvecs fnpveclens fnpoffsets fnssize fnsalign);

# Defines from optab.h

$OT_MAXOFS = 2;
$OT_LAST = 0xff;
$OT_MAXLEN = 0xff;

$OT_BITFIELD = "x";
$OT_BOOLEAN = "o";
$OT_BYTE = "b";
$OT_DOUBLE = "d";
$OT_ENUM = "e";
$OT_FLOAT = "f";
$OT_INT = "i";
$OT_SHORT = "s";
$OT_UBYTE = "B";
$OT_UINT = "I";
$OT_USHORT = "S";
$OT_VOIDP = "v";
$OT_ASTART = "{";
$OT_AFINISH = "}";

# Subroutine for getting a format character, given a type

sub typemap {
    my %map = ("GLTbitfield", $OT_BITFIELD,
	       "GLTboolean", $OT_BOOLEAN,
	       "GLTbyte", $OT_BYTE,
	       "GLTclampd", $OT_DOUBLE,
	       "GLTclampf", $OT_FLOAT,
	       "GLTdouble", $OT_DOUBLE,
	       "GLTenum", $OT_ENUM,
	       "GLTfloat", $OT_FLOAT,
	       "GLTint", $OT_INT,
	       "GLTshort", $OT_SHORT,
	       "GLTsizei", $OT_INT,
	       "GLTubyte", $OT_UBYTE,
	       "GLTuint", $OT_UINT,
	       "GLTushort", $OT_USHORT,
	       "GLTvoid *", $OT_VOIDP);
    my $type = $map{$_[0]};
    Carp::croak("Uh-oh, no format mapping for $_[0]!") if (!$type);
    return $type;
}

# Compute everything up front

@valid = ();
foreach $op (@ops) {
    $name = optofn($op);
    next if ($ignore{$name} || $noencode{$name});
    $valid[$op] = $name;
}

%ents = %strents = %labels = ();

foreach $name (@valid) {
    if (defined($name)) {
	$opcode = fntoop($name);
	$label = fntolb($name);
	$ssize = fnssize($name);
	$salign = fnsalign($name);
	@pstypes = split(':', fnpstypes($name));
	@pisvecs = split(':', fnpisvecs($name));
	@pveclens = split(':', fnpveclens($name));
	@poffsets = split(':', fnpoffsets($name));

	if ($pstypes[0] eq 'void') {
	    @format = ();
	    @offsets = ();
	}
	else {
	    @offsets = ();
	    @format = ();
	    while (@pstypes) {
		$pstype = shift(@pstypes);
		$pisvec = shift(@pisvecs);
		$pveclen = shift(@pveclens);
		$poffset = shift(@poffsets);
		if ($pisvec && !$pveclen) {
		    if ($poffset & 3) {
			Carp::croak("Uh-oh, vector not int aligned for " .
				"function $name");
		    }
		    push(@offsets, $poffset / 4); # offset is measured in ints
		}
		if ($pisvec) {
		    if ($pveclen) {
			$format = typemap($pstype) x $pveclen;
			push(@format, "$OT_ASTART$format$OT_AFINISH");
		    }
		    else {
			# maybe define a new format for $pstype != GLTvoid
			push(@format, $OT_VOIDP);
		    }
		}
		else {
		    push(@format, typemap($pstype));
		}
	    }
	}

	@flags = ();
	push(@flags, "OT_DALIGN") if ($salign > 4);
	push(@flags, "OT_HASV") if (@offsets);
	push(@flags, "OT_BAD") if ($name eq "gltBad");

	if ($#offsets >= $OT_MAXOFS) {
	    Carp::croak("Uh-oh, OT_MAXOFS exceeded for function $name"); 
	}

	while ($#offsets + 1 < $OT_MAXOFS) {
	    push(@offsets, $OT_LAST);
	}

	$len = $ssize;

	if ($len > $OT_MAXLEN) {
	    Carp::croak("Uh-oh, OT_MAXLEN exceeded for function $name");
	}

	$flags = join(' | ', @flags) if (@flags);
	$flags = "OT_NONE" unless (@flags);
	$offsets = join(', ', @offsets);

	$format = join('', @format);

	$ent = "{ $flags, $len, {$offsets} },";
	$strent = "{ \"$label\", \"$name\", \"$format\" },";

	if (!defined($ents{$opcode})) {
	    $ents{$opcode} = sprintf("%-38s /* $label */", $ent);
	    $strents{$opcode} = $strent;
	}
    }
}

# The string entry for GLT_OP_BAD is special

$strents{lbtoop("GLT_OP_BAD")} = 
  "{ bad_opname, bad_funcname, bad_format },";

# Find max index

$last = 0;
foreach $key (keys(%ents)) {
    $last = $key if ($key > $last);
}

# Write output file

print("/*\n");
print(" * optab.c\n");
print(" *\n");
print(" * Machine generated opcode table\n");
print(" *\n");
print(" */\n");
print("\n");

print("/* Include files */\n");
print("\n");

print("#include \"optab.h\"\n");
print("\n");

print("/* Opcode table */\n");
print("\n");

print("GLT_op_tab_ent __glt_op_tab_ents[] = {\n");

foreach $index (0..$last) {
    $index = 0 if (!defined($ents{$index}));
    print("    $ents{$index}\n");
}

print("};\n");
print("\n");

print("/* Special strings */\n");
print("\n");

print("static char bad_opname[] = \"GLT_OP_BAD\";\n");
print("static char bad_funcname[] = \"gltBad\";\n");
print("static char bad_format[] = \"\";\n");
print("\n");

print("/* Opcode string table */\n");
print("\n");

print("GLT_op_tab_str_ent __glt_op_tab_str_ents[] = {\n");

foreach $index (0..$last) {
    $index = 0 if (!defined($strents{$index}));
    print("    $strents{$index}\n");
}

print("};\n");
