/*
 *  open.c
 *
 *  Function for opening a glt trace file
 *
 *  Kekoa Proudfoot
 *  4/30/98, 5/20/98
 */

#include <stdio.h>
#include <string.h>
#include <errno.h>

#ifdef WIN32
#include <io.h>
#include <fcntl.h>
#include <process.h>
#else
#include <unistd.h>
#endif

#include "glt.h"
#include "optab.h"
#include "file.h"
#include "memory.h"
#include "dlist.h"
#include "swap.h"

#define GZIP "gzip"
#define GOPT "-dc"

/* Function to read bytes from a trace */

static int
trace_read(GLT_trace *trace, void *data, int len)
{
    int got = fread(data, 1, len, trace->file);
    trace->pos += got;
    if (got == len)
        return 1;
    trace->flags |= GLT_EOF;
    if (got)
        trace->flags |= GLT_ERR;
    return 0;
}

/* Function to read header from trace */

static void
get_header(GLT_trace *trace)
{
    char header[4];
    char native[4];
    char swap[4];
    char gzip[4];

    GLT_header hdr;

    *(GLTshort *)&native[0] = GLT_OP_HEADER;
    *(GLTshort *)&native[2] = sizeof(GLT_header);

    *(GLTshort *)&swap[0] = __glt_swap_short(GLT_OP_HEADER);
    *(GLTshort *)&swap[2] = __glt_swap_short(sizeof(GLT_header));

    gzip[0] = (GLT_GZMAGIC >> 24) & 0xff;
    gzip[1] = (GLT_GZMAGIC >> 16) & 0xff;
    gzip[2] = (GLT_GZMAGIC >> 8) & 0xff;
    gzip[3] = (GLT_GZMAGIC >> 0) & 0xff;

    if (!trace_read(trace, header, 4)) {
        trace->flags |= GLT_ERR;
    }
    else if (!memcmp(header, native, 4)) {
        if (!trace_read(trace, &hdr, sizeof(GLT_header)))
            trace->flags |= GLT_ERR;
        else {
            trace->magic = hdr.magic;
            trace->version = hdr.version;
        }
    }
    else if (!memcmp(header, swap, 4)) {
        trace->flags |= GLT_RSWAP;
        if (!trace_read(trace, &hdr, sizeof(GLT_header)))
            trace->flags |= GLT_ERR;
        else {
            trace->magic = hdr.magic;
            trace->version = __glt_swap_long(hdr.version);
        }
    }
    /* Hack! Detects .gz files */
    else if (!memcmp(header, gzip, 3)) {
        trace->flags |= GLT_GZIP;
        trace->magic = GLT_GZMAGIC;
    }
    else
        trace->flags |= GLT_ERR;
}

/* Custom popen for gzipped input */

static FILE *
gzopen(const char *filename)
{
    FILE *file;
    int fds[2];
#ifdef WIN32
    int outfd;
    int errfd;
    int newerr;
    int fail = 0;
#endif
    int pid;
    int c;

    char *argv[] = { GZIP, GOPT, NULL, NULL };

    argv[2] = (char *)filename;

    if (!filename)
        return NULL;

#ifdef WIN32
    if (_pipe(fds, 16384, O_BINARY | O_NOINHERIT) < 0) /* create pipe */
        return NULL;
    if ((newerr = open("nul", O_WRONLY)) < 0) /* open nul */
        return NULL;
    if ((outfd = dup(fileno(stdout))) < 0) /* save stdout */
        return NULL;
    if ((errfd = dup(fileno(stderr))) < 0) /* save stderr */
        return NULL;
    if ((dup2(fds[1], fileno(stdout))) < 0) /* set stdout for child */
        return NULL;
    close(fds[1]); /* close pipe input */
    /* Hack?  On fail, try to restore stderr and stdout, if possible */
    if (dup2(newerr, fileno(stderr)) < 0) /* set stderr for child */
        fail = 1;
    else if (spawnvp(P_NOWAIT, argv[0], argv) < 0) /* spawn child */
        fail = 1;
    if (dup2(errfd, fileno(stderr)) < 0) /* restore stderr */
        fail = 1;
    if (dup2(outfd, fileno(stdout)) < 0) /* restore stdout */
        return NULL;
    if (fail) /* check for fail */
        return NULL;
    close(errfd); /* close saved stderr */
    close(outfd); /* close saved stdout */
    close(newerr); /* close new stderr */
#else
    if (pipe(fds) < 0)
        return NULL;
    if ((pid = fork()) < 0)
        return NULL;
    if (!pid) {
        /* child process */
        close(fds[0]); /* close pipe output */
        if (dup2(fds[1], 1) < 0) /* copy pipe input to stdout */
            exit(1);
        close(fds[1]); /* close original copy of pipe input */
        close(2); /* no stderr */
        execvp(argv[0], argv); /* execute program */
        exit(1); /* or fail */
    }
    /* parent process */
    close(fds[1]); /* close pipe input */
#endif

    file = fdopen(fds[0], "rb"); /* let stdio control pipe output */

    /* Check for empty output */

    if ((c = getc(file)) == EOF) {
        fclose(file);
        return NULL;
    }
    ungetc(c, file);

    return file;
}

/* Open function */

void
glt_open(GLT_trace *trace, char *filename, int flags)
{
    /* Determine internal flags */

    flags &= (GLT_WRONLY | GLT_MEM | GLT_ERROK | GLT_WSWAP |
              GLT_UNWIND | GLT_RETLIST | GLT_UNBUF);

    if ((flags & GLT_UNWIND) && (flags & (GLT_WRONLY | GLT_MEM | GLT_WSWAP)))
        glt_fatal("glt_open: GLT_UNWIND must be single flag.");

    if (!filename || !filename[0])
        flags |= GLT_NFILE;
    if (flags & GLT_WRONLY)
        flags |= GLT_WROK;
    else
        flags |= GLT_RDOK;

    /* Try to be intelligent and not read from or write to a tty */

    if (flags & GLT_NFILE) {
        FILE *f = ( flags & GLT_WRONLY ) ? stdout : stdin;
        if ( isatty( fileno(f) ) ) {
            glt_fatal( "glt_open: trace data not %s a terminal.",
                       ( flags & GLT_WRONLY ) ? "written to" : "read from" );
        }
    }

    /* Fix filename if no file */

    if (flags & GLT_NFILE) {
        filename = (flags & GLT_WRONLY) ? "stdout" : "stdin";
#ifdef WIN32
        setmode(fileno((flags & GLT_WRONLY) ? stdout : stdin), O_BINARY);
#endif
    }

    /* Open file */

    if (flags & GLT_WRONLY)
        trace->file = (flags & GLT_NFILE) ? stdout : fopen(filename, "wb");
    else
        trace->file = (flags & GLT_NFILE) ? stdin : fopen(filename, "rb");

    if (!trace->file) {
        char *msg = strerror(errno);
        if (!msg)
            msg = "open failed";
        glt_fatal("glt_open: %s: %s", filename, msg);
    }

    /* Turn off buffering */
    if (flags & GLT_UNBUF) {
        setbuf(trace->file,NULL);
    }

    /* Determine if this file allows seeking */

    if (fseek(trace->file, 0, SEEK_SET) != 0) {
        flags |= GLT_NSEEK;
        clearerr(trace->file);
    }

    /* Initialize trace struct */

    trace->flags = flags;
    trace->pos = 0;
    trace->len = 0;
    trace->data = NULL;
    trace->header.op = GLT_OP_BAD;
    trace->nallocvecs = 0;
    trace->ngetvecs = 0;

    /* Read the header if reading */

    if (flags & GLT_RDOK) {
        get_header(trace);
        /* Check for gzipped input */
        if (trace->flags & GLT_GZIP) {
            if (trace->flags & GLT_NFILE)
                glt_fatal("glt_open: %s: pipe through gzcat to decompress",
                          filename);
            if (fclose(trace->file))
                glt_mesg(GLT_DEBUG, "glt_open: close failed");
            glt_mesg_va(GLT_DEBUG, ("%s: opening with gzopen", filename));
            trace->file = gzopen(filename);
            if (!trace->file)
                glt_fatal("glt_open: %s: gzopen failed", filename);
            trace->flags |= GLT_NFILE | GLT_NSEEK;
            trace->pos = 0;
            get_header(trace);
        }
        /* Report on error */
        if (!(flags & GLT_ERROK)) {
            char magic[5] = {0,0,0,0,0};
            if (trace->flags & GLT_ERR)
                glt_fatal("glt_open: %s: not a GLT file", filename);
            if (!(trace->flags & GLT_RSWAP) && trace->magic != GLT_MAGIC)
                glt_fatal("glt_open: %s: bad magic number", filename);
            if ((trace->flags & GLT_RSWAP) && trace->magic != GLT_SMAGIC)
                glt_fatal("glt_open: %s: bad magic number", filename);
            if (trace->version < GLT_COMPAT)
                glt_fatal("glt_open: %s: incompatible version %d (min is %d)",
                          filename, trace->version, GLT_COMPAT);
            if (trace->version > GLT_VERSION)
                glt_fatal("glt_open: %s: incompatible version %d (max is %d)",
                          filename, trace->version, GLT_VERSION);
            strncpy(magic, (char *)&trace->magic, 4);
            glt_mesg_va(GLT_DEBUG, ("File %s contains magic %s, version %d",
                                    filename, magic, trace->version));
        }
    }

    /* Set function pointers */

    if (flags & GLT_UNWIND)
        __glt_open_dlist(trace, filename);
    else if (flags & GLT_MEM)
        __glt_open_mem(trace, filename);
    else
        __glt_open_file(trace, filename);

    /* Write the header if writing */

    if (flags & GLT_WROK) {
        GLT_data data;
        data.header.magic = GLT_MAGIC;
        data.header.version = GLT_VERSION;
        glt_put(trace, GLT_OP_HEADER, &data);
    }
}
