/*
 *  memory.c
 *
 *  Internal glt functions for memory traces
 *
 *  Missing are optimized reads and writes that don't reqire data to be
 *  copied.  It's definitely possible to add these, but the glt function
 *  interface would need to change, and this would likely mess up glimpl,
 *  making there be two versions (one for memory tracing and one for file
 *  tracing).
 *
 *  Kekoa Proudfoot
 *  5/20/98
 */

/* Include files */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>

#include "glt.h"
#include "optab.h"
#include "file.h"
#include "memory.h"

/* Configuration macros */

/* Min block size should be considerably larger than the max texture size? */

#define MIN_BLOCK_SIZE 4194264 /* almost 4MB, but not quite */

/* Types */

typedef struct GLT_memblk {
    int pos;
    int len;
    void *data;
    struct GLT_memblk *next;
    struct GLT_memblk *prev;
    double dstart[1];
} GLT_memblk;

/* Macros */

#define offset(p,ofs)       (&(((int *)p)[ofs]))
#define vec_len(st,ent,vec) (*(int *)offset(st,ent->offsets[vec]))
#ifndef GLT64
#define vec_ptr(st,ent,vec) (*(void **)offset(st,ent->offsets[vec]+1))
#else
#define vec_ptr(st,ent,vec) (*((void **)((char *)st+ent->len+dalign(ent->len))+vec))
#endif
#define align(p,type)       (-(int)(p)&(sizeof(type)-1))
#define walign(p)           (align(p,int))
#define dalign(p)           (align(p,double))

/* Function prototypes for open */

static GLT_memblk *extend_mem(GLT_trace *trace, int size);
static void glt_close_mem(GLT_trace *trace);
static void *glt_allocv_mem(GLT_trace *trace, int size);
static void glt_put_mem(GLT_trace *trace, GLT_opcode op, GLT_data *data);
static GLT_opcode glt_peek_mem(GLT_trace *trace);
static GLT_opcode glt_get_mem(GLT_trace *trace, GLT_data *data);
static void glt_skip_mem(GLT_trace *trace);
static int glt_tell_mem(GLT_trace *trace);
static void glt_seek_mem(GLT_trace *trace, int pos);
static int glt_eof_mem(GLT_trace *trace);
static int glt_err_mem(GLT_trace *trace);

/* Open function */

void
__glt_open_mem(GLT_trace *trace, char *filename)
{
    if (trace->flags & GLT_WRONLY)
        trace->data = extend_mem(trace, MIN_BLOCK_SIZE);
    else {
        if (trace->flags & GLT_NSEEK) {
            glt_warn("glt: %s: memory open failed, using file open instead",
                     filename);
            __glt_open_file(trace, filename);
            trace->flags &= ~GLT_MEM;
            return;
        }
        else if (trace->flags & GLT_RSWAP) {
            glt_warn("glt: %s: non-native byte order, using file open instead",
                     filename);
            __glt_open_file(trace, filename);
            trace->flags &= ~GLT_MEM;
            return;
        }
        else {
            fseek(trace->file, 0, SEEK_END);
            trace->len = ftell(trace->file);
            fseek(trace->file, 0, SEEK_SET);
            /* if we could catch this malloc, could revert to file open... */
            if (trace->len)
                trace->data = (void *)glt_malloc(trace->len+4);
            else
                trace->data = (void *)glt_malloc(4);
            /* put a GLT_OP_BAD header at the end of the memory as
             * a safety net */
            *(GLTuint *) ((GLTubyte *) trace->data + trace->len) = 0;
            glt_mesg_va(GLT_WARN, ("glt: reading %d bytes...", trace->len));
            /* Hack! should check the number of bytes actually read */
            fread(trace->data, trace->len, 1, trace->file);
        }
    }

    /* Hack! GLT_WSWAP is easy to implement, but it has been put off */
    if (trace->flags & GLT_WSWAP)
        glt_fatal("glt: memory traces don't support GLT_WSWAP");

    trace->close = glt_close_mem;
    trace->allocv = glt_allocv_mem;
    trace->put = glt_put_mem;
    trace->peek = glt_peek_mem;
    trace->get = glt_get_mem;
    trace->skip = glt_skip_mem;
    trace->tell = glt_tell_mem;
    trace->seek = glt_seek_mem;
    trace->eof = glt_eof_mem;
    trace->err = glt_err_mem;
}

/* Internal functions */

static GLT_memblk *
extend_mem(GLT_trace *trace, int size)
{
    GLT_memblk *blk = (GLT_memblk *)glt_malloc(sizeof(GLT_memblk) + size);

    blk->pos = 0;
    blk->len = size;
    blk->data = blk->dstart;
    blk->next = NULL;
    blk->prev = trace->data;
    if (blk->prev) {
        assert(blk->prev->next == NULL);
        blk->prev->next = blk;
    }

    trace->data = blk;

    return blk;
}

static void
flush_mem(GLT_trace *trace)
{
    GLT_memblk *blk;
    assert(trace->data);
    glt_mesg_va(GLT_WARN, ("glt: writing %d bytes...", trace->pos));
    for (blk = trace->data; blk->prev; blk = blk->prev);
    do {
        GLT_memblk *tmp;
        if (blk->len)
            fwrite(blk->data, blk->pos, 1, trace->file);
        tmp = blk;
        blk = blk->next;
        glt_free(tmp);
    } while (blk);
}

static void
glt_close_mem(GLT_trace *trace)
{
    while (trace->nallocvecs)
        glt_free(trace->allocvecs[--trace->nallocvecs]);
    while (trace->ngetvecs)
        glt_free(trace->getvecs[--trace->ngetvecs]);
    if (trace->flags & GLT_WRONLY)
        flush_mem(trace);
    if (trace->flags & GLT_RDOK)
        if (trace->data)
            glt_free(trace->data);
    if (trace->flags & GLT_NFILE)
        return;
    if (fclose(trace->file))
        glt_mesg(GLT_DEBUG, "glt_close: close failed");
}

static void *
glt_allocv_mem(GLT_trace *trace, int size)
{
    void *p;
    assert(trace->nallocvecs < GLT_MAXV);
    trace->allocvecs[trace->nallocvecs++] = p = glt_malloc(size);
    assert(p);
    return p;
}

static __inline void
trace_write(GLT_trace *trace, void *data, int len)
{
    GLT_memblk *blk = (GLT_memblk *)trace->data;
    if (blk->pos + len > blk->len)
        blk = extend_mem(trace, (len < MIN_BLOCK_SIZE) ? MIN_BLOCK_SIZE : len);
    memcpy(&((char *)blk->data)[blk->pos], data, len);
    blk->pos += len;
}

static void
glt_put_mem(GLT_trace *trace, GLT_opcode op, GLT_data *data)
{
    GLT_dheader header;
    GLT_op_tab_ent *ent = ot_getent(op);
    int len, hlen;
    int i;

    assert(trace->flags & GLT_WROK);
    assert(walign(trace->pos) == 0);
    assert(op < GLT_NUM_OPS);
    assert(!(ent->flags & OT_BAD));

    glt_mesg_va(GLT_PUT, ("tracing %s", glt_label(op)));

    glt_mesg(GLT_PUTF, glt_text(op, data));

    /* Pad if necessary */

    if (ent->flags & OT_DALIGN) {
        if (dalign(trace->pos + ((ent->flags & OT_HASV) ? 8 : 4)) == 4) {
            header.op = GLT_OP_PAD;
            header.len = 0;
            trace_write(trace, &header, 4);
            trace->pos += 4;
        }
    }

    /* Compute length */

    if (ent->flags & OT_HASV) {
        int vlen_save[OT_MAXOFS];
        void *vptr_save[OT_MAXOFS];
        int zero[2] = { 0, 0 };

        len = ent->len;
        for (i = 0; i < OT_MAXOFS; i++) {
            int vlen;
            void *vptr;

            if (ent->offsets[i] == OT_LAST)
                break;

            vlen = vec_len(data, ent, i);
            vptr = vec_ptr(data, ent, i);

            glt_mesg_va(GLT_PUT, ("vector, len = %d", vlen));

            vlen_save[i] = vlen;
            vlen += (ent->flags & OT_DALIGN) ? dalign(vlen) : walign(vlen);
            len += vlen;
            vec_len(data, ent, i) = vlen;

            vptr_save[i] = vptr;
            vec_ptr(data, ent, i) = NULL;
        }

        /* Write long header */

        header.op = op;
        header.len = 0xffff;
        header.llen = len;

        glt_mesg_va(GLT_PUT, ("writing header op=%d llen=%d", op, len));

        trace_write(trace, &header, 8);
        hlen = 8;

        /* Update trace position */

        trace->pos += hlen;

        assert(!(ent->flags & OT_DALIGN) || (dalign(trace->pos) == 0));

        /* Write data */

        trace_write(trace, data, ent->len);

        /* Update trace position */

        trace->pos += ent->len;

        /* Write vectors */

        for (i = 0; i < OT_MAXOFS; i++) {
            int vlen;
            if (ent->offsets[i] == OT_LAST)
                break;
            vlen = vec_len(data, ent, i);
            if (vlen != 0) {
                int pad = vlen - vlen_save[i];
                trace_write(trace, vptr_save[i], vlen_save[i]);
                if (pad)
                    trace_write(trace, zero, pad);
                trace->pos += vlen;
            }
            vec_len(data, ent, i) = vlen_save[i];
            vec_ptr(data, ent, i) = vptr_save[i];
        }

        /* Free allocvecs */

        while (trace->nallocvecs)
            glt_free(trace->allocvecs[--trace->nallocvecs]);
    }
    else {

        len = ent->len;

        /* Write short header */

        assert(len < 0xffff);
        header.op = op;
        header.len = len;

        glt_mesg_va(GLT_PUT, ("writing header op=%d len=%d", op, len));

        trace_write(trace, &header, 4);
        hlen = 4;

        /* Update trace position */

        trace->pos += hlen;

        assert(!(ent->flags & OT_DALIGN) || (dalign(trace->pos) == 0));

        /* Write data */

        trace_write(trace, data, ent->len);

        /* Update trace position */

        trace->pos += ent->len;
    }
}

static __inline int
trace_read(GLT_trace *trace, void *data, int len)
{
    int got;
    if (trace->pos + len > trace->len)
        got = trace->len - trace->pos;
    else
        got = len;
    memcpy(data, &((char *)trace->data)[trace->pos], got);
    trace->pos += got;
    if (got == len)
        return 1;
    trace->flags |= GLT_EOF;
    if (got)
        trace->flags |= GLT_ERR;
    return 0;
}

static __inline void
trace_get_header(GLT_trace *trace)
{
    assert(!trace->header.op);
    assert(walign(trace->pos) == 0);

    if (!trace_read(trace, &trace->header, 4)) {
        trace->header.op = GLT_OP_BAD;
        trace->header.len = 0;
    }
    if (trace->header.len == 0xffff) {
        if (!trace_read(trace, &trace->header.llen, 4)) {
            trace->flags |= GLT_ERR;
            trace->header.op = GLT_OP_BAD;
            trace->header.llen = 0;
        }
    }
    else {
        trace->header.llen = trace->header.len;
    }
    if (trace->header.op >= GLT_NUM_OPS) {
        glt_mesg_va(GLT_DEBUG, ("Bad opcode %d near file offset 0x%08x",
                                trace->header.op, trace->pos));
        trace->flags |= GLT_ERR;
        trace->header.op = GLT_OP_BAD;
        trace->header.llen = 0;
    }
#ifndef NDEBUG
    else if (ot_getent(trace->header.op)->flags & OT_BAD) {
        if (trace->header.op != GLT_OP_BAD) {
            glt_mesg_va(GLT_DEBUG, ("Bad opcode %d near file offset 0x%08x",
                                    trace->header.op, trace->pos));
            trace->header.op = GLT_OP_BAD;
            trace->header.llen = 0;
        }
    }
#endif
}

static GLT_opcode
glt_peek_mem(GLT_trace *trace)
{
    if (!trace->header.op)
        trace_get_header(trace);

    return trace->header.op;
}

/* Really have all these ifs in get? */
static GLT_opcode
glt_get_mem(GLT_trace *trace, GLT_data *data)
{
    GLT_op_tab_ent *ent;
    GLT_opcode op;
    int start;

    if (!trace->header.op)
        trace_get_header(trace);

    op = trace->header.op;

#if 0
    if (trace->header.op > GLT_NUM_OPS) {
        trace->flags |= GLT_ERR; /* really do this? */
        trace->header.op = GLT_OP_BAD; /* really do this? */
        trace->header.llen;
        /* unrecognized, so skip and return real opcode? */
    }
#endif

    if (trace->flags & (GLT_EOF | GLT_ERR))
        return GLT_OP_BAD;

    ent = ot_getent(trace->header.op);

    /* Read data */

    assert(!(ent->flags & OT_DALIGN) || (dalign(trace->pos) == 0));

    start = trace->pos;

    if (ent->len)
        trace_read(trace, data, ent->len);

    /* Read vectors */

    if (ent->flags & OT_HASV) {
        int i;
        /* Hack? Really free trace->getvecs this late? */
        while (trace->ngetvecs)
            glt_free(trace->getvecs[--trace->ngetvecs]);
        for (i = 0; i < OT_MAXOFS; i++) {
            int vlen;
            if (ent->offsets[i] == OT_LAST)
                break;
            vlen = vec_len(data, ent, i);
            if (vlen != 0) {
                void *vptr;
                assert(trace->ngetvecs < GLT_MAXV);
                trace->getvecs[trace->ngetvecs++] = vptr = glt_malloc(vlen);
                trace_read(trace, vptr, vlen);
                vec_ptr(data, ent, i) = vptr;
            }
            else {
                vec_ptr(data, ent, i) = NULL;
            }
        }
    }

    if (trace->pos - start != trace->header.llen) {
        trace->flags |= GLT_ERR;
        op = GLT_OP_BAD;
    }

    /* Clear header */

    trace->header.op = GLT_OP_BAD;

    return op;
}

static void
glt_skip_mem(GLT_trace *trace)
{
    if (!trace->header.op)
        trace_get_header(trace);

    trace->pos += trace->header.llen;

    if (trace->pos > trace->len)
        trace->pos = trace->len;

    trace->header.op = GLT_OP_BAD;
}

static int
glt_eof_mem(GLT_trace *trace)
{
    if (!trace->header.op)
        trace_get_header(trace);

    return (trace->flags & GLT_EOF);
}

static int
glt_err_mem(GLT_trace *trace)
{
    return (trace->flags & GLT_ERR);
}

static int
glt_tell_mem(GLT_trace *trace)
{
    return trace->pos;
}

static void
glt_seek_mem(GLT_trace *trace, int pos)
{
    trace->pos = pos;
    if (trace->pos > trace->len)
        glt_fatal("glt_seek: seek failed");
    trace->header.op = GLT_OP_BAD;
}
