/*
 *  malloc.c
 *
 *  GLT memory allocation routines
 *
 *  Kekoa Proudfoot
 *  3/19/98, 5/20/98
 */

#include <stdlib.h>
#include <memory.h>

#include "glt.h"

/* Two implementations, debug and standard */

#ifndef NDEBUG

/* Debug implementation */

/* Type definitions */

#define MEM_BLK_MAGIC (('M'<<24)|('B'<<16)|('L'<<8)|('K'<<0))

typedef struct mem_blk_s {
    unsigned int       magic;
    const char        *file;
    int                line;
    unsigned int       size;
    struct mem_blk_s  *prev;
    struct mem_blk_s  *next;
} mem_blk_t;

/* Static global variables */

static mem_blk_t *head = NULL;

void *
__glt_malloc(int size, const char *file, int line)
{
    mem_blk_t *blk;

    if ((blk = (mem_blk_t *)malloc(size + sizeof(mem_blk_t))) == NULL) {
        glt_fatal("glt_malloc: failed to allocate %d bytes, %s:%d", size,
                  file, line);
    }

    blk->magic = MEM_BLK_MAGIC;
    blk->file = file;
    blk->line = line;
    blk->size = size;
    blk->prev = NULL;
    blk->next = head;

    if (head)
        head = head->prev = blk;
    else
        head = blk;

    return (void *)(blk + 1);
}

void *
__glt_realloc(void *ptr, int size, const char *file, int line)
{
    mem_blk_t *blk = ((mem_blk_t *)ptr) - 1;
    void *temp;

    if ( !ptr )
        glt_fatal("glt_realloc: null realloc, %s:%d", file, line );

    if ( blk->magic != MEM_BLK_MAGIC )
        glt_fatal( "glt_realloc: invalid realloc, %s:%d", file, line );

    temp = __glt_malloc( size, file, line );
    memcpy( temp, ptr, blk->size );

    __glt_free( ptr, file, line );

    return temp;
}

void
__glt_free(void *ptr, const char *file, int line)
{
    mem_blk_t *blk = ((mem_blk_t *)ptr) - 1;

    if (!ptr)
        glt_fatal("glt_free: null free, %s:%d", file, line);

    if (blk->magic != MEM_BLK_MAGIC)
        glt_fatal("glt_free: invalid free, %s:%d", file, line);

    if (blk->prev) blk->prev->next = blk->next;
    if (blk->next) blk->next->prev = blk->prev;

    if (head == blk) head = blk->next;

    blk->magic = 0;

    free(blk);
}

void
__glt_printmem(void)
{
    mem_blk_t *blk;
    if (head) {
        glt_mesg(GLT_MALLOC, "glt: outstanding memory:");
        glt_mesg_va(GLT_MALLOC, ("glt:   %-10s  %-8s  %s", "addr",
                                 "bytes", "file"));
        for (blk = head; blk; blk = blk->next) {
            glt_mesg_va(GLT_MALLOC, ("glt:   0x%08x  %8u  %s:%d", blk + 1,
                                     blk->size, blk->file, blk->line));
        }
    }
}

#else /* !NDEBUG */

/* Standard implementation */

void *
__glt_malloc(int size)
{
    void *ptr;
    if ((ptr = malloc(size)) == NULL)
        glt_fatal("glt_malloc: failed to allocate %d bytes", size);
    return ptr;
}

void *
__glt_realloc(void *ptr, int size)
{
    ptr = realloc(ptr, size);
    if ( ptr == NULL )
        glt_fatal( "glt_realloc: failed to realloc %d bytes", size );
    return ptr;
}

#endif /* !NDEBUG */
