/*
 *  file.c
 *
 *  Internal glt functions for file traces
 *
 *  Kekoa Proudfoot
 *  4/30/98, 5/20/98
 */

/* Include files */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include "glt.h"
#include "optab.h"
#include "file.h"
#include "swap.h"

/* Macros */

#define offset(p,ofs)       (&(((int *)p)[ofs]))
#define vec_len(st,ent,vec) (*(int *)offset(st,ent->offsets[vec]))
#ifndef GLT64
#define vec_ptr(st,ent,vec) (*(void **)offset(st,ent->offsets[vec]+1))
#else
#define vec_ptr(st,ent,vec) (*((void **)((char *)st+ent->len+dalign(ent->len))+vec))
#endif
#define align(p,type)       (-(int)(p)&(sizeof(type)-1))
#define walign(p)           (align(p,int))
#define dalign(p)           (align(p,double))

/* Function prototypes */

static void glt_close_file(GLT_trace *trace);
static void *glt_allocv_file(GLT_trace *trace, int size);
static void glt_put_file(GLT_trace *trace, GLT_opcode op, GLT_data *data);
static void glt_put_file_swap(GLT_trace *trace, GLT_opcode op, GLT_data *data);
static GLT_opcode glt_peek_file(GLT_trace *trace);
static GLT_opcode glt_get_file(GLT_trace *trace, GLT_data *data);
static void glt_skip_file(GLT_trace *trace);
static void glt_skip_file_nseek(GLT_trace *trace);
static int glt_tell_file(GLT_trace *trace);
static void glt_seek_file(GLT_trace *trace, int pos);
static int glt_eof_file(GLT_trace *trace);
static int glt_err_file(GLT_trace *trace);

/* Open function */

void
__glt_open_file(GLT_trace *trace, char *filename)
{
    if (trace->flags & GLT_RSWAP)
        glt_warn("glt: %s: non-native byte order, swapping", filename);

    trace->close = glt_close_file;
    trace->allocv = glt_allocv_file;
    trace->put = (trace->flags & GLT_WSWAP) ?
        glt_put_file_swap : glt_put_file;
    trace->peek = glt_peek_file;
    trace->get = glt_get_file;
    trace->skip = (trace->flags & GLT_NSEEK) ?
        glt_skip_file_nseek : glt_skip_file;
    trace->tell = glt_tell_file;
    trace->seek = glt_seek_file;
    trace->eof = glt_eof_file;
    trace->err = glt_err_file;
}

/* Internal functions */

static void
glt_close_file(GLT_trace *trace)
{
    while (trace->nallocvecs)
        glt_free(trace->allocvecs[--trace->nallocvecs]);
    while (trace->ngetvecs)
        glt_free(trace->getvecs[--trace->ngetvecs]);
    if (trace->flags & GLT_NFILE)
        return;
    if (fclose(trace->file))
        glt_mesg(GLT_DEBUG, "glt_close: close failed");
}

static void *
glt_allocv_file(GLT_trace *trace, int size)
{
    void *p;
    assert(trace->nallocvecs < GLT_MAXV);
    trace->allocvecs[trace->nallocvecs++] = p = glt_malloc(size);
    assert(p);
    return p;
}

static void
glt_put_file(GLT_trace *trace, GLT_opcode op, GLT_data *data)
{
    GLT_dheader header;
    GLT_op_tab_ent *ent = ot_getent(op);
    int len;
    int i;

    assert(trace->flags & GLT_WROK);
    assert(walign(trace->pos) == 0);
    assert(op < GLT_NUM_OPS);
    assert(!(ent->flags & OT_BAD));

    glt_mesg_va(GLT_PUT, ("tracing %s", glt_label(op)));

    glt_mesg(GLT_PUTF, glt_text(op, data));

    /* Pad if necessary */

    if (ent->flags & OT_DALIGN) {
        if (dalign(trace->pos + ((ent->flags & OT_HASV) ? 8 : 4)) == 4) {
            header.op = GLT_OP_PAD;
            header.len = 0;
            fwrite(&header, 4, 1, trace->file);
            trace->pos += 4;
        }
    }

    /* Vectors are more complicated, process them separately */

    if (ent->flags & OT_HASV) {
        int vlen_save[OT_MAXOFS];
        void *vptr_save[OT_MAXOFS];
        int zero[2] = { 0, 0 };

        /* Compute length */

        len = ent->len;
        for (i = 0; i < OT_MAXOFS; i++) {
            int vlen;
            void *vptr;

            if (ent->offsets[i] == OT_LAST)
                break;

            vlen = vec_len(data, ent, i);
            vptr = vec_ptr(data, ent, i);

            glt_mesg_va(GLT_PUT, ("vector, len = %d", vlen));

            vlen_save[i] = vlen;
            vlen += (ent->flags & OT_DALIGN) ? dalign(vlen) : walign(vlen);
            len += vlen;
            vec_len(data, ent, i) = vlen;

            vptr_save[i] = vptr;
            vec_ptr(data, ent, i) = NULL;
        }

        /* Write long header */

        header.op = op;
        header.len = 0xffff;
        header.llen = len;

        glt_mesg_va(GLT_PUT, ("writing header op=%d llen=%d", op, len));

        fwrite(&header, 8, 1, trace->file);

        /* Update trace position */

        trace->pos += 8;

        /* Verify alignment */

        assert(!(ent->flags & OT_DALIGN) || (dalign(trace->pos) == 0));
#if 0
        assert(ftell(trace->file) == trace->pos);
#endif

        /* Write data */

        fwrite(data, ent->len, 1, trace->file);

        /* Update trace position */

        trace->pos += ent->len;

        /* Write vectors, updating trace position */

        for (i = 0; i < OT_MAXOFS; i++) {
            int vlen;
            if (ent->offsets[i] == OT_LAST)
                break;
            vlen = vec_len(data, ent, i);
            if (vlen != 0) {
                int pad = vlen - vlen_save[i];
                fwrite(vptr_save[i], vlen_save[i], 1, trace->file);
                if (pad)
                    fwrite(zero, pad, 1, trace->file);
                trace->pos += vlen;
            }
            vec_len(data, ent, i) = vlen_save[i];
            vec_ptr(data, ent, i) = vptr_save[i];
        }

        /* Free allocvecs */

        while (trace->nallocvecs)
            glt_free(trace->allocvecs[--trace->nallocvecs]);
    }
    else {

        /* Compute length */

        len = ent->len;

        /* Write short header */

        assert(len < 0xffff);
        header.op = op;
        header.len = len;

        glt_mesg_va(GLT_PUT, ("writing header op=%d len=%d", op, len));

        fwrite(&header, 4, 1, trace->file);

        /* Update trace position */

        trace->pos += 4;

        /* Verify alignment */

        assert(!(ent->flags & OT_DALIGN) || (dalign(trace->pos) == 0));
#if 0
        assert(ftell(trace->file) == trace->pos);
#endif

        /* Write data */

        fwrite(data, ent->len, 1, trace->file);

        /* Update trace position */

        trace->pos += ent->len;
    }
}

static void
glt_put_file_swap(GLT_trace *trace, GLT_opcode op, GLT_data *data)
{
    GLT_dheader header;
    GLT_op_tab_ent *ent = ot_getent(op);
    int len;
    int i;

    assert(trace->flags & GLT_WROK);
    assert(walign(trace->pos) == 0);
    assert(op < GLT_NUM_OPS);
    assert(!(ent->flags & OT_BAD));

    glt_mesg_va(GLT_PUT, ("tracing %s", glt_label(op)));

    glt_mesg(GLT_PUTF, glt_text(op, data));

    /* Pad if necessary */

    if (ent->flags & OT_DALIGN) {
        if (dalign(trace->pos + ((ent->flags & OT_HASV) ? 8 : 4)) == 4) {
            header.op = GLT_OP_PAD;
            header.len = 0;
            __glt_swap_header(&header);
            fwrite(&header, 4, 1, trace->file);
            trace->pos += 4;
        }
    }

    /* Vectors are more complicated, process them separately */

    if (ent->flags & OT_HASV) {
        int vlen_save[OT_MAXOFS];
        void *vptr_save[OT_MAXOFS];
        int zero[2] = { 0, 0 };

        /* Compute length */

        len = ent->len;
        for (i = 0; i < OT_MAXOFS; i++) {
            int vlen;
            void *vptr;

            if (ent->offsets[i] == OT_LAST)
                break;

            vlen = vec_len(data, ent, i);
            vptr = vec_ptr(data, ent, i);

            glt_mesg_va(GLT_PUT, ("vector, len = %d", vlen));

            vlen_save[i] = vlen;
            vlen += (ent->flags & OT_DALIGN) ? dalign(vlen) : walign(vlen);
            len += vlen;
            vec_len(data, ent, i) = vlen;

            vptr_save[i] = vptr;
            vec_ptr(data, ent, i) = NULL;
        }

        /* Write long header */

        header.op = op;
        header.len = 0xffff;
        header.llen = len;

        glt_mesg_va(GLT_PUT, ("writing header op=%d llen=%d", op, len));

        __glt_swap_header(&header);
        fwrite(&header, 8, 1, trace->file);

        /* Update trace position */

        trace->pos += 8;

        /* Verify alignment */

        assert(!(ent->flags & OT_DALIGN) || (dalign(trace->pos) == 0));
#if 0
        assert(ftell(trace->file) == trace->pos);
#endif
        /* Swap data, write it, then unswap it */

        __glt_swap_data(op, data);
        fwrite(data, ent->len, 1, trace->file);
        __glt_unswap_data(op, data);

        /* Update trace position */

        trace->pos += ent->len;

        /* Restore vector pointers */

        for (i = 0; i < OT_MAXOFS; i++) {
            if (ent->offsets[i] == OT_LAST)
                break;
            vec_ptr(data, ent, i) = vptr_save[i];
        }


        /* Swap and write vectors, updating trace position, then unswap */

        __glt_swap_vectors(op, data);
        for (i = 0; i < OT_MAXOFS; i++) {
            int vlen;
            if (ent->offsets[i] == OT_LAST)
                break;
            vlen = vec_len(data, ent, i);
            if (vlen != 0) {
                int pad = vlen - vlen_save[i];
                fwrite(vptr_save[i], vlen_save[i], 1, trace->file);
                if (pad)
                    fwrite(zero, pad, 1, trace->file);
                trace->pos += vlen;
            }
            vec_len(data, ent, i) = vlen_save[i];
        }
        __glt_unswap_vectors(op, data);

        /* Free allocvecs */

        while (trace->nallocvecs)
            glt_free(trace->allocvecs[--trace->nallocvecs]);
    }
    else {

        /* Compute length */

        len = ent->len;

        /* Write short header */

        assert(len < 0xffff);
        header.op = op;
        header.len = len;

        glt_mesg_va(GLT_PUT, ("writing header op=%d len=%d", op, len));

        __glt_swap_header(&header);
        fwrite(&header, 4, 1, trace->file);

        /* Update trace position */

        trace->pos += 4;

        /* Verify alignment */

        assert(!(ent->flags & OT_DALIGN) || (dalign(trace->pos) == 0));
#if 0
        assert(ftell(trace->file) == trace->pos);
#endif
        /* Swap data, write, then unswap */

        __glt_swap_data(op, data);
        fwrite(data, ent->len, 1, trace->file);
        __glt_unswap_data(op, data);

        /* Update trace position */

        trace->pos += ent->len;
    }
}

static __inline int
trace_read(GLT_trace *trace, void *data, int len)
{
    int got = fread(data, 1, len, trace->file);
    trace->pos += got;
    if (got == len)
        return 1;
    trace->flags |= GLT_EOF;
    if (got)
        trace->flags |= GLT_ERR;
    return 0;
}

static __inline void
trace_get_header(GLT_trace *trace)
{
    assert(!trace->header.op);
    assert(walign(trace->pos) == 0);

    if (!trace_read(trace, &trace->header, 4)) {
        trace->header.op = GLT_OP_BAD;
        trace->header.len = 0;
    }
    else if (trace->header.len == 0xffff) {
        if (!trace_read(trace, &trace->header.llen, 4)) {
            trace->flags |= GLT_ERR;
            trace->header.op = GLT_OP_BAD;
            trace->header.llen = 0;
        }
        else {
            if (trace->flags & GLT_RSWAP)
                __glt_swap_header(&trace->header);
        }
    }
    else {
        if (trace->flags & GLT_RSWAP)
            __glt_swap_header(&trace->header);
        trace->header.llen = trace->header.len;
    }
    if (trace->header.op >= GLT_NUM_OPS) {
        glt_mesg_va(GLT_DEBUG, ("Bad opcode %d near file offset 0x%08x",
                                trace->header.op, trace->pos));
        trace->flags |= GLT_ERR; /* really do this? */
        trace->header.op = GLT_OP_BAD; /* really do this? */
        trace->header.llen = 0;
        /* unrecognized opcode, it should be skipped */
    }
#ifndef NDEBUG
    else if (ot_getent(trace->header.op)->flags & OT_BAD) {
        if (trace->header.op != GLT_OP_BAD) {
            glt_mesg_va(GLT_DEBUG, ("Bad opcode %d near file offset 0x%08x",
                                    trace->header.op, trace->pos));
            trace->header.op = GLT_OP_BAD; /* really do this? */
            trace->header.llen = 0;
            /* unsupported opcode, it should be skipped? */
        }
    }
#endif
}

static GLT_opcode
glt_peek_file(GLT_trace *trace)
{
    if (!trace->header.op)
        trace_get_header(trace);

    return trace->header.op;
}

/* Really have all these ifs in get? Ok because of branch prediction? */
/* Change semantics with respect to unrecognized opcodes? */

static GLT_opcode
glt_get_file(GLT_trace *trace, GLT_data *data)
{
    GLT_op_tab_ent *ent;
    GLT_opcode op;
    int start;

    if (!trace->header.op)
        trace_get_header(trace);

    op = trace->header.op;

#if 0
    if (trace->header.op > GLT_NUM_OPS) {
        trace->flags |= GLT_ERR; /* really do this? */
        trace->header.op = GLT_OP_BAD; /* really do this? */
        trace->header.llen;
        /* unrecognized, so skip and return real opcode? */
    }
#endif

    if (trace->flags & (GLT_EOF | GLT_ERR))
        return GLT_OP_BAD;

    ent = ot_getent(trace->header.op);

    /* Read data */

    assert(!(ent->flags & OT_DALIGN) || (dalign(trace->pos) == 0));

    start = trace->pos;

    if (ent->len) {
        trace_read(trace, data, ent->len);
        if (trace->flags & GLT_RSWAP)
            __glt_swap_data(op, data);
    }

    /* Read vectors */

    if (ent->flags & OT_HASV) {
        int i;
        /* Hack? Really free trace->getvecs this late? */
        while (trace->ngetvecs)
            glt_free(trace->getvecs[--trace->ngetvecs]);
        for (i = 0; i < OT_MAXOFS; i++) {
            int vlen;
            if (ent->offsets[i] == OT_LAST)
                break;
            vlen = vec_len(data, ent, i);
            if (vlen != 0) {
                void *vptr;
                assert(trace->ngetvecs < GLT_MAXV);
                trace->getvecs[trace->ngetvecs++] = vptr = glt_malloc(vlen);
                trace_read(trace, vptr, vlen);
                vec_ptr(data, ent, i) = vptr;
            }
            else {
                vec_ptr(data, ent, i) = NULL;
            }
        }
        if (trace->flags & GLT_RSWAP)
            __glt_swap_vectors(op, data);
    }

    /* Hack? This doesn't work if pad contains anything but header */
    if (trace->pos - start != trace->header.llen) {
        trace->flags |= GLT_ERR;
        op = GLT_OP_BAD;
    }

    /* Clear header */

    trace->header.op = GLT_OP_BAD;

    return op;
}

static void
glt_skip_file(GLT_trace *trace)
{
    if (!trace->header.op)
        trace_get_header(trace);

    fseek(trace->file, (long) trace->header.llen, SEEK_CUR);

    trace->pos += trace->header.llen;

    trace->header.op = GLT_OP_BAD;
}

static void
glt_skip_file_nseek(GLT_trace *trace)
{
    char buf[1024];
    int len;

    if (!trace->header.op)
        trace_get_header(trace);

    for (len = trace->header.llen; len > 0; len -= sizeof(buf)) {
        int read = len;
        if (read > sizeof(buf))
            read = sizeof(buf);
        if (!trace_read(trace, buf, read))
            break;
    }

    trace->header.op = GLT_OP_BAD;
}

static int
glt_eof_file(GLT_trace *trace)
{
    if (!trace->header.op)
        trace_get_header(trace);

    return (trace->flags & GLT_EOF);
}

static int
glt_err_file(GLT_trace *trace)
{
    return (trace->flags & GLT_ERR);
}

static int
glt_tell_file(GLT_trace *trace)
{
#ifndef NDEBUG
    if (!(trace->flags & (GLT_NSEEK | GLT_NFILE))) {
        int pos = ftell(trace->file);
        assert(trace->pos == pos);
    }
#endif
    return trace->pos;
}

static void
glt_seek_file(GLT_trace *trace, int pos)
{
    trace->pos = pos;
    /* look at trace->flags & GLT_NSEEK ? */
    if (fseek(trace->file, pos, SEEK_SET) != 0)
        glt_fatal("glt_seek: seek failed");
    trace->header.op = GLT_OP_BAD;
}
