/*
 *  dlist.c
 *
 *  Display list management routines for glt library
 *
 *  Matthew Eldridge
 *  3/29/2000
 */

#ifdef WIN32
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif
#include <assert.h>
#include <memory.h>
#include <GL/gl.h>

#include "glt.h"
#include "optab.h"
#include "dlist.h"
#include "swap.h"

/* types */

#define DLIST_GROW            2048
#define DLIST_MAX_STACK_DEPTH 32

typedef struct GLT_dlist {
    GLTuint           name;
    int               pos;
    int               len;
    void             *data;
    struct GLT_dlist *next;
} GLT_dlist;

typedef struct GLT_dlist_state {

    struct {
        GLT_opcode op;
        GLT_data   data;
    } save;

    int depth;

    struct {
        int   pos;
        int   len;
        void *data;
    } stack[ DLIST_MAX_STACK_DEPTH ];

    GLT_dlist *lists;

} GLT_dlist_state;

/* Macros */

#ifndef NDEBUG
#define ONCE(cmd) \
    do { \
        static int once = 0; \
        if (!once && ++once) cmd ; \
    } while (0)
#else
#define ONCE(cmd)
#endif

#define offset(p,ofs)       (&(((int *)p)[ofs]))
#define vec_len(st,ent,vec) (*(int *)offset(st,ent->offsets[vec]))
#ifndef GLT64
#define vec_ptr(st,ent,vec) (*(void **)offset(st,ent->offsets[vec]+1))
#else
#define vec_ptr(st,ent,vec) (*((void **)((char *)st+ent->len+dalign(ent->len))+vec))
#endif
#define align(p,type)       (-(int)(p)&(sizeof(type)-1))
#define walign(p)           (align(p,int))
#define dalign(p)           (align(p,double))

/* Function prototypes */

static void       list_close( GLT_trace *trace );
static void      *list_allocv( GLT_trace *trace, int size );
static GLT_opcode list_peek( GLT_trace *trace );
static GLT_opcode list_get( GLT_trace *trace, GLT_data *data );
static void       list_put( GLT_trace *trace, GLT_opcode op, GLT_data *data );
static void       list_skip( GLT_trace *trace );
static int        list_eof( GLT_trace *trace );
static int        list_err( GLT_trace *trace );
static int        list_tell( GLT_trace *trace );
static void       list_seek( GLT_trace *trace, int pos );

static void       file_get_header( GLT_trace *trace );
static GLT_opcode file_get( GLT_trace *trace, GLT_data *data );

static void       mem_get_header( GLT_trace *trace );
static void       mem_put( GLT_dlist *dlist, GLT_opcode op, GLT_data *data );
static GLT_opcode mem_get( GLT_trace *trace, GLT_data *data );

/* interface */

void
__glt_open_dlist( GLT_trace *trace, char *filename )
{
    if ( trace->flags & GLT_RSWAP )
        glt_warn( "glt: %s: non-native byte order, swapping", filename );

    trace->close  = list_close;
    trace->allocv = list_allocv;
    trace->put    = list_put;
    trace->peek   = list_peek;
    trace->get    = list_get;
    trace->skip   = list_skip;
    trace->tell   = list_tell;
    trace->seek   = list_seek;
    trace->eof    = list_eof;
    trace->err    = list_err;

    trace->dlist  = glt_malloc( sizeof(GLT_dlist_state) );

    trace->dlist->depth = 0;
    trace->dlist->lists = NULL;
}

static GLT_dlist *
create_list( GLT_trace *trace, GLTuint name )
{
    GLT_dlist *dlist;

    dlist = (GLT_dlist *) glt_malloc( sizeof(GLT_dlist) );

    dlist->name = name;
    dlist->pos  = 0;
    dlist->len  = DLIST_GROW;
    dlist->data = glt_malloc( dlist->len );

    dlist->next = trace->dlist->lists;
    trace->dlist->lists = dlist;

    return dlist;
}

static GLT_dlist *
lookup_list( GLT_trace *trace, GLTuint name )
{
    GLT_dlist *dlist;

    dlist = trace->dlist->lists;
    while ( dlist && dlist->name != name )
        dlist = dlist->next;

    return dlist;
}

static void
delete_list( GLT_trace *trace, GLTuint name )
{
    GLT_dlist **prev, *dlist;

    prev  = &trace->dlist->lists;
    dlist = *prev;
    while ( dlist && dlist->name != name ) {
        prev = &dlist->next;
        dlist = *prev;
    }

    if ( dlist ) {
        *prev = dlist->next;
        glt_free( dlist->data );
        glt_free( dlist );
    }
}

static void
delete_lists( GLT_trace *trace, GLTuint list, GLTsizei range )
{
    GLTsizei i;

    for ( i = 0; i < range; i++ )
        delete_list( trace, list + i );
}

static void
new_list( GLT_trace *trace, GLTuint name )
{
    GLT_dlist *dlist;
    GLT_opcode op;
    GLT_data   data;

    delete_list( trace, name );

    dlist = create_list( trace, name );

    do {
        op = file_get( trace, &data );
        if ( !op )
            glt_fatal( "glt: hit an error before glEndList" );
        mem_put( dlist, op, &data );
    } while ( op != GLT_OP_END_LIST );
}

static void
call_list( GLT_trace *trace, GLTuint name )
{
    GLT_dlist *dlist;

    dlist = lookup_list( trace, name );
    if ( !dlist ) {
        glt_warn( "glt: glCallList( %u ), no such list", name );
        return;
    }

    if ( trace->dlist->depth >= DLIST_MAX_STACK_DEPTH )
        glt_fatal( "glt: display list stack depth exceeded" );

    trace->dlist->stack[ trace->dlist->depth ].pos  = trace->pos;
    trace->dlist->stack[ trace->dlist->depth ].len  = trace->len;
    trace->dlist->stack[ trace->dlist->depth ].data = trace->data;
    trace->dlist->depth++;

    trace->pos  = 0;
    trace->len  = dlist->pos;
    trace->data = dlist->data;
}

void
end_list( GLT_trace *trace )
{
    assert( trace->dlist->depth > 0 );

    trace->dlist->depth--;
    trace->pos  = trace->dlist->stack[ trace->dlist->depth ].pos;
    trace->len  = trace->dlist->stack[ trace->dlist->depth ].len;
    trace->data = trace->dlist->stack[ trace->dlist->depth ].data;
}

static __inline int
display_list_op( GLT_opcode op )
{
    switch ( op ) {
      case GLT_OP_NEW_LIST:
      case GLT_OP_CALL_LIST:
      case GLT_OP_CALL_LISTS:
      case GLT_OP_DELETE_LISTS:
      case GLT_OP_END_LIST:
      case GLT_OP_LIST_BASE:
        return 1;
    }

    return 0;
}

static void
handle_list_op( GLT_trace *trace, GLT_opcode op, GLT_data *data )
{
    switch ( op ) {

      case GLT_OP_NEW_LIST:
        if ( trace->dlist->depth )
            glt_fatal( "glt: glNewList() inside a display list" );
        new_list( trace, data->new_list.list );
        if ( data->new_list.mode == GL_COMPILE_AND_EXECUTE ||
             ( trace->flags & GLT_RETLIST ) )
            call_list( trace, data->new_list.list );
        break;

      case GLT_OP_CALL_LIST:
        call_list( trace, data->call_list.list );
        break;

      case GLT_OP_DELETE_LISTS:
        if ( trace->dlist->depth )
            glt_fatal( "glt: glDeleteLists() inside a display list" );
        delete_lists( trace,
                      data->delete_lists.list,
                      data->delete_lists.range );
        break;

      case GLT_OP_END_LIST:
        if ( trace->dlist->depth )
            end_list( trace );
        else
            glt_fatal( "glt: glEndList() not inside a list" );
        break;

      case GLT_OP_CALL_LISTS:
        ONCE( glt_warn( "glt: ignoring glCallLists" ) );
        break;

      case GLT_OP_LIST_BASE:
        ONCE( glt_warn( "glt: ignoring glListBase" ) );
        break;

      default:
        glt_fatal( "glt: handle_list_op( op=%s )", glt_label( op ) );
    }

    trace->header.op = GLT_OP_BAD;
}

static void
list_close( GLT_trace *trace )
{
    GLT_dlist *dlist;

    while ( trace->nallocvecs )
        glt_free( trace->allocvecs[--trace->nallocvecs] );

    while ( trace->ngetvecs )
        glt_free( trace->getvecs[--trace->ngetvecs] );

    if ( fclose( trace->file ) )
        glt_mesg( GLT_DEBUG, "glt_close: close failed" );

    dlist = trace->dlist->lists;
    while ( dlist ) {
        GLT_dlist *temp = dlist;
        dlist = dlist->next;
        glt_free( temp->data );
        glt_free( temp );
    }

    glt_free( trace->dlist );
}

static __inline void
list_get_header( GLT_trace *trace )
{
    if ( trace->dlist->depth )
        mem_get_header( trace );
    else
        file_get_header( trace );
}

static void *
list_allocv( GLT_trace *trace, int size )
{
    void *p;
    assert( trace->nallocvecs < GLT_MAXV );
    trace->allocvecs[trace->nallocvecs++] = p = glt_malloc( size );
    assert( p );
    return p;
}

static GLT_opcode
list_peek( GLT_trace *trace )
{
    if ( trace->flags & GLT_RETLIST )
        glt_fatal( "glt: peek() when unwinding" );

    if ( !trace->header.op )
        list_get_header( trace );

    while ( display_list_op( trace->header.op ) ) {

        GLT_opcode op;
        GLT_data   data;

        if ( trace->dlist->depth )
            op = mem_get( trace, &data );
        else
            op = file_get( trace, &data );

        handle_list_op( trace, op, &data );
        if ( !trace->header.op )
            list_get_header( trace );
    }

    return trace->header.op;
}

static GLT_opcode
list_get( GLT_trace *trace, GLT_data *data )
{
    GLT_opcode op;

    if ( trace->dlist->depth )
        op = mem_get( trace, data );
    else
        op = file_get( trace, data );

    while ( display_list_op( op ) ) {

        handle_list_op( trace, op, data );

        if ( trace->flags & GLT_RETLIST )
            return op;

        if ( trace->dlist->depth )
            op = mem_get( trace, data );
        else
            op = file_get( trace, data );
    }

    return op;
}

static void
list_put( GLT_trace *trace, GLT_opcode op, GLT_data *data )
{
    (void) trace;
    (void) op;
    (void) data;
    glt_fatal( "glt: put() when unwinding" );
}

static void
list_skip( GLT_trace *trace )
{
    (void) trace;
    glt_fatal( "glt: skip() when unwinding" );
}

static int
list_eof( GLT_trace *trace )
{
    if ( !trace->header.op )
        list_get_header( trace );

    return ( trace->flags & GLT_EOF );
}

static int
list_err( GLT_trace *trace )
{
    return ( trace->flags & GLT_ERR );
}

static int
list_tell( GLT_trace *trace )
{
    (void) trace;
    glt_fatal( "glt: tell() when unwinding" );
    return 0;
}

static void
list_seek( GLT_trace *trace, int pos )
{
    (void) trace;
    (void) pos;
    glt_fatal( "glt: seek() when unwinding" );
}

/* file functions */

static __inline int
file_read( GLT_trace *trace, void *data, int len )
{
    int got = fread( data, 1, len, trace->file );
    trace->pos += got;
    if ( got == len )
        return 1;
    trace->flags |= GLT_EOF;
    if ( got )
        trace->flags |= GLT_ERR;
    return 0;
}

static void
file_get_header( GLT_trace *trace )
{
    assert( !trace->header.op );
    assert( walign( trace->pos ) == 0 );

    if ( !file_read( trace, &trace->header, 4 ) ) {
        trace->header.op = GLT_OP_BAD;
        trace->header.len = 0;
    }
    else if ( trace->header.len == 0xffff ) {
        if ( !file_read( trace, &trace->header.llen, 4 ) ) {
            trace->flags |= GLT_ERR;
            trace->header.op = GLT_OP_BAD;
            trace->header.llen = 0;
        }
        else {
            if ( trace->flags & GLT_RSWAP )
                __glt_swap_header( &trace->header );
        }
    }
    else {
        if ( trace->flags & GLT_RSWAP )
            __glt_swap_header( &trace->header );
        trace->header.llen = trace->header.len;
    }
    if ( trace->header.op >= GLT_NUM_OPS ) {
        glt_mesg_va( GLT_DEBUG, ( "Bad opcode %d near file offset 0x%08x",
                                  trace->header.op, trace->pos ) );
        trace->flags |= GLT_ERR; /* really do this? */
        trace->header.op = GLT_OP_BAD; /* really do this? */
        trace->header.llen = 0;
        /* unrecognized opcode, it should be skipped */
    }
#ifndef NDEBUG
    else if ( ot_getent( trace->header.op )->flags & OT_BAD ) {
        if ( trace->header.op != GLT_OP_BAD ) {
            glt_mesg_va( GLT_DEBUG, ( "Bad opcode %d near file offset 0x%08x",
                                    trace->header.op, trace->pos ) );
            trace->header.op = GLT_OP_BAD; /* really do this? */
            trace->header.llen = 0;
            /* unsupported opcode, it should be skipped? */
        }
    }
#endif
}

/* Really have all these ifs in get? Ok because of branch prediction? */
/* Change semantics with respect to unrecognized opcodes? */

static GLT_opcode
file_get( GLT_trace *trace, GLT_data *data )
{
    GLT_op_tab_ent *ent;
    GLT_opcode op;
    int start;

    if ( !trace->header.op )
        file_get_header( trace );

    op = trace->header.op;

#if 0
    if (trace->header.op > GLT_NUM_OPS) {
        trace->flags |= GLT_ERR; /* really do this? */
        trace->header.op = GLT_OP_BAD; /* really do this? */
        trace->header.llen;
        /* unrecognized, so skip and return real opcode? */
    }
#endif

    if ( trace->flags & ( GLT_EOF | GLT_ERR ) )
        return GLT_OP_BAD;

    ent = ot_getent( trace->header.op );

    /* Read data */

    assert(!(ent->flags & OT_DALIGN) || (dalign(trace->pos) == 0));

    start = trace->pos;

    if ( ent->len ) {
        file_read( trace, data, ent->len );
        if ( trace->flags & GLT_RSWAP )
            __glt_swap_data( op, data );
    }

    /* Read vectors */

    if ( ent->flags & OT_HASV ) {
        int i;
        /* Hack? Really free trace->getvecs this late? */
        while ( trace->ngetvecs )
            glt_free( trace->getvecs[--trace->ngetvecs] );
        for ( i = 0; i < OT_MAXOFS; i++ ) {
            int vlen;
            if ( ent->offsets[i] == OT_LAST )
                break;
            vlen = vec_len( data, ent, i );
            if ( vlen != 0 ) {
                void *vptr;
                assert( trace->ngetvecs < GLT_MAXV );
                trace->getvecs[trace->ngetvecs++] = vptr = glt_malloc( vlen );
                file_read( trace, vptr, vlen );
                vec_ptr( data, ent, i ) = vptr;
            }
            else {
                vec_ptr( data, ent, i ) = NULL;
            }
        }
        if ( trace->flags & GLT_RSWAP )
            __glt_swap_vectors( op, data );
    }

    /* Hack? This doesn't work if pad contains anything but header */
    if ( trace->pos - start != trace->header.llen ) {
        trace->flags |= GLT_ERR;
        op = GLT_OP_BAD;
    }

    /* Clear header */

    trace->header.op = GLT_OP_BAD;

    return op;
}

/* memory functions */

static __inline void
mem_write( GLT_dlist *dlist, void *data, int len )
{
    if ( dlist->pos + len > dlist->len ) {
        dlist->len  += len + DLIST_GROW;
        dlist->data = glt_realloc( dlist->data, dlist->len );
    }

    memcpy( (char *) dlist->data + dlist->pos, data, len );
    dlist->pos += len;
}

static void
mem_put( GLT_dlist *dlist, GLT_opcode op, GLT_data *data )
{
    GLT_dheader header;
    GLT_op_tab_ent *ent = ot_getent( op );
    int len;
    int i;

    assert( walign( dlist->pos ) == 0 );
    assert( op < GLT_NUM_OPS );
    assert( !( ent->flags & OT_BAD ) );

    glt_mesg_va( GLT_PUT, ( "tracing %s", glt_label( op ) ) );

    glt_mesg( GLT_PUTF, glt_text( op, data ) );

    /* Pad if necessary */

    if ( ent->flags & OT_DALIGN ) {
        if ( dalign(dlist->pos + ( ( ent->flags & OT_HASV ) ? 8 : 4 )) == 4 ) {
            header.op = GLT_OP_PAD;
            header.len = 0;
            mem_write( dlist, &header, 4 );
        }
    }

    /* Compute length */

    if ( ent->flags & OT_HASV ) {
        int vlen_save[OT_MAXOFS];
        void *vptr_save[OT_MAXOFS];
        int zero[2] = { 0, 0 };

        len = ent->len;
        for ( i = 0; i < OT_MAXOFS; i++ ) {
            int vlen;
            void *vptr;

            if ( ent->offsets[i] == OT_LAST )
                break;

            vlen = vec_len( data, ent, i );
            vptr = vec_ptr( data, ent, i );

            glt_mesg_va( GLT_PUT, ( "vector, len = %d", vlen ) );

            vlen_save[i] = vlen;
            vlen += ( ent->flags & OT_DALIGN ) ? dalign(vlen) : walign(vlen);
            len += vlen;
            vec_len( data, ent, i ) = vlen;

            vptr_save[i] = vptr;
            vec_ptr( data, ent, i ) = NULL;
        }

        /* Write long header */

        header.op = op;
        header.len = 0xffff;
        header.llen = len;

        glt_mesg_va( GLT_PUT, ( "writing header op=%d llen=%d", op, len ) );

        mem_write( dlist, &header, 8 );

        assert( !( ent->flags & OT_DALIGN ) || ( dalign( dlist->pos ) == 0 ) );

        /* Write data */

        mem_write( dlist, data, ent->len );

        /* Write vectors */

        for ( i = 0; i < OT_MAXOFS; i++ ) {
            int vlen;
            if ( ent->offsets[i] == OT_LAST )
                break;
            vlen = vec_len( data, ent, i );
            if ( vlen != 0 ) {
                int pad = vlen - vlen_save[i];
                mem_write( dlist, vptr_save[i], vlen_save[i] );
                if ( pad )
                    mem_write( dlist, zero, pad );
            }
            vec_len( data, ent, i ) = vlen_save[i];
            vec_ptr( data, ent, i ) = vptr_save[i];
        }

    }
    else {

        len = ent->len;

        /* Write short header */

        assert(len < 0xffff);
        header.op = op;
        header.len = len;

        glt_mesg_va( GLT_PUT, ( "writing header op=%d len=%d", op, len ) );

        mem_write( dlist, &header, 4 );

        assert( !( ent->flags & OT_DALIGN ) || ( dalign( dlist->pos ) == 0 ) );

        /* Write data */

        mem_write( dlist, data, ent->len );
    }
}

static __inline int
mem_read( GLT_trace *trace, void *data, int len )
{
    int got;
    if ( trace->pos + len > trace->len )
        got = trace->len - trace->pos;
    else
        got = len;
    memcpy( data, (char *) trace->data + trace->pos, got );
    trace->pos += got;
    if ( got == len )
        return 1;
    trace->flags |= GLT_EOF;
    if ( got )
        trace->flags |= GLT_ERR;
    return 0;
}

static void
mem_get_header( GLT_trace *trace )
{
    assert( !trace->header.op );
    assert( walign( trace->pos ) == 0 );

    if ( !mem_read( trace, &trace->header, 4 ) ) {
        trace->header.op = GLT_OP_BAD;
        trace->header.len = 0;
    }
    if ( trace->header.len == 0xffff ) {
        if ( !mem_read( trace, &trace->header.llen, 4 ) ) {
            trace->flags |= GLT_ERR;
            trace->header.op = GLT_OP_BAD;
            trace->header.llen = 0;
        }
    }
    else {
        trace->header.llen = trace->header.len;
    }
    if ( trace->header.op >= GLT_NUM_OPS ) {
        glt_mesg_va( GLT_DEBUG, ( "Bad opcode %d near file offset 0x%08x",
                                  trace->header.op, trace->pos ) );
        trace->flags |= GLT_ERR;
        trace->header.op = GLT_OP_BAD;
        trace->header.llen = 0;
    }
#ifndef NDEBUG
    else if ( ot_getent(trace->header.op)->flags & OT_BAD ) {
        if ( trace->header.op != GLT_OP_BAD ) {
            glt_mesg_va( GLT_DEBUG, ( "Bad opcode %d near file offset 0x%08x",
                                      trace->header.op, trace->pos ) );
            trace->header.op = GLT_OP_BAD;
            trace->header.llen = 0;
        }
    }
#endif
}

/* Really have all these ifs in get? */
static GLT_opcode
mem_get( GLT_trace *trace, GLT_data *data )
{
    GLT_op_tab_ent *ent;
    GLT_opcode op;
    int start;

    if ( !trace->header.op )
        mem_get_header( trace );

    op = trace->header.op;

#if 0
    if ( trace->header.op > GLT_NUM_OPS ) {
        trace->flags |= GLT_ERR; /* really do this? */
        trace->header.op = GLT_OP_BAD; /* really do this? */
        trace->header.llen;
        /* unrecognized, so skip and return real opcode? */
    }
#endif

    if ( trace->flags & ( GLT_EOF | GLT_ERR ) )
        return GLT_OP_BAD;

    ent = ot_getent( trace->header.op );

    /* Read data */

    assert(!(ent->flags & OT_DALIGN) || (dalign(trace->pos) == 0));

    start = trace->pos;

    if ( ent->len )
        mem_read( trace, data, ent->len );

    /* Read vectors */

    if ( ent->flags & OT_HASV ) {
        int i;
        /* Hack? Really free trace->getvecs this late? */
        while ( trace->ngetvecs )
            glt_free( trace->getvecs[--trace->ngetvecs] );
        for ( i = 0; i < OT_MAXOFS; i++ ) {
            int vlen;
            if ( ent->offsets[i] == OT_LAST )
                break;
            vlen = vec_len( data, ent, i );
            if ( vlen != 0 ) {
                void *vptr;
                assert( trace->ngetvecs < GLT_MAXV );
                trace->getvecs[trace->ngetvecs++] = vptr = glt_malloc( vlen );
                mem_read( trace, vptr, vlen );
                vec_ptr( data, ent, i ) = vptr;
            }
            else {
                vec_ptr( data, ent, i ) = NULL;
            }
        }
    }

    if ( trace->pos - start != trace->header.llen ) {
        trace->flags |= GLT_ERR;
        op = GLT_OP_BAD;
    }

    /* Clear header */

    trace->header.op = GLT_OP_BAD;

    return op;
}
