/****************************************************************************

Routines to write the SHRMs to a file.  This includes a number of 
additional debugging routines not relevant to the main program that are not
well documented here.

Ravi Ramamoorthi: May 22, 2002

***************************************************************************/

#include <stdio.h>
#include <math.h>
#include <complex>
#include <assert.h>
#include "legendre.h"
#include "globals.h"
#include "vectors.h"
#include "transforms.h"
#include "write.h"

//--------------------------------------------------------------------------
// Main routine to print the SHRM.
// Format is two bytes [high and low] for compactness and precision
//--------------------------------------------------------------------------

void generate(double SHRM[maxtheta][maxphi][maxp+1][2*maxp+1][3], int ourmaxp, const char *filename, const char *cmdline,int size) {

  int i,j,k,l,m,p,q ;
  FILE *fp2 ;
  assert(fp2 = fopen(filename,"wb")) ;
  double maxvals[maxp+1][2*maxp+1][3] ;

  for (i = 0 ; i < size ; i++)
    for (j = 0 ; j < size ; j++)
      for (k = 0 ; k < 3 ; k++)
	for (p = 0 ; p <= ourmaxp ; p++)
	  for (q = -p ; q <= p ; q++) {
	    double v = SHRM[i][j][p][q+maxp][k] ;
	    if ((i == 0 && j == 0) || (fabs(v) > maxvals[p][q+maxp][k]))
	      maxvals[p][q+maxp][k] = fabs(v) ;
	  }
  
  fprintf(fp2,"#%s\n",cmdline) ;
  for (k = 0 ; k < 3 ; k++) {
    for (p = 0 ; p <= ourmaxp ; p++)
      for (q = -p ; q <= p ; q++)
	fprintf(fp2,"%lf ",mulfac*maxvals[p][q+maxp][k]) ;
    fprintf(fp2,"\n") ;
  }
 
  for (i = 0 ; i < size ; i++)
    for (j = 0 ; j < size ; j++)
      for (k = 0 ; k < 3 ; k++)
	for (p = 0 ; p <= ourmaxp ; p++)
	  for (q = -p ; q <= p ; q++) {
	    double d ;
	    if (maxvals[p][q+maxp][k]) 
	      d = SHRM[i][j][p][q+maxp][k]/maxvals[p][q+maxp][k] ;
	    else d = 0.0 ;
	    d = (d+1.0)/2.0 ;
	    int val = (int)(65535.999*d) ;
	    assert(val >= 0 && val < 65536) ;
	    unsigned char big,small ;
	    big = (unsigned char) (val/256) ;
	    small = (unsigned char) (val - ((int) big)*256) ;
	    fprintf(fp2,"%c%c",big,small) ;
	  }
  fclose(fp2) ;
}

//--------------------------------------------------------------------------
// Routine to generate SHRM cubemaps
//--------------------------------------------------------------------------

void generatecube(double SHRM[cubesize][cubesize][maxp+1][2*maxp+1][3], int ourmaxp, const char *filename,int size) {

  int i,j,k,l,m,p,q ;
  FILE *fp2 ;
  assert(fp2 = fopen(filename,"wb")) ;
  double maxvals[maxp+1][2*maxp+1][3] ;

  for (i = 0 ; i < size ; i++)
    for (j = 0 ; j < size ; j++)
      for (k = 0 ; k < 3 ; k++)
	for (p = 0 ; p <= ourmaxp ; p++)
	  for (q = -p ; q <= p ; q++) {
	    double v = SHRM[i][j][p][q+maxp][k] ;
	    if ((i == 0 && j == 0) || (fabs(v) > maxvals[p][q+maxp][k]))
	      maxvals[p][q+maxp][k] = fabs(v) ;
	  }
  
  for (k = 0 ; k < 3 ; k++) {
    for (p = 0 ; p <= ourmaxp ; p++)
      for (q = -p ; q <= p ; q++)
	fprintf(fp2,"%lf ",mulfac*maxvals[p][q+maxp][k]) ;
    fprintf(fp2,"\n") ;
  }
 
  for (i = 0 ; i < size ; i++)
    for (j = 0 ; j < size ; j++)
      for (k = 0 ; k < 3 ; k++)
	for (p = 0 ; p <= ourmaxp ; p++)
	  for (q = -p ; q <= p ; q++) {
	    double d = SHRM[i][j][p][q+maxp][k]/maxvals[p][q+maxp][k] ;
	    d = (d+1.0)/2.0 ;
	    int val = (int)(65535.999*d) ;
	    assert(val >= 0 && val < 65536) ;
	    unsigned char big,small ;
	    big = (unsigned char) (val/256) ;
	    small = (unsigned char) (val - ((int) big)*256) ;
	    fprintf(fp2,"%c%c",big,small) ;
	  }
  fclose(fp2) ;
}

//-------------------------------------------------------------
// The main SHRM printer.  
//-------------------------------------------------------------

void printbigshrm(const char * base, const char *cmdline) {
  char s[100] ;
  sprintf(s,"%s.shrm",base) ;
  assert(maxtheta == maxphi) ;
  generate(refwarp,maxp,s,cmdline,maxtheta) ;
  //  sprintf(s,"%s3D.shrm",base) ;
  //  generate(kautzrefwarp,mymaxp,s,cmdline,maxtheta) ;  
}

//-------------------------------------------------------------
// Code for setting the variables for cubemaps
//-------------------------------------------------------------

void setst(int i, int j, double &s, double &t) {
      s = (double) ((cubesize-1.0)/2.0-i) / (double) ((cubesize-1.0)/ 2.0) ;
      t = (double) (j-(cubesize-1.0)/2.0) / (double) ((cubesize-1.0)/ 2.0) ;
}

void setvars(double x, double y, double z, double theta, double
	     phi, double SHRM[maxtheta][maxphi][maxp+1][2*maxp+1][3], int ourmaxp, double cube[maxp+1][2*maxp+1][3]) {
  double imgx, imgy ;
  double vec[3] = {x,y,z} ;
  VecNormalize(vec) ;
  theta = acos(vec[2]) ;
  phi = atan2(vec[1],vec[0]) ;
  if (phi < 0) phi += 2.0*M_PI ;
  imgy = (theta/M_PI*maxtheta) ;
  imgx = (phi/(2.0*M_PI)*maxphi) ;
  if (imgy > maxtheta - 1.0) imgy = maxtheta - 1.0001 ;
  if (imgx > maxphi - 1.0) imgx = maxphi - 1.0001 ;
  double fracx, fracy ;
  int intx, inty, intxp, intyp ;
  intx = (int) imgx ; inty = (int) imgy ;
  intxp = intx + 1 ; intyp = inty + 1 ;
  //      if (intxp == maxphi) --intxp ;
  //      if (intyp == maxtheta) --intyp ;
  assert(intxp < maxphi) ; assert(intyp < maxtheta) ;

  fracx = imgx - intx ; fracy = imgy - inty ;
  int k,p,q ;
  for (k = 0 ; k < 3 ; k++) {
    for (p = 0 ; p <= ourmaxp ; p++)
      for (q = -p ; q <= p ; q++) {
	double val ;
	val = fracx*fracy*     SHRM[intyp][intxp][p][q+maxp][k] +
	  (1-fracx)*fracy*     SHRM[intyp][intx] [p][q+maxp][k] +
	  fracx*(1-fracy)*     SHRM[inty][intxp] [p][q+maxp][k] +
	  (1-fracx)*(1-fracy)* SHRM[inty][intx]  [p][q+maxp][k] ;
	cube[p][q+maxp][k] = val ;
      }
  }
}

//--------------------------------------------------------------
// Printing of the SHRM cubemaps
//--------------------------------------------------------------

void printcubeshrm(const char *base, int ourmaxp, double SHRM[maxtheta][maxphi][maxp+1][2*maxp+1][3], const int size) {
  
  char str[300] ;
  double x,y,z,theta,phi,s,t ;
  int i,j,k,l,m,n,p,q ;
  int imgx, imgy ;

  sprintf(str,"%s_nx.shrm",base) ;
  for (i = 0 ; i < size ; i++)
    for (j = 0 ; j < size ; j++) {
      setst(i,j,s,t) ;
      x = -1.0 ; y = s ; z = t ;
      setvars(x,y,z,theta,phi,SHRM,ourmaxp,cubemapshrm[i][j]) ;
    }
  generatecube(cubemapshrm,ourmaxp,str,cubesize) ;

  sprintf(str,"%s_px.shrm",base) ;
  for (i = 0 ; i < size ; i++)
    for (j = 0 ; j < size ; j++) {
      setst(i,j,s,t) ;
      x = 1.0 ; y = s ; z = -t ;
      setvars(x,y,z,theta,phi,SHRM,ourmaxp,cubemapshrm[i][j]) ;
    }
  generatecube(cubemapshrm,ourmaxp,str,cubesize) ;

  sprintf(str,"%s_ny.shrm",base) ;
   for (i = 0 ; i < size ; i++)
    for (j = 0 ; j < size ; j++) {
      setst(i,j,s,t) ;
      x = t ; y = -1.0 ; z = s ;
      setvars(x,y,z,theta,phi,SHRM,ourmaxp,cubemapshrm[i][j]) ;
    }
  generatecube(cubemapshrm,ourmaxp,str,cubesize) ;

  sprintf(str,"%s_py.shrm",base) ;
   for (i = 0 ; i < size ; i++)
    for (j = 0 ; j < size ; j++) {
      setst(i,j,s,t) ;
      x = t ; y = 1.0 ; z = -s ;
      setvars(x,y,z,theta,phi,SHRM,ourmaxp,cubemapshrm[i][j]) ;
    }
  generatecube(cubemapshrm,ourmaxp,str,cubesize) ;

  sprintf(str,"%s_nz.shrm",base) ;
   for (i = 0 ; i < size ; i++)
    for (j = 0 ; j < size ; j++) {
      setst(i,j,s,t) ;
      x = -t ; y = s ; z = -1.0 ;
      setvars(x,y,z,theta,phi,SHRM,ourmaxp,cubemapshrm[i][j]) ;
    }
  generatecube(cubemapshrm,ourmaxp,str,cubesize) ;

  sprintf(str,"%s_pz.shrm",base) ;
   for (i = 0 ; i < size ; i++)
    for (j = 0 ; j < size ; j++) {
      setst(i,j,s,t) ;
      x = t ; y = s ; z = 1.0 ;
      setvars(x,y,z,theta,phi,SHRM,ourmaxp,cubemapshrm[i][j]) ;
    }
  generatecube(cubemapshrm,ourmaxp,str,cubesize) ;
}

//-------------------------------------------------------------------------
// The main routine to print the SHRM
//------------------------------------------------------------------------

void printshrms(const char *base, const char *cmdline) {
  printbigshrm(base,cmdline) ; // The main full SHRM
  printcubeshrm(base,mymaxp,refwarp,cubesize) ; // Cubemaps used in the shading
                                                // Language
 
  //  char kautz[100] ; sprintf(kautz,"%s3D",base) ;
  //  printcubeshrm(kautz,mymaxp,kautzrefwarp,cubesize) ;
}


//---------------------------------------------------------------------------
// Additional debugging routines
//---------------------------------------------------------------------------

void dumpimageref (double refwarp[maxtheta][maxphi][maxp+1][2*maxp+1][3], double view[3]) {
  double thetaview, phiview, th, ph ;
  if (fabs(view[2]) > 1.0) view[2] *= .999 ;
  assert(fabs(view[2] <= 1.0)) ;
  thetaview = acos(view[2]) ;
  phiview = atan2(view[1],view[0]) ;
  if (phiview < 0) phiview += 2.0*M_PI ;
  
  th = thetaview/M_PI*maxtheta ; if (th > maxtheta - 1) th = maxtheta - 1.0001 ;
  ph = phiview/(2.0*M_PI)*maxphi ; if (ph > maxphi - 1) ph = maxphi - 1.0001 ;
  assert(th >= 0 && ph >= 0) ;

  FILE *fp ;
  assert(fp = fopen("tmp.ppm","wb")) ;
  fprintf(fp,"P6\n%d %d\n255\n",maxphi*2,maxtheta) ;
  int i,j,k,p,q ;
  for (i = 0 ; i < maxtheta ; i++)
    for (j = 0 ; j < maxphi*2 ; j++) 
      for (k = 0 ; k < 3 ; k++) {
	double val = 0 ;
	for (p = 0 ; p <= maxp ; p++)
	  for (q = -p ; q <= p ; q++) {
	    int jj1, jj2 ;
	    jj1 = jj2 = j/2 ; 
	    if (j%2 == 1 && j != (maxphi-1)) ++jj2 ;
	    val += 0.5*mulfac*
	          (refwarp[i][jj1][p][q+maxp][k]*Yreal(p,q,th,ph) 
		  +refwarp[i][jj2][p][q+maxp][k]*Yreal(p,q,th,ph)) ;
	  }
	if (val > 255) val = 255 ;
	if (val < 0) val = 0 ;
	unsigned char c = (unsigned char) val ;
	fprintf(fp,"%c",c) ;
      }
  fclose(fp) ;
}

void dumpimagerefall (double refwarp[maxtheta][maxphi][maxp+1][2*maxp+1][3], double view[3]) {
  double thetaview, phiview, th, ph ;
  if (fabs(view[2]) > 1.0) view[2] *= .999 ;
  assert(fabs(view[2] <= 1.0)) ;
  thetaview = acos(view[2]) ;
  phiview = atan2(view[1],view[0]) ;
  if (phiview < 0) phiview += 2.0*M_PI ;
  
  th = thetaview/M_PI*maxtheta ; if (th > maxtheta - 1) th = maxtheta - 1.0001 ;
  ph = phiview/(2.0*M_PI)*maxphi ; if (ph > maxphi - 1) ph = maxphi - 1.0001 ;
  assert(th >= 0 && ph >= 0) ;

  int i,j,k,p,q ;
  char s[300] ;
  FILE *fp[maxp+1] ;
  for (p = 0 ; p <= maxp ; p++) {
    sprintf(s,"refp%d.ppm",p) ;
    assert(fp[p] = fopen(s,"wb")) ;
    fprintf(fp[p],"P6\n%d %d\n255\n",maxphi*2,maxtheta) ;
  }

  for (i = 0 ; i < maxtheta ; i++)
    for (j = 0 ; j < maxphi*2 ; j++) 
      for (k = 0 ; k < 3 ; k++) {
	double val = 0 ;
	for (p = 0 ; p <= maxp ; p++) {
	  for (q = -p ; q <= p ; q++) {
	    int jj1, jj2 ;
	    jj1 = jj2 = j/2 ; 
	    if (j%2 == 1 && j != (maxphi-1)) ++jj2 ;
	    val += 0.5*mulfac*
	          (refwarp[i][jj1][p][q+maxp][k]*Yreal(p,q,th,ph) 
		  +refwarp[i][jj2][p][q+maxp][k]*Yreal(p,q,th,ph)) ;
	  }
	  unsigned char c ;
	  if (val > 255) c = (unsigned char) 255 ;
	  else if (val < 0) c = (unsigned char) 0 ;
	  else c = (unsigned char) val ;
	  fprintf(fp[p],"%c",c) ;
	}
      }
  for (p = 0 ; p <= maxp ; p++) fclose(fp[p]) ;
}

void dumpimagecabral(double refwarp[maxtheta][maxphi][maxp+1][2*maxp+1][3], double view[3]) {
  // Assume view = 0 0 1 
  view[0] = 0 ; view[1] = 0 ; view[2] = 1 ;
  
  double icos1[3] = {0.000000, 0.525731, 0.850651} ;
  double icos2[3] = {0.000000, -0.525731, 0.850651} ;
  double icos3[3] = {-0.850651, 0.000000, 0.525731} ;
  

  /*
  double icos1[3] = {0,0.607061858,0.79465458} ;
  double icos2[3] = {-0.525730991,-0.303530929,0.79465458} ;
  double icos3[3] = {+0.525730991,-0.303530929,0.79465458} ;
  */

  double thetaview1, phiview1, th1, ph1 ;
  double thetaview2, phiview2, th2, ph2 ;
  double thetaview3, phiview3, th3, ph3 ;
  if (fabs(icos1[2]) > 1.0) icos1[2] *= .999 ;
  assert(fabs(icos1[2] <= 1.0)) ;
  thetaview1 = acos(icos1[2]) ;
  phiview1 = atan2(icos1[1],icos1[0]) ;
  if (phiview1 < 0) phiview1 += 2.0*M_PI ;
  
  th1 = thetaview1/M_PI*maxtheta ; if (th1 > maxtheta - 1) th1 = maxtheta - 1.0001 ;
  ph1 = phiview1/(2.0*M_PI)*maxphi ; if (ph1 > maxphi - 1) ph1 = maxphi - 1.0001 ;
  assert(th1 >= 0 && ph1 >= 0) ;

  if (fabs(icos2[2]) > 1.0) icos2[2] *= .999 ;
  assert(fabs(icos2[2] <= 1.0)) ;
  thetaview2 = acos(icos2[2]) ;
  phiview2 = atan2(icos2[1],icos2[0]) ;
  if (phiview2 < 0) phiview2 += 2.0*M_PI ;
  
  th2 = thetaview2/M_PI*maxtheta ; if (th2 > maxtheta - 1) th2 = maxtheta - 1.0001 ;
  ph2 = phiview2/(2.0*M_PI)*maxphi ; if (ph2 > maxphi - 1) ph2 = maxphi - 1.0001 ;
  assert(th2 >= 0 && ph2 >= 0) ;

  if (fabs(icos3[2]) > 1.0) icos3[2] *= .999 ;
  assert(fabs(icos3[2] <= 1.0)) ;
  thetaview3 = acos(icos3[2]) ;
  phiview3 = atan2(icos3[1],icos3[0]) ;
  if (phiview3 < 0) phiview3 += 2.0*M_PI ;
  
  th3 = thetaview3/M_PI*maxtheta ; if (th3 > maxtheta - 1) th3 = maxtheta - 1.0001 ;
  ph3 = phiview3/(2.0*M_PI)*maxphi ; if (ph3 > maxphi - 1) ph3 = maxphi - 1.0001 ;
  assert(th3 >= 0 && ph3 >= 0) ;

  double thetaview, phiview, th, ph ;
  if (fabs(view[2]) > 1.0) view[2] *= .999 ;
  assert(fabs(view[2] <= 1.0)) ;
  thetaview = acos(view[2]) ;
  phiview = atan2(view[1],view[0]) ;
  if (phiview < 0) phiview += 2.0*M_PI ;
  
  th = thetaview/M_PI*maxtheta ; if (th > maxtheta - 1) th = maxtheta - 1.0001 ;
  ph = phiview/(2.0*M_PI)*maxphi ; if (ph > maxphi - 1) ph = maxphi - 1.0001 ;
  assert(th >= 0 && ph >= 0) ;

  FILE *fp ;
  assert(fp = fopen("tmpcabral.ppm","wb")) ;
  fprintf(fp,"P6\n%d %d\n255\n",maxtheta,maxphi) ;
  int i,j,k,p,q ;
  double diff[3] = {0,0,0}, sqval[3] = {0,0,0} ;
  for (i = 0 ; i < maxtheta ; i++) {
    double st = sin(i*M_PI/maxtheta)*M_PI/2.0/(maxtheta*maxphi) ;
    for (j = 0 ; j < maxphi ; j++) 
      for (k = 0 ; k < 3 ; k++) {
	double val = 0, val2 = 0 ;
	for (p = 0 ; p <= maxp ; p++)
	  for (q = -p ; q <= p ; q++) {
	    val += 1.0/3.0*mulfac*(refwarp[i][j][p][q+maxp][k]*Yreal(p,q,th1,ph1) 
                              +refwarp[i][j][p][q+maxp][k]*Yreal(p,q,th2,ph2)
                              +refwarp[i][j][p][q+maxp][k]*Yreal(p,q,th3,ph3));
	    val2 += mulfac*refwarp[i][j][p][q+maxp][k]*Yreal(p,q,th,ph) ;
	  }
	diff[k] += st*(val-val2)*(val-val2) ;
	sqval[k]+= st*val2*val2 ;
	if (val > 255) val = 255 ;
	if (val < 0) val = 0 ;
	unsigned char c = (unsigned char) val ;
	fprintf(fp,"%c",c) ;
      }
  }
  fclose(fp) ;
  fprintf(stderr,"Diff = %lf %lf %lf Sqval = %lf %lf %lf\n",diff[0],diff[1],diff[2],sqval[0],sqval[1],sqval[2]) ;
}

void dumpimagespherecabral(double refwarp[maxtheta][maxphi][maxp+1][2*maxp+1][3], double view[3]) {

  // Assume view = 0 0 1 
  view[0] = 0 ; view[1] = 0 ; view[2] = 1 ;
  
  double icos1[3] = {0.000000, 0.525731, 0.850651} ;
  double icos2[3] = {0.000000, -0.525731, 0.850651} ;
  double icos3[3] = {-0.850651, 0.000000, 0.525731} ;
  

  //  double wt[3] = {1.0/3.0,1.0/3.0,1.0/3.0} ;
  double wt[3] = {1.0/2.0,1.0/2.0,0.0} ;

  /*
  double icos1[3] = {0,0.607061858,0.79465458} ;
  double icos2[3] = {-0.525730991,-0.303530929,0.79465458} ;
  double icos3[3] = {+0.525730991,-0.303530929,0.79465458} ;
  */

  double thetaview1, phiview1, th1, ph1 ;
  double thetaview2, phiview2, th2, ph2 ;
  double thetaview3, phiview3, th3, ph3 ;
  if (fabs(icos1[2]) > 1.0) icos1[2] *= .999 ;
  assert(fabs(icos1[2] <= 1.0)) ;
  thetaview1 = acos(icos1[2]) ;
  phiview1 = atan2(icos1[1],icos1[0]) ;
  if (phiview1 < 0) phiview1 += 2.0*M_PI ;
  
  th1 = thetaview1/M_PI*maxtheta ; if (th1 > maxtheta - 1) th1 = maxtheta - 1.0001 ;
  ph1 = phiview1/(2.0*M_PI)*maxphi ; if (ph1 > maxphi - 1) ph1 = maxphi - 1.0001 ;
  assert(th1 >= 0 && ph1 >= 0) ;

  if (fabs(icos2[2]) > 1.0) icos2[2] *= .999 ;
  assert(fabs(icos2[2] <= 1.0)) ;
  thetaview2 = acos(icos2[2]) ;
  phiview2 = atan2(icos2[1],icos2[0]) ;
  if (phiview2 < 0) phiview2 += 2.0*M_PI ;
  
  th2 = thetaview2/M_PI*maxtheta ; if (th2 > maxtheta - 1) th2 = maxtheta - 1.0001 ;
  ph2 = phiview2/(2.0*M_PI)*maxphi ; if (ph2 > maxphi - 1) ph2 = maxphi - 1.0001 ;
  assert(th2 >= 0 && ph2 >= 0) ;

  if (fabs(icos3[2]) > 1.0) icos3[2] *= .999 ;
  assert(fabs(icos3[2] <= 1.0)) ;
  thetaview3 = acos(icos3[2]) ;
  phiview3 = atan2(icos3[1],icos3[0]) ;
  if (phiview3 < 0) phiview3 += 2.0*M_PI ;
  
  th3 = thetaview3/M_PI*maxtheta ; if (th3 > maxtheta - 1) th3 = maxtheta - 1.0001 ;
  ph3 = phiview3/(2.0*M_PI)*maxphi ; if (ph3 > maxphi - 1) ph3 = maxphi - 1.0001 ;
  assert(th3 >= 0 && ph3 >= 0) ;

  double thetaview, phiview, th, ph ;
  if (fabs(view[2]) > 1.0) view[2] *= .999 ;
  assert(fabs(view[2] <= 1.0)) ;
  thetaview = acos(view[2]) ;
  phiview = atan2(view[1],view[0]) ;
  if (phiview < 0) phiview += 2.0*M_PI ;
  
  th = thetaview/M_PI*maxtheta ; if (th > maxtheta - 1) th = maxtheta - 1.0001 ;
  ph = phiview/(2.0*M_PI)*maxphi ; if (ph > maxphi - 1) ph = maxphi - 1.0001 ;
  assert(th >= 0 && ph >= 0) ;

  FILE *fp ;
  assert(fp = fopen("tmpcabral.ppm","wb")) ;
  double Z[3],Y[3],X[3] ;
  double theta = acos(view[2]) ;
  double phi = atan2(view[1],view[0]) ; if (phi < 0) phi += 2.0*M_PI ;
  Z[0] = view[0] ; Z[1] = view[1] ; Z[2] = view[2] ;
  Y[0] = -sin(phi) ; Y[1] = cos(phi) ; Y[2] = 0 ;
  X[0] = cos(theta)*cos(phi) ; X[1] = cos(theta)*sin(phi); 
  X[2] = -sin(theta);

  int size, imgsize ;
  size = 388 ; imgsize = 512 ;
  int start, end ;
  start = (imgsize - size)/2 ;
  end = start + size - 1 ;
  double sub = (size - 1)/2.0 ;
  double mid = (imgsize - 1.0) / 2.0 ;
  fprintf(fp,"P6\n%d %d\n255\n",imgsize,imgsize) ;

  int i,j,k,l,p,q ;

  for (i = 0 ; i < imgsize ; i++)
    for (j = 0 ; j < imgsize ; j++)
      for (k = 0 ; k < 3 ; k++) {
	if (i < start || j < start || i > end || j > end) {
	  unsigned char c = (unsigned char) 0 ;
	  fprintf(fp,"%c",c) ;
	  continue ;
	}
	if (i == 270 && j == 94) {
	  fprintf(stderr,"Here\n") ;
	}
	double x,y,z ;
	double xx,yy,zz ;
	xx = (j-mid)/sub ;
	yy = (mid-i)/sub ;
	if (xx*xx + yy*yy >= 1.0) {
	  unsigned char c = (unsigned char) 0 ;
	  fprintf(fp,"%c",c) ;
	  continue ;
	}
	zz = sqrt(1.0-xx*xx-yy*yy) ;
	x = xx*X[0] + yy*Y[0] + zz*Z[0] ;
	y = xx*X[1] + yy*Y[1] + zz*Z[1] ;
	z = xx*X[2] + yy*Y[2] + zz*Z[2] ;

	double N[3] ; 
	N[0] = x ; N[1] = y ; N[2] = z ;
	double R[3] ;
	reflectionvector(N,view,R) ;
	double posn[3][3] ;
	posn[2][0] = R[0] ; posn[2][1] = R[1] ; posn[2][2] = R[2] ;
	double alpha, beta ;
	if (fabs(R[2])>1.0) R[2] *= 0.9999 ;
	alpha = acos(R[2]) ; beta = atan2(R[1],R[0]) ; 
	if (beta < 0) beta += 2.0*M_PI;
	posn[1][0] = -sin(beta) ; posn[1][1] = cos(beta) ; posn[1][2] = 0 ;
	posn[0][0] = cos(alpha)*cos(beta) ; posn[0][1] = cos(alpha)*sin(beta) ; posn[0][2] = -sin(alpha);
	double lightpos[3], viewpos[3] ;
	for (l = 0 ; l < 3 ; l++) { 
	  viewpos[l]  = dotp(view, posn[l]) ; 
	  lightpos[l] = dotp(light,posn[l]) ;
	}
	
	double thetain, phin, thetaout, phiout, a, b, phi ;
	if (fabs(viewpos[2])>1.0) viewpos[2] *= 0.9999 ;
	thetaout= acos(viewpos[2]) ;
	phiout = atan2(viewpos[1],viewpos[0]) ; 
	if (phiout<0) phiout +=2.0*M_PI;
	a = alpha/M_PI*maxtheta ;
	b = beta/(2.0*M_PI)*maxphi ;
	if (a > maxtheta - 1.0) a = maxtheta - 1.0 ;
	if (b > maxphi - 1.0) b = maxphi - 1.0 ;
	assert (a >= 0 && b >= 0) ;
	if (fabs(lightpos[2])>1.0) lightpos[2] *= 0.9999  ;
	thetain = acos(lightpos[2]) ;
	phin = atan2(lightpos[1],lightpos[0]) ; if (phin < 0) phin += 2.0*M_PI;
	phi = phiout - phin ; if (phi < 0) phi += 2.0*M_PI ;

        double ti = thetain/M_PI*maxtheta ;
	double phh = phi/(2.0*M_PI)*maxphi;

        double th, ph ;
	th = (thetaout/M_PI*maxtheta) ;
	ph = (phiout/(2.0*M_PI)*maxphi) ; 
	if (th > maxtheta - 1) th = maxtheta - 1.0001 ;
	if (ph > maxphi - 1) ph = maxphi - 1.0001 ;
	assert(th >= 0 && ph >= 0) ;

	double thetaview, phiview, tv, pv ;
	if (fabs(view[2]) > 1.0) view[2] *= .999 ;
	assert(fabs(view[2] <= 1.0)) ;
	thetaview = acos(view[2]) ;
	phiview = atan2(view[1],view[0]) ;
	if (phiview < 0) phiview += 2.0*M_PI ;
	double val = 0 ;
	for (p = 0 ; p <= maxp ; p++)
	  for (q = -p ; q <= p ; q++) {
	    double factor1 = wt[0]*Yreal(p,q,th1,ph1) + wt[1]*Yreal(p,q,th2,ph2) + wt[2]*Yreal(p,q,th3,ph3) ;
	    double factor2 ;
	    double fraca, fracb ;
	    fraca = a - (int) a ; fracb = b - (int) b ;
	    factor2=(1-fraca)*(1-fracb)*refwarp[(int)a][(int)b][p][q+maxp][k] 
                   +fraca*(1-fracb)*refwarp[1+(int)a][(int)b][p][q+maxp][k] 
	           +(1-fraca)*fracb*refwarp[(int)a][1+(int)b][p][q+maxp][k] 
                   +fraca*fracb*refwarp[1+(int)a][1+(int)b][p][q+maxp][k] ;
	    val += factor1*factor2 ;
	  }
	val *= mulfac ;
	if (val > 255) val = 255 ;
	if (val < 0) val = 0 ;
	unsigned char c = (unsigned char) val ;
	fprintf(fp,"%c",c) ;
      }
  fclose(fp) ;
}

void dumpimage (double refwarp[maxtheta][maxphi][maxp+1][2*maxp+1][3], double view[3]) {
  double thetaview, phiview, th, ph ;
  if (fabs(view[2]) > 1.0) view[2] *= .999 ;
  assert(fabs(view[2] <= 1.0)) ;
  thetaview = acos(view[2]) ;
  phiview = atan2(view[1],view[0]) ;
  if (phiview < 0) phiview += 2.0*M_PI ;
  
  th = thetaview/M_PI*maxtheta ; 
  if (th > maxtheta - 1) th = maxtheta - 1.0001 ;
  ph = phiview/(2.0*M_PI)*maxphi ; if (ph > maxphi - 1) ph = maxphi - 1.0001 ;
  assert(th >= 0 && ph >= 0) ;

  FILE *fp ;
  assert(fp = fopen("tmp.ppm","wb")) ;
  fprintf(fp,"P6\n%d %d\n255\n",maxtheta,maxphi) ;
  int i,j,k,p,q ;
  for (i = 0 ; i < maxtheta ; i++)
    for (j = 0 ; j < maxphi ; j++) { 

      double a,b ;
      double normal[3] ;
      normal[0] = sintheta[i]*cosphi[j] ;
      normal[1] = sintheta[i]*sinphi[j] ;
      normal[2] = costheta[i] ;

      if (dotp(normal,view)<0) {
	unsigned char c = (unsigned char) 0 ;
	fprintf(fp,"%c%c%c",c,c,c) ;
	continue ;
      }
      
      double refl[3] ;
      reflectionvector(normal,view,refl) ;

      a = acos(refl[2])/M_PI*maxtheta ;
      b = atan2(refl[1],refl[0]) ; if (b < 0) b += 2.0*M_PI ;
      b = b/(2.0*M_PI)*maxphi ;
      if (a > maxtheta - 1.0) a = maxtheta - 1.0 ;
      if (b > maxphi - 1.0) b = maxphi - 1.0 ;
      assert (a >= 0 && b >= 0) ;
      
      for (k = 0 ; k < 3 ; k++) {
	double val = 0 ;
	for (p = 0 ; p <= maxp ; p++)
	  for (q = -p ; q <= p ; q++) {
	    val += mulfac*refwarp[(int) a][(int) b][p][q+maxp][k]*Yreal(p,q,th,ph) ;
	  }
	if (val > 255) val = 255 ;
	if (val < 0) val = 0 ;
	unsigned char c = (unsigned char) val ;
	fprintf(fp,"%c",c) ;
      }
    }
  fclose(fp) ;
}

void dumpimagesphereangular(double view[3]) {
  int i,j,k,l,p,q ;

  FILE *fp ;
  assert(fp  = fopen("tmpang.ppm","wb")) ;
  int flag = 0 ;

  double Z[3],Y[3],X[3] ;
  double theta = acos(view[2]) ;
  double phi = atan2(view[1],view[0]) ; if (phi < 0) phi += 2.0*M_PI ;
  Z[0] = view[0] ; Z[1] = view[1] ; Z[2] = view[2] ;
  Y[0] = -sin(phi) ; Y[1] = cos(phi) ; Y[2] = 0 ;
  X[0] = cos(theta)*cos(phi) ; X[1] = cos(theta)*sin(phi); 
  X[2] = -sin(theta);

  int size, imgsize ;
  //  size = 388 ; imgsize = 512 ;
      size = 194 ; imgsize = 256 ;
  //      size = 76 ; imgsize = 100 ;
  int start, end ;
  start = (imgsize - size)/2 ;
  end = start + size - 1 ;
  double sub = (size - 1)/2.0 ;
  double mid = (imgsize - 1.0) / 2.0 ;
  fprintf(fp,"P6\n%d %d\n255\n",imgsize,imgsize) ;

  double thetaview, phiview, tv, pv ;
  if (fabs(view[2]) > 1.0) view[2] *= .999 ;
  assert(fabs(view[2] <= 1.0)) ;
  thetaview = acos(view[2]) ;
  phiview = atan2(view[1],view[0]) ;
  if (phiview < 0) phiview += 2.0*M_PI ;
  
  tv = thetaview/M_PI*maxtheta ; if (tv > maxtheta - 1) tv = maxtheta - 1.0001 ;
  pv = phiview/(2.0*M_PI)*maxphi ; if (pv > maxphi - 1) pv = maxphi - 1.0001 ;
  assert(tv >= 0 && pv >= 0) ;
  
  for (i = 0 ; i < imgsize ; i++) {
    for (j = 0 ; j < imgsize ; j++) {
      if (i == 53 && j == 18) {
	fprintf(stderr, "Here\n") ;
      }
      double val[3] = {0,0,0} ;

      if (i < start || j < start || i > end || j > end) {
	unsigned char c = (unsigned char) 0 ;
	  fprintf(fp,"%c%c%c",c,c,c) ;
	  continue ;
      }
      double x,y,z ;
      double xx,yy,zz ;
      xx = (j-mid)/sub ;
      yy = (mid-i)/sub ;
      if (xx*xx + yy*yy >= 1.0) {
	unsigned char c = (unsigned char) 0 ;
	fprintf(fp,"%c%c%c",c,c,c) ;
	continue ;
      }
      zz = sqrt(1.0-xx*xx-yy*yy) ;
      x = xx*X[0] + yy*Y[0] + zz*Z[0] ;
      y = xx*X[1] + yy*Y[1] + zz*Z[1] ;
      z = xx*X[2] + yy*Y[2] + zz*Z[2] ;
      
      double N[3] ; 
      N[0] = x ; N[1] = y ; N[2] = z ;
      double R[3] ;
      reflectionvector(N,view,R) ;
      double posn[3][3] ;
      posn[2][0] = R[0] ; posn[2][1] = R[1] ; posn[2][2] = R[2] ;
      double alpha, beta ;
      if (fabs(R[2])>1.0) R[2] *= 0.9999 ;
      alpha = acos(R[2]) ; beta = atan2(R[1],R[0]) ; 
      if (beta < 0) beta += 2.0*M_PI;
      posn[1][0] = -sin(beta) ; posn[1][1] = cos(beta) ; posn[1][2] = 0 ;
      posn[0][0] = cos(alpha)*cos(beta) ; posn[0][1] = cos(alpha)*sin(beta) ; posn[0][2] = -sin(alpha);
      double viewpos[3] ;
      for (l = 0 ; l < 3 ; l++) { 
	viewpos[l]  = dotp(view, posn[l]) ; 
      }
      double thetain, phin, thetaout, phiout, a, b, phi ;
      thetaout= acos(viewpos[2]) ;
      phiout = atan2(viewpos[1],viewpos[0]) ; 
      if (phiout<0) phiout +=2.0*M_PI;
      a = alpha/M_PI*maxtheta ;
      b = beta/(2.0*M_PI)*maxphi ;
      if (a > maxtheta - 1.0) a = maxtheta - 1.0 ;
      if (b > maxphi - 1.0) b = maxphi - 1.0 ;
      assert (a >= 0 && b >= 0) ;
      
      double th, ph ;
      th = (thetaout/M_PI*maxtheta) ;
      ph = (phiout/(2.0*M_PI)*maxphi) ; 
      if (th > maxtheta - 1) th = maxtheta - 1.0001 ;
      if (ph > maxphi - 1) ph = maxphi - 1.0001 ;
      assert(th >= 0 && ph >= 0) ;
      
      
      // Light directions
      double valtest1, valtest2, valtest3, valtest4,valtest5[3] = {0,0,0} ;

      for (p = 0 ; p < mysamp ; p++) {
	double thetalight, philight ;
	thetalight = p*M_PI/mysamp ;
	double stl = sin(thetalight) ;
	double ctl = cos(thetalight) ;
	double mf = (2.0*M_PI*M_PI*stl)/(mysamp*mysamp) ;
	for (q = 0 ; q < mysamp ; q++) {
	  double cpl = cosphi[q] ; double spl = sinphi[q] ;
	  double mylight[3], lightpos[3] ;
	  mylight[0] = stl*cpl ; 
	  mylight[1] = stl*spl ; 
	  mylight[2] = ctl ;
 	  
	  for (l = 0 ; l < 3 ; l++) { 
	    lightpos[l] = dotp(mylight,posn[l]) ;
	  }
	  thetain = acos(lightpos[2]) ;
	  phin = atan2(lightpos[1],lightpos[0]) ; if (phin < 0) phin += 2.0*M_PI;
	  phi = phiout - phin ; if (phi < 0) phi += 2.0*M_PI ;
	  
	  double ti = thetain/M_PI*maxtheta ;
	  double phh = phi/(2.0*M_PI)*maxphi;
	  
	  
	  for (k = 0 ; k < 3 ; k++) {
	    valtest1 = brdf[k][(int) ti][(int) th][(int) phh] ;
	    val[k] += mf*valtest1*mulfac*sampfile[k][p][q] ;

	    if (flag) {
	      valtest2 = 0 ;
	      for (q = 0 ; q <= maxp ; q++)
		for (l = q ; l <= mymaxl ; l++)
		  for (p = q ; p <= maxp ; p++) {
		    double mv = 1.0 ;
		  if (q == 0) mv = sqrt2 ;
		  valtest2 += mv*brdfcoeffs[k][l][p][q]*sqrtpi*
		    plmvalreal(l,q,(int)ti)*plmvalreal(p,q,(int)th)
		    *cosphi[(q*(int)phh)%maxphi] ;
		  }
	      valtest5[k] += mf*valtest2*mulfac*sampfile[k][p][q] ;
	      //	      printf("%lf %lf\n",valtest1,valtest2) ;
	      //	      fflush(stdout) ;
	    }
	  }
	}
      }
	  


      for (k = 0 ; k < 3 ; k++) {

	if (flag) {
	  valtest3 = val[k] ;
	  valtest4 = 0 ;
	  for (p = 0 ; p <= maxp ; p++)
	    for (q = -p ; q <= p ; q++) {
	      double factor1 = Yreal(p,q,tv,pv) ;
	      double factor2 ;
	      double fraca, fracb ;
	      fraca = a - (int) a ; fracb = b - (int) b ;
	      factor2=(1-fraca)*(1-fracb)*refwarp[(int)a][(int)b][p][q+maxp][k] 
		+fraca*(1-fracb)*refwarp[1+(int)a][(int)b][p][q+maxp][k] 
		+(1-fraca)*fracb*refwarp[(int)a][1+(int)b][p][q+maxp][k] 
		+fraca*fracb*refwarp[1+(int)a][1+(int)b][p][q+maxp][k] ;
	      valtest4 += factor1*factor2 ;
	    }
	  valtest4 *= mulfac ;
	  printf("%lf %lf\n",valtest3,valtest4) ;
	  fflush(stdout) ;
	}

	if (val[k] > 255) val[k] = 255 ;
	else if (val[k] < 0) val[k] = 0;
	unsigned char c = (unsigned char) (val[k]) ;
	fprintf(fp,"%c",c) ;
      }
    }
    printf("%d ",i) ;
    fflush(stdout) ;
  }
}

void dumpimages(double refwarp[maxtheta][maxphi][maxp+1][2*maxp+1][3], double imagelist[maxv][maxv][maxtheta][maxphi][3]) {
  int a,b,p,q,i,j,c ;
  int multheta = maxtheta/maxv ;
  int mulphi = maxphi/maxv ;

  for (i = 0 ; i < maxv ; i++) 
    for (a = 0 ; a < maxtheta ; a++)
      for (b = 0 ; b < maxphi ; b++)
	for (q = -maxp ; q <= maxp ; q++)
	  for (c = 0 ; c < 3 ; c++) 
	    for (p = (q > 0) ? q : -q ; p <= maxp ; p++) 
	      imagetmp[i][a][b][q+maxp][c] += refwarp[a][b][p][q+maxp][c] * plmvalreal(p,q,i*multheta) ;

  for (i = 0 ; i < maxv ; i++) 
    for (j = 0 ; j < maxv ; j++)
      for (a = 0 ; a < maxtheta ; a++)
	for (b = 0 ; b < maxphi ; b++)
	  for (c = 0 ; c < 3 ; c++) 
	    for (q = -maxp ; q <= maxp ; q++)
	      imagelist[i][j][a][b][c] += imagetmp[i][a][b][q+maxp][c] * expreal(q,j*mulphi) ; 
}


void dumpimagesphere(double refwarp[maxtheta][maxphi][maxp+1][2*maxp+1][3], double view[3],int kk) {
  int i,j,k,l,p,q ;

  FILE *fp ;
  if (kk == 0) assert(fp  = fopen("tmpfreq.ppm","wb")) ;
  else assert(fp  = fopen("tmp3D.ppm","wb")) ;
  int flag = 0 ;

  double Z[3],Y[3],X[3] ;
  double theta = acos(view[2]) ;
  double phi = atan2(view[1],view[0]) ; if (phi < 0) phi += 2.0*M_PI ;
  Z[0] = view[0] ; Z[1] = view[1] ; Z[2] = view[2] ;
  Y[0] = -sin(phi) ; Y[1] = cos(phi) ; Y[2] = 0 ;
  X[0] = cos(theta)*cos(phi) ; X[1] = cos(theta)*sin(phi); 
  X[2] = -sin(theta);

  int size, imgsize ;
  //  size = 388 ; imgsize = 512 ;
  size = 194 ; imgsize = 256 ;
  int start, end ;
  start = (imgsize - size)/2 ;
  end = start + size - 1 ;
  double sub = (size - 1)/2.0 ;
  double mid = (imgsize - 1.0) / 2.0 ;
  fprintf(fp,"P6\n%d %d\n255\n",imgsize,imgsize) ;

  for (i = 0 ; i < imgsize ; i++)
    for (j = 0 ; j < imgsize ; j++)
      for (k = 0 ; k < 3 ; k++) {
	if (i < start || j < start || i > end || j > end) {
	  unsigned char c = (unsigned char) 0 ;
	  fprintf(fp,"%c",c) ;
	  continue ;
	}
	if (i == 270 && j == 94) {
	  fprintf(stderr,"Here\n") ;
	}
	double x,y,z ;
	double xx,yy,zz ;
	xx = (j-mid)/sub ;
	yy = (mid-i)/sub ;
	if (xx*xx + yy*yy >= 1.0) {
	  unsigned char c = (unsigned char) 0 ;
	  fprintf(fp,"%c",c) ;
	  continue ;
	}
	zz = sqrt(1.0-xx*xx-yy*yy) ;
	x = xx*X[0] + yy*Y[0] + zz*Z[0] ;
	y = xx*X[1] + yy*Y[1] + zz*Z[1] ;
	z = xx*X[2] + yy*Y[2] + zz*Z[2] ;

	double N[3] ; 
	N[0] = x ; N[1] = y ; N[2] = z ;
	double R[3] ;
	reflectionvector(N,view,R) ;
	double posn[3][3] ;
	posn[2][0] = R[0] ; posn[2][1] = R[1] ; posn[2][2] = R[2] ;
	double alpha, beta ;
	if (fabs(R[2])>1.0) R[2] *= 0.9999 ;
	alpha = acos(R[2]) ; beta = atan2(R[1],R[0]) ; 
	if (beta < 0) beta += 2.0*M_PI;
	posn[1][0] = -sin(beta) ; posn[1][1] = cos(beta) ; posn[1][2] = 0 ;
	posn[0][0] = cos(alpha)*cos(beta) ; posn[0][1] = cos(alpha)*sin(beta) ; posn[0][2] = -sin(alpha);
	double lightpos[3], viewpos[3] ;
	for (l = 0 ; l < 3 ; l++) { 
	  viewpos[l]  = dotp(view, posn[l]) ; 
	  lightpos[l] = dotp(light,posn[l]) ;
	}
	
	double thetain, phin, thetaout, phiout, a, b, phi ;
	if (fabs(viewpos[2])>1.0) viewpos[2] *= 0.9999 ;
	thetaout= acos(viewpos[2]) ;
	phiout = atan2(viewpos[1],viewpos[0]) ; 
	if (phiout<0) phiout +=2.0*M_PI;
	a = alpha/M_PI*maxtheta ;
	b = beta/(2.0*M_PI)*maxphi ;
	if (a > maxtheta - 1.0) a = maxtheta - 1.0 ;
	if (b > maxphi - 1.0) b = maxphi - 1.0 ;
	assert (a >= 0 && b >= 0) ;
	if (fabs(lightpos[2])>1.0) lightpos[2] *= 0.9999  ;
	thetain = acos(lightpos[2]) ;
	phin = atan2(lightpos[1],lightpos[0]) ; if (phin < 0) phin += 2.0*M_PI;
	phi = phiout - phin ; if (phi < 0) phi += 2.0*M_PI ;

        double ti = thetain/M_PI*maxtheta ;
	double phh = phi/(2.0*M_PI)*maxphi;

        double th, ph ;
	th = (thetaout/M_PI*maxtheta) ;
	ph = (phiout/(2.0*M_PI)*maxphi) ; 
	if (th > maxtheta - 1) th = maxtheta - 1.0001 ;
	if (ph > maxphi - 1) ph = maxphi - 1.0001 ;
	assert(th >= 0 && ph >= 0) ;

	double thetaview, phiview, tv, pv ;
	if (fabs(view[2]) > 1.0) view[2] *= .999 ;
	assert(fabs(view[2] <= 1.0)) ;
	thetaview = acos(view[2]) ;
	phiview = atan2(view[1],view[0]) ;
	if (phiview < 0) phiview += 2.0*M_PI ;
  
	tv = thetaview/M_PI*maxtheta ; if (tv > maxtheta - 1) tv = maxtheta - 1.0001 ;
	pv = phiview/(2.0*M_PI)*maxphi ; if (pv > maxphi - 1) pv = maxphi - 1.0001 ;
	assert(tv >= 0 && pv >= 0) ;


	double valtest1, valtest2, valtest3, valtest4 ;
	if (flag) {
	valtest1 = brdf[k][(int) ti][(int) th][(int) phh] ;
	valtest2 = 0 ;

	for (q = 0 ; q <= maxp ; q++)
	  for (l = q ; l <= mymaxl ; l++)
	    for (p = q ; p <= maxp ; p++) {
	      double mv = 1.0 ;
	      if (q == 0) mv = sqrt2 ;
	      valtest2 += mv*brdfcoeffs[k][l][p][q]*sqrtpi*
		          plmvalreal(l,q,(int)ti)*plmvalreal(p,q,(int)th)
                          *cosphi[(q*(int)phh)%maxphi] ;
	    }

	valtest3 = valtest4 = 0 ;
	int ii,jj,m ;
	for (ii = 0 ; ii < maxtheta ; ii++) 
	  for (jj = 0 ; jj < maxphi ; jj++) {
	    double ltpos[3] ;
	    ltpos[0] = sintheta[ii]*cosphi[jj] ;
	    ltpos[1] = sintheta[ii]*sinphi[jj] ;
	    ltpos[2] = costheta[ii] ;
	    double mf = (2.0*M_PI*M_PI*sintheta[ii])/(maxtheta*maxphi) ;
	    double tin = acos(dotp(ltpos,R)) ;
	    int tinval = (int) (tin/M_PI*maxtheta) ;
	    double lightval = 0 ;
	    for (l = 0 ; l <= mymaxl ; l++)
	      for (m = - l ; m <= l ; m++) 
		lightval += lightcoeffs[l][m+maxl][0]*Yreal(l,m,ii,jj) ;
	    
		valtest3 += mf*brdf[k][tinval][(int) th][(int) phh]*lightval ;
		valtest4 += mf*brdftest[k][tinval][(int) th][(int) phh]*lightval ;
	      
	  }
	}
	

	double val = 0 ;
	for (p = 0 ; p <= maxp ; p++)
	  for (q = -p ; q <= p ; q++) {
	    double factor1 = Yreal(p,q,tv,pv) ;
	    double factor2 ;
	    double fraca, fracb ;
	    fraca = a - (int) a ; fracb = b - (int) b ;
	    factor2=(1-fraca)*(1-fracb)*refwarp[(int)a][(int)b][p][q+maxp][k] 
                   +fraca*(1-fracb)*refwarp[1+(int)a][(int)b][p][q+maxp][k] 
	           +(1-fraca)*fracb*refwarp[(int)a][1+(int)b][p][q+maxp][k] 
                   +fraca*fracb*refwarp[1+(int)a][1+(int)b][p][q+maxp][k] ;
	    val += factor1*factor2 ;
	  }
	val *= mulfac ;
	if (val > 255) val = 255 ;
	if (val < 0) val = 0 ;
	unsigned char c = (unsigned char) val ;
	fprintf(fp,"%c",c) ;
      }
  fclose(fp) ;
}

void dumpimagesphereall(double refwarp[maxtheta][maxphi][maxp+1][2*maxp+1][3], double view[3],int kk) {
  int i,j,k,l,p,q ;

  int flag = 0 ;

  double Z[3],Y[3],X[3] ;
  double theta = acos(view[2]) ;
  double phi = atan2(view[1],view[0]) ; if (phi < 0) phi += 2.0*M_PI ;
  Z[0] = view[0] ; Z[1] = view[1] ; Z[2] = view[2] ;
  Y[0] = -sin(phi) ; Y[1] = cos(phi) ; Y[2] = 0 ;
  X[0] = cos(theta)*cos(phi) ; X[1] = cos(theta)*sin(phi); 
  X[2] = -sin(theta);

  int size, imgsize ;
  //  size = 388 ; imgsize = 512 ;
  size = 194 ; imgsize = 256 ;
  int start, end ;
  start = (imgsize - size)/2 ;
  end = start + size - 1 ;
  double sub = (size - 1)/2.0 ;
  double mid = (imgsize - 1.0) / 2.0 ;

  FILE *fp[maxp+1] ;
  char s[300] ;
  for (p = 0 ; p <= maxp ; p++) {
    if (kk == 0) sprintf(s,"tmpfreq%d.ppm",p) ;
    else sprintf(s,"tmp3D%d.ppm",p) ;
    assert(fp[p] = fopen(s,"wb")) ;
    fprintf(fp[p],"P6\n%d %d\n255\n",imgsize,imgsize) ;
  }

  for (i = 0 ; i < imgsize ; i++)
    for (j = 0 ; j < imgsize ; j++)
      for (k = 0 ; k < 3 ; k++) {
	if (i < start || j < start || i > end || j > end) {
	  unsigned char c = (unsigned char) 0 ;
	  for (p = 0 ; p <= maxp ; p++) fprintf(fp[p],"%c",c) ;
	  continue ;
	}
	if (i == 270 && j == 94) {
	  fprintf(stderr,"Here\n") ;
	}
	double x,y,z ;
	double xx,yy,zz ;
	xx = (j-mid)/sub ;
	yy = (mid-i)/sub ;
	if (xx*xx + yy*yy >= 1.0) {
	  unsigned char c = (unsigned char) 0 ;
	  for (p = 0 ; p <= maxp ; p++) fprintf(fp[p],"%c",c) ;
	  continue ;
	}
	zz = sqrt(1.0-xx*xx-yy*yy) ;
	x = xx*X[0] + yy*Y[0] + zz*Z[0] ;
	y = xx*X[1] + yy*Y[1] + zz*Z[1] ;
	z = xx*X[2] + yy*Y[2] + zz*Z[2] ;

	double N[3] ; 
	N[0] = x ; N[1] = y ; N[2] = z ;
	double R[3] ;
	reflectionvector(N,view,R) ;
	double posn[3][3] ;
	posn[2][0] = R[0] ; posn[2][1] = R[1] ; posn[2][2] = R[2] ;
	double alpha, beta ;
	if (fabs(R[2])>1.0) R[2] *= 0.9999 ;
	alpha = acos(R[2]) ; beta = atan2(R[1],R[0]) ; 
	if (beta < 0) beta += 2.0*M_PI;
	posn[1][0] = -sin(beta) ; posn[1][1] = cos(beta) ; posn[1][2] = 0 ;
	posn[0][0] = cos(alpha)*cos(beta) ; posn[0][1] = cos(alpha)*sin(beta) ; posn[0][2] = -sin(alpha);
	double lightpos[3], viewpos[3] ;
	for (l = 0 ; l < 3 ; l++) { 
	  viewpos[l]  = dotp(view, posn[l]) ; 
	  lightpos[l] = dotp(light,posn[l]) ;
	}
	
	double thetain, phin, thetaout, phiout, a, b, phi ;
	if (fabs(viewpos[2])>1.0) viewpos[2] *= 0.9999 ;
	thetaout= acos(viewpos[2]) ;
	phiout = atan2(viewpos[1],viewpos[0]) ; 
	if (phiout<0) phiout +=2.0*M_PI;
	a = alpha/M_PI*maxtheta ;
	b = beta/(2.0*M_PI)*maxphi ;
	if (a > maxtheta - 1.0) a = maxtheta - 1.0 ;
	if (b > maxphi - 1.0) b = maxphi - 1.0 ;
	assert (a >= 0 && b >= 0) ;
	if (fabs(lightpos[2])>1.0) lightpos[2] *= 0.9999  ;
	thetain = acos(lightpos[2]) ;
	phin = atan2(lightpos[1],lightpos[0]) ; if (phin < 0) phin += 2.0*M_PI;
	phi = phiout - phin ; if (phi < 0) phi += 2.0*M_PI ;

        double ti = thetain/M_PI*maxtheta ;
	double phh = phi/(2.0*M_PI)*maxphi;

        double th, ph ;
	th = (thetaout/M_PI*maxtheta) ;
	ph = (phiout/(2.0*M_PI)*maxphi) ; 
	if (th > maxtheta - 1) th = maxtheta - 1.0001 ;
	if (ph > maxphi - 1) ph = maxphi - 1.0001 ;
	assert(th >= 0 && ph >= 0) ;

	double thetaview, phiview, tv, pv ;
	if (fabs(view[2]) > 1.0) view[2] *= .999 ;
	assert(fabs(view[2] <= 1.0)) ;
	thetaview = acos(view[2]) ;
	phiview = atan2(view[1],view[0]) ;
	if (phiview < 0) phiview += 2.0*M_PI ;
  
	tv = thetaview/M_PI*maxtheta ; if (tv > maxtheta - 1) tv = maxtheta - 1.0001 ;
	pv = phiview/(2.0*M_PI)*maxphi ; if (pv > maxphi - 1) pv = maxphi - 1.0001 ;
	assert(tv >= 0 && pv >= 0) ;


	double valtest1, valtest2, valtest3, valtest4 ;
	if (flag) {
	valtest1 = brdf[k][(int) ti][(int) th][(int) phh] ;
	valtest2 = 0 ;

	for (q = 0 ; q <= maxp ; q++)
	  for (l = q ; l <= mymaxl ; l++)
	    for (p = q ; p <= maxp ; p++) {
	      double mv = 1.0 ;
	      if (q == 0) mv = sqrt2 ;
	      valtest2 += mv*brdfcoeffs[k][l][p][q]*sqrtpi*
		          plmvalreal(l,q,(int)ti)*plmvalreal(p,q,(int)th)
                          *cosphi[(q*(int)phh)%maxphi] ;
	    }

	valtest3 = valtest4 = 0 ;
	int ii,jj,m ;
	for (ii = 0 ; ii < maxtheta ; ii++) 
	  for (jj = 0 ; jj < maxphi ; jj++) {
	    double ltpos[3] ;
	    ltpos[0] = sintheta[ii]*cosphi[jj] ;
	    ltpos[1] = sintheta[ii]*sinphi[jj] ;
	    ltpos[2] = costheta[ii] ;
	    double mf = (2.0*M_PI*M_PI*sintheta[ii])/(maxtheta*maxphi) ;
	    double tin = acos(dotp(ltpos,R)) ;
	    int tinval = (int) (tin/M_PI*maxtheta) ;
	    double lightval = 0 ;
	    for (l = 0 ; l <= mymaxl ; l++)
	      for (m = - l ; m <= l ; m++) 
		lightval += lightcoeffs[l][m+maxl][0]*Yreal(l,m,ii,jj) ;
	    
		valtest3 += mf*brdf[k][tinval][(int) th][(int) phh]*lightval ;
		valtest4 += mf*brdftest[k][tinval][(int) th][(int) phh]*lightval ;
	      
	  }
	}
	

	double val = 0 ;
	for (p = 0 ; p <= maxp ; p++) {
	  for (q = -p ; q <= p ; q++) {
	    double factor1 = Yreal(p,q,tv,pv) ;
	    double factor2 ;
	    double fraca, fracb ;
	    fraca = a - (int) a ; fracb = b - (int) b ;
	    factor2=(1-fraca)*(1-fracb)*refwarp[(int)a][(int)b][p][q+maxp][k] 
                   +fraca*(1-fracb)*refwarp[1+(int)a][(int)b][p][q+maxp][k] 
	           +(1-fraca)*fracb*refwarp[(int)a][1+(int)b][p][q+maxp][k] 
                   +fraca*fracb*refwarp[1+(int)a][1+(int)b][p][q+maxp][k] ;
	    val += mulfac*factor1*factor2 ;
	  }
	  unsigned char c ;
	  if (val > 255) c = (unsigned char)  255 ;
	  else if (val < 0) c = (unsigned char) 0 ;
	  else c = (unsigned char) val ;
	  fprintf(fp[p],"%c",c) ;
	}
      }
  for (p = 0 ; p <= maxp ; p++) fclose(fp[p]) ;
}


