/****************************************************************************

This code operates on each of the BRDFs, taking the user input and creating
an appropriate BRDF.  

Ravi Ramamoorthi: May 22, 2002

***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include <complex>
#include <string.h>
#include "legendre.h"
#include "globals.h"
#include "brdfs.h"
#include "vectors.h"

//-------------------------------------------------------------------------
// An N.H model as per equation 11 in the paper
//-------------------------------------------------------------------------

void sigma(int argc, char ** argv, double brdfnorm[3]) {
  if (argc != 2) {
    fprintf(stderr,"sigma sigmaval\n") ;
  }
  double sigma = atof(argv[1]) ;
  double sigmaden = 1.0/(4.0*M_PI*sigma*sigma) ;
  int i,j,k ;
  brdfnorm[0] = brdfnorm[1] = brdfnorm[2] = 0 ;
  for (i = 0 ; i < maxtheta ; i++)
    for (j = 0 ; j < maxtheta ; j++)
      for (k = 0 ; k < maxphi ; k++) {


	double thetain, thetaout, phi ;
	thetain = M_PI*i/maxtheta ;
	thetaout= M_PI*j/maxtheta ;
	phi = 2.0*M_PI*k/maxphi ;

	double L[3], N[3], R[3], V[3], H[3] ;
	R[0] = 0 ; R[1] = 0 ; R[2] = 1.0 ;
	
	V[0] = sin(thetaout) ; V[1] = 0 ; V[2] = cos(thetaout) ;
	
	N[0] = sin(thetaout/2) ; N[1] = 0 ; N[2] = cos(thetaout/2) ;
	
	L[0] = sin(thetain)*cos(phi) ;
	L[1] = sin(thetain)*sin(phi) ;
	L[2] = cos(thetain) ;

	double norm = 0 ;
	int t ;
	for (t = 0 ; t < 3 ; t++) { 
	  H[t] = L[t] + V[t] ;
	  norm = norm + H[t]*H[t] ;	  
	}
	norm = sqrt(norm) ;
	for (t = 0 ; t < 3 ; t++) {
	  H[t] /= norm ;
	}

	double NdotV, NdotH, NdotL ;
	NdotV = dotp(N,V) ;
	NdotH = dotp(N,H) ;
	NdotL = dotp(N,L) ;
	if (fabs(NdotH)>1.0) NdotH *= 0.999 ;
	double thetah = acos(NdotH) ;

	if (NdotL < 0)
	  brdf[0][i][j][k] = brdf[1][i][j][k] = brdf[2][i][j][k] = 0  ;
	else brdf[0][i][j][k] = brdf[1][i][j][k] = brdf[2][i][j][k] = 
	       sigmaden*exp(-thetah*thetah/sigma/sigma) ;

	assert(NdotV >= 0) ;
	valbrdf[0][i][j][k] = valbrdf[1][i][j][k] = valbrdf[2][i][j][k] = true ;
	
	double dtheta1, dtheta2, dphi ;
	dtheta1 = dtheta2 = M_PI/maxtheta ;
	dphi = 2.0*M_PI/maxphi ;
	
	double bplus = sin(thetain)*sin(thetaout)*dtheta1*dtheta2*dphi*brdf[0][i][j][k]*brdf[0][i][j][k]*valbrdf[0][i][j][k] ;
	brdfnorm[0] += bplus ; brdfnorm[1] += bplus ; brdfnorm[2] += bplus ;
      }
}

//------------------------------------------------------------------------
// The Lafortune model
//------------------------------------------------------------------------

void lafortune(int argc, char ** argv, double brdfnorm[3]) {
  int i,j,k,l,m,p,q ;
  FILE *fp1 ;

  const int maxlobes = 5 ;
  int numlobes ;

  typedef struct {
    double cx[3], cz[3], n[3] ;
  } LOBE ;

  LOBE lobes[maxlobes] ;

  double light[3] = {.866,0,0.5} ;

  if (argc != 2) {
    fprintf(stderr,"lafortune dir/file\n") ;
    exit(1) ;
  }
  
  assert(fp1 = fopen(argv[1],"rt")) ;
  assert(fscanf(fp1,"%d",&numlobes)) ;
  char s[100] ;

  for (i = 0 ; i < numlobes ; i++) {
    assert(fscanf(fp1,"%s",s)) ;
    assert(!strcmp(s,"lobe")) ;
    for (j = 0 ; j < 3 ; j++) 
      assert(fscanf(fp1,"%lf",&(lobes[i].cx[j]))) ;
    for (j = 0 ; j < 3 ; j++) 
      assert(fscanf(fp1,"%lf",&(lobes[i].cz[j]))) ;
    for (j = 0 ; j < 3 ; j++) 
      assert(fscanf(fp1,"%lf",&(lobes[i].n[j]))) ;    
  }

  brdfnorm[0] = brdfnorm[1] = brdfnorm[2] = 0 ;

  for (i = 0 ; i < maxtheta ; i++)
    for (j = 0 ; j < maxtheta ; j++)
      for (k = 0 ; k < maxphi ; k++) {

	double thetain, thetaout, phi ;
	thetain = M_PI*i/maxtheta ;
	thetaout= M_PI*j/maxtheta ;
	phi = 2.0*M_PI*k/maxphi ;

	double posn[3][3] ;
	double L[3], N[3], R[3], V[3], H[3] ;
	double Lnorm[3], Vnorm[3], Hnorm[3] ;

	R[0] = 0 ; R[1] = 0 ; R[2] = 1.0 ;
	
	V[0] = sin(thetaout) ; V[1] = 0 ; V[2] = cos(thetaout) ;
	
	N[0] = sin(thetaout/2) ; N[1] = 0 ; N[2] = cos(thetaout/2) ;

	posn[2][0] = N[0] ; posn[2][1] = 0 ; posn[2][2] = N[2] ;
	posn[1][0] = 0 ; posn[1][1] = 1.0 ; posn[1][2] = 0 ; 
	posn[0][0] = N[2] ; posn[0][1] = 0 ; posn[0][2] = -N[0] ;
	
	L[0] = sin(thetain)*cos(phi) ;
	L[1] = sin(thetain)*sin(phi) ;
	L[2] = cos(thetain) ;

	double norm = 0 ;
	int t ;
	for (t = 0 ; t < 3 ; t++) { 
	  H[t] = L[t] + V[t] ;
	  norm = norm + H[t]*H[t] ;	  
	}
	norm = sqrt(norm) ;
	for (t = 0 ; t < 3 ; t++) {
	  H[t] /= norm ;
	}

	for (l = 0 ; l < 3 ; l++) {
	  Lnorm[l] = dotp(L,posn[l]) ;
	  Vnorm[l] = dotp(V,posn[l]) ;
	  Hnorm[l] = dotp(H,posn[l]) ;
	}

	double NdotV, NdotH, NdotL ;
	NdotV = dotp(N,V) ;
	NdotH = dotp(N,H) ;
	NdotL = dotp(N,L) ;
	if (fabs(NdotH)>1.0) NdotH *= 0.999 ;
	double thetah = acos(NdotH) ;
	assert(NdotV >= 0) ;

	assert(fabs(NdotL-Lnorm[2])<.01) ;

	for (l = 0 ; l < 3 ; l++) {

	  if (NdotL < 0)
	    brdf[l][i][j][k] = 0  ;
	  else {
	    for (m = 0 ; m < numlobes ; m++) {
	      double expand =  lobes[m].cx[l]*Lnorm[0]*Vnorm[0] +
		               lobes[m].cx[l]*Lnorm[1]*Vnorm[1] +
	                       lobes[m].cz[l]*Lnorm[2]*Vnorm[2] ;
	      if (expand < 0) expand = 0 ;
	      brdf[l][i][j][k] += pow(expand,lobes[m].n[l])*NdotL ;
	    }
	  }
	  valbrdf[l][i][j][k] = true ;
	
	  double dtheta1, dtheta2, dphi ;
	  dtheta1 = dtheta2 = M_PI/maxtheta ;
	  dphi = 2.0*M_PI/maxphi ;
	  brdfnorm[l] += sin(thetain)*sin(thetaout)*dtheta1*dtheta2*dphi*brdf[l][i][j][k]*brdf[l][i][j][k]*valbrdf[l][i][j][k] ;
	}
      }  
}

// Project utility function for McCool reflection maps

void project(double vec[3], double &u, double &v, int width) {
  double pv, ar, x, y, z ;
  x = vec[0] ; y = vec[1] ; z = vec[2] ;
  pv = 7.0/8.0 ;
  ar = 1.0/(2.0*(1.0+z)) ;
  u  = pv*x*ar + 0.5 ;
  v  = pv*y*ar + 0.5 ;
  u *= width ;
  v = width*(1.0-v) ;
}

//-------------------------------------------------------------------------
// McCool factored model.  
//-------------------------------------------------------------------------

void mccool(int argc, char ** argv, double brdfnorm[3]) {
 
  FILE *fp1, *fp2, *fp3 ;
  int i,j,k,l,m,p,q ;

  const int width = 256 ;
  unsigned char krylonp[width][width][3] ;
  unsigned char krylonq[width][width][3] ;
  unsigned char krylonh[width][width][3] ;
  //double mult[3] = {11.4187,0.0555595,0.0841939} ;
  //double mult[3] = {0.043248,0.0725744,0.0527886} ;
  double mult[3] ; // {0.0304027,0.000378451,1.46881e-06} ;
  double multh ;
  double light[3] = {.866,0,0.5} ;
  //double light[3] = {0,0,1} ;
  if (argc != 6) {
    fprintf(stderr,"mccool dir/file multr multg multb multh\n") ;
    exit(1) ;
  }  

  char name[300] ;
  sprintf(name,"%sp.ppm",argv[1]) ;
  assert(fp1 = fopen(name,"rb")) ;
  sprintf(name,"%sq.ppm",argv[1]) ;
  assert(fp2 = fopen(name,"rb")) ;
  sprintf(name,"%sh.ppm",argv[1]) ;
  assert(fp3 = fopen(name,"rb")) ;
  fscanf(fp1,"%*s %*d %*d %*d%*c") ;
  fscanf(fp2,"%*s %*d %*d %*d%*c") ;
  fscanf(fp3,"%*s %*d %*d %*d%*c") ;

  brdfnorm[0] = brdfnorm[1] = brdfnorm[2] = 0 ;
  mult[0] = atof(argv[2]) ; mult[1] = atof(argv[3]) ; mult[2] = atof(argv[4]);
  multh = atof(argv[5]) ;

  for (i = 0 ; i < width ; i++)
    for (j = 0 ; j < width ; j++)
      for (k = 0 ; k < 3 ; k++) {
	fscanf(fp1,"%c",&(krylonp[i][j][k])) ;
	fscanf(fp2,"%c",&(krylonq[i][j][k])) ;
	fscanf(fp3,"%c",&(krylonh[i][j][k])) ;
      }

  for (i = 0 ; i < maxtheta ; i++)
    for (j = 0 ; j < maxtheta ; j++)
      for (k = 0 ; k < maxphi ; k++) {

	double thetain, thetaout, phi ;
	thetain = M_PI*i/maxtheta ;
	thetaout= M_PI*j/maxtheta ;
	phi = 2.0*M_PI*k/maxphi ;

	double posn[3][3] ;
	double L[3], N[3], R[3], V[3], H[3] ;
	double Lnorm[3], Vnorm[3], Hnorm[3] ;

	R[0] = 0 ; R[1] = 0 ; R[2] = 1.0 ;
	
	V[0] = sin(thetaout) ; V[1] = 0 ; V[2] = cos(thetaout) ;
	
	N[0] = sin(thetaout/2) ; N[1] = 0 ; N[2] = cos(thetaout/2) ;

	posn[2][0] = N[0] ; posn[2][1] = 0 ; posn[2][2] = N[2] ;
	posn[1][0] = 0 ; posn[1][1] = 1.0 ; posn[1][2] = 0 ; 
	posn[0][0] = N[2] ; posn[0][1] = 0 ; posn[0][2] = -N[0] ;
	
	L[0] = sin(thetain)*cos(phi) ;
	L[1] = sin(thetain)*sin(phi) ;
	L[2] = cos(thetain) ;

	double norm = 0 ;
	int t ;
	for (t = 0 ; t < 3 ; t++) { 
	  H[t] = L[t] + V[t] ;
	  norm = norm + H[t]*H[t] ;	  
	}
	norm = sqrt(norm) ;
	for (t = 0 ; t < 3 ; t++) {
	  H[t] /= norm ;
	}

	for (l = 0 ; l < 3 ; l++) {
	  Lnorm[l] = dotp(L,posn[l]) ;
	  Vnorm[l] = dotp(V,posn[l]) ;
	  Hnorm[l] = dotp(H,posn[l]) ;
	}

	double NdotV, NdotH, NdotL ;
	NdotV = dotp(N,V) ;
	NdotH = dotp(N,H) ;
	NdotL = dotp(N,L) ;
	if (fabs(NdotH)>1.0) NdotH *= 0.999 ;
	double thetah = acos(NdotH) ;
	assert(NdotV >= 0) ;

	for (l = 0 ; l < 3 ; l++) {

	  if (NdotL < 0)
	    brdf[l][i][j][k] = 0  ;
	  else {
	    double ul,vl,uh,vh,uv,vv ;
	    project(Lnorm,ul,vl,width) ;
	    project(Hnorm,uh,vh,width) ;
	    project(Vnorm,uv,vv,width) ;
	    double v,h,w,h2 ;
	    assert(vv>=0&&vv<width&&uv>=0&&uv<width) ;
	    assert(vl>=0&&vl<width&&ul>=0&&ul<width) ;
	    assert(vh>=0&&vh<width&&uh>=0&&uh<width) ;
	    
	    v = krylonp[(int) vv][(int) uv][l] ; 
	    w = krylonp[(int) vl][(int) ul][l] ;
	    h = krylonq[(int) vh][(int) uh][l] ;
	   h2 = krylonh[(int) vh][(int) uh][l] ;
	   brdf[l][i][j][k] = (mult[l]*v*w*h)*Lnorm[2]/2550.0 ;
	   brdf[l][i][j][k] += multh*h2*0.25 ;
	    assert(brdf[l][i][j][k] >= 0) ;
	  }
	  valbrdf[l][i][j][k] = true ;
	
	  double dtheta1, dtheta2, dphi ;
	  dtheta1 = dtheta2 = M_PI/maxtheta ;
	  dphi = 2.0*M_PI/maxphi ;
	  brdfnorm[l] += sin(thetain)*sin(thetaout)*dtheta1*dtheta2*dphi*brdf[l][i][j][k]*brdf[l][i][j][k]*valbrdf[l][i][j][k] ;
	}
      }
}

//-------------------------------------------------------------------------


void findangles(double posn[3], double &theta, double &phi) {
  theta = acos(posn[2]) ;
  phi = atan2(posn[1],posn[0]) ;
  if (phi < 0) phi += 2.0*M_PI ;
}

//-------------------------------------------------------------------------
// Curet BRDF using Zernike polynomial expansion
//-------------------------------------------------------------------------

void curet(int argc, char ** argv, double brdfnorm[3]) {
  const int zmaxl = 8 ;
  const int numsamps = 61 ;
  double zercoeffs[1+numsamps][maxl+1][maxl+1][maxl+1][3] ;
  float zvals[maxtheta][maxtheta][maxphi][55] ;
  int sampnum ;
  brdfnorm[0] = brdfnorm[1] = brdfnorm[2] = 0 ;
  int i,j,k,l,m,p,q,r,s ;
  FILE *fp1 ;

  if (argc != 3) {
    fprintf(stderr,"curetbrdf zercoeffs sampnum\n") ;
    exit(1) ;
  }

  sampnum = atoi(argv[2]) ;
  assert(fp1 = fopen(argv[1],"rt")) ;

  int col ;
  for (i = 1 ; i <= numsamps ; i++) {
    for (col = 0 ; col < 3 ; col++) {
      for (l = 0 ; l <= zmaxl ; l++)
	for (r = l ; r <= zmaxl ; r++)
	  for (s = 0 ; s <= zmaxl ; s++) {
	     if (s > r || s > l || ((l+s)%2 == 1) || ((r+s)%2 == 1)) continue ;
	     assert(fscanf(fp1,"%lf",&(zercoeffs[i][l][r][s][col]))) ;
	  }
    }
  }
  fclose(fp1) ;

  for (i = 0 ; i < maxtheta ; i++) {
    for (j = 0 ; j < maxtheta ; j++)
      for (k = 0 ; k < maxphi ; k++) {

	double thetain, thetaout, phi ;
	thetain = M_PI*i/maxtheta ;
	thetaout= M_PI*j/maxtheta ;
	phi = 2.0*M_PI*k/maxphi ;

	double posn[3][3] ;
	double L[3], N[3], R[3], V[3], H[3] ;
	double Lnorm[3], Vnorm[3], Hnorm[3] ;

	
	R[0] = 0 ; R[1] = 0 ; R[2] = 1.0 ;
	
	V[0] = sin(thetaout) ; V[1] = 0 ; V[2] = cos(thetaout) ;
	
	N[0] = sin(thetaout/2) ; N[1] = 0 ; N[2] = cos(thetaout/2) ;

	posn[2][0] = N[0] ; posn[2][1] = 0 ; posn[2][2] = N[2] ;
	posn[1][0] = 0 ; posn[1][1] = 1.0 ; posn[1][2] = 0 ; 
	posn[0][0] = N[2] ; posn[0][1] = 0 ; posn[0][2] = -N[0] ;
	
	L[0] = sin(thetain)*cos(phi) ;
	L[1] = sin(thetain)*sin(phi) ;
	L[2] = cos(thetain) ;

	double norm = 0 ;
	int t ;
	for (t = 0 ; t < 3 ; t++) { 
	  H[t] = L[t] + V[t] ;
	  norm = norm + H[t]*H[t] ;	  
	}
	norm = sqrt(norm) ;
	for (t = 0 ; t < 3 ; t++) {
	  H[t] /= norm ;
	}

	for (l = 0 ; l < 3 ; l++) {
	  Lnorm[l] = dotp(L,posn[l]) ;
	  Vnorm[l] = dotp(V,posn[l]) ;
	  Hnorm[l] = dotp(H,posn[l]) ;
	}

	double NdotV, NdotH, NdotL ;
	NdotV = dotp(N,V) ;
	NdotH = dotp(N,H) ;
	NdotL = dotp(N,L) ;
	if (fabs(NdotH)>1.0) NdotH *= 0.999 ;
	double thetah = acos(NdotH) ;
	assert(NdotV >= 0) ;
	if (NdotL < 0) 
	  valbrdf[0][i][j][k] = valbrdf[1][i][j][k] = valbrdf[2][i][j][k] = 0;
	else {
	  valbrdf[0][i][j][k] = valbrdf[1][i][j][k] = valbrdf[2][i][j][k] = 1 ;

	    double tinorm, tonorm, pinorm, ponorm, pnorm ;
	    tinorm = thetain ; tonorm = thetaout ; pnorm = phi ;
	    
	    findangles(Lnorm,tinorm, pinorm) ;
	    findangles(Vnorm,tonorm, ponorm) ;
	    pnorm = ponorm - pinorm ; if (pnorm < 0) pnorm += 2.0*M_PI ;
	    
	    double ci = cos(tinorm) ;
	    
	    double val = 0 ;
	    int ll ;
	    double ti,to,p ;
	    ti = tinorm/(M_PI/2.0)*maxtheta ;
	    if (ti > maxtheta - 1) ti = maxtheta - 1 ;
	    to = tonorm/(M_PI/2.0)*maxtheta ;
	    if (to > maxtheta - 1) to = maxtheta - 1 ;
	    p = pnorm/(2.0*M_PI)*maxphi ;
	    if (p > maxphi - 1) p = maxphi - 1 ;

	    int ctr = 0 ;
	    for (ll = 0 ; ll <= zmaxl ; ll++)
	      for (r = ll ; r <= zmaxl ; r++)
		for (s = 0 ; s <= zmaxl ; s++) {
		  if (s > r || s > ll || ((ll+s)%2 == 1) || ((r+s)%2 == 1)) continue ;
		  zvals[i][j][k][ctr] =  Zrealiso(ll,r,s,p,ti,to)*ci ;
		  if (ll != r) zvals[i][j][k][ctr] += Zrealiso(r,ll,s,p,ti,to)*ci ;
		  ++ctr ;
		}
	    assert(ctr == 55) ;
	}
      }
      fprintf(stderr,"%d ",i) ;      
  }
    for (l = 0 ; l < 3 ; l++) {
      brdfnorm[l] = 0 ;
      for (i = 0 ; i < maxtheta ; i++) {
      for (j = 0 ; j < maxtheta ; j++) {

	double thetain, thetaout, phi ;
	thetain = M_PI*i/maxtheta ;
	thetaout= M_PI*j/maxtheta ;
	double dtheta1, dtheta2, dphi ;
	dtheta1 = dtheta2 = M_PI/maxtheta ;
	dphi = 2.0*M_PI/maxphi ;
	double si, so ;
	si = sin(thetain) ; so = sin(thetaout) ;

	for (k = 0 ; k < maxphi ; k++) {
	  brdf[l][i][j][k] = 0 ;
	    if (valbrdf[l][i][j][k]) {
	      int ctr = 0 ;
	      for (int ll = 0 ; ll <= zmaxl ; ll++)
		for (r = ll ; r <= zmaxl ; r++)
		  for (s = 0 ; s <= zmaxl ; s++) {
		    if (s > r || s > ll || ((ll+s)%2 == 1) || ((r+s)%2 == 1)) continue ;
		    brdf[l][i][j][k] += 
		      zvals[i][j][k][ctr]*zercoeffs[sampnum][ll][r][s][l];
		    ++ctr ;
		  }
	      assert(ctr == 55) ;
	    }
	  brdfnorm[l] += si*so*dtheta1*dtheta2*dphi*brdf[l][i][j][k]*brdf[l][i][j][k] ;	    
	}
      }
      fprintf(stderr,"%d ",i) ;      
      }
    }
}

//--------------------------------------------------------------------------
// Kay-Kajiya model
//--------------------------------------------------------------------------

void kaykajiya(int argc, char ** argv, double brdfnorm[3]) {
  int i,j,k,l;

  if (argc !=2) {
    fprintf(stderr,"kaykajiya exponent\n") ;
  }
  double exponent = atof(argv[1]) ;
  brdfnorm[0] = brdfnorm[1] = brdfnorm[2] = 0 ;

  for (i = 0 ; i < maxtheta ; i++)
    for (j = 0 ; j < maxtheta ; j++)
      for (k = 0 ; k < maxphi ; k++) {

	double thetain, thetaout, phi ;
	thetain = M_PI*i/maxtheta ;
	thetaout= M_PI*j/maxtheta ;
	phi = 2.0*M_PI*k/maxphi ;
	double c = cos(thetain - thetaout) ;
	if (c < 0) c = 0 ;
	else 
	  c = pow(c,exponent) ;
	for (l = 0 ; l < 3 ; l++) {
	  brdf[l][i][j][k] = c ;
	  valbrdf[l][i][j][k] = true ;
	  double dtheta1, dtheta2, dphi ;
	  dtheta1 = dtheta2 = M_PI/maxtheta ;
	  dphi = 2.0*M_PI/maxphi ;
	  brdfnorm[l] += sin(thetain)*sin(thetaout)*dtheta1*dtheta2*dphi*brdf[l][i][j][k]*brdf[l][i][j][k]*valbrdf[l][i][j][k] ;
	}
      }
}
