function DEC = discreteExteriorCalculus(X,T)
% Sets up all necessary Discrete Exterior Calculus matrices for a mesh
%
% Naming convention:  DEC.operatorij is the "operator" part of DEC from
%   i-forms to j-forms; the suffix D will represent dual.  For example,
%   the operator ijD will take primal i-forms to dual j-forms.
% 
% Original code written by Miri Ben-Chen; some parts stripped to make the
% homework problem nontrivial!  Also adds a few convenience matrices for
% visualization

nv = size(X,1);
nf = size(T,1);

% I = edge vertex 1, J = edge vertex 2
I = [T(:,1);T(:,2);T(:,3)];
J = [T(:,2);T(:,3);T(:,1)];
S = [1:nf,1:nf,1:nf]';

% E goes from edge (i,j) to sum of face indices of two adjacent triangles
E = sparse(I,J,S',nv,nv); 

% DEC datastructures, to get from edges to vertices to triangles and so on
Elisto = [I,J]; % list of edges
sElist = sort(Elisto,2); % sort indices in increasing order

s = (rowNorms(Elisto - sElist) > 1e-12); % s = 1 when edge is flipped / 0 otherwise
t = S.*(-1).^s; % t = -face_index for flipped edge / +face_index otherwise

[Elist,une] = unique(sElist, 'rows'); % Get rid of repeated edges -- each appears 2x
e2t = zeros(length(Elist),4);
t2e = zeros(nf,3);

for m=1:length(Elist) % for each edge m
    i = Elist(m,1); j = Elist(m,2); % get endpoints of edge (i,j)
    t1 = t(une(m)); % triangle 1 comes from face index
    t2 = -(E(i,j) + E(j,i) - abs(t1))*sign(t1); % triangle 2 comes from array sum
    e2t(m,1:2) = [t1, t2]; % edge to triangle -- store two search results
    
    f = T(abs(t1),:); % f = vertices of the first triangle
    loc = find(f == (sum(f) - i - j)); % loc = which vertex (1 to 3) is not i or j
    t2e(abs(t1),loc) = m*sign(t1); % triangle-to-edge store with sign from orientation
    e2t(m,3) = loc; % store edge-to-triangle
    if t2 ~= 0 % not a boundary edge, do the same thing again
        f = T(abs(t2),:); loc = find(f == (sum(f) - i - j));
        t2e(abs(t2),loc) = m*sign(t2);
        e2t(m,4) = loc;
    end
end

% Sparse matrix -- (i,j)-->position in edge list
v2e = sparse(Elist(:,1),Elist(:,2),1:length(Elist),nv,nv);

% Number of edges
ne = length(Elist);

% Boundary operator from 1 forms to 0 forms
I = [Elist(:,1);Elist(:,2)]; % vertex indices
J = [1:ne,1:ne]'; % edge indices
S = [-ones(ne,1); ones(ne,1)];
DEC.boundary10 = sparse(I,J,S,nv,ne); % 1 forms to 0 forms

% Boundary operator from 2 forms to 1 forms
I = [1:ne, 1:ne]'; % edge indices
J = [abs(e2t(:,1)); abs(e2t(:,2))]; % triangle indices
S = [sign(e2t(:,1)); sign(e2t(:,2))];
locs = find(J ~= 0);
I = I(locs); J = J(locs); S = S(locs);
DEC.boundary21 = sparse(I,J,S,ne,nf);

% Differential -- transpose of boundary
DEC.d01 = DEC.boundary10';
DEC.d12 = DEC.boundary21';

% Hodge star
L1 = rowNorms(X(T(:,2),:)-X(T(:,3),:)); % Three edge lengths
L2 = rowNorms(X(T(:,1),:)-X(T(:,3),:));
L3 = rowNorms(X(T(:,1),:)-X(T(:,2),:));
A1 = (L2.^2 + L3.^2 - L1.^2) ./ (2.*L2.*L3); % Invert Law of Cosines
A2 = (L1.^2 + L3.^2 - L2.^2) ./ (2.*L1.*L3);
A3 = (L1.^2 + L2.^2 - L3.^2) ./ (2.*L1.*L2);
interiorAngles = [A1,A2,A3];
interiorAngles = acos(interiorAngles);
DEC.interiorAngles = interiorAngles;
triangleEdgeLengths = [L1, L2, L3];
DEC.triangleEdgeLengths = triangleEdgeLengths;

% *0 - dual voronoi areas
ar = (1/8)*cot(interiorAngles).*triangleEdgeLengths.^2;
I = [T(:,1);T(:,2);T(:,3)];
J = [T(:,2);T(:,3);T(:,1)];
S = [ar(:,3);ar(:,1);ar(:,2)];
In = [I;J];
Jn = In;
Sn = [S;S];
DEC.star02D = sparse(In,Jn,Sn,nv,nv);
star02Ddiag = full(diag(DEC.star02D));
DEC.star2D0 = spdiags(1./star02Ddiag,0,nv,nv);

% *1 - cot weights
I = [T(:,1);T(:,2);T(:,3)];
J = [T(:,2);T(:,3);T(:,1)];
S = 0.5*cot([interiorAngles(:,3);interiorAngles(:,1);interiorAngles(:,2)]);
In = [I;J];
Jn = [J;I];
Sn = [S;S];
W = sparse(In,Jn,Sn,nv,nv);
h1d = zeros(length(Elist),1);
for i=1:length(h1d)
    h1d(i) = W(Elist(i,1),Elist(i,2));
end
DEC.star11D = spdiags(h1d,0,ne,ne); % Check that thse are in right order
DEC.star1D1 = spdiags(1./h1d,0,ne,ne);

% *2 - 1/triangle areas
Ar = rowNorms(cross(X(T(:,1),:) - X(T(:,2),:), X(T(:,1),:) - X(T(:,3),:)))/2;
DEC.star20D = spdiags(1./Ar, 0, nf, nf);

DEC.t2e = t2e;
    