#ifdef WIN32
#include <windows.h>
#endif

#include <GL/gl.h>
#include <GL/glut.h>
#include <stdlib.h>
#include <stdio.h>

void Init (void);
void Display (void);
void Reshape (int, int);
void Key (unsigned char, int, int);
float fill_rate_bench (void);
float tri_rate_bench (float edgelength, int ntri);
void check_gl_errors (void);

/* The window size */
#define WIN_WIDTH 800
#define WIN_HEIGHT 800

int
main(int argc, char **argv)
{
  glutInit(&argc, argv);
  glutInitWindowSize(WIN_WIDTH, WIN_HEIGHT);
  glutInitDisplayMode(GLUT_RGB | GLUT_DOUBLE | GLUT_DEPTH);
  glutCreateWindow("CS448a: GfxBench");
  Init();
  glutDisplayFunc(Display);
  glutReshapeFunc(Reshape);
  glutKeyboardFunc(Key);
  glutShowWindow();
  glutMainLoop();
  return 0;
}

void Display (void) {
  float fill_rate;
  float tri_rate;

  /* Display the gfx card information */
  printf( "--------------------------------------------------\n");
  printf ("Vendor:      %s\n", glGetString(GL_VENDOR));
  printf ("Renderer:    %s\n", glGetString(GL_RENDERER));
  printf ("Version:     %s\n", glGetString(GL_VERSION));
  printf( "Visual:      RGBA=<%d,%d,%d,%d>  Z=<%d>  double=%d\n",
          glutGet( GLUT_WINDOW_RED_SIZE ),
          glutGet( GLUT_WINDOW_GREEN_SIZE ),
          glutGet( GLUT_WINDOW_BLUE_SIZE ),
          glutGet( GLUT_WINDOW_ALPHA_SIZE ),
          glutGet( GLUT_WINDOW_DEPTH_SIZE ),
          glutGet( GLUT_WINDOW_DOUBLEBUFFER ) );
  printf( "Geometry:    %dx%d+%d+%d\n",
          glutGet( GLUT_WINDOW_WIDTH ),
          glutGet( GLUT_WINDOW_HEIGHT ),
          glutGet( GLUT_WINDOW_X ),
          glutGet( GLUT_WINDOW_Y ) );
  printf( "Screen:      %dx%d\n",
          glutGet( GLUT_SCREEN_WIDTH ),
          glutGet( GLUT_SCREEN_HEIGHT ) );
  printf( "--------------------------------------------------\n");

  /* Perform the fill rate test.
  ** Returns MPix/second. 
  */
  fill_rate = fill_rate_bench();
  printf ("Fill Rate: \t%3.2f MPix/second\n", fill_rate);

  /* Triangle rate test.
  ** First argument is the edgelength of the 
  ** triangle in pixels.
  ** Second argument is number of triangles on 
  ** the edge of a triangle mesh to be rendered.
  ** Returns Mtris/second.
  **
  ** Here we do 1 pixel triangles in a 50x50 mesh 
  */
  tri_rate = tri_rate_bench(1, 50);
  printf ("Triangle Rate: \t%3.2f Mtri/second\n", tri_rate);
  
  exit(0);
}


float fill_rate_bench (void) {
  int start;
  int end;
  int i;
  int tricount;
  int milliseconds;
  int screensize;

  /* Load Identity Matrix */
  glMatrixMode( GL_MODELVIEW );
  glPushMatrix();
  glLoadIdentity( );
  
  glMatrixMode( GL_PROJECTION );
  glPushMatrix();
  glLoadIdentity( );

  /* Make sure we are drawing to the front buffer */
  glDrawBuffer(GL_FRONT);

  /* Clear the screen */
  glClear(GL_COLOR_BUFFER_BIT);

  /* Make sure that there are no GL commands
  ** in the pipeline.
  */
  glFinish();

  /* Reset the counter to zero */
  tricount = 0;

  /* Get the elapsed time in milliseconds */
  start = glutGet(GLUT_ELAPSED_TIME);
  do {
    /* Draw a full screen polygon 1000 times.
    **
    */
    for (i=0; i<1000; i++) {
      glBegin(GL_POLYGON);
      glColor3f (1.0f, 0.0f, 0.0f);
      glVertex2f (-1.0f, -1.0f);
      glColor3f (0.0f, 1.0f, 0.0f);
      glVertex2f (-1.0f, 1.0f);
      glColor3f (0.0f, 0.0f, 1.0f);
      glVertex2f (1.0f, 1.0f);
      glColor3f (1.0f, 0.0f, 1.0f);
      glVertex2f (1.0f, -1.0f);
      glEnd();
      tricount++;
    }
    /* Get the timer and quit if we've 
    ** rendered for a second.
    */
    end = glutGet(GLUT_ELAPSED_TIME);
  } while (end - start < 1000);

  /* IMPORTANT!  Make sure all graphics commands
  ** have completed using glFinish.  Triangles
  ** may still be in the pipeline.
  */
  glFinish();
  
  /* Get the time (in milliseconds) after the finish */
  end = glutGet(GLUT_ELAPSED_TIME);
  milliseconds = end - start;
  screensize = WIN_WIDTH*WIN_HEIGHT;

  /* Restore the matrix stack */
  glMatrixMode( GL_MODELVIEW );
  glPopMatrix();
  
  glMatrixMode( GL_PROJECTION );
  glPopMatrix();

  /* Before printing the results, make sure we didn't have
  ** any GL related errors. */
  check_gl_errors();

  /* Return Mpix per second */
  return ((float) tricount) * screensize / milliseconds / 1000000.0f * 1000.0f;
}


float tri_rate_bench (float edgelength, int ntri) {
  int i, j;
  int start, end, time;
  int tricount;
  float step, x, y;
  GLfloat *vtx;
  GLfloat *v;
  GLfloat *color;
  GLfloat *c;
  GLuint *index;
  GLuint *ind;
  int numvtx;
 
  /* We allocate a mesh of with (ntri x ntri) triangles.
  ** This means we need ntri+1 vertices on an edge
  */
  numvtx = (ntri+1)*(ntri+1);
  
  /* Each vertex is:
  ** x, y: position
  ** r, g, b: color
  **
  ** We also compute a connection list (integer offsets)
  ** for doing a triangle strip.
  */
  
  vtx =   (GLfloat *) malloc (numvtx*2*sizeof(GLfloat));
  color = (GLfloat *) malloc (numvtx*3*sizeof(GLfloat));
  index = (GLuint *)  malloc (numvtx*2*sizeof(GLuint));
 
  v = vtx;
  c = color;
  ind = index;

  /* Compute the step size for the given screen size
  ** Note that this assumes a square screen
  */
  step = edgelength / WIN_WIDTH * 2.0f;
  
  /* Create the mesh points and the triangle 
  ** strip list 
  */
  for (j=0; j <= ntri; j++) {

    y = -1.0f + step * j;
    for (i=0; i <= ntri; i++) {
   
      x = -1.0f + step * i;

      *v++ = x;
      *v++ = y;

      /* A random color */
      *c++ = ((float)rand()) / RAND_MAX;
      *c++ = ((float)rand()) / RAND_MAX;
      *c++ = ((float)rand()) / RAND_MAX;
  
      /* Index into vtx array */
      *ind++ = (j)   * (ntri+1) + i;
      *ind++ = (j+1) * (ntri+1) + i;
    }
  }

  /* Clear the screen */
  glClear(GL_COLOR_BUFFER_BIT);

  /* Load identities */
  glMatrixMode( GL_PROJECTION );
  glPushMatrix();
  glLoadIdentity( );

  glMatrixMode( GL_MODELVIEW );
  glPushMatrix();
  glLoadIdentity( );

  /* Set up the pointers */
  glVertexPointer(2, GL_FLOAT, 0, vtx);
  glEnableClientState(GL_VERTEX_ARRAY);
  glColorPointer(3, GL_FLOAT, 0, color);
  glEnableClientState(GL_COLOR_ARRAY);

  /* Clear the GL pipe */
  glFinish();

  /* Check for any errors */
  check_gl_errors();

  /* Reset the counter to zero */
  tricount = 0;
  
  /* Get the elapsed time in milliseconds */
  start = glutGet(GLUT_ELAPSED_TIME);
  do {

    /* Render the mesh 2000 times */
    for (i=0; i<2000; i++) {
 
      ind = index;
      
      /* Rotate the mesh to ensure 
      ** an even rasterization.
      */
      glLoadIdentity( );
      glTranslatef( 0.0f, -1.0f, 0.0f);
      glRotatef( i*0.0225f, 0.0f, 0.0f, 1.0f);
      glTranslatef( 1.0f, 1.0f, 0.0f);

      /* Draw a strip */
      for (j = 0; j < ntri; j++) {
      
        glDrawElements(GL_TRIANGLE_STRIP, (ntri+1)*2, GL_UNSIGNED_INT, ind);
        ind+=(ntri+1)*2;
      }

      /* Keep track of how many triangles */
      tricount +=  ntri*ntri*2;
    }
    /* End loop is a second has passed. */
    end = glutGet(GLUT_ELAPSED_TIME);
  } while (end - start < 1000);

  /* IMPORTANT: Flush the GL pipe before 
  ** getting the time!
  */
  glFinish();
  end = glutGet(GLUT_ELAPSED_TIME);

  /* Restore the gl stack */
  glMatrixMode( GL_MODELVIEW );
  glPopMatrix();
  
  glMatrixMode( GL_PROJECTION );
  glPopMatrix();

  /* Free the arrays */
  free(vtx);
  free(color);
  free(index);

  /* Before printing the results, make sure we didn't have
  ** any GL related errors. */
  check_gl_errors();

  /* Return the Mtri/seconds */
  time = end - start;
  return ((float) tricount) / time * 1000.0f / 1000000.0f;
}


/* A simple routine which checks for GL errors. */
void check_gl_errors (void) {
  GLenum err;
  err = glGetError();
  if (err != GL_NO_ERROR) {
    fprintf (stderr, "GL Error: ");
    switch (err) {
    case GL_INVALID_ENUM:
      fprintf (stderr, "Invalid Enum\n");
      break;      
    case GL_INVALID_VALUE:
      fprintf (stderr, "Invalid Value\n");
      break;
    case GL_INVALID_OPERATION:
      fprintf (stderr, "Invalid Operation\n");
      break;
    case GL_STACK_OVERFLOW:
      fprintf (stderr, "Stack Overflow\n");
      break;
    case GL_STACK_UNDERFLOW:
      fprintf (stderr, "Stack Underflow\n");
      break;
    case GL_OUT_OF_MEMORY:
      fprintf (stderr, "Out of Memory\n");
      break;
    default:
      fprintf (stderr, "Unknown\n");
    }
    exit(1);
  }
}

void Init (void) {
  /* Init does nothing */
}

void Reshape (int x, int y) {
  /* Reshape does nothing */
  (void) x;
  (void) y;
}


void Key(unsigned char a, int x, int y) {
  /* Key does nothing */
  (void) x;
  (void) y;
  (void) a;
}

