/* CS348C PROGRAMMING ASSIGNMENT 1: USER INTERFACE MODULE.

Copyright (c) 1995 The Board of Trustees of The Leland Stanford Junior
University. All rights reserved.

Permission to use, copy, modify and distribute this software for any
purpose is hereby granted without fee, provided that the above
copyright notice and this permission notice appear in all copies of
this software and that you do not sell the software.  Commercial
licensing is available by contacting the author.

THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT WARRANTY OF ANY KIND,
EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.

Author:
   Apostolos Lerios
   Computer Science Department
   Stanford University
   U.S.A.
   http://graphics.stanford.edu/~tolis/ */


#include "polly.h"
#include "xsupport.h"
#include <stream.h>
#include <Xm/Frame.h>
#include <Xm/Form.h>
#include <Xm/RowColumn.h>
#include <Inventor/SoDB.h>
#include <Inventor/nodes/SoSelection.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoSphere.h>
#include <Inventor/nodes/SoCoordinate3.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/sensors/SoNodeSensor.h>
#include <Inventor/sensors/SoTimerSensor.h>
#include <Inventor/manips/SoHandleBoxManip.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>


/* TYPE. */

typedef SoSeparator *SoSeparatorP;


/* FORWARD DECLARATION. */

static void Stop(void);


/* GLOBALS. */

struct Global {
  SoXtExaminerViewer *Viewer;  // 3D scene viewer.
  SoHandleBoxManip *HandleBox; // Control point marker manipulator.
  SoPath *ActiveTransform;     // Path to selected transform node.

  int Bezier;             // Flag: are the control points, Bezier points?
  int PointCount;         // Number of control points.
  Point *Position;        // Control point positions.
  SoSeparatorP *Marker;   // Control point markers.
  SoCoordinate3 *Polygon; // Control polygon.

  int SegmentCount;          // Segment count in linear approximation of curve.
  SoCoordinate3 *Coordinate; // Coordinates of these segments' endpoints.

  double FrameStep;      // Parameter value change between animation frames.
  double LastFrame;      // Parameter value at next animation frame.
  SoTimerSensor *Timer;  // Timer for frame synchronization.
  SoSeparator *Particle; // Animation marker.

  int MainLoopStarted; // Flag: has the application displayed its windows?

  Global(void)
  : ActiveTransform(0), Bezier(0), PointCount(4), SegmentCount(100),
  FrameStep(0.01), Timer(0), MainLoopStarted(0) {}
};

Global G;


/* ELEMENTARY CALLBACKS. */

/* Terminates program execution. */

static void Quit(void) {

  Cleanup();
  exit(0);
}

/* Prints the curve parameters. */

static void Print(void) {

  cerr << "Curve type: " << (G.Bezier?"Bezier":"Interpolating") << endl;
  cerr << "Control point positions:" << endl;
  for (int i=0;i<G.PointCount;i++) {
    cerr << " " << i << ": ";
    cerr << G.Position[i][0] << " " << G.Position[i][1] << " " <<
      G.Position[i][2] << endl;
  }
  cerr << endl;
}


/* SELECTION CALLBACKS. */

/* Attaches the manipulator to the selected control point marker. Path
   leads to the selected node. */

static void SelectionCB(void *,
			SoPath *Path) {

  G.ActiveTransform=Path->copy(0,Path->getLength()-1);
  G.ActiveTransform->push(0);
  G.ActiveTransform->ref();

  G.HandleBox=new(SoHandleBoxManip);
  G.HandleBox->replaceNode(G.ActiveTransform);
}

/* Detaches the manipulator from the selected control point marker. */

static void DeselectionCB(void *,
			  SoPath *) {

  if (G.ActiveTransform) {
    G.HandleBox->replaceManip(G.ActiveTransform,new(SoTransform));

    G.ActiveTransform->unref();
    G.ActiveTransform=0;
  }
}


/* ANIMATION CALLBACKS. */

/* Moves the animation marker to its next position along the curve. */

static void Move(void) {

  Point P;  
  GetCurvePoint(G.Bezier,G.LastFrame,P);  
  ((SoTransform *)(G.Particle->getChild(0)))->
    translation.setValue(P[0],P[1],P[2]);

  G.LastFrame+=G.FrameStep;
  if (G.LastFrame>1.0)
    Stop();
}

/* Animates the path of a particle traveling along the curve. */

static void Animate(void) {

  if (G.Timer)
    return;

  SoSphere *Marker=new(SoSphere);
  Marker->radius=0.5;
  G.Particle->addChild(Marker);

  G.Timer=new(SoTimerSensor)((SoSensorCB *)Move,0);
  G.Timer->schedule();
  G.LastFrame=0.0;
  Move();
}

/* Terminates an on-going animation. */

static void Stop(void) {

  if (!G.Timer)
    return;
  delete(G.Timer);
  G.Timer=0;

  G.Particle->removeChild(2);
}


/* CURVE DRAWING CALLBACKS. */

/* Draws the curve. */

static void DrawCurve(void) {

  // Disable screen updates.

  if (G.MainLoopStarted)
    G.Viewer->setAutoRedraw(FALSE);

  // Find points of curve's linear approximation.

  for (int i=0;i<=G.SegmentCount;i++) {
    Point P;
    GetCurvePoint(G.Bezier,double(i)/G.SegmentCount,P);
    G.Coordinate->point.set1Value(i,P[0],P[1],P[2]);
  }
  G.Coordinate->point.setNum(G.SegmentCount+1);

  // Update screen.

  if (G.MainLoopStarted) {
    G.Viewer->setAutoRedraw(TRUE);
    G.Viewer->render();
  }
}

/* Moves the i+1 st control point. */

static void ExtractPosition(int i) {

  const SbVec3f &Position=
    ((SoTransform *)(G.Marker[i]->getChild(0)))->translation.getValue();
  G.Polygon->point.set1Value(i,Position);
  for (int j=0;j<3;j++)
    G.Position[i][j]=Position[j];
  DrawCurve();
}


/* CHOICE BOX CALLBACKS. */

/* Sets the control point type to interpolated points. Set indicates
   whether the button was turned on or off. */

static void SetInterpolatingType(int Set) {

  if (Set) {
    G.Bezier=0;
    DrawCurve();
  }
}

/* Sets the control point type to Bezier points. Set indicates whether
   the button was turned on or off. */

static void SetBezierType(int Set) {

  if (Set) {
    G.Bezier=1;
    DrawCurve();
  }
}


/* SLIDER CALLBACKS. */

/* Sets the segment count to Value. */

static void NewSegmentCount(float Value,
			    void *) {

  G.SegmentCount=int(Value);
  DrawCurve();
}

/* Sets the frame step to Value. */

static void NewFrameStep(float Value,
                         void *) {

  G.FrameStep=Value;
}


/* MAIN PROGRAM. */

int main(int argc,
	 char *argv[]) {

  // Copyright.

  cerr << endl;
  cerr << "Parametric polynomial curve visualization." << endl;
  cerr << endl;
  cerr << "            Apostolos Lerios" << endl;
  cerr << "           Stanford University" << endl;
  cerr << "   http://graphics.stanford.edu/~tolis/" << endl;
  cerr << endl;

  // Parse command-line for simple switches.

  for (int i=1;i<argc;i++) {
    if (!strcmp(argv[i],"-help")) {
      cerr << "Usage: " << endl;
      cerr << endl;
      cerr << argv[0] << endl;
      cerr << endl;
      cerr << "-help:   shows this help screen." << endl;
      cerr << "-count:  number of control points." << endl;
      cerr << "-points: initial control point positions." << endl;
      cerr << endl;
      return 1;
    } else if (!strcmp(argv[i],"-count")) {
      if (++i==argc) {
        cerr << "-count should precede number of control points." << endl;
        return 1;
      }
      G.PointCount=atoi(argv[i]);
      if (G.PointCount<2) {
        cerr << "At least two control points must be specified." << endl;
        return 1;
      }
    }
  }

  // Parse command-line for point specification.

  int PointIndex=0;
  for (i=1;i<argc;i++) {
    if (!strcmp(argv[i],"-points")) {
      if (i+3*G.PointCount>=argc) {
        cerr << "-points should precede initial control point positions." <<
	  endl;
        return 1;
      }
      PointIndex=i+1;
      i+=3*G.PointCount;
    }
  }

  // Initialize Inventor and Xt.

  Widget ApplicationWindow=SoXt::init(argv[0]);
  if (!ApplicationWindow) {
    cerr << "Could not create application window." << endl;
    return 1;
  }

  // Create main window.

  Widget Main=XtVaCreateWidget("",xmFormWidgetClass,ApplicationWindow,
                               NULL);
  Widget Controls=XtVaCreateWidget("",xmRowColumnWidgetClass,Main,
				   XmNtopAttachment,XmATTACH_FORM,
				   XmNleftAttachment,XmATTACH_FORM,
				   XmNrightAttachment,XmATTACH_FORM,
				   NULL);

  // Buttons.

  Widget Row=XtVaCreateWidget("",xmRowColumnWidgetClass,Controls,
                              XmNorientation,XmHORIZONTAL,
                              NULL);
  new(PushButton)(Row,"Quit",PushButtonCallback(Quit));
  new(PushButton)(Row,"Print",PushButtonCallback(Print));
  new(PushButton)(Row,"Animate",PushButtonCallback(Animate));
  new(PushButton)(Row,"Stop",PushButtonCallback(Stop));
  XtManageChild(Row);

  // Choice box.

  ChoiceButtonSet *Buttons=new(ChoiceButtonSet)(Controls,"Curve Type:",1);
  Buttons->AddButton("Interpolating",1,SetInterpolatingType);
  Buttons->AddButton("Bezier",0,SetBezierType);  

  // Sliders.
  
  Row=XtVaCreateWidget("",xmRowColumnWidgetClass,Controls,
		       XmNorientation,XmHORIZONTAL,
		       NULL);
  new(Slider)(Row,"Segments",10,1000,G.SegmentCount,0,
	      SliderCallback(NewSegmentCount));
  new(Slider)(Row,"Frame Step",1,50,int(G.FrameStep*1000.0),3,
              SliderCallback(NewFrameStep));
  XtManageChild(Row);

  XtManageChild(Controls);

  // Scene graph (control points).

  SoSeparator *Scene=new(SoSeparator);

  SoSelection *Selection=new(SoSelection);
  Scene->addChild(Selection);
  Selection->addSelectionCallback((SoSelectionPathCB *)SelectionCB,0);
  Selection->addDeselectionCallback((SoSelectionPathCB *)DeselectionCB,0);

  G.Marker=new(SoSeparatorP[G.PointCount]);
  G.Position=new(Point[G.PointCount]);
  for (i=0;i<G.PointCount;i++) {
    G.Marker[i]=new(SoSeparator);

    // Marker movement callback.

    new(SoNodeSensor)((SoSensorCB *)ExtractPosition,(void *)i)->
      attach(G.Marker[i]);

    // Position of control point.

    SoTransform *Transform=new(SoTransform);
    if (PointIndex) {
      int Offset=PointIndex+3*i;
      for (int j=0;j<3;j++)
	G.Position[i][j]=atof(argv[Offset+j]);
    } else {
      G.Position[i][0]=i*3.0;
      G.Position[i][1]=0.0;
      G.Position[i][2]=0.0;
    }
    Transform->translation.setValue(G.Position[i][0],
				    G.Position[i][1],
				    G.Position[i][2]);
    G.Marker[i]->addChild(Transform);

    // Color of control point marker.

    SoMaterial *Material=new(SoMaterial);
    Material->diffuseColor.setHSVValue(0.66+0.33*i/(G.PointCount-1.0),1.0,1.0);
    Material->transparency.setValue(0.1);
    G.Marker[i]->addChild(Material);

    // Shape of marker.

    G.Marker[i]->addChild(new(SoSphere));

    Selection->addChild(G.Marker[i]);
  }

  Scene->addChild(Selection);

  // Scene graph (control polygon).

  SoSeparator *Polygon=new(SoSeparator);

  SoMaterial *Material=new(SoMaterial);
  Material->diffuseColor.setValue(1.0,1.0,0.0);
  Polygon->addChild(Material);

  SoLightModel *LightModel=new(SoLightModel);
  LightModel->model=SoLightModel::BASE_COLOR;
  Polygon->addChild(LightModel);

  G.Polygon=new(SoCoordinate3);
  for (i=0;i<G.PointCount;i++)
    G.Polygon->point.set1Value(i,G.Position[i][0],G.Position[i][1],
			       G.Position[i][2]);
  Polygon->addChild(G.Polygon);

  Polygon->addChild(new(SoLineSet));

  Scene->addChild(Polygon);

  // Scene graph (curve visualization).

  SoSeparator *Curve=new(SoSeparator);
  
  LightModel=new(SoLightModel);
  LightModel->model=SoLightModel::BASE_COLOR;
  Curve->addChild(LightModel);

  G.Coordinate=new(SoCoordinate3);
  Curve->addChild(G.Coordinate);

  Curve->addChild(new(SoLineSet));

  Scene->addChild(Curve);

  // Scene graph (animation).

  G.Particle=new(SoSeparator);
  
  G.Particle->addChild(new(SoTransform));

  Material=new(SoMaterial);
  Material->diffuseColor.setValue(0.0,1.0,0.0);
  G.Particle->addChild(Material);

  Scene->addChild(G.Particle);

  // Initialize curve drawing module.

  InitSystem(G.PointCount,G.Position);
  DrawCurve();

  // Viewer.

  Widget Frame=XtVaCreateWidget("",xmFrameWidgetClass,Main,
				XmNshadowType,XmSHADOW_ETCHED_IN,
				XmNtopAttachment,XmATTACH_WIDGET,
				XmNtopWidget,Controls,
				XmNleftAttachment,XmATTACH_FORM,
				XmNrightAttachment,XmATTACH_FORM,
				XmNbottomAttachment,XmATTACH_FORM,
				NULL);
  G.Viewer=new(SoXtExaminerViewer)(Frame);
  G.Viewer->setSceneGraph(Scene);
  G.Viewer->setSize(SbVec2s(600,600));
  G.Viewer->show();

  // Pop up window and yield control to Inventor.

  XtManageChild(Frame);
  XtManageChild(Main);
  SoXt::show(ApplicationWindow);
  G.MainLoopStarted=1;
  SoXt::mainLoop();
  return 0;
}
